/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Mirco Müller <macslow@bangang.de>
 */

#ifndef __PGM_TEXT_H__
#define __PGM_TEXT_H__

#include "pgmdrawable.h"

G_BEGIN_DECLS

#define PGM_TYPE_TEXT (pgm_text_get_type ())
#define PGM_TEXT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), PGM_TYPE_TEXT, PgmText))
#define PGM_TEXT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), PGM_TYPE_TEXT, PgmTextClass))
#define PGM_IS_TEXT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), PGM_TYPE_TEXT))
#define PGM_IS_TEXT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), PGM_TYPE_TEXT))
#define PGM_TEXT_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS((obj), PGM_TYPE_TEXT, PgmTextClass))

typedef struct _PgmText      PgmText;
typedef struct _PgmTextClass PgmTextClass;

/**
 * PgmTextEllipsize:
 * @PGM_TEXT_ELLIPSIZE_NONE: No ellipsization.
 * @PGM_TEXT_ELLIPSIZE_START: Omit characters at the start of the text.
 * @PGM_TEXT_ELLIPSIZE_MIDDLE: Omit characters in the middle of the text.
 * @PGM_TEXT_ELLIPSIZE_END: Omit characters at the end of the text.
 *
 * The different text ellipsizings.
 */
typedef enum {
  PGM_TEXT_ELLIPSIZE_NONE,
  PGM_TEXT_ELLIPSIZE_START,
  PGM_TEXT_ELLIPSIZE_MIDDLE,
  PGM_TEXT_ELLIPSIZE_END
} PgmTextEllipsize;

/**
 * PgmTextAlignment:
 * @PGM_TEXT_ALIGN_LEFT: Put all available space on the right.
 * @PGM_TEXT_ALIGN_CENTER: Center the line within the available space.
 * @PGM_TEXT_ALIGN_RIGHT: Put all available space on the left.
 *
 * The different text alignments.
 */
typedef enum {
  PGM_TEXT_ALIGN_LEFT,
  PGM_TEXT_ALIGN_CENTER,
  PGM_TEXT_ALIGN_RIGHT
} PgmTextAlignment;

/**
 * PgmTextWrap:
 * @PGM_TEXT_WRAP_WORD: Wrap lines at word boundaries.
 * @PGM_TEXT_WRAP_CHAR: Wrap lines at character boundaries.
 * @PGM_TEXT_WRAP_WORD_CHAR: Wrap lines at word boundaries, but fall back to
 * character boundaries if there is not enough space for a full word.
 *
 * The different text wrappings.
 */
typedef enum {
  PGM_TEXT_WRAP_WORD,
  PGM_TEXT_WRAP_CHAR,
  PGM_TEXT_WRAP_WORD_CHAR
} PgmTextWrap;

/**
 * PgmTextGravity:
 * @PGM_TEXT_GRAVITY_SOUTH: Glyphs stand upright.
 * @PGM_TEXT_GRAVITY_EAST: Glyphs are rotated 90 degrees clockwise.
 * @PGM_TEXT_GRAVITY_NORTH: Glyphs are upside-down.
 * @PGM_TEXT_GRAVITY_WEST: Glyphs are rotated 90 degrees counter-clockwise.
 * @PGM_TEXT_GRAVITY_AUTO: Gravity is resolved automatically.
 *
 * The different text gravities.
 */
typedef enum {
  PGM_TEXT_GRAVITY_SOUTH,
  PGM_TEXT_GRAVITY_EAST,
  PGM_TEXT_GRAVITY_NORTH,
  PGM_TEXT_GRAVITY_WEST,
  PGM_TEXT_GRAVITY_AUTO
} PgmTextGravity;

/**
 * PgmTextStretch:
 * @PGM_TEXT_STRETCH_CONDENSED: Condensed width.
 * @PGM_TEXT_STRETCH_NORMAL: Normal width.
 * @PGM_TEXT_STRETCH_EXPANDED: Expanded width.
 *
 * The different text stretchings.
 */
typedef enum {
  PGM_TEXT_STRETCH_CONDENSED,
  PGM_TEXT_STRETCH_NORMAL,
  PGM_TEXT_STRETCH_EXPANDED
} PgmTextStretch;

/**
 * PgmTextStyle:
 * @PGM_TEXT_STYLE_NORMAL: The font is upright.
 * @PGM_TEXT_STYLE_OBLIQUE: The font is slanted, but in a roman style.
 * @PGM_TEXT_STYLE_ITALIC: The font is slanted in an italic style.
 *
 * The different text styles.
 */
typedef enum {
  PGM_TEXT_STYLE_NORMAL,
  PGM_TEXT_STYLE_OBLIQUE,
  PGM_TEXT_STYLE_ITALIC
} PgmTextStyle;

/**
 * PgmTextVariant:
 * @PGM_TEXT_VARIANT_NORMAL: Normal font.
 * @PGM_TEXT_VARIANT_SMALL_CAPS: Font with the lower case characters replaced
 * by smaller variants of the capital characters.
 *
 * The different text variants.
 */
typedef enum {
  PGM_TEXT_VARIANT_NORMAL,
  PGM_TEXT_VARIANT_SMALL_CAPS
} PgmTextVariant;

/**
 * PgmTextWeight:
 * @PGM_TEXT_WEIGHT_LIGHT: The light weight.
 * @PGM_TEXT_WEIGHT_NORMAL: The normal weight.
 * @PGM_TEXT_WEIGHT_BOLD: The bold weight.
 *
 * The different text weights.
 */
typedef enum {
  PGM_TEXT_WEIGHT_LIGHT,
  PGM_TEXT_WEIGHT_NORMAL,
  PGM_TEXT_WEIGHT_BOLD
} PgmTextWeight;

/**
 * PgmTextShadowPosition:
 * @PGM_TEXT_SHADOW_NORTH: Shadow casted to the North.
 * @PGM_TEXT_SHADOW_SOUTH: Shadow casted to the South.
 * @PGM_TEXT_SHADOW_WEST: Shadow casted to the West.
 * @PGM_TEXT_SHADOW_EAST: Shadow casted to the East.
 * @PGM_TEXT_SHADOW_NORTH_WEST: Shadow casted to the North/West.
 * @PGM_TEXT_SHADOW_NORTH_EAST: Shadow casted to the North/East.
 * @PGM_TEXT_SHADOW_SOUTH_WEST: Shadow casted to the South/West.
 * @PGM_TEXT_SHADOW_SOUTH_EAST: Shadow casted to the South/East.
 *
 * The text's shadow cast position.
 */
typedef enum {
  PGM_TEXT_SHADOW_NORTH      = (1 << 0),
  PGM_TEXT_SHADOW_SOUTH      = (1 << 1),
  PGM_TEXT_SHADOW_WEST       = (1 << 2),
  PGM_TEXT_SHADOW_EAST       = (1 << 3),
  PGM_TEXT_SHADOW_NORTH_WEST = (PGM_TEXT_SHADOW_NORTH | PGM_TEXT_SHADOW_WEST),
  PGM_TEXT_SHADOW_NORTH_EAST = (PGM_TEXT_SHADOW_NORTH | PGM_TEXT_SHADOW_EAST),
  PGM_TEXT_SHADOW_SOUTH_WEST = (PGM_TEXT_SHADOW_SOUTH | PGM_TEXT_SHADOW_WEST),
  PGM_TEXT_SHADOW_SOUTH_EAST = (PGM_TEXT_SHADOW_SOUTH | PGM_TEXT_SHADOW_EAST)
} PgmTextShadowPosition;

/**
 * PgmText:
 * @label: the text label.
 * @font_family: the text font family.
 * @gravity: the text gravity.
 * @stretch: the text stretching.
 * @style: the text style.
 * @variant: the text variant.
 * @weight: the text weight.
 * @height: the text height in proportion to the drawable height.
 * @ellipsize: the text ellipsizing.
 * @alignment: the text alignment.
 * @wrap: the text wrapping.
 * @line_spacing: the space between to lines of text in canvas coordinates.
 * @justify: the text justification.
 * @multiline: the text multilining.
 * @cast_shadow: the text shadow casting state.
 * @shadow_position: the text shadow position.
 * @outline_width: the text outline width in canvas coordinates.
 * @r: the text red color component.
 * @g: the text blue color component.
 * @b: the text green color component.
 * @a: the text alpha color component.
 * @shadow_r: the text shadow red color component.
 * @shadow_g: the text shadow blue color component.
 * @shadow_b: the text shadow green color component.
 * @shadow_a: the text shadow alpha color component.
 * @outline_r: the text outline red color component.
 * @outline_g: the text outline blue color component.
 * @outline_b: the text outline green color component.
 * @outline_a: the text outline alpha color component.
 *
 * The #PgmText structure.
 */
struct _PgmText {
  /*< private >*/
  PgmDrawable parent;

  /*< public >*/ /* with LOCK */
  /* Label */
  gchar *label;

  /* Font properties */
  gchar          *font_family;
  PgmTextGravity  gravity;
  PgmTextStretch  stretch;
  PgmTextStyle    style;
  PgmTextVariant  variant;
  PgmTextWeight   weight;
  gfloat          height;

  /* Text adjustment */
  PgmTextEllipsize ellipsize;
  PgmTextAlignment alignment;
  PgmTextWrap      wrap;
  gfloat           line_spacing;
  gboolean         justify;
  gboolean         multiline;

  gboolean cast_shadow;
  PgmTextShadowPosition shadow_position;
  gfloat outline_width;

  /* Colors */
  guchar r, g, b, a;
  guchar shadow_r, shadow_g, shadow_b, shadow_a;
  guchar outline_r, outline_g, outline_b, outline_a;

  /*< private >*/
  gboolean use_markup;
};

struct _PgmTextClass {
  PgmDrawableClass parent_class;
};

GType        pgm_text_get_type            (void);

PgmDrawable *pgm_text_new                 (const gchar *markup);

PgmError     pgm_text_set_label           (PgmText *text,
                                           const gchar *label);

PgmError     pgm_text_set_markup          (PgmText *text,
                                           const gchar *markup);

PgmError     pgm_text_get_label           (PgmText *text,
                                           gchar **label);

PgmError     pgm_text_set_font_family     (PgmText *text,
                                           const gchar *font_family);

PgmError     pgm_text_get_font_family     (PgmText *text,
                                           gchar **font_family);

PgmError     pgm_text_set_font_height     (PgmText *text,
                                           gfloat font_height);

PgmError     pgm_text_get_font_height     (PgmText *text,
                                           gfloat *font_height);

PgmError     pgm_text_set_ellipsize       (PgmText *text,
                                           PgmTextEllipsize ellipsize);

PgmError     pgm_text_get_ellipsize       (PgmText *text,
                                           PgmTextEllipsize *ellipsize);

PgmError     pgm_text_set_justify         (PgmText *text,
                                           gboolean justify);

PgmError     pgm_text_get_justify         (PgmText *text,
                                           gboolean *justify);

PgmError     pgm_text_set_alignment       (PgmText *text,
                                           PgmTextAlignment alignment);

PgmError     pgm_text_get_alignment       (PgmText *text,
                                           PgmTextAlignment *alignment);

PgmError     pgm_text_set_wrap            (PgmText *text,
                                           PgmTextWrap wrap);

PgmError     pgm_text_get_wrap            (PgmText *text,
                                           PgmTextWrap *wrap);

PgmError     pgm_text_set_gravity         (PgmText *text,
                                           PgmTextGravity gravity);

PgmError     pgm_text_get_gravity         (PgmText *text,
                                           PgmTextGravity *gravity);

PgmError     pgm_text_set_stretch         (PgmText *text,
                                           PgmTextStretch stretch);

PgmError     pgm_text_get_stretch         (PgmText *text,
                                           PgmTextStretch *stretch);

PgmError     pgm_text_set_style           (PgmText *text,
                                           PgmTextStyle style);

PgmError     pgm_text_get_style           (PgmText *text,
                                           PgmTextStyle *style);

PgmError     pgm_text_set_variant         (PgmText *text,
                                           PgmTextVariant variant);

PgmError     pgm_text_get_variant         (PgmText *text,
                                           PgmTextVariant *variant);

PgmError     pgm_text_set_weight          (PgmText *text,
                                           PgmTextWeight weight);

PgmError     pgm_text_get_weight          (PgmText *text,
                                           PgmTextWeight *weight);

PgmError     pgm_text_set_line_spacing    (PgmText *text,
                                           gfloat line_spacing);

PgmError     pgm_text_get_line_spacing    (PgmText *text,
                                           gfloat *line_spacing);

PgmError     pgm_text_set_multiline       (PgmText *text,
                                           gboolean multiline);

PgmError     pgm_text_get_multiline       (PgmText *text,
                                           gboolean *multiline);

PgmError     pgm_text_set_color           (PgmText *text,
                                           guchar red,
                                           guchar green,
                                           guchar blue,
                                           guchar alpha);

PgmError     pgm_text_get_color           (PgmText *text,
                                           guchar *red,
                                           guchar *green,
                                           guchar *blue,
                                           guchar *alpha);

PgmError     pgm_text_set_outline_color   (PgmText *text,
                                           guchar red,
                                           guchar green,
                                           guchar blue,
                                           guchar alpha);

PgmError     pgm_text_get_outline_color   (PgmText *text,
                                           guchar *red,
                                           guchar *green,
                                           guchar *blue,
                                           guchar *alpha);

PgmError     pgm_text_set_outline_width   (PgmText *text,
                                           gfloat width);

PgmError     pgm_text_get_outline_width   (PgmText *text,
                                           gfloat *width);

PgmError     pgm_text_set_cast_shadow     (PgmText *text,
                                           gboolean cast_shadow);

PgmError     pgm_text_get_cast_shadow     (PgmText *text,
                                           gboolean *cast_shadow);

PgmError     pgm_text_set_shadow_position (PgmText *text,
                                           PgmTextShadowPosition position);

PgmError     pgm_text_get_shadow_position (PgmText *text,
                                           PgmTextShadowPosition *position);

PgmError     pgm_text_set_shadow_color    (PgmText *text,
                                           guchar red,
                                           guchar green,
                                           guchar blue,
                                           guchar alpha);

PgmError     pgm_text_get_shadow_color    (PgmText *text,
                                           guchar *red,
                                           guchar *green,
                                           guchar *blue,
                                           guchar *alpha);

G_END_DECLS

#endif /* __PGM_TEXT_H__ */

