# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

"""
Models managed by TheMovieDB.org resource_provider
"""

from elisa.core.components.model import Model
from elisa.core import media_uri
from elisa.plugins.themoviedb.constants import *

class MoviesListModel(Model):

    def __init__(self):
        super(MoviesListModel, self).__init__()
        self.movies = []

class MovieModel(Model):

    def __init__(self):
        """
        A MovieModel represents a search result on themoviedb.org or a
        detailled set of informations as reported per the getInfo API
        call. All fields are usually non-empty excepted these
        optional ones:

        - runtime
        - budget
        - revenue
        - homepage
        - trailers
        - production_countries
        - people
        - categories

        @ivar score: movie relevance compared to the search term on the API.
                     Will be 1.0 if getInfo was called (perfect match)
        @type score: C{float}
        @ivar popularity: popularity on website (?) (we assume between
                          0 and 100 undocumented in online API docs)
        @type popularity: C{int}
        @ivar rating: user rating (between 0 and 10)
        @type rating: C{float}
        @ivar title: movie main title
        @type title: C{unicode}
        @ivar alternative_title: movie alternate title
        @type alternative_title: C{unicode}
        @ivar id: movie id in themoviedb database
        @type id: C{int}
        @ivar imdbid: IMDB.com ID for the movie
        @type imdbid: C{unicode}
        @ivar url: themoviedb.org url of the movie
        @type url: L{elisa.core.media_uri.MediaUri}
        @ivar short_overview: overview of the movie
        @type short_overview: C{unicode}
        @ivar release_date: official date of the movie release
        @type release_date: L{datetime.date}
        @ivar posters: movie posters, sorted by size. Possible sizes
                       are: original, mid, thummb, cover
        @type posters: C{dict}. Keys: sizes (C{unicode}),
                       values: urls (L{elisa.core.media_uri.MediaUri})
        @ivar backdrops: movie backdrops, sorted by size. Possible sizes
                         are original, mid, thumb
        @type backdrops: C{dict}. Keys: sizes (C{unicode}),
                         values: urls (L{elisa.core.media_uri.MediaUri})
        @ivar runtime: movie duration, in minutes
        @type runtime: C{int}
        @ivar budget: movie budget, in $ ?
        @type budget: C{int}
        @ivar revenue: movie revenue, in $ ?
        @type revenue: C{int}
        @ivar homepage: official website
        @type homepage: L{elisa.core.media_uri.MediaUri}
        @ivar trailers: movie trailers sorted by source ("youtube" is
                        currently the only supported source).
        @type trailers: C{dict}. Keys sources (C{unicode}),
                        values: urls (L{elisa.core.media_uri.MediaUri})
        @ivar production_countries: countries where the movie was made (?)
        @type production_countries: C{list} of tuples(name: C{unicode},
                                    short_name: C{unicode},
                                    url: L{elisa.core.media_uri.MediaUri})
        @ivar people: people involved in the movie, sorted by job
        @type people: C{dict}. Keys job (C{unicode}),
                      values: tuples(name: C{unicode}, role: C{unicode},
                                     url: L{elisa.core.media_uri.MediaUri})
        @ivar categories: user-supplied tags
        @type categories: C{dict}. Keys: name (C{unicode}),
                          values: moviedb_url (L{elisa.core.media_uri.MediaUri})
        """
        super(MovieModel, self).__init__()
        self.score = None
        self.rating = None
        self.popularity = None
        self.title = None
        self.alternative_title = None
        self.id = None
        self.imdbid = None
        self.url = None
        self.short_overview = None
        self.release_date = None
        self.posters = {}
        self.backdrops = {}

        # optional:
        self.runtime = None
        self.budget = None
        self.revenue = None
        self.homepage = None
        self.trailers = {}
        self.production_countries = []
        self.people = {}
        self.categories = {}

    @property
    def api_url(self):
        """
        A little helper giving the API url call to use to retrieve
        extended informations about the movie.

        @returns: the getInfo URL that can be used in a get() call of the
                  resource_provider.
        @rtype:   L{elisa.core.media_uri.MediaUri}
        """
        url = "http://%s/2.0/Movie.getInfo?id=%s&api_key=%s" % \
              (API_SERVER, self.id, API_KEY)
        return media_uri.MediaUri(url)

    @property
    def imdb_url(self):
        """
        A little helper giving the API url call to use to retrieve
        informations about the movie on imdb.com.

        @returns: the imdbLookup URL that can be used in a get() call of the
                  resource_provider.
        @rtype:   L{elisa.core.media_uri.MediaUri}
        """
        url = "http://%s/2.0/Movie.imdbLookup?imdb_id=%s&api_key=%s" % \
              (API_SERVER, self.imdbid, API_KEY)
        return media_uri.MediaUri(url)
