<?php
	/**
	 * Module
	 * Superclass of every module. Many default functions are defined in this class.
	 * @author Johnny Biemans <j.biemans@connectux.com>
	 */
	class Module
	{
		/**
		 * @var int unique id of the class.
		 */
		var $id;
		
		/**
		 * @var string entryid, which will be registered by the bus object.
		 */
		var $entryid;
		
		/**
		 * @var array list of all actions, which is received from the client.
		 */
		var $data;
		
		/**
		 * @var array list of the results, which is send to the client.
		 */
		var $responseData;
		
		/**
		 * @var array list of all the errors occurred.
		 */
		var $errors;
		
		/**
		 * Constructor
		 * @param int $id unique id.
		 * @param array $data list of all actions.
		 */
		function Module($id, $data, $events = false)
		{
			$this->id = $id;
			$this->data = $data;
			$this->errors = array();
			$this->responseData = array();
			$this->responseData["attributes"] = array("name" => strtolower(get_class($this)), "id" => $id);
			$this->responseData["action"] = array();

			if($events) {
				if(isset($GLOBALS["bus"])) {
					$this->entryid = $this->getEntryID();

					if($this->entryid) {
						if(is_array($this->entryid)) {
							foreach($this->entryid as $entryid)
							{
								$GLOBALS["bus"]->register($this, $entryid, $events, true);
							}
						} else {
							$GLOBALS["bus"]->register($this, $this->entryid, $events);
						}
					}
				}
			}
		}
		
		/**
		 * Executes all the actions in the $data variable.
		 * @return boolean true on success of false on fialure.
		 */
		function execute()
		{
			$result = false;
			
			// you must implement this function for each module
			
			return $result;
		}

	


		/**
		 * Function which returns an entryid, which is used to register this module. It
		 * searches in the class variable $data for a ParentEntryID or an EntryID.
		 * @return string an entryid if found, false if entryid not found.
		 */
		function getEntryID()
		{
			$entryid = 0;
			
			foreach($this->data as $action)
			{
				if(isset($action["parententryid"])) {
					$entryid = $action["parententryid"];
				} else if(isset($action["entryid"])) {
					$entryid = $action["entryid"];
				}
			}
			
			return $entryid;
		}
		
		/**
		 * Returns all the errors, which occurred.
		 * @return array An array of all the errors, which occurred.
		 */
		function getErrors()
		{
			return $this->errors;
		}
		
		/**
		 * Returns the response data.
		 * @return array An array of the response data. This data is send to the client.
		 */
		function getData()
		{
			return $this->responseData;
		}
		
		/**
		 * Sets the action data, which will be executed.
		 * @param array $data array of all the actions.
		 */
		function setData($data)
		{
			$this->data = $data;
		}
		
		/**
		 * Returns the id.
		 * @return int id.
		 */
		function getId()
		{
			return $this->id;
		}
		
		/**
		 * Function which resets the data and the response data class variable.
		 */
		function reset()
		{
			$this->data = array();
			
			$this->responseData = array();
			$this->responseData["attributes"] = array("name" => strtolower(get_class($this)), "id" => $this->id);
			$this->responseData["action"] = array();
		}
		
		/**
		 * Function which returns MAPI Message Store Object. It
		 * searches in the variable $action for a storeid.
		 * @param array $action the XML data retrieved from the client
		 * @return object MAPI Message Store Object, false if storeid is not found in the $action variable 
		 */
		function getActionStore($action)
		{
			$store = false;
			
			if(isset($action["store"])) {
				$store = $GLOBALS["mapisession"]->openMessageStore(hex2bin($action["store"]));
			}
			
			return $store;
		}
		
		/**
		 * Function which returns a parent entryid. It
		 * searches in the variable $action for a parententryid.
		 * @param array $action the XML data retrieved from the client
		 * @return object MAPI Message Store Object, false if parententryid is not found in the $action variable 
		 */
		function getActionParentEntryID($action)
		{
			$parententryid = false;
			
			if(isset($action["parententryid"])) {
				$parententryid = hex2bin($action["parententryid"]);
			}
			
			return $parententryid;
		}
		
		/**
		 * Function which returns an entryid. It
		 * searches in the variable $action for an entryid.
		 * @param array $action the XML data retrieved from the client
		 * @return object MAPI Message Store Object, false if entryid is not found in the $action variable 
		 */
		function getActionEntryID($action)
		{
			$entryid = false;
			
			if(isset($action["entryid"]) && !is_array($action["entryid"])) {
				$entryid = hex2bin($action["entryid"]);
			} else if(isset($action["entryid"]) && is_array($action["entryid"])) {
				$entryid = array();
				foreach($action["entryid"] as $action_entryid)
				{
					array_push($entryid, hex2bin($action_entryid));
				}
			}
			
			return $entryid;
		}
	}
?>
