<?php
	/**
	 * Language handling class
	 *
	 * @package core
	 * @author Michael Erkens <michael@connectux.com>
	 */
	class Language {
		var $languages;
		var $languagetable;
		var $lang;
		var $loaded;
			
        /**
        * Default constructor
        *
        * By default, the Language class only knows about en_EN (English). If you want more languages, you
        * must call loadLanguages().
        *
        */
		function Language()
		{
			$this->languages = array("en_EN"=>"English");
			$this->languagetable = array("en_EN"=>"eng_ENG");
			$this->loaded = false;
		}
		
		/**
		* Loads languages from disk
		*
		* loadLanguages() reads the languages from disk by reading LANGUAGE_DIR and opening all directories
		* in that directory. Each directory must contain a 'language.txt' file containing:
		*
		* <language display name>
		* <win32 language name>
		*
		* For example:
		* <code>
		* Nederlands
		* nld_NLD
		* </code>
		*
		* Also, the directory names must have a name that is:
		* 1. Available to the server's locale system
		* 2. In the UTF-8 charset
		*
		* For example, nl_NL.utf-8
		*
		*/
		function loadLanguages()
		{
			if($this->loaded)
				return;
			$dh = opendir(LANGUAGE_DIR);
			while (($entry = readdir($dh))!==false){
				if (is_dir(LANGUAGE_DIR.$entry."/LC_MESSAGES") && is_file(LANGUAGE_DIR.$entry."/language.txt")){
					$fh = fopen(LANGUAGE_DIR.$entry."/language.txt", "r");
					$lang_title = fgets($fh);
					$lang_table = fgets($fh);
					fclose($fh);
					$this->languages[$entry] = $lang_title;
					$this->languagetable[$entry] = $lang_table;
				}
			}
			$this->loaded = true;		
		}

		/**
		* Attempt to set language
		*
		* setLanguage attempts to set the language to the specified language. The language passed
		* is the name of the directory containing the language.
		*
		* For setLanguage() to success, the language has to have been loaded via loadLanguages() AND
		* the gettext system on the system must 'know' the language specified.
		*
		* @param string $lang Language (eg nl_NL.utf-8)
		*/
		function setLanguage($lang)
		{
			$lang = (empty($lang)||$lang=="C")?LANG:$lang; // default language fix
			if ($this->is_language($lang)){
				$this->lang = $lang;
		
				if (strtoupper(substr(PHP_OS, 0, 3)) === "WIN"){
					$this->loadLanguages(); // we need the languagetable for windows...
					setlocale(LC_ALL,$this->languagetable[$lang]);
				}else{
					setlocale(LC_ALL,$lang);
				}

				bindtextdomain($lang , LANGUAGE_DIR);

				// All text from gettext() and _() is in UTF-8 so if you're saving to
				// MAPI directly, don't forget to convert to windows-1252 if you're writing
				// to PT_STRING8
				bind_textdomain_codeset($lang, "UTF-8");

				textdomain($lang);
			}else{
				trigger_error("Unknown language: '".$lang."'", E_USER_WARNING);
			}
		}

		/**
		* Return a list of supported languages
		*
		* Returns an associative array in the format langid -> langname, for example "nl_NL.utf8" -> "Nederlands"
		*
		* @return array List of supported languages
		*/
		function getLanguages()
		{
			$this->loadLanguages();
			return $this->languages;
		}
	
		/**
		* Returns the ID of the currently selected language
		*
		* @return string ID of selected language
		*/
		function getSelected()
		{
			return $this->lang;
		}
	
		/**
		* Returns if the specified language is valid or not
		*
		* @param string $lang 
		* @return boolean TRUE if the language is valid
		*/
		function is_language($lang)
		{
			return $lang=="en_EN" || is_dir(LANGUAGE_DIR . "/" . $lang);
		}
	}
?>
