/*
 * Copyright (C) 2005 - 2008  Zarafa B.V.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of the Program
 * under the AGPL does not imply a trademark license. Therefore any rights,
 * title and interest in our trademarks remain entirely with us.
 * However, if you propagate an unmodified version of the Program you are 
 * required to use the term "Zarafa" to indicate that you distribute the 
 * Program. Furthermore you may use our trademarks where it is necessary to 
 * indicate the intended purpose of a product or service provided you use it in 
 * accordance with honest practices in industrial or commercial matters.
 * If you want to propagate modified versions of the Program under the name
 * "Zarafa" or "Zarafa Server", you may only do so if you have a written
 * permission by Zarafa B.V. (to acquire a permission please contact Zarafa at
 * trademark@zarafa.com).
 * The user interface of the software displays a attribution notice containing
 * the term "Zarafa" and/or the Logo of Zarafa. You have to preserve these
 * attribution notices when you propagate unmodified or modified versions of
 * the Program.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifndef ECLOGGER_H
#define ECLOGGER_H

#include <string>
#include <pthread.h>
#include <stdio.h>
#include <stdarg.h>
#include "ECConfig.h"

#define EC_LOGLEVEL_NONE	0
#define EC_LOGLEVEL_FATAL	1
#define EC_LOGLEVEL_ERROR	2
#define EC_LOGLEVEL_WARNING	3
#define EC_LOGLEVEL_INFO	4
#define EC_LOGLEVEL_ANY		5
#define EC_LOGLEVEL_DEBUG	6

#define _LOG_BUFSIZE		10240
#define _LOG_TSSIZE			64

class ECLogger {
protected:
	char* MakeTimestamp();
	int max_loglevel;
	char *msgbuffer;
	char timestring[_LOG_TSSIZE];

public:
	ECLogger(int max_ll);
	virtual ~ECLogger();

	void SetLoglevel(int max_ll);

	virtual void Reset() = 0;
	virtual void Log(int loglevel, const std::string &message) = 0;
	virtual void Log(int loglevel, const char *format, ...) = 0;
};


/* Dummy null logger */
class ECLogger_Null : public ECLogger {
public:
	ECLogger_Null();
	~ECLogger_Null();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string &message);
	virtual void Log(int Loglevel, const char *format, ...);
};

/* File logger. Use "-" for stderr logging */
class ECLogger_File : public ECLogger {
private:
	FILE *log;
	pthread_mutex_t filelock;
	char *logname;
	int timestamp;

public:
	ECLogger_File(int max_ll, int add_timestamp, const char *filename);
	~ECLogger_File();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string &message);
	virtual void Log(int loglevel, const char *format, ...);
};

/*	Linux syslog logger */
class ECLogger_Syslog : public ECLogger {
private:
	int levelmap[EC_LOGLEVEL_DEBUG+1];	/* converts to syslog levels */

public:
	ECLogger_Syslog(int max_ll, const char *ident);
	~ECLogger_Syslog();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string &message);
	virtual void Log(int loglevel, const char *format, ...);
};

/* Windows eventlog logger */

ECLogger* CreateLogger(ECConfig *config, char *argv0, const char *lpszServiceName);
void LogConfigErrors(ECConfig *lpConfig, ECLogger *lpLogger);

#endif
