<?php

	/**
	 * Folder Properties Module
	 * @author Michael Erkens <m.erkens@connectux.com>
	 */
	class PropertiesModule extends Module
	{
		/**
		 * Constructor
		 * @param int $id unique id.
		 * @param string $folderentryid Entryid of the folder. Data will be selected from this folder.
		 * @param array $data list of all actions.
		 */
		function PropertiesModule($id, $data)
		{
			parent::Module($id, $data);
		}
		
		/**
		 * Executes all the actions in the $data variable.
		 * @return boolean true on success of false on fialure.
		 */
		function execute()
		{
			$result = false;
			foreach($this->data as $action)
			{
				if(isset($action["attributes"]) && isset($action["attributes"]["type"])) {
					switch($action["attributes"]["type"])
					{
						case "folderprops":
							$store = $GLOBALS["mapisession"]->openMessageStore(hex2bin($action["store"]));
							$folder = mapi_msgstore_openentry($store, hex2bin($action["entryid"]));
							
							$data = $this->getFolderProps($store, $folder);

							// return response
							$data["attributes"] = array("type"=>"folderprops");
							array_push($this->responseData["action"], $data);
							$GLOBALS["bus"]->addData($this->responseData);
							$result = true;
							break;

						case "save":
							$store = $GLOBALS["mapisession"]->openMessageStore(hex2bin($action["store"]));
							$folder = mapi_msgstore_openentry($store, hex2bin($action["entryid"]));
							
							$result = $this->save($folder,$action);

							if($result != NOERROR) {
								// return error
								$data = array();
								$data["attributes"] = array("type" => "error");
								$data["error"] = array();
								$data["error"]["hresult"] = $result;
								$data["error"]["hresult_name"] = get_mapi_error_name($result);
								$data["error"]["message"] = _("Unable to set the permission on folder.");

								array_push($this->responseData["action"], $data);
								$GLOBALS["bus"]->addData($this->responseData);
								$result = true;
							} else {
								$data = array();
								$data["attributes"] = array("type" => "saved");

								array_push($this->responseData["action"], $data);
								$GLOBALS["bus"]->addData($this->responseData);
							}
							break;
					}
				}
			}
			
			return $result;
		}

		/**
		* returns properties of a folder, used by the properties dialog
		*/
		function getFolderProps($store, $folder)
		{
			$data = $GLOBALS["operations"]->getProps($store, $folder, $GLOBALS["properties"]->getFolderProperties());	
							
			// adding container_class if missing
			if (!isset($data["container_class"])){
				$data["container_class"] = "IPF.Note";
			}

			// adding missing comment
			if (!isset($data["comment"])){
				$data["comment"] = "";
			}
							
			// replace "IPM_SUBTREE" with the display name of the store, and use the store message size
			if ($data["display_name"] == "IPM_SUBTREE"){
				$store_props = mapi_getprops($store, array(PR_DISPLAY_NAME, PR_MESSAGE_SIZE_EXTENDED, PR_QUOTA_WARNING_THRESHOLD, PR_QUOTA_SEND_THRESHOLD, PR_QUOTA_RECEIVE_THRESHOLD));
				$data["display_name"] = windows1252_to_utf8($store_props[PR_DISPLAY_NAME]);
				$data["message_size"] = round($store_props[PR_MESSAGE_SIZE_EXTENDED]/1024). " "._("kb");
				$data["store_size"] = round($store_props[PR_MESSAGE_SIZE_EXTENDED]/1024);

				if (isset($store_props[PR_QUOTA_WARNING_THRESHOLD]))
					$data["quota_warning"] = round($store_props[PR_QUOTA_WARNING_THRESHOLD]);
				if (isset($store_props[PR_QUOTA_SEND_THRESHOLD]))
					$data["quota_soft"] = round($store_props[PR_QUOTA_SEND_THRESHOLD]);
				if (isset($store_props[PR_QUOTA_RECEIVE_THRESHOLD]))
					$data["quota_hard"] = round($store_props[PR_QUOTA_RECEIVE_THRESHOLD]);
			}

			// retrieve parent folder name
			if (is_array($data["parent_entryid"]) && isset($data["parent_entryid"]["_content"])){
				$data["parent_entryid"] = $data["parent_entryid"]["_content"];
			}
			$parent_folder = mapi_msgstore_openentry($store, hex2bin($data["parent_entryid"]));
			$parent_props = mapi_getprops($parent_folder, array(PR_DISPLAY_NAME));
			$data["parent_display_name"] = "";
			if (isset($parent_props[PR_DISPLAY_NAME])){
				$data["parent_display_name"] = windows1252_to_utf8($parent_props[PR_DISPLAY_NAME]);
				if ($data["parent_display_name"] == "IPM_SUBTREE"){
					// this must be the root folder, so get the name of the store
					$store_props = mapi_getprops($store, array(PR_DISPLAY_NAME));
					$data["parent_display_name"] = windows1252_to_utf8($store_props[PR_DISPLAY_NAME]);
				}
			}
			
			// calculating missing message_size
			if (!isset($data["message_size"])){
				$data["message_size"] = round($GLOBALS["operations"]->calcFolderMessageSize($folder, false)/1024). " "._("kb");
			}
			
			// retrieving folder permissions
			$data["permissions"] = $this->getFolderPermissions($folder);

			return $data;
		}

		/**
		 * Function which saves changed properties to a folder.
		 * @param object $folder MAPI object of the folder
		 * @param array $props the properties to save
		 * @return boolean true on success or false on failure		 		 
		 */
		function save($folder, $action)
		{
			mapi_setprops($folder, array(PR_COMMENT=>utf8_to_windows1252($action["comment"])));
			$result = mapi_last_hresult();
			
			if (isset($action["permissions"])){
				$returnValue = $this->setFolderPermissions($folder, $action["permissions"]);
				if($returnValue != NOERROR) {
					$result = $returnValue;
				}
			}
			
			mapi_savechanges($folder);
			$result = ($result == NOERROR) ? mapi_last_hresult() : $result;

			return $result;
		}


		function getFolderPermissions($folder)
		{
			// check if folder is rootFolder, then we need the permissions from the store
			$folderProps = mapi_getprops($folder, array(PR_DISPLAY_NAME, PR_STORE_ENTRYID));

			$store = $GLOBALS["mapisession"]->openMessageStore($folderProps[PR_STORE_ENTRYID]);
			if ($folderProps[PR_DISPLAY_NAME] == "IPM_SUBTREE"){
				$folder = $store; 
			}

			$grants = mapi_zarafa_getpermissionrules($folder, ACCESS_TYPE_GRANT);
			foreach($grants as $id=>$grant){
				unset($grant["type"]);
				unset($grant["state"]);
				$userinfo = $this->getUserInfo($grant["userid"]);

				$grant["username"] = $userinfo["username"];
				$grant["fullname"] = $userinfo["fullname"];
				$grant["usertype"] = $userinfo["type"];
				$grant["entryid"] = $userinfo["entryid"];

				$grants[$id] = $grant;
			}

			$result = array("grant"=>$grants);
			return $result;			
		}

		function setFolderPermissions($folder, $permissions)
		{
			// first, get the current permissions because we need to delete all current acl's 
			$current_perms = $this->getFolderPermissions($folder);

			$folderProps = mapi_getprops($folder, array(PR_DISPLAY_NAME, PR_STORE_ENTRYID, PR_ENTRYID));
			$store = $GLOBALS["mapisession"]->openMessageStore($folderProps[PR_STORE_ENTRYID]);
		
			// check if the folder is the default calendar, if so we also need to set the same permissions on the freebusy folder
			$inbox = mapi_msgstore_getreceivefolder($store);
			// public store can not have inbox folder
			if(isset($inbox) && $inbox) {
				$inboxProps = mapi_getprops($inbox, array(PR_IPM_APPOINTMENT_ENTRYID, PR_FREEBUSY_ENTRYIDS));
				if ($folderProps[PR_ENTRYID] == $inboxProps[PR_IPM_APPOINTMENT_ENTRYID]){
					if(isset($inboxProps[PR_FREEBUSY_ENTRYIDS]) && isset($inboxProps[PR_FREEBUSY_ENTRYIDS][3])){
						$freebusy = mapi_msgstore_openentry($store, $inboxProps[PR_FREEBUSY_ENTRYIDS][3]);
					}
				}
			}

			// check if folder is rootFolder, then we need the permissions from the store
			if ($folderProps[PR_DISPLAY_NAME] == "IPM_SUBTREE"){
				$folder = $store; 
			}

			// delete current acl's
			foreach($current_perms as $cur_tmp=>$cur_perms){
				foreach($cur_perms as $i=>$cur_perm){
					$userinfo = $this->getUserInfo($cur_perm["userid"]);

					$acls = array(
								array(
									"userid" => $userinfo["id"],
									"type" => ACCESS_TYPE_GRANT,
									"rights" => 0,
									"state" => RIGHT_DELETED|RIGHT_AUTOUPDATE_DENIED
								)
							);
					mapi_zarafa_setpermissionrules($folder, $acls);
					if (isset($freebusy)){
						mapi_zarafa_setpermissionrules($freebusy, $acls);
						mapi_savechanges($freebusy);
					}
				}
			}

			if (is_array($permissions)){
				$acls = array();
				foreach($permissions as $type=>$perms){
					switch($type){
						case "denied":
							$type = ACCESS_TYPE_DENIED;
							break;
						case "grant":
						default:
							$type = ACCESS_TYPE_GRANT;
							break;
					}
					
					// when there is just one user in the permission, wrap it in an array
					if (!is_array($perms[key($perms)])){
						$perms = array($perms);
					}
	
					foreach($perms as $i=>$perm){
						$userinfo = entryid2userid(hex2bin($perms[$i]["entryid"]));
						$acl = array(
							"type" => (int)$type,
							"userid" => (int)$userinfo["userid"],
							"rights" => (int)$perms[$i]["rights"],
							"state" => RIGHT_NEW | RIGHT_AUTOUPDATE_DENIED
						);
						$acls[] = $acl;
					}		
				}
				mapi_zarafa_setpermissionrules($folder, $acls);
				if(mapi_last_hresult() != NOERROR) {
					// unable to set permissions
					return mapi_last_hresult(); 
				}
				mapi_savechanges($folder);
				if (isset($freebusy)){
					// set permissions on free/busy message
					foreach($acls as $acl) {
						if( $acl["type"] == ACCESS_TYPE_GRANT && ($acl["rights"] & ecRightsEditOwned))
							$acl["rights"] |= ecRightsEditAny;
					}
					mapi_zarafa_setpermissionrules($freebusy, $acls);
					if(mapi_last_hresult() != NOERROR) {
						// unable to set permissions
						return mapi_last_hresult(); 
					}
					mapi_savechanges($freebusy);
				}
			}
		}

		function getUserInfo($userid){

			// default return stuff
			$result = array("fullname"=>_("Unknown user/group"),
							"username"=>_("unknown"),
							"entryid"=>null,
							"type"=>MAPI_MAILUSER,
							"id"=>$userid
							);

			// open the addressbook
			$ab = $GLOBALS["mapisession"]->getAddressbook();

			// try userid as normal user
			$entryid = userid2entryid($userid, MAPI_MAILUSER);
			$user = mapi_ab_openentry($ab, $entryid);

			if (mapi_last_hresult()==MAPI_E_NOT_FOUND){
				// try it again, but now as group
				$entryid = userid2entryid($userid, MAPI_DISTLIST);
				$user = mapi_ab_openentry($ab, $entryid);
			}

			if ($user){
				$props = mapi_getprops($user, array(PR_ACCOUNT, PR_DISPLAY_NAME, PR_OBJECT_TYPE));
				$result["username"] = windows1252_to_utf8($props[PR_ACCOUNT]);
				$result["fullname"] = windows1252_to_utf8($props[PR_DISPLAY_NAME]);
				$result["entryid"] = bin2hex($entryid);
				$result["type"] = $props[PR_OBJECT_TYPE];
			}
			return $result;
		}
	}
?>
