/**
 * --Calendar Week View--
 * @author	Frans van Veen <f.vanveen@zarafa.com>
 * @type	View
 * @classDescription	This view can be used for appointement
 * list module to display the calendar items
 * 
 * +--------------------------+
 * | mon        | thu         |
 * |            |             |
 * |------------|-------------|
 * | tue        | fri         |
 * |            |             | 
 * |------------|-------------|
 * | wed        | sat         |
 * |            |-------------|
 * |            | sun         |
 * +--------------------------+
 * 
 * DEPENDS ON:
 * |------> view.js
 * |----+-> *listmodule.js
 * |    |----> listmodule.js
 */

CalendarWeekView.prototype = new View;
CalendarWeekView.prototype.constructor = CalendarWeekView;
CalendarWeekView.superclass = View.prototype;

// PUBLIC
/**
 * @constructor This view can be used for appointement list module to display the calendar items
 * @param {Int} moduleID
 * @param {HtmlElement} element
 * @param {Object} events
 * @param {XmlElement} data
 */
function CalendarWeekView(moduleID, element, events, data)
{
	this.element = element;
	this.moduleID = moduleID;
	this.events = events;
	this.data = data;
	
	this.setData(data);
	
	this.dayElements = new Array();
}

CalendarWeekView.prototype.setData = function(data)
{
	this.startdate = data["startdate"];
	this.duedate = data["duedate"];
	this.selecteddate = data["selecteddate"];
}

/**
 * Function will render the view and execute this.resizeView when done
 */
CalendarWeekView.prototype.initView = function()
{
	// clear old elements
	dhtml.deleteAllChildren(this.element);
	
	// add day elments
	this.dayElements[0] = new Object();
	this.dayElements[0].element = dhtml.addElement(this.element,"div","week_view day_monday");
	this.dayElements[3] = new Object();
	this.dayElements[3].element = dhtml.addElement(this.element,"div","week_view day_thursday");
	this.dayElements[1] = new Object();
	this.dayElements[1].element = dhtml.addElement(this.element,"div","week_view day_tuesday");
	this.dayElements[4] = new Object();
	this.dayElements[4].element = dhtml.addElement(this.element,"div","week_view day_friday");
	this.dayElements[2] = new Object();
	this.dayElements[2].element = dhtml.addElement(this.element,"div","week_view day_wednesday");
	this.dayElements[5] = new Object();
	this.dayElements[5].element = dhtml.addElement(this.element,"div","week_view day_saturday");
	this.dayElements[6] = new Object();
	this.dayElements[6].element = dhtml.addElement(this.element,"div","week_view day_sunday");
	
	// add header info
	var tmpStart = new Date(this.selecteddate).getStartDateOfWeek();
	for(var i=0;i < this.dayElements.length;i++){
		this.dayElements[i].element.id = "day_"+this.moduleID+"_"+(tmpStart.getTime()/1000);
		dragdrop.addTarget(this.element,this.dayElements[i].element,"appointment",true);
		if(i == 6){
			var dayTitle = dhtml.addElement(this.dayElements[i].element,"span","day_header","",DAYS[0]+" "+tmpStart.getDate()+" "+MONTHS[tmpStart.getMonth()]);
		} else {
			var dayTitle = dhtml.addElement(this.dayElements[i].element,"span","day_header","",DAYS[i+1]+" "+tmpStart.getDate()+" "+MONTHS[tmpStart.getMonth()]);
		}
		dhtml.addEvent(this.moduleID,dayTitle,"click",eventCalenderWeekViewChangeViewToDay);
		tmpStart.addDays(1);
	}

	// week picker
	var extraElement = dhtml.addElement("","span");
	// previous button 
	var prevButton = dhtml.addElement(extraElement,"span","prev_button","",NBSP);
	dhtml.addEvent(this.moduleID,prevButton,"click",eventDayViewClickPrev);
	prevButton.title = _("Previous week");
	// title string
	var startDateObj = new Date(this.startdate);
	var dueDateObj = new Date(this.duedate-ONE_DAY);			
	var titleString = startDateObj.getDate()+" "+MONTHS[startDateObj.getMonth()]+" - "+dueDateObj.getDate()+" "+MONTHS[dueDateObj.getMonth()];
	dhtml.addElement(extraElement,"span","","",titleString);	
	// next button
	var nextButton =dhtml.addElement(extraElement,"span","next_button","",NBSP);
	dhtml.addEvent(this.moduleID,nextButton,"click",eventDayViewClickNext);
	nextButton.title = _("Next week");
	webclient.getModule(this.moduleID).setExtraTitle(extraElement);

	this.resizeView()
}

/**
 * Function will resize all elements in the view
 */
CalendarWeekView.prototype.resizeView = function()
{
	for(var i in this.dayElements){
		if(i == "5" || i == "6"){
			this.dayElements[i].element.style.height = (this.element.offsetHeight/3)/2 + "px";
		} else {
			this.dayElements[i].element.style.height = (this.element.offsetHeight/3) + "px";	
		}
		this.dayElements[i].element.style.width = (this.element.offsetWidth/2) - 1 + "px";
	}
	dragdrop.updateTargets("appointment");
	this.checkForMoreItems();
}

/**
 * Function will adds items to the view
 * @param {Object} items Object with items
 * @param {Array} properties property list
 * @param {Object} action the action tag
 * @return {Array} list of entryids
 */
CalendarWeekView.prototype.execute = function(items, properties, action)
{
	var entryids = false;

	for(var i=0;i<items.length;i++){
		if (!entryids) {
			entryids = new Object();
		}
		var item = this.createItem(items[i]);
		entryids[item["id"]]= item["entryid"];
	}
	this.resizeView();
	dragdrop.updateTargets("appointment");
	
	return entryids;
}

// PRIVATE
/**
 * Function will check if there are items outside the view that
 * have to be hidden
 */
CalendarWeekView.prototype.checkForMoreItems = function()
{
	for(var i in this.dayElements){
		var items = dhtml.getElementsByClassNameInElement(this.dayElements[i].element,"event","div");
		var dayHeight = this.dayElements[i].element.offsetHeight;
		var	dayCurrentHeight = (items.length*15)+15;
		var maxItems = Math.floor((dayHeight-18)/15)-1;
		
		// remove more_items icon		
		var moreItem = dhtml.getElementsByClassNameInElement(this.dayElements[i].element,"more_items","div")[0];
		if(moreItem){
			dhtml.deleteElement(moreItem);
		}

		// hide/show items
		var moreItemCount = 0;
		for(var j=0; j<items.length; j++){
			if(j < maxItems){
				items[j].style.display = "block";
			} else{
				items[j].style.display = "none";
				moreItemCount++;
			}
		}

		// show more_items icon
		if(items.length > maxItems && items.length > 0){
			var moreItem = dhtml.addElement(this.dayElements[i].element,"div","more_items");
			var unixTime = this.dayElements[i].element.getAttribute("id").split("_")[2];
			moreItem.setAttribute("unixtime",unixTime);
			if (moreItemCount==1){
				moreItem.title = _("There is one more item");
			} else {
				moreItem.title = _("There are %s more items").sprintf(moreItemCount);
			}

			dhtml.addEvent(this.moduleID,moreItem,"click",eventChangeViewToDay);
		}
		
	}
}

CalendarWeekView.prototype.addItem = function()
{
	return false;
}

/**
 * Function will add one "item" tot the view or replace "element" 
 * @param {Object} item
 * @param {HtmlElement} element changed item
 * @return {Object} entry item for entryID list
 */
CalendarWeekView.prototype.createItem = function(item, element)
{
	var entry = Object();
	// get properties
	var startDate = new Date(item.getElementsByTagName("startdate")[0].getAttribute("unixtime")*1000);
	var dueDate = new Date(item.getElementsByTagName("duedate")[0].getAttribute("unixtime")*1000);
	var unixTimeStart = item.getElementsByTagName("startdate")[0].getAttribute("unixtime");
	var unixTimeDue = item.getElementsByTagName("duedate")[0].getAttribute("unixtime");	
	var subject = item.getElementsByTagName("subject")[0].hasChildNodes()?item.getElementsByTagName("subject")[0].firstChild.nodeValue:NBSP;
	var location = dhtml.getTextNode(item.getElementsByTagName("location")[0],"");
	var parent_entryid = dhtml.getTextNode(item.getElementsByTagName("parent_entryid")[0],"");
	var alldayevent = dhtml.getTextNode(item.getElementsByTagName("alldayevent")[0],"");
	var entryid = dhtml.getTextNode(item.getElementsByTagName("entryid")[0],"");
	var label =  dhtml.getTextNode(item.getElementsByTagName("label")[0],"");
	var recurring = dhtml.getTextNode(item.getElementsByTagName("recurring")[0],"");
	var basedate = item.getElementsByTagName("basedate")[0]?item.getElementsByTagName("basedate")[0].getAttribute("unixtime"):false;	
	var exception = item.getElementsByTagName("exception")[0];
	var privateAppointment = dhtml.getTextNode(item.getElementsByTagName("private")[0], 0);
	var privateSensitivity = dhtml.getTextNode(item.getElementsByTagName("sensitivity")[0], 0);

	var startTime = new Date(startDate);
	var AppointmentElement;
	
	startTime.setHours(0);
	startTime.setMinutes(0);
	startTime.setSeconds(0);

	// Check how many days this event takes TODO: check DST
	var numberOfItems = Math.ceil((dueDate-startDate)/ONE_DAY);
	var timeDiff = dueDate.strftime(_("%H%M%S")) - startDate.strftime(_("%H%M%S"));
	if(timeDiff <= 0 && alldayevent != "1") {
		numberOfItems++;
	}

	if(numberOfItems < 1)
	    numberOfItems = 1;
	for(var i=0;i<numberOfItems;i++){
		var unixTimeId = startDate.getTime()/1000;
		var parent = dhtml.getElementById("day_"+this.moduleID+"_"+startTime.getTime()/1000);
		if(parent){
			if(element){
				dhtml.deleteElement(element);
			}
			AppointmentElement = dhtml.addElement(parent,"div","event");
			AppointmentElement.is_disabled = dhtml.getTextNode(item.getElementsByTagName("disabled_item")[0],0) != 0; // for private items, no events may be added to this item

			// set some properties
			AppointmentElement.id = entryid+"_"+unixTimeId;
			AppointmentElement.startdate = startDate.getTime()/1000;
			AppointmentElement.duedate = dueDate.getTime()/1000;
			
			// allday event
			if(alldayevent ==  "1"){
				dhtml.addClassName(AppointmentElement, "event_day_item");
			} else {
				dhtml.addElement(AppointmentElement,"span","event_time","",startDate.toTime()+" "+dueDate.toTime());
			}
	
			// private item
			if(privateAppointment == "1") {
				dhtml.addElement(AppointmentElement, "span", "private", false, NBSP);
			} else if(privateSensitivity == "2") {
				dhtml.addElement(AppointmentElement, "span", "private", false, NBSP);
			}
	
			
			// Recurring items
			if(recurring.length > 0 && basedate.length > 0) {
				AppointmentElement.setAttribute("basedate", basedate);
				if(exception && exception.firstChild) {
					dhtml.addElement(AppointmentElement, "span", "recurring_exception", false, NBSP);			
				} else if(recurring == "1") {
					dhtml.addElement(AppointmentElement, "span", "recurring", false, NBSP);
				}
			}
			
			// add subject
			if(location.length > 0){
				dhtml.addElement(AppointmentElement,"span","event_subject","",subject+" ("+location+")");
			} else {
				dhtml.addElement(AppointmentElement,"span","event_subject","",subject);
			}			

			// meeting
			var meeting = item.getElementsByTagName("meeting")[0];
			if(meeting && meeting.firstChild) {
				var responseStatus = dhtml.getTextNode(item.getElementsByTagName("responsestatus")[0], 0);
				AppointmentElement.meetingrequest = responseStatus; // store responseStatus in DOM tree
				switch(meeting.firstChild.nodeValue)
				{
					case "1":
					case "3":
					case "5":
						dhtml.addElement(AppointmentElement, "span", "meetingrequest", false, NBSP);
						break;
				}
			}

			// Appointment labels
			switch(label){
				case "1": dhtml.addClassName(AppointmentElement,"label_important"); break;
				case "2": dhtml.addClassName(AppointmentElement,"label_work"); break;
				case "3": dhtml.addClassName(AppointmentElement,"label_personal"); break;
				case "4": dhtml.addClassName(AppointmentElement,"label_holiday"); break;
				case "5": dhtml.addClassName(AppointmentElement,"label_required"); break;
				case "6": dhtml.addClassName(AppointmentElement,"label_travel_required"); break;
				case "7": dhtml.addClassName(AppointmentElement,"label_prepare_required"); break;
				case "8": dhtml.addClassName(AppointmentElement,"label_birthday"); break;
				case "9": dhtml.addClassName(AppointmentElement,"label_special_date"); break;
				case "10": dhtml.addClassName(AppointmentElement,"label_phone_interview"); break;
			}

			if (!AppointmentElement.is_disabled){
			
				// add events
				if (this.events && this.events["row"]){
					dhtml.setEvents(this.moduleID, AppointmentElement, this.events["row"]);
				}
				
				// single click   TODO: check DST
				if(dueDate-startDate-1 < ONE_DAY){
					dragdrop.addDraggable(AppointmentElement,"appointment",true,APPOINTMENT_NOT_RESIZABLE);
				}
			}
			
			entry["id"] = entryid+"_"+unixTimeId;
			entry["entryid"] = entryid;
		}
		
		// add one_day for more day event
		startTime.addDays(1);
	}
	return entry;
}

/**
 * Function will update a item
 * @param {Object} element
 * @param {Object} item
 * @param {Object} properties
 * @return {Object} entry item for entryID list
 */
CalendarWeekView.prototype.updateItem = function(element, item, properties)
{
	if(item) {
		dhtml.deleteAllChildren(element);
		var result = this.createItem(item,element);
		this.checkForMoreItems()
		return result;
	}
	return undefined;
}

/**
 * Function will show Loading text in view
 */
CalendarWeekView.prototype.loadMessage = function()
{
	dhtml.removeEvents(this.element);
	dhtml.deleteAllChildren(this.element);

	this.element.innerHTML = "<center>" + _("Loading") + "...</center>";
	document.body.style.cursor = "wait";
}

/**
 * Function will delete load text in view
 */
CalendarWeekView.prototype.deleteLoadMessage = function()
{
	dhtml.deleteAllChildren(this.element);
	this.initView();
	document.body.style.cursor = "default";
}

// EVENTS
/**
 * Function will change the view to day view
 * @param {Object} moduleObject
 * @param {Object} element
 * @param {Object} event
 */
function eventCalenderWeekViewChangeViewToDay(moduleObject, element, event)
{
	moduleObject.changeView("day",element.parentNode.getAttribute("id").split("_")[2]*1000);
}

/**
 * Function will modify that item in the view and server(php)
 * @param {Object} moduleObject
 * @param {Object} targetElement
 * @param {Object} element
 * @param {Object} event
 */
function eventCalenderWeekViewDragDropTarget(moduleObject, targetElement, element, event)
{	
	var draggableStartdate = element.startdate;
	var draggableDuedate = element.duedate;
	var draggableEntryID = element.getAttribute("id").split("_")[0];
	var targetDate = new Date(targetElement.getAttribute("id").split("_")[2]*1000);
	var duration = draggableDuedate-draggableStartdate;		

	var newDraggableStartdate = new Date(draggableStartdate*1000);
	newDraggableStartdate.setDate(targetDate.getDate());
	newDraggableStartdate.setMonth(targetDate.getMonth());
	newDraggableStartdate.setFullYear(targetDate.getFullYear());
	newDraggableStartdate = newDraggableStartdate/1000;//convert to unixtimestamp
	
	var newDraggableDuedate = newDraggableStartdate+duration;
	
	// move item
	targetElement.appendChild(element);
	element.setAttribute("resizable",APPOINTMENT_NOT_RESIZABLE);// TODO fixed this in dragdrop.js
																															// this workaround is not working in ie6
	// send update to server
	var props = new Object();
	props["entryid"] = draggableEntryID;
	props["startdate"] = newDraggableStartdate;
	props["duedate"] = newDraggableDuedate;
	props["commonstart"] = newDraggableStartdate;
	props["commonend"] = newDraggableDuedate;

	var send = false;
	if (element.meetingrequest && element.meetingrequest==1) // 1 = olResponseOrganized
		if( newDraggableDuedate>(new Date().getTime()/1000))
			send = confirm(_("Would you like to send an update to the attendees regarding changes to this meeting?"));

	moduleObject.save(props, send);
}
