"""This module implements a host completer for commands like ping, ssh and
scp.

BE AWARE THAT ENABLE THIS MODULE WILL EXPOSE YOU TO SECURITY RISK

Have a look at the original discussion about this module:
http://lists.ipython.scipy.org/pipermail/ipython-dev/2007-July/003070.html
and read the doc below carefully.

To enable this completer you need a decoded .ssh/known_hosts which is
not recommended.

Try this command::

    $ cat ~/.ssh/known_hosts

If you can't find something human readable then your known_hosts file is
encrypted. You need to put the option `HashKnownHosts no` in your
~/.ssh/config. This will expose you to a security risk. If an attacker crack
your password then he will be able to access ALL hosts found in your
known_hosts file (if you're using the same password for some of them)

If you can see some host names then your system is unsecured... and you
can use the completer.

Now you are aware of this. Let's see how to enable host completion. You just
need to add this lines to your ipy_user_conf.py::

    import ipy_host_completers.py

By default host completion is enable for ping, ssh, scp and sftp but
you can add more. Here is some examples::

    ip.set_hook('complete_command',host_completer,str_key='custom_ssh')
    ip.set_hook('complete_command',scp_completer,str_key='custom_scp')

The only difference between host and scp completer is that the scp one add
`:` after the hostname. This is useful for scp/sftp.
"""

import IPython.ipapi
import os,re

ip = IPython.ipapi.get()

known_hosts = re.compile(r'(\S+),[0-9]\S+\sssh-') #regexp to find hosts


def host_completer(self,event):
    """ Search possible hosts in ~/.ssh/known_hosts """

    # print event # dbg

    if os.name == 'posix':
        filename = os.path.expanduser('~/.ssh/known_hosts')
        if os.path.isfile(filename):
            fd = open(filename)
            hosts = known_hosts.findall(fd.read())
            fd.close()
            hosts = [h for h in hosts if h.startswith(event.symbol)]
            if len(hosts) == 1:
                return ['%s ' % hosts[0]]
    raise IPython.ipapi.TryNext


def scp_completer(self,event):
    hosts = host_completer(self,event)
    if len(hosts) == 1:
        return ['%s:' % hosts[0].strip()]
    return hosts


ip.set_hook('complete_command',host_completer,str_key='ping')
ip.set_hook('complete_command',host_completer,str_key='ssh')
ip.set_hook('complete_command',scp_completer,str_key='scp')
ip.set_hook('complete_command',scp_completer,str_key='sftp')

