/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __netdude_dialog_h
#define __netdude_dialog_h

#include <gtk/gtk.h>
#include <nd_packet.h>

typedef enum {
  ND_DIALOG_QUESTION,
  ND_DIALOG_WARNING
} ND_DialogType;

typedef enum {
  ND_BASE_DEC,
  ND_BASE_HEX
} ND_Base;


/**
 * ND_NumberCallback - callback signature used when closing number dialog.
 * @packet: the packet edited when the dialog was opened.
 * @user_data: arbitrary user data.
 * @value: the numeric value the user selected.
 *
 * This is the signature of callbacks that can be passed to
 * nd_dialog_number() and that gets called when the user
 * clicks "OK".
 */
typedef void (*ND_NumberCallback) (LND_Packet *packet,
				   void       *user_data,
				   guint       value);

/**
 * ND_AddressCallback - callback signature used when closing address dialog.
 * @address: input address data.
 * @address_len: length in bytes of @address.
 * @packet: packet edited when address dialog was opened.
 * @user_data: arbitrary user data.
 *
 * This is the signature of callbacks that can be passed to
 * nd_dialog_hardware_address() and that gets called when the
 * user clicks "OK".
 */
typedef void (*ND_AddressCallback) (guchar     *address,
				    int         address_len,
				    LND_Packet *packet,
				    void       *user_data);

/**
 * ND_StringCallback - callback signature used when closing a dialog that obtained a string from the user.
 * @string: the string the user provided.
 * @user_data: arbitrary user data.
 * 
 * This is the signature of callbacks that can be passed to
 * nd_dialog_string() and others, and that gets called when the user
 * clicks "OK".
 */
typedef void (*ND_StringCallback) (const char *string,
				   void *user_data);


/**
 * ND_DialogCallback - generic dialog callback.
 * @user_data: arbitrary user data.
 *
 * This is the signature of the generic callbacks used in the dialogs.
 * Also used in the generic number and address dialogs for
 * the "Cancel" callbacks, where the user can clean up any
 * @user_data if necessary.
 */
typedef void (*ND_DialogCallback) (void *user_data);


/**
 * nd_dialog_ync - shows a yes-no-cancel dialog.
 * @title: text to display in window title.
 * @message: message content.
 * @yes_cb: callback called when user clicks "yes"-button.
 * @no_cb: callback called when user clicks "no"-button.
 * @cancel_cb: callback called when user clicks "no"-button.
 * @user_data: arbitrary user data.
 *
 * Shows a yes-no-cancel dialog and calls the appriopriate callback
 * when a button is clicked. The dialog is displayed and hidden
 * automatically. You can pass %NULL for callbacks you don't need.
 */
void      nd_dialog_ync(const char *title, const char *message,
			ND_DialogCallback yes_cb,
			ND_DialogCallback no_cb,
			ND_DialogCallback cancel_cb,
			void *user_data);

/**
 * nd_dialog_message - shows a simple message dialog.
 * @title: text to display in window title.
 * @message: message content.
 * @modal: whether the dialog will be modal.
 *
 * The function shows a simple message dialog with an okay
 * button. No callbacks. If @modal is %TRUE, no other windows
 * will be active while the dialog is shown.
 */
void      nd_dialog_message(const char *title,
			    const char *message,
			    gboolean    modal);


/**
 * nd_dialog_number - shows a generic number entry dialog
 * @message: message content.
 * @base: whether to initially use hex or decimal display.
 * @value: initial value to display.
 * @max: maximum value allowed.
 * @callback_okay: callback called when user clicks "okay"-button.
 * @callback_cancel: callback called when user clicks "cancel"-button.
 * @packet: currently edited packet, passed through to callbacks.
 * @user_data: arbitrary user data.
 *
 * The function shows a number-editing dialog with the given label
 * and initial value. The largest possible value is given through
 * @max. You can pass %NULL for @callback_cancel if you don't need it.
 * The dialog also handles 32-bit values nicely, in contrast to
 * GTK's entry widget.
 */
void      nd_dialog_number(const char *message,
			   ND_Base base,
			   guint value, guint max,
			   ND_NumberCallback callback_okay,
			   ND_DialogCallback callback_cancel,
			   LND_Packet *packet,
			   void *user_data);

/* Callbacks used internally by the number dialog.
 * You don't need this.
 */
void      nd_dialog_number_update(void);
void      nd_dialog_number_set_base(ND_Base base);
void      nd_dialog_number_ok(GtkWidget *dialog);
void      nd_dialog_number_cancel(GtkWidget *dialog);

/* GtkSpinButton is buggy for large values, at least in Gtk 1.2+.
 * These are callbacks for the replacement dialog. The user should
 * hardly notice any difference.
 */
void      nd_dialog_largenumber_ok(GtkWidget *dialog);
void      nd_dialog_largenumber_up(GtkWidget *dialog);
void      nd_dialog_largenumber_down(GtkWidget *dialog);


void      nd_dialog_string(const char *title,
			   const char *message,
			   ND_StringCallback callback_ok,
			   ND_DialogCallback callback_cancel,
			   void *user_data);

void      nd_dialog_string_ok(GtkWidget *dialog);
void      nd_dialog_string_cancel(GtkWidget *dialog);

/**
 * nd_dialog_hardware_address - shows a dialog to enter hardware addresses.
 * @message: message content.
 * @address: pointer to initial address.
 * @address_len: length of initial address.
 * @callback_ok: callback called when user clicks "okay"-button.
 * @callback_cancel: callback called when user clicks "cancel"-button.
 * @packet: currently edited packet.
 * @user_data: arbitrary data.
 *
 * The function shows a generic hardware address entry dialog
 * that allows you to edit an address with the given @address_len
 * (in bytes, e.g. 6 for Ethernet). When the "okay"-button is clicked,
 * @callback_ok is called with the new value. You can pass %NULL
 * for @callback_cancel if you don't need it.
 */
void      nd_dialog_hardware_address(const char *message,
				     guchar *address, int address_len,
				     ND_AddressCallback callback_ok,
				     ND_DialogCallback  callback_cancel,
				     LND_Packet *packet,
				     void *user_data);

/**
 * nd_dialog_ip - shows a dialog to enter IPv4 addresses.
 * @message: message content.
 * @ip_address: pointer to initial IP address.
 * @callback_ok: callback called when user clicks "okay"-button.
 * @callback_cancel: callback called when user clicks "cancel"-button.
 * @packet: currently edited packet.
 * @user_data: arbitrary user data.
 *
 * The function shows a dialog to edit an IPv4 address. This is
 * needed sufficiently often that it is provided withing Netdude
 * itself, not by a plugin. The @ip_address memory is not used
 * internally. You can pass %NULL for @callback_cancel if you
 * don't need it.
 */
void      nd_dialog_ip(const char *message, guchar *ip_address,
		       ND_AddressCallback callback_ok,
		       ND_DialogCallback  callback_cancel,
		       LND_Packet *packet,
		       void *user_data);		       

/* callbacks for the IP dialog - nothing you need.
 */
void      nd_dialog_ip_ok(void);
void      nd_dialog_ip_cancel(void);

/**
 * nd_dialog_filesel - shows a dialog that allows choosing files.
 * @title: text displayed in window title.
 * @file_name: initial file/directory name to display.
 * @callback_fname: callback called when user clicks "okay"-button.
 * @user_data: arbitrary user data.
 *
 * The function shows a file selection dialog, initialized to
 * @file_name, if provided. You can also pass %NULL for @file_name.
 * @callback_fname if called when the user clicks the "okay"-button.
 */
void      nd_dialog_filesel(const char *title,
			    const char *file_name,
			    ND_StringCallback callback_fname,
			    void *user_data);

void      nd_dialog_filesel_close();

/* This one's used internally. */
void      nd_dialog_filesel_ok(GtkWidget *dialog);


/**
 * nd_dialog_exit - shows Netdude's exit dialog.
 *
 * This function opens the exit dialog, which shows a list of unsaved
 * files, which the user can save directly from the dialog before
 * Netdude exits.
 */
void      nd_dialog_exit(void);


/**
 * nd_dialog_generic - generic way to create a dialog.
 * @type: type of the dialog, determines what icon is displayed in dialog.
 * @title: text displayed in window title.
 * @message: message content.
 * @modal: whether dialog is modal or not.
 * @extra_area: arbitrary widget that is hooked into the dialog.
 * @user_data: arbitrary user data passed through to callbacks.
 * @num_buttons: number of buttons passed as varargs.
 * @...: sequence of button labels and callbacks.
 *
 * This function is the most generic way to generate a dialog, allowing you
 * to basically specify all parts of it. The varargs specify the
 * buttons and must be a sequence of char*s for buttons labels and
 * #ND_DialogCallbacks for the button callbacks.
 */
void      nd_dialog_generic(ND_DialogType  type,
			    const char    *title,
			    const char    *message,
			    gboolean       modal,
			    GtkWidget     *extra_area,
			    void          *user_data,
			    int            num_buttons,
			    ...);
		    

/**
 * nd_dialog_about - shows Netude's About-dialog.
 *
 * The function displays Netdude's About-dialog, containing
 * version and author info etc.
 */
void      nd_dialog_about(void);


/**
 * nd_dialog_fontsel - shows font selection dialog.
 * @title: text displayed in window title.
 * @font_name: initially selected font name, can be %NULL.
 * @callback_ok: callback called when user clicks "okay"-button.
 * @user_data: arbitrary user data.
 *
 * The function shows a font selection dialog, initialized
 * to @font_name, if provided. @callback_ok is called when
 * the user clicks the "okay"-button.
 */
void      nd_dialog_fontsel(const char *title,
			    const char *font_name,
			    ND_StringCallback callback_ok,
			    void *user_data);		       

/* callbacks for the font selection dialog - nothing you need.
 */
void      nd_dialog_fontsel_ok(GtkFontSelectionDialog *dialog);
void      nd_dialog_fontsel_cancel(GtkFontSelectionDialog *dialog);
void      nd_dialog_fontsel_apply(GtkFontSelectionDialog *dialog);

#endif
