/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal.sql;

import org.eclipse.jface.text.*;

/**
 * This class implements an edit strategy for indenting that is sensitive to brackets.
 */
public class SQLAutoIndentStrategy extends DefaultAutoIndentStrategy {

	public SQLAutoIndentStrategy() {
	}
	
	/**
     * Allows the strategy to manipulate the document command.
     * 
	 * @see org.eclipse.jface.text.IAutoEditStrategy#customizeDocumentCommand(org.eclipse.jface.text.IDocument, org.eclipse.jface.text.DocumentCommand)
	 */
	public void customizeDocumentCommand(IDocument d, DocumentCommand c) {
		if (c.length == 0 && c.text != null && endsWithDelimiter(d, c.text))
			smartIndentAfterNewLine(d, c);
		else if ("}".equals(c.text)) { //$NON-NLS-1$
			smartInsertAfterBracket(d, c);
		}
	}
	
	/**
	 * Returns whether or not the given text ends with one of the delimiters
     * valid for the given document.
     * 
     * @param d the document from which delimiters are obtained
     * @param txt the text to check
     * @return true when the text ends with a delimiter, otherwise false
	 */
	private boolean endsWithDelimiter( IDocument doc, String txt ) {
		String[] delimiters = doc.getLegalLineDelimiters();
		if (delimiters != null)
			return TextUtilities.endsWith( delimiters, txt ) > -1;
		return false;
	}

    /**
     * Returns the first offset greater than <code>offset</code> and smaller than 
     * <code>end</code> whose character is not a space or tab character. If no such
     * offset is found, <code>end</code> is returned.
     *
     * @param document the document to search in
     * @param offset the offset at which searching start
     * @param end the offset at which searching stops
     * @return the offset in the specified range whose character is not a space or tab
     * @exception BadLocationException if position is an invalid range in the given document
     */
    protected int findEndOfWhiteSpace( IDocument doc, int offset, int end ) throws BadLocationException {
        while (offset < end) {
            char c= doc.getChar( offset );
            if (c != ' ' && c != '\t') {
                return offset;
            }
            offset++;
        }
        return end;
    } 

	/**
	 * Returns the line number of the next bracket after <code>end</code>.
     * 
	 * @param doc the document being parsed
	 * @param line the line to start searching back from
	 * @param end the end position to search back from
	 * @param closingBracketIncrease the number of brackets to skip
     * @return the line number of the next matching bracket after end
     * @exception BadLocationException if position is an invalid range in the given document
	 */
	 protected int findMatchingOpenBracket( IDocument doc, int line, int end, int closingBracketIncrease ) throws BadLocationException {
		int start = doc.getLineOffset(line);
		int brackcount = getBracketCount( doc, start, end, false ) - closingBracketIncrease;

		// Sum up the bracket counts of each line (closing brackets count negative, 
		// opening positive) until we find a line the brings the count to zero.
		while (brackcount < 0) {
			line--;
			if (line < 0) {
				return -1;
			}
			start = doc.getLineOffset( line );
			end = start + doc.getLineLength( line ) - 1;
			brackcount += getBracketCount( doc, start, end, false );
		}
		return line;
	}
	
	/**
	 * Returns the bracket value of a section of text. Closing brackets have a value of -1 and 
	 * open brackets have a value of 1.
     * 
	 * @param doc the document being parsed
	 * @param start the start position for the search
	 * @param end the end position for the search
	 * @param ignoreCloseBrackets whether or not to ignore closing brackets in the count
     * @return the bracket count value
     * @exception BadLocationException if position is an invalid range in the given document 
	 */
	 private int getBracketCount( IDocument doc, int start, int end, boolean ignoreCloseBrackets ) throws BadLocationException {
		int begin = start;
		int bracketcount = 0;
		while (begin < end) {
			char curr = doc.getChar( begin );
			begin++;
			switch (curr) {
				case '/' :
					if (begin < end) {
						char next = doc.getChar( begin );
						if (next == '*') {
							// found a comment start, advance to the comment end.
							begin = getCommentEnd( doc, begin + 1, end );
						}
					}
					break;
				case '*' :
					if (begin < end) {
						char next = doc.getChar(begin);
						if (next == '/') {
							// we have been in a comment: forget what we read before
							bracketcount = 0;
							begin++;
						}
					}
					break;
                case '-' :
                    if (begin < end) {
                        char next = doc.getChar( begin );
                        if (next == '-') {
                            // '--'-comment: nothing to do anymore on this line 
                            begin = end;
                        }
                    }
                    break;
				case '{' :
					bracketcount++;
					ignoreCloseBrackets= false;
					break;
				case '}' :
					if (!ignoreCloseBrackets) {
						bracketcount--;
					}
					break;
				case '"' :
				case '\'' :
					begin = getStringEnd( doc, begin, end, curr);
					break;
				default :
					}
		}
		return bracketcount;
	}
	
	/**
	 * Returns the end position a comment starting at the given position in the 
     * given document.
     * 
	 * @param doc the document being parsed
	 * @param start the start position for the search
	 * @param end the end position for the search
     * @return the end position a comment starting at pos
     * @exception BadLocationException if position is an invalid range in the given document	 
     */
	 private int getCommentEnd( IDocument doc, int start, int end) throws BadLocationException {
		int currentPos = start;
		while (currentPos < end) {
			char currentChar = doc.getChar( currentPos );
			currentPos++;
			if (currentChar == '*') {
				if (currentPos < end && doc.getChar( currentPos) == '/') {
					return currentPos + 1;
				}
			}
		}
		return end;
	}
	
	/**
	 * Returns the string which is the leading whitespace of the given line of 
     * the given document. 
     * 
	 * @param doc the document being parsed
	 * @param line the line being searched
     * @return the leading whitespace of the line
     * @exception BadLocationException if position is an invalid range in the given document
     */
	 protected String getIndentOfLine( IDocument doc, int line ) throws BadLocationException {
		if (line > -1) {
			int start = doc.getLineOffset( line );
			int end = start + doc.getLineLength( line ) - 1;
			int whiteend = findEndOfWhiteSpace( doc, start, end );
			return doc.get( start, whiteend - start);
		} 

        return ""; //$NON-NLS-1$
	}
	
	/**
	 * Returns the position of the character in the document after position.
     * 
	 * @param doc the document being parsed
	 * @param start the position to start searching from
	 * @param end the end of the document
	 * @param character the character we are trying to find
     * @return the next location of the character
     * @exception BadLocationException if position is an invalid range in the given document
     */
	 private int getStringEnd( IDocument doc, int start, int end, char character ) throws BadLocationException {
		int currentPos = start;
		while (currentPos < end) {
			char currentChar = doc.getChar( currentPos );
			currentPos++;
			if (currentChar == '\\') {
				// ignore escaped characters
				currentPos++;
			} else if (currentChar == character) {
				return currentPos;
			}
		}
		return end;
	}
	
	/**
	 * Sets the indent of a new line based on the command provided in the given document.
     * 
	 * @param doc the document being parsed
	 * @param command the command being performed
	 */
	 protected void smartIndentAfterNewLine( IDocument doc, DocumentCommand command ) {
		int docLength = doc.getLength();
		if (command.offset == -1 || docLength == 0)
			return;

		try {
			int p = (command.offset == docLength ? command.offset - 1 : command.offset);
			int line = doc.getLineOfOffset( p );

			StringBuffer buf = new StringBuffer( command.text );
			if (command.offset < docLength && doc.getChar( command.offset ) == '}') {
				int indLine= findMatchingOpenBracket( doc, line, command.offset, 0 );
				if (indLine == -1) {
					indLine = line;
				}
				buf.append( getIndentOfLine( doc, indLine ));
			} else {
				int start = doc.getLineOffset( line );
				int whiteend = findEndOfWhiteSpace( doc, start, command.offset);
				buf.append( doc.get( start, whiteend - start ));
				if (getBracketCount( doc, start, command.offset, true ) > 0) {
					buf.append( '\t' );
				}
			}
			command.text = buf.toString();

		} 
        catch (BadLocationException excp) {
		    // TODO:
		}
	}
	
	/**
	 * Sets the indent of a bracket based on the command provided in the given document.
     * 
	 * @param doc the document being parsed
	 * @param command the command being performed
	 */
	 protected void smartInsertAfterBracket( IDocument doc, DocumentCommand command ) {
		if (command.offset == -1 || doc.getLength() == 0)
			return;

		try {
			int p = (command.offset == doc.getLength() ? command.offset - 1 : command.offset);
			int line = doc.getLineOfOffset( p );
			int start = doc.getLineOffset( line );
			int whiteend = findEndOfWhiteSpace( doc, start, command.offset );

			// shift only when line does not contain any text up to the closing bracket
			if (whiteend == command.offset) {
				// evaluate the line with the opening bracket that matches out closing bracket
				int indLine = findMatchingOpenBracket( doc, line, command.offset, 1 );
				if (indLine != -1 && indLine != line) {
					// take the indent of the found line
					StringBuffer replaceText = new StringBuffer( getIndentOfLine( doc, indLine ));
					// add the rest of the current line including the just added close bracket
					replaceText.append( doc.get( whiteend, command.offset - whiteend ));
					replaceText.append( command.text );
					// modify document command
					command.length = command.offset - start;
					command.offset = start;
					command.text = replaceText.toString();
				}
			}
		} 
        catch (BadLocationException excp) {
		    // TODO:
		}
	}
}
