-module(snmpc_mib_gram).
-export([parse/1, parse_and_scan/1, format_error/1]).
-file("snmpc_mib_gram.yrl", 623).
%%----------------------------------------------------------------------

-include("snmp_types.hrl").
-include("snmpc.hrl").

% value
val(Token) -> element(3, Token).

line_of(Token) -> element(2, Token).

%% category
cat(Token) -> element(1, Token). 

statusv1(Tok) ->
    case val(Tok) of
        mandatory -> mandatory;
        optional -> optional;
        obsolete -> obsolete;
        deprecated -> deprecated;
        Else -> return_error(line_of(Tok),
                             "syntax error before: " ++ atom_to_list(Else))
    end.

statusv2(Tok) ->
    case val(Tok) of
        current -> current;
        deprecated -> deprecated;
        obsolete -> obsolete;
        Else -> return_error(line_of(Tok),
                             "syntax error before: " ++ atom_to_list(Else))
    end.

accessv1(Tok) ->
    case val(Tok) of
        'read-only' -> 'read-only';
        'read-write' -> 'read-write';
        'write-only' -> 'write-only';
        'not-accessible' -> 'not-accessible';
        Else -> return_error(line_of(Tok),
                             "syntax error before: " ++ atom_to_list(Else))
    end.

accessv2(Tok) ->
    case val(Tok) of
        'not-accessible' -> 'not-accessible';
        'accessible-for-notify' -> 'accessible-for-notify';
        'read-only' -> 'read-only';
        'read-write' -> 'read-write';
        'read-create' -> 'read-create';
        Else -> return_error(line_of(Tok),
                             "syntax error before: " ++ atom_to_list(Else))
    end.

%% ---------------------------------------------------------------------
%% Various basic record build functions
%% ---------------------------------------------------------------------

make_module_identity(Name, LU, Org, CI, Desc, Revs, NA) ->
    #mc_module_identity{name         = Name,
                        last_updated = LU,
	                organization = Org,
	                contact_info = CI,
	                description  = Desc,
	                revisions    = Revs, 
	                name_assign  = NA}.

make_revision(Rev, Desc) ->
    #mc_revision{revision    = Rev,
	         description = Desc}.

make_object_type(Name, Syntax, MaxAcc, Status, Desc, Ref, Kind, NA) ->
    #mc_object_type{name        = Name,
                    syntax      = Syntax,
	            max_access  = MaxAcc,
	            status      = Status,
	            description = Desc,
	            reference   = Ref,
	            kind        = Kind, 
	            name_assign = NA}.

make_object_type(Name, Syntax, Units, MaxAcc, Status, Desc, Ref, Kind, NA) ->
    #mc_object_type{name        = Name,
                    syntax      = Syntax, 
                    units       = Units, 
	            max_access  = MaxAcc,
	            status      = Status,
	            description = Desc,
	            reference   = Ref,
	            kind        = Kind, 
	            name_assign = NA}.

make_new_type(Name, Macro, Syntax) ->
    #mc_new_type{name   = Name, 
	         macro  = Macro,
                 syntax = Syntax}.

make_new_type(Name, Macro, DisplayHint, Status, Desc, Ref, Syntax) ->
    #mc_new_type{name         = Name, 
	         macro        = Macro,
                 status       = Status,
                 description  = Desc,
                 reference    = Ref,
	         display_hint = DisplayHint,
                 syntax       = Syntax}.

make_trap(Name, Ent, Vars, Desc, Ref, Num) ->
    #mc_trap{name        = Name,
             enterprise  = Ent,
             vars        = Vars,
             description = Desc,
	     reference   = Ref,
	     num         = Num}.

make_notification(Name, Vars, Status, Desc, Ref, NA) ->
    #mc_notification{name        = Name,
                     vars        = Vars,
                     status      = Status,
                     description = Desc,
	             reference   = Ref,
	             name_assign = NA}.

make_module_compliance(Name, Status, Desc, Ref, Mod, NA) ->
    #mc_module_compliance{name        = Name,
                          status      = Status,
                          description = Desc,
	                  reference   = Ref,
                          module      = Mod,
	                  name_assign = NA}.

make_object_group(Name, Objs, Status, Desc, Ref, NA) ->
    #mc_object_group{name        = Name,
                     objects     = Objs,
                     status      = Status,
                     description = Desc,
	             reference   = Ref,
	             name_assign = NA}.

make_notification_group(Name, Objs, Status, Desc, Ref, NA) ->
    #mc_notification_group{name        = Name,
                           objects     = Objs,
                           status      = Status,
                           description = Desc,
	                   reference   = Ref,
	                   name_assign = NA}.

make_sequence(Name, Fields) ->
    #mc_sequence{name   = Name, 
                 fields = Fields}.

make_internal(Name, Macro, Parent, SubIdx) ->
    #mc_internal{name      = Name, 
                 macro     = Macro, 
                 parent    = Parent, 
                 sub_index = SubIdx}.



%% ---------------------------------------------------------------------


%%----------------------------------------------------------------------
%% Purpose: Find how much room needs to be allocated for the data type
%%          (when sending it in a PDU (the maximum difference will be 
%%           the size allocated)).
%%          This is applicable for OCTET STRINGs and OBJECT IDENTIFIERs.
%%
%%     Or : Find the range of integers in the integer list.
%%          This is applicable for INTEGERs
%%
%% Arg: A list of integers.
%%----------------------------------------------------------------------

make_range_integer(RevHexStr, h) ->
    erlang:list_to_integer(lists:reverse(RevHexStr), 16);
make_range_integer(RevHexStr, 'H') ->
    erlang:list_to_integer(lists:reverse(RevHexStr), 16);
make_range_integer(RevBitStr, b) ->
    erlang:list_to_integer(lists:reverse(RevBitStr), 2);
make_range_integer(RevBitStr, 'B') ->
    erlang:list_to_integer(lists:reverse(RevBitStr), 2);
make_range_integer(RevStr, Base) ->
    throw({error, {invalid_base, Base, lists:reverse(RevStr)}}).

make_range(XIntList) ->
    IntList = lists:flatten(XIntList),
    {range, lists:min(IntList), lists:max(IntList)}.

make_defval_for_string(Line, Str, Atom) ->
    case lists:member(Atom, [h, 'H', b, 'B']) of
	true ->
	    case catch make_defval_for_string2(Str, Atom) of
		Defval when list(Defval) ->
		    Defval;
		{error, ErrStr} ->
		    snmpc_lib:print_error("Bad DEFVAL ~w string ~p - ~s",
						 [Atom, Str, ErrStr],
						 Line),
		    "";
		_Else ->
		    snmpc_lib:print_error("Bad DEFVAL ~w string ~p",
						 [Atom, Str],
						 Line),
		    ""
	    end;
	false ->
	    snmpc_lib:print_error("Bad DEFVAL string type ~w for ~p",
					 [Atom, Str],
					 Line),
	    ""
    end.
	    

make_defval_for_string2([], h) -> [];
make_defval_for_string2([X16,X|HexString], h) ->
    lists:append(hex_to_bytes(snmpc_misc:to_upper([X16,X])),
		 make_defval_for_string2(HexString, h));
make_defval_for_string2([_Odd], h) ->
    throw({error, "odd number of bytes in hex string"});
make_defval_for_string2(HexString, 'H') ->
    make_defval_for_string2(HexString,h);

make_defval_for_string2(BitString, 'B') ->
    bits_to_bytes(BitString);
make_defval_for_string2(BitString, b) ->
    make_defval_for_string2(BitString, 'B').

bits_to_bytes(BitStr) ->
    lists:reverse(bits_to_bytes(lists:reverse(BitStr), 1, 0)).

bits_to_bytes([], 1, _Byte) ->   % empty bitstring
    [];
bits_to_bytes([], 256, _Byte) -> % correct; multiple of 8
    [];
% If we are to support arbitrary length of bitstrings.  This migth
% be needed in the new SMI.
%bits_to_bytes([], N, Byte) ->
%    [Byte];
bits_to_bytes([], _N, _Byte) ->
    throw({error, "not a multiple of eight bits in bitstring"});
bits_to_bytes(Rest, 256, Byte) ->
    [Byte | bits_to_bytes(Rest, 1, 0)];
bits_to_bytes([$1 | T], N, Byte) ->
    bits_to_bytes(T, N*2, N + Byte);
bits_to_bytes([$0 | T], N, Byte) ->
    bits_to_bytes(T, N*2, Byte);
bits_to_bytes([_BadChar | _T], _N, _Byte) ->
    throw({error, "bad character in bit string"}).

%%----------------------------------------------------------------------
%% These HEX conversion routines are stolen from module asn1_bits by 
%% klacke@erix.ericsson.se
%% I didn't want to ship the entire asn1-compiler so I used cut-and-paste.
%%----------------------------------------------------------------------
hex_to_bytes(HexNumber) when atom(HexNumber) ->
    hex_to_bytes(atom_to_list(HexNumber));

hex_to_bytes(HexNumber) ->
    case length(HexNumber) rem 2 of
	1 ->  %% Odd
	    hex_to_bytes(lists:append(HexNumber,[$0]),[]);
	0 ->  %% even
	    hex_to_bytes(HexNumber,[])
    end.

hex_to_bytes([],R) ->
    lists:reverse(R);
hex_to_bytes([Hi,Lo|Rest],Res) ->
    hex_to_bytes(Rest,[hex_to_byte(Hi,Lo)|Res]).

hex_to_four_bits(Hex) ->
    if
	Hex == $0 -> 0;
	Hex == $1 -> 1;
	Hex == $2 -> 2;
	Hex == $3 -> 3;
	Hex == $4 -> 4;
	Hex == $5 -> 5;
	Hex == $6 -> 6;
	Hex == $7 -> 7;
	Hex == $8 -> 8;
	Hex == $9 -> 9;
	Hex == $A -> 10;
	Hex == $B -> 11;
	Hex == $C -> 12;
	Hex == $D -> 13;
	Hex == $E -> 14;
	Hex == $F -> 15;
	true -> throw({error, "bad hex character"})
    end.

hex_to_byte(Hi,Lo) ->
    (hex_to_four_bits(Hi) bsl 4) bor hex_to_four_bits(Lo).

kind(DefValPart,IndexPart) ->
    case DefValPart of
	undefined ->
	    case IndexPart of
		{indexes, undefined} -> {variable, []};
		{indexes, Indexes}  ->
		    {table_entry, {indexes, Indexes}};
		{augments,Table} ->
		    {table_entry,{augments,Table}}
	    end;
	{defval, DefVal} -> {variable, [{defval, DefVal}]}
    end.    

display_hint(Val) ->
    case val(Val) of
        Str when list(Str) ->
            lists:reverse(Str);
        _ ->
            throw({error, {invalid_display_hint, Val}})
    end.

units(Val) ->
    case val(Val) of
        Str when list(Str) ->
            lists:reverse(Str);
        _ ->
            throw({error, {invalid_units, Val}})
    end.

ensure_ver(Ver, Line, What) ->
    case get(snmp_version) of
	Ver -> ok;
	_Other ->
	    snmpc_lib:print_error(
	      "~s is only allowed in SNMPv~p.",[What,Ver],Line)
    end.


ensure_ver(Ver,Token) ->
    ensure_ver(Ver,line_of(Token), atom_to_list(cat(Token))).

filter_v2imports(2,'Integer32')  -> {builtin, 'Integer32'};
filter_v2imports(2,'Counter32')  -> {builtin, 'Counter32'};
filter_v2imports(2,'Gauge32')    -> {builtin, 'Gauge32'};
filter_v2imports(2,'Unsigned32') -> {builtin, 'Unsigned32'};
filter_v2imports(2,'Counter64')  -> {builtin, 'Counter64'};
filter_v2imports(_,Type)         -> {type, Type}.
    
w(F, A) ->
    snmpc_lib:w(F, A).

%i(F, A) ->
%    io:format("~w:" ++ F ++ "~n", [?MODULE|A]).


-file("/usr/obj/ports/erlang-11b.0/otp_src_R11B-0/bootstrap/lib/parsetools/include/yeccpre.hrl", 0).
%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id $
%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The parser generator will insert appropriate declarations before this line.%

parse(Tokens) ->
    yeccpars0(Tokens, false).

parse_and_scan({F, A}) -> % Fun or {M, F}
    yeccpars0([], {F, A});
parse_and_scan({M, F, A}) ->
    yeccpars0([], {{M, F}, A}).

format_error(Message) ->
    case io_lib:deep_char_list(Message) of
	true ->
	    Message;
	_ ->
	    io_lib:write(Message)
    end.

% To be used in grammar files to throw an error message to the parser
% toplevel. Doesn't have to be exported!
-compile({nowarn_unused_function,{return_error,2}}).
return_error(Line, Message) ->
    throw({error, {Line, ?MODULE, Message}}).

yeccpars0(Tokens, MFA) ->
    try yeccpars1(Tokens, MFA, 0, [], [])
    catch 
        throw: {error, {_Line, ?MODULE, _M}} = Error -> 
                   Error % probably from return_error/1
    end.

% Don't change yeccpars1/6 too much, it is called recursively by yeccpars2/8!
yeccpars1([Token | Tokens], Tokenizer, State, States, Vstack) ->
    yeccpars2(State, element(1, Token), States, Vstack, Token, Tokens,
	      Tokenizer);
yeccpars1([], {F, A}, State, States, Vstack) ->
    case apply(F, A) of
        {ok, Tokens, _Endline} ->
	    yeccpars1(Tokens, {F, A}, State, States, Vstack);
        {eof, _Endline} ->
            yeccpars1([], false, State, States, Vstack);
        {error, Descriptor, _Endline} ->
            {error, Descriptor}
    end;
yeccpars1([], false, State, States, Vstack) ->
    yeccpars2(State, '$end', States, Vstack, {'$end', 999999}, [], false).

% For internal use only.
yeccerror(Token) ->
    {error,
     {element(2, Token), ?MODULE,
      ["syntax error before: ", yecctoken2string(Token)]}}.

yecctoken2string({atom, _, A}) -> io_lib:write(A);
yecctoken2string({integer,_,N}) -> io_lib:write(N);
yecctoken2string({float,_,F}) -> io_lib:write(F);
yecctoken2string({char,_,C}) -> io_lib:write_char(C);
yecctoken2string({var,_,V}) -> io_lib:format('~s', [V]);
yecctoken2string({string,_,S}) -> io_lib:write_string(S);
yecctoken2string({reserved_symbol, _, A}) -> io_lib:format('~w', [A]);
yecctoken2string({_Cat, _, Val}) -> io_lib:format('~w', [Val]);
yecctoken2string({'dot', _}) -> io_lib:format('~w', ['.']);
yecctoken2string({'$end', _}) ->
    [];
yecctoken2string({Other, _}) when is_atom(Other) ->
    io_lib:format('~w', [Other]);
yecctoken2string(Other) ->
    io_lib:write(Other).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



-file("./snmpc_mib_gram.erl", 445).

yeccpars2(0, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 3, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(1, 'DEFINITIONS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 4, [1 | __Ss], [__T | __Stack]);
yeccpars2(1, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(2, '$end', _, __Stack, _, _, _) ->
 {ok, hd(__Stack)};
yeccpars2(2, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(3, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_3_(__Stack),
 yeccpars2(yeccgoto(mibname, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(4, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [4 | __Ss], [__T | __Stack]);
yeccpars2(4, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(5, 'BEGIN', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 10, [5 | __Ss], [__T | __Stack]);
yeccpars2(5, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(6, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 8, [6 | __Ss], [__T | __Stack]);
yeccpars2(6, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(7, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(implies, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(8, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 9, [8 | __Ss], [__T | __Stack]);
yeccpars2(8, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(9, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_9_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(implies, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(10, 'IMPORTS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 12, [10 | __Ss], [__T | __Stack]);
yeccpars2(10, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_10_(__Stack),
 yeccpars2(11, __Cat, [10 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(11, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 68, [11 | __Ss], [__T | __Stack]);
yeccpars2(11, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 69, [11 | __Ss], [__T | __Stack]);
yeccpars2(11, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(12, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 17, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 18, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 19, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'MODULE-COMPLIANCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'MODULE-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'NOTIFICATION-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'NOTIFICATION-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'OBJECT-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'OBJECT-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'OBJECT-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 34, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 35, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 36, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 37, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 38, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 39, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TEXTUAL-CONVENTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 40, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TRAP-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 41, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 42, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [12 | __Ss], [__T | __Stack]);
yeccpars2(12, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(13, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'FROM', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(14, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 17, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 18, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 19, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'MODULE-COMPLIANCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'MODULE-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'NOTIFICATION-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'NOTIFICATION-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'OBJECT-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'OBJECT-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'OBJECT-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 34, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 35, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 36, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 37, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 38, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 39, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TEXTUAL-CONVENTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 40, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TRAP-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 41, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 42, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_14_(__Stack),
 yeccpars2(yeccgoto(imports, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(15, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [15 | __Ss], [__T | __Stack]);
yeccpars2(15, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(16, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_16_(__Stack),
 yeccpars2(yeccgoto(listofimports, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(17, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_17_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(18, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_18_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(19, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_19_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(20, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_20_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(21, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_21_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(22, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_22_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(23, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_23_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(24, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_24_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(25, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_25_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(26, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_26_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(27, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_27_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(28, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_28_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(29, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_29_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(30, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_30_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(31, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_31_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(32, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_32_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(33, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_33_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(34, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_34_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(35, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_35_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(36, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_36_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(37, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_37_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(38, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_38_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(39, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_39_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(40, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_40_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(41, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_41_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(42, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_42_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(43, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_43_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(44, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_44_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(45, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_45_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(46, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_46_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(47, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_47_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(48, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_48_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(49, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_49_(__Stack),
 yeccpars2(yeccgoto(import_stuff, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(50, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_50_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(import, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(51, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_51_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(imports, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(52, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 17, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 18, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 19, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'MODULE-COMPLIANCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'MODULE-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'NOTIFICATION-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 27, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'NOTIFICATION-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'OBJECT-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'OBJECT-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 31, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'OBJECT-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 33, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 34, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 35, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 36, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 37, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 38, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 39, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TEXTUAL-CONVENTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 40, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TRAP-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 41, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 42, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(53, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [53 | __Ss], [__T | __Stack]);
yeccpars2(53, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(54, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_54_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(imports_from_one_mib, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(55, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_55_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(listofimports, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(56, 'END', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 399, [56 | __Ss], [__T | __Stack]);
yeccpars2(56, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(57, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definition, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(58, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definition, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(59, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definition, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(60, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 228, [60 | __Ss], [__T | __Stack]);
yeccpars2(60, 'OBJECT-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 361, [60 | __Ss], [__T | __Stack]);
yeccpars2(60, 'TRAP-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 362, [60 | __Ss], [__T | __Stack]);
yeccpars2(60, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(61, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definition, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(62, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [62 | __Ss], [__T | __Stack]);
yeccpars2(62, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [62 | __Ss], [__T | __Stack]);
yeccpars2(62, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(63, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definition, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(64, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 69, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(65, 'MODULE-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 72, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(66, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [66 | __Ss], [__T | __Stack]);
yeccpars2(66, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 69, [66 | __Ss], [__T | __Stack]);
yeccpars2(66, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_66_(__Stack),
 yeccpars2(yeccgoto(v1orv2, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(67, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_67_(__Stack),
 yeccpars2(yeccgoto(listofdefinitions, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(68, 'MODULE-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = 'yeccpars2_68_MODULE-IDENTITY'(__Stack),
 yeccpars2(yeccgoto(mibid, hd(__Ss)), 'MODULE-IDENTITY', __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(68, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_68_(__Stack),
 yeccpars2(yeccgoto(objectname, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(69, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_69_(__Stack),
 yeccpars2(yeccgoto(newtypename, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(70, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_70_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(listofdefinitions, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(71, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_71_(__Stack),
 yeccpars2(yeccgoto(objectname, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(72, 'LAST-UPDATED', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 73, [72 | __Ss], [__T | __Stack]);
yeccpars2(72, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(73, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 75, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(74, 'ORGANIZATION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 76, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(75, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_75_(__Stack),
 yeccpars2(yeccgoto(last_updated, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(76, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 78, [76 | __Ss], [__T | __Stack]);
yeccpars2(76, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(77, 'CONTACT-INFO', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 79, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(78, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_78_(__Stack),
 yeccpars2(yeccgoto(oranization, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(79, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 81, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(80, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 82, [80 | __Ss], [__T | __Stack]);
yeccpars2(80, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(81, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_81_(__Stack),
 yeccpars2(yeccgoto(contact_info, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(82, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 84, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_82_(__Stack),
 yeccpars2(83, __Cat, [82 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(83, 'REVISION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 88, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_83_(__Stack),
 yeccpars2(86, __Cat, [83 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(84, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_84_(__Stack),
 yeccpars2(yeccgoto(descriptionfield, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(85, 'REVISION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 88, [85 | __Ss], [__T | __Stack]);
yeccpars2(85, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_85_(__Stack),
 yeccpars2(yeccgoto(revisionpart, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(86, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(87, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_87_(__Stack),
 yeccpars2(yeccgoto(revisions, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(88, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 90, [88 | __Ss], [__T | __Stack]);
yeccpars2(88, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(89, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 91, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(90, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_90_(__Stack),
 yeccpars2(yeccgoto(revision_string, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(91, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 93, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(92, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_92_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(revision, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(93, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_93_(__Stack),
 yeccpars2(yeccgoto(revision_desc, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(94, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_94_(__Stack),
 __Nss = lists:nthtail(11, __Ss),
 yeccpars2(yeccgoto(moduleidentity, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(95, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 96, [95 | __Ss], [__T | __Stack]);
yeccpars2(95, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(96, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(97, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(fatherobjectname, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(98, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 100, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 101, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(99, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 107, [99 | __Ss], [__T | __Stack]);
yeccpars2(99, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(100, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 103, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(101, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 100, [101 | __Ss], [__T | __Stack]);
yeccpars2(101, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 101, [101 | __Ss], [__T | __Stack]);
yeccpars2(101, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_101_(__Stack),
 yeccpars2(yeccgoto(parentintegers, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(102, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_102_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(parentintegers, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(103, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 104, [103 | __Ss], [__T | __Stack]);
yeccpars2(103, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(104, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 105, [104 | __Ss], [__T | __Stack]);
yeccpars2(104, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(105, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 100, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 101, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_105_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(parentintegers, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(106, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_106_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(parentintegers, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(107, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_107_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(nameassign, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(108, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_108_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(revisions, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(109, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(110, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(111, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(112, 'MODULE-COMPLIANCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, 'NOTIFICATION-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 226, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, 'NOTIFICATION-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 227, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 228, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, 'OBJECT-GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 229, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, 'OBJECT-IDENTITY', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, 'OBJECT-TYPE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(113, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(114, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(115, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(116, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(117, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(118, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [118 | __Ss], [__T | __Stack]);
yeccpars2(118, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [118 | __Ss], [__T | __Stack]);
yeccpars2(118, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(119, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(120, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(definitionv2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(121, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [121 | __Ss], [__T | __Stack]);
yeccpars2(121, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 69, [121 | __Ss], [__T | __Stack]);
yeccpars2(121, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_121_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(v1orv2, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(122, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_122_(__Stack),
 yeccpars2(yeccgoto(listofdefinitionsv2, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(123, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_123_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(listofdefinitionsv2, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(124, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 130, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 146, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TEXTUAL-CONVENTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 150, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [124 | __Ss], [__T | __Stack]);
yeccpars2(124, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(125, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 205, [125 | __Ss], [__T | __Stack]);
yeccpars2(125, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_125_(__Stack),
 yeccpars2(yeccgoto(syntax, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(126, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 205, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_126_(__Stack),
 yeccpars2(yeccgoto(syntax, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(127, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_127_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(newtype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(128, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_128_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(129, 'STRING', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 203, [129 | __Ss], [__T | __Stack]);
yeccpars2(129, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(130, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 182, [130 | __Ss], [__T | __Stack]);
yeccpars2(130, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(131, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_131_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(132, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_132_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(133, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_133_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(134, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_134_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(135, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 200, [135 | __Ss], [__T | __Stack]);
yeccpars2(135, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(136, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_136_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(137, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(138, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_138_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(139, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(140, 'IDENTIFIER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 199, [140 | __Ss], [__T | __Stack]);
yeccpars2(140, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(141, 'STRING', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 198, [141 | __Ss], [__T | __Stack]);
yeccpars2(141, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(142, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(143, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_143_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(144, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_144_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(145, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_145_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(146, 'OF', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 173, [146 | __Ss], [__T | __Stack]);
yeccpars2(146, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 175, [146 | __Ss], [__T | __Stack]);
yeccpars2(146, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(147, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_147_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(148, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_148_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(149, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_149_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(150, 'DISPLAY-HINT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 159, [150 | __Ss], [__T | __Stack]);
yeccpars2(150, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_150_(__Stack),
 yeccpars2(158, __Cat, [150 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(151, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_151_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(152, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_152_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(153, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_153_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(154, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(155, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_155_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(156, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_156_(__Stack),
 yeccpars2(yeccgoto(type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(157, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(usertype, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(158, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 161, [158 | __Ss], [__T | __Stack]);
yeccpars2(158, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(159, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 160, [159 | __Ss], [__T | __Stack]);
yeccpars2(159, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(160, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_160_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(displaypart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(161, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [161 | __Ss], [__T | __Stack]);
yeccpars2(161, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(162, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_162_(__Stack),
 yeccpars2(164, __Cat, [162 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(163, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_163_(__Stack),
 yeccpars2(yeccgoto(statusv2, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(164, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [164 | __Ss], [__T | __Stack]);
yeccpars2(164, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_164_(__Stack),
 yeccpars2(167, __Cat, [164 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(165, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 166, [165 | __Ss], [__T | __Stack]);
yeccpars2(165, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(166, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_166_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(description, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(167, 'SYNTAX', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 170, [167 | __Ss], [__T | __Stack]);
yeccpars2(167, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(168, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 169, [168 | __Ss], [__T | __Stack]);
yeccpars2(168, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(169, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_169_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(referpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(170, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 130, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(171, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_171_(__Stack),
 __Nss = lists:nthtail(9, __Ss),
 yeccpars2(yeccgoto(textualconvention, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(172, 'OF', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 173, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(173, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [173 | __Ss], [__T | __Stack]);
yeccpars2(173, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(174, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_174_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(syntax, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(175, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 178, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(176, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 194, [176 | __Ss], [__T | __Stack]);
yeccpars2(176, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 195, [176 | __Ss], [__T | __Stack]);
yeccpars2(176, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(177, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 181, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(178, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(fieldname, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(179, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(fsyntax, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(180, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_180_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(fields, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(181, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 182, [181 | __Ss], [__T | __Stack]);
yeccpars2(181, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_181_(__Stack),
 yeccpars2(yeccgoto(fsyntax, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(182, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 184, [182 | __Ss], [__T | __Stack]);
yeccpars2(182, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(183, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 188, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 189, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(184, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 185, [184 | __Ss], [__T | __Stack]);
yeccpars2(184, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(185, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 186, [185 | __Ss], [__T | __Stack]);
yeccpars2(185, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(186, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 187, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(187, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_187_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(namedbits, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(188, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 190, [188 | __Ss], [__T | __Stack]);
yeccpars2(188, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(189, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_189_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(syntax, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(190, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 191, [190 | __Ss], [__T | __Stack]);
yeccpars2(190, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(191, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 192, [191 | __Ss], [__T | __Stack]);
yeccpars2(191, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(192, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 193, [192 | __Ss], [__T | __Stack]);
yeccpars2(192, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(193, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_193_(__Stack),
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(namedbits, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(194, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 178, [194 | __Ss], [__T | __Stack]);
yeccpars2(194, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(195, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_195_(__Stack),
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(tableentrydefinition, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(196, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 181, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(197, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_197_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(fields, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(198, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_198_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(type, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(199, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_199_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(type, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(200, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 184, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(201, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 188, [201 | __Ss], [__T | __Stack]);
yeccpars2(201, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 202, [201 | __Ss], [__T | __Stack]);
yeccpars2(201, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(202, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_202_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(syntax, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(203, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_203_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(type, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(204, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_204_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(syntax, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(205, 'SIZE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 208, [205 | __Ss], [__T | __Stack]);
yeccpars2(205, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 209, [205 | __Ss], [__T | __Stack]);
yeccpars2(205, quote, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 210, [205 | __Ss], [__T | __Stack]);
yeccpars2(205, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(206, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [206 | __Ss], [__T | __Stack]);
yeccpars2(206, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 216, [206 | __Ss], [__T | __Stack]);
yeccpars2(206, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(207, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 219, [207 | __Ss], [__T | __Stack]);
yeccpars2(207, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_207_(__Stack),
 yeccpars2(yeccgoto(sizedescr, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(208, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 213, [208 | __Ss], [__T | __Stack]);
yeccpars2(208, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(209, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_209_(__Stack),
 yeccpars2(yeccgoto(range_num, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(210, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 211, [210 | __Ss], [__T | __Stack]);
yeccpars2(210, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 212, [210 | __Ss], [__T | __Stack]);
yeccpars2(210, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(211, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_211_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(range_num, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(212, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_212_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(range_num, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(213, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 209, [213 | __Ss], [__T | __Stack]);
yeccpars2(213, quote, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 210, [213 | __Ss], [__T | __Stack]);
yeccpars2(213, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(214, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 215, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 216, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(215, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 218, [215 | __Ss], [__T | __Stack]);
yeccpars2(215, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(216, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 209, [216 | __Ss], [__T | __Stack]);
yeccpars2(216, quote, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 210, [216 | __Ss], [__T | __Stack]);
yeccpars2(216, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(217, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 216, [217 | __Ss], [__T | __Stack]);
yeccpars2(217, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_217_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(sizedescr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(218, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_218_(__Stack),
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(size, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(219, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 220, [219 | __Ss], [__T | __Stack]);
yeccpars2(219, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(220, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 209, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, quote, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 210, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(221, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 209, [221 | __Ss], [__T | __Stack]);
yeccpars2(221, quote, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 210, [221 | __Ss], [__T | __Stack]);
yeccpars2(221, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_221_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(sizedescr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(222, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 216, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_222_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(sizedescr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(223, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_223_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(size, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(224, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_224_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(syntax, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(225, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 322, [225 | __Ss], [__T | __Stack]);
yeccpars2(225, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(226, 'NOTIFICATIONS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 313, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(227, 'OBJECTS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 292, [227 | __Ss], [__T | __Stack]);
yeccpars2(227, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_227_(__Stack),
 yeccpars2(306, __Cat, [227 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(228, 'IDENTIFIER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 304, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(229, 'OBJECTS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 292, [229 | __Ss], [__T | __Stack]);
yeccpars2(229, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_229_(__Stack),
 yeccpars2(291, __Cat, [229 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(230, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 285, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(231, 'SYNTAX', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 232, [231 | __Ss], [__T | __Stack]);
yeccpars2(231, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(232, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 130, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(233, 'UNITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 235, [233 | __Ss], [__T | __Stack]);
yeccpars2(233, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_233_(__Stack),
 yeccpars2(234, __Cat, [233 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(234, 'MAX-ACCESS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 237, [234 | __Ss], [__T | __Stack]);
yeccpars2(234, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(235, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 236, [235 | __Ss], [__T | __Stack]);
yeccpars2(235, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(236, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_236_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(unitspart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(237, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 239, [237 | __Ss], [__T | __Stack]);
yeccpars2(237, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(238, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 240, [238 | __Ss], [__T | __Stack]);
yeccpars2(238, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(239, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_239_(__Stack),
 yeccpars2(yeccgoto(accessv2, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(240, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(241, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 242, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(242, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 84, [242 | __Ss], [__T | __Stack]);
yeccpars2(242, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_242_(__Stack),
 yeccpars2(243, __Cat, [242 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(243, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [243 | __Ss], [__T | __Stack]);
yeccpars2(243, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_243_(__Stack),
 yeccpars2(244, __Cat, [243 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(244, 'AUGMENTS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 246, [244 | __Ss], [__T | __Stack]);
yeccpars2(244, 'INDEX', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 247, [244 | __Ss], [__T | __Stack]);
yeccpars2(244, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_244_(__Stack),
 yeccpars2(245, __Cat, [244 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(245, 'DEFVAL', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 263, [245 | __Ss], [__T | __Stack]);
yeccpars2(245, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_245_(__Stack),
 yeccpars2(262, __Cat, [245 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(246, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 258, [246 | __Ss], [__T | __Stack]);
yeccpars2(246, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(247, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 248, [247 | __Ss], [__T | __Stack]);
yeccpars2(247, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(248, 'IMPLIED', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 253, [248 | __Ss], [__T | __Stack]);
yeccpars2(248, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [248 | __Ss], [__T | __Stack]);
yeccpars2(248, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(249, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(index, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(250, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_250_(__Stack),
 yeccpars2(yeccgoto(indextypesv2, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(251, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 255, [251 | __Ss], [__T | __Stack]);
yeccpars2(251, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 256, [251 | __Ss], [__T | __Stack]);
yeccpars2(251, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(252, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(indextypev2, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(253, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(254, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_254_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(indextypev2, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(255, 'IMPLIED', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 253, [255 | __Ss], [__T | __Stack]);
yeccpars2(255, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [255 | __Ss], [__T | __Stack]);
yeccpars2(255, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(256, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_256_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(indexpartv2, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(257, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_257_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(indextypesv2, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(258, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [258 | __Ss], [__T | __Stack]);
yeccpars2(258, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(259, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(entry, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(260, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 261, [260 | __Ss], [__T | __Stack]);
yeccpars2(260, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(261, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_261_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(indexpartv2, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(262, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [262 | __Ss], [__T | __Stack]);
yeccpars2(262, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [262 | __Ss], [__T | __Stack]);
yeccpars2(262, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(263, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 264, [263 | __Ss], [__T | __Stack]);
yeccpars2(263, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(264, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 265, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 266, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, quote, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 267, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 268, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 269, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(265, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 283, [265 | __Ss], [__T | __Stack]);
yeccpars2(265, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(266, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 282, [266 | __Ss], [__T | __Stack]);
yeccpars2(266, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(267, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 278, [267 | __Ss], [__T | __Stack]);
yeccpars2(267, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 279, [267 | __Ss], [__T | __Stack]);
yeccpars2(267, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(268, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 277, [268 | __Ss], [__T | __Stack]);
yeccpars2(268, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(269, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 272, [269 | __Ss], [__T | __Stack]);
yeccpars2(269, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_269_(__Stack),
 yeccpars2(270, __Cat, [269 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(270, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 275, [270 | __Ss], [__T | __Stack]);
yeccpars2(270, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(271, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 273, [271 | __Ss], [__T | __Stack]);
yeccpars2(271, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(defbitsvalue, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(272, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_272_(__Stack),
 yeccpars2(yeccgoto(defbitsnames, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(273, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 274, [273 | __Ss], [__T | __Stack]);
yeccpars2(273, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(274, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_274_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(defbitsnames, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(275, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 276, [275 | __Ss], [__T | __Stack]);
yeccpars2(275, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(276, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_276_(__Stack),
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(defvalpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(277, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_277_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(defvalpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(278, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 281, [278 | __Ss], [__T | __Stack]);
yeccpars2(278, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(279, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 280, [279 | __Ss], [__T | __Stack]);
yeccpars2(279, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(280, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_280_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(defvalpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(281, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_281_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(defvalpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(282, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_282_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(defvalpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(283, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_283_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(defvalpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(284, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_284_(__Stack),
 __Nss = lists:nthtail(14, __Ss),
 yeccpars2(yeccgoto(objecttypev2, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(285, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [285 | __Ss], [__T | __Stack]);
yeccpars2(285, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(286, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 287, [286 | __Ss], [__T | __Stack]);
yeccpars2(286, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(287, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 288, [287 | __Ss], [__T | __Stack]);
yeccpars2(287, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(288, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [288 | __Ss], [__T | __Stack]);
yeccpars2(288, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_288_(__Stack),
 yeccpars2(289, __Cat, [288 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(289, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [289 | __Ss], [__T | __Stack]);
yeccpars2(289, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [289 | __Ss], [__T | __Stack]);
yeccpars2(289, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(290, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_290_(__Stack),
 __Nss = lists:nthtail(7, __Ss),
 yeccpars2(yeccgoto(objectidentity, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(291, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 299, [291 | __Ss], [__T | __Stack]);
yeccpars2(291, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(292, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 293, [292 | __Ss], [__T | __Stack]);
yeccpars2(292, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(293, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [293 | __Ss], [__T | __Stack]);
yeccpars2(293, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(294, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 296, [294 | __Ss], [__T | __Stack]);
yeccpars2(294, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 297, [294 | __Ss], [__T | __Stack]);
yeccpars2(294, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(295, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_295_(__Stack),
 yeccpars2(yeccgoto(objects, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(296, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [296 | __Ss], [__T | __Stack]);
yeccpars2(296, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(297, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_297_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(objectspart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(298, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_298_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(objects, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(299, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [299 | __Ss], [__T | __Stack]);
yeccpars2(299, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(300, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [300 | __Ss], [__T | __Stack]);
yeccpars2(300, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_300_(__Stack),
 yeccpars2(301, __Cat, [300 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(301, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [301 | __Ss], [__T | __Stack]);
yeccpars2(301, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_301_(__Stack),
 yeccpars2(302, __Cat, [301 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(302, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [302 | __Ss], [__T | __Stack]);
yeccpars2(302, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [302 | __Ss], [__T | __Stack]);
yeccpars2(302, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(303, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_303_(__Stack),
 __Nss = lists:nthtail(7, __Ss),
 yeccpars2(yeccgoto(objectgroup, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(304, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [304 | __Ss], [__T | __Stack]);
yeccpars2(304, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [304 | __Ss], [__T | __Stack]);
yeccpars2(304, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(305, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_305_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(objectidentifier, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(306, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 307, [306 | __Ss], [__T | __Stack]);
yeccpars2(306, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(307, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [307 | __Ss], [__T | __Stack]);
yeccpars2(307, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(308, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 309, [308 | __Ss], [__T | __Stack]);
yeccpars2(308, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(309, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 84, [309 | __Ss], [__T | __Stack]);
yeccpars2(309, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_309_(__Stack),
 yeccpars2(310, __Cat, [309 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(310, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [310 | __Ss], [__T | __Stack]);
yeccpars2(310, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_310_(__Stack),
 yeccpars2(311, __Cat, [310 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(311, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [311 | __Ss], [__T | __Stack]);
yeccpars2(311, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [311 | __Ss], [__T | __Stack]);
yeccpars2(311, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(312, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_312_(__Stack),
 __Nss = lists:nthtail(8, __Ss),
 yeccpars2(yeccgoto(notification, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(313, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 314, [313 | __Ss], [__T | __Stack]);
yeccpars2(313, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(314, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [314 | __Ss], [__T | __Stack]);
yeccpars2(314, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(315, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 296, [315 | __Ss], [__T | __Stack]);
yeccpars2(315, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 316, [315 | __Ss], [__T | __Stack]);
yeccpars2(315, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(316, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 317, [316 | __Ss], [__T | __Stack]);
yeccpars2(316, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(317, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [317 | __Ss], [__T | __Stack]);
yeccpars2(317, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(318, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [318 | __Ss], [__T | __Stack]);
yeccpars2(318, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_318_(__Stack),
 yeccpars2(319, __Cat, [318 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(319, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [319 | __Ss], [__T | __Stack]);
yeccpars2(319, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_319_(__Stack),
 yeccpars2(320, __Cat, [319 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(320, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [320 | __Ss], [__T | __Stack]);
yeccpars2(320, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [320 | __Ss], [__T | __Stack]);
yeccpars2(320, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(321, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_321_(__Stack),
 __Nss = lists:nthtail(10, __Ss),
 yeccpars2(yeccgoto(notificationgroup, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(322, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [322 | __Ss], [__T | __Stack]);
yeccpars2(322, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(323, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [323 | __Ss], [__T | __Stack]);
yeccpars2(323, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_323_(__Stack),
 yeccpars2(324, __Cat, [323 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(324, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [324 | __Ss], [__T | __Stack]);
yeccpars2(324, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_324_(__Stack),
 yeccpars2(325, __Cat, [324 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(325, 'MODULE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 329, [325 | __Ss], [__T | __Stack]);
yeccpars2(325, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_325_(__Stack),
 yeccpars2(327, __Cat, [325 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(326, 'MODULE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 329, [326 | __Ss], [__T | __Stack]);
yeccpars2(326, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_326_(__Stack),
 yeccpars2(yeccgoto(modulepart, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(327, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [327 | __Ss], [__T | __Stack]);
yeccpars2(327, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [327 | __Ss], [__T | __Stack]);
yeccpars2(327, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(328, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_328_(__Stack),
 yeccpars2(yeccgoto(modules, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(329, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 3, [329 | __Ss], [__T | __Stack]);
yeccpars2(329, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_329_(__Stack),
 yeccpars2(330, __Cat, [329 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(330, 'MANDATORY-GROUPS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 333, [330 | __Ss], [__T | __Stack]);
yeccpars2(330, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_330_(__Stack),
 yeccpars2(332, __Cat, [330 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(331, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_331_(__Stack),
 yeccpars2(yeccgoto(modulenamepart, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(332, 'GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 342, [332 | __Ss], [__T | __Stack]);
yeccpars2(332, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 343, [332 | __Ss], [__T | __Stack]);
yeccpars2(332, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_332_(__Stack),
 yeccpars2(339, __Cat, [332 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(333, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 334, [333 | __Ss], [__T | __Stack]);
yeccpars2(333, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(334, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [334 | __Ss], [__T | __Stack]);
yeccpars2(334, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(335, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 296, [335 | __Ss], [__T | __Stack]);
yeccpars2(335, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 336, [335 | __Ss], [__T | __Stack]);
yeccpars2(335, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(336, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_336_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(mandatorypart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(337, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_337_(__Stack),
 yeccpars2(yeccgoto(compliance, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(338, 'GROUP', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 342, [338 | __Ss], [__T | __Stack]);
yeccpars2(338, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 343, [338 | __Ss], [__T | __Stack]);
yeccpars2(338, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_338_(__Stack),
 yeccpars2(yeccgoto(compliancepart, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(339, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_339_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(module, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(340, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_340_(__Stack),
 yeccpars2(yeccgoto(compliance, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(341, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_341_(__Stack),
 yeccpars2(yeccgoto(compliances, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(342, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [342 | __Ss], [__T | __Stack]);
yeccpars2(342, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(343, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [343 | __Ss], [__T | __Stack]);
yeccpars2(343, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(344, 'SYNTAX', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 346, [344 | __Ss], [__T | __Stack]);
yeccpars2(344, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_344_(__Stack),
 yeccpars2(345, __Cat, [344 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(345, 'WRITE-SYNTAX', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 349, [345 | __Ss], [__T | __Stack]);
yeccpars2(345, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_345_(__Stack),
 yeccpars2(348, __Cat, [345 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(346, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 130, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [346 | __Ss], [__T | __Stack]);
yeccpars2(346, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(347, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_347_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(syntaxpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(348, 'MIN-ACCESS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 352, [348 | __Ss], [__T | __Stack]);
yeccpars2(348, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_348_(__Stack),
 yeccpars2(351, __Cat, [348 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(349, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 130, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [349 | __Ss], [__T | __Stack]);
yeccpars2(349, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(350, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_350_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(writesyntaxpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(351, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [351 | __Ss], [__T | __Stack]);
yeccpars2(351, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_351_(__Stack),
 yeccpars2(354, __Cat, [351 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(352, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 239, [352 | __Ss], [__T | __Stack]);
yeccpars2(352, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(353, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_353_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(accesspart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(354, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_354_(__Stack),
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(object, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(355, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [355 | __Ss], [__T | __Stack]);
yeccpars2(355, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_355_(__Stack),
 yeccpars2(356, __Cat, [355 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(356, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_356_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(compliancegroup, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(357, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_357_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(compliances, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(358, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_358_(__Stack),
 __Nss = lists:nthtail(7, __Ss),
 yeccpars2(yeccgoto(modulecompliance, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(359, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_359_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(modules, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(360, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 130, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 146, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [360 | __Ss], [__T | __Stack]);
yeccpars2(360, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(361, 'SYNTAX', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 377, [361 | __Ss], [__T | __Stack]);
yeccpars2(361, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(362, 'ENTERPRISE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 363, [362 | __Ss], [__T | __Stack]);
yeccpars2(362, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(363, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [363 | __Ss], [__T | __Stack]);
yeccpars2(363, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(364, 'VARIABLES', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 366, [364 | __Ss], [__T | __Stack]);
yeccpars2(364, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_364_(__Stack),
 yeccpars2(365, __Cat, [364 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(365, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [365 | __Ss], [__T | __Stack]);
yeccpars2(365, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_365_(__Stack),
 yeccpars2(373, __Cat, [365 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(366, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 367, [366 | __Ss], [__T | __Stack]);
yeccpars2(366, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(367, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [367 | __Ss], [__T | __Stack]);
yeccpars2(367, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(368, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 370, [368 | __Ss], [__T | __Stack]);
yeccpars2(368, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 371, [368 | __Ss], [__T | __Stack]);
yeccpars2(368, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(369, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_369_(__Stack),
 yeccpars2(yeccgoto(variables, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(370, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [370 | __Ss], [__T | __Stack]);
yeccpars2(370, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(371, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_371_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(varpart, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(372, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_372_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(variables, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(373, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [373 | __Ss], [__T | __Stack]);
yeccpars2(373, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_373_(__Stack),
 yeccpars2(374, __Cat, [373 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(374, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [374 | __Ss], [__T | __Stack]);
yeccpars2(374, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [374 | __Ss], [__T | __Stack]);
yeccpars2(374, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(375, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 376, [375 | __Ss], [__T | __Stack]);
yeccpars2(375, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(376, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_376_(__Stack),
 __Nss = lists:nthtail(8, __Ss),
 yeccpars2(yeccgoto(traptype, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(377, 'AutonomousType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'BIT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'BITS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 130, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'Counter', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 131, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'DateAndTime', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'DisplayString', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'Gauge', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'INTEGER', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 135, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'InstancePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'IpAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'MacAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'NetworkAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'OBJECT', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'OCTET', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'Opaque', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'PhysAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'RowPointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'RowStatus', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'SEQUENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'StorageType', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'TAddress', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'TDomain', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'TestAndIncr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'TimeInterval', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'TimeStamp', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'TimeTicks', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'TruthValue', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, 'VariablePointer', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 156, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, variable, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [377 | __Ss], [__T | __Stack]);
yeccpars2(377, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(378, 'ACCESS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 379, [378 | __Ss], [__T | __Stack]);
yeccpars2(378, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(379, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 381, [379 | __Ss], [__T | __Stack]);
yeccpars2(379, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(380, 'STATUS', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 382, [380 | __Ss], [__T | __Stack]);
yeccpars2(380, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(381, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_381_(__Stack),
 yeccpars2(yeccgoto(accessv1, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(382, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 384, [382 | __Ss], [__T | __Stack]);
yeccpars2(382, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(383, 'DESCRIPTION', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 385, [383 | __Ss], [__T | __Stack]);
yeccpars2(383, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(384, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_384_(__Stack),
 yeccpars2(yeccgoto(statusv1, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(385, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 84, [385 | __Ss], [__T | __Stack]);
yeccpars2(385, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_385_(__Stack),
 yeccpars2(386, __Cat, [385 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(386, 'REFERENCE', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [386 | __Ss], [__T | __Stack]);
yeccpars2(386, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_386_(__Stack),
 yeccpars2(387, __Cat, [386 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(387, 'INDEX', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 389, [387 | __Ss], [__T | __Stack]);
yeccpars2(387, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_387_(__Stack),
 yeccpars2(388, __Cat, [387 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(388, 'DEFVAL', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 263, [388 | __Ss], [__T | __Stack]);
yeccpars2(388, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_388_(__Stack),
 yeccpars2(397, __Cat, [388 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(389, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 390, [389 | __Ss], [__T | __Stack]);
yeccpars2(389, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(390, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [390 | __Ss], [__T | __Stack]);
yeccpars2(390, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(391, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_391_(__Stack),
 yeccpars2(yeccgoto(indextypesv1, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(392, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 394, [392 | __Ss], [__T | __Stack]);
yeccpars2(392, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 395, [392 | __Ss], [__T | __Stack]);
yeccpars2(392, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(393, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(indextypev1, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(394, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 71, [394 | __Ss], [__T | __Stack]);
yeccpars2(394, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(395, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_395_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(indexpartv1, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(396, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_396_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(indextypesv1, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(397, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 6, [397 | __Ss], [__T | __Stack]);
yeccpars2(397, '::=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 7, [397 | __Ss], [__T | __Stack]);
yeccpars2(397, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(398, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_398_(__Stack),
 __Nss = lists:nthtail(13, __Ss),
 yeccpars2(yeccgoto(objecttypev1, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(399, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_399_(__Stack),
 __Nss = lists:nthtail(6, __Ss),
 yeccpars2(yeccgoto(mib, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(__Other, _, _, _, _, _, _) ->
 erlang:error({yecc_bug,"1.1",{missing_state_in_action_table, __Other}}).

yeccgoto(accesspart, 348) ->
 351;
yeccgoto(accessv1, 379) ->
 380;
yeccgoto(accessv2, 237) ->
 238;
yeccgoto(accessv2, 352) ->
 353;
yeccgoto(compliance, 332) ->
 341;
yeccgoto(compliance, 338) ->
 357;
yeccgoto(compliancegroup, 332) ->
 340;
yeccgoto(compliancegroup, 338) ->
 340;
yeccgoto(compliancepart, 332) ->
 339;
yeccgoto(compliances, 332) ->
 338;
yeccgoto(contact_info, 79) ->
 80;
yeccgoto(defbitsnames, 269) ->
 271;
yeccgoto(defbitsvalue, 269) ->
 270;
yeccgoto(definition, 11) ->
 67;
yeccgoto(definition, 66) ->
 70;
yeccgoto(definitionv2, 64) ->
 122;
yeccgoto(definitionv2, 121) ->
 123;
yeccgoto(defvalpart, 245) ->
 262;
yeccgoto(defvalpart, 388) ->
 397;
yeccgoto(description, 162) ->
 164;
yeccgoto(description, 300) ->
 301;
yeccgoto(description, 318) ->
 319;
yeccgoto(description, 323) ->
 324;
yeccgoto(description, 351) ->
 354;
yeccgoto(description, 355) ->
 356;
yeccgoto(description, 365) ->
 373;
yeccgoto(descriptionfield, 82) ->
 83;
yeccgoto(descriptionfield, 242) ->
 243;
yeccgoto(descriptionfield, 309) ->
 310;
yeccgoto(descriptionfield, 385) ->
 386;
yeccgoto(displaypart, 150) ->
 158;
yeccgoto(entry, 258) ->
 260;
yeccgoto(fatherobjectname, 96) ->
 98;
yeccgoto(fieldname, 175) ->
 177;
yeccgoto(fieldname, 194) ->
 196;
yeccgoto(fields, 175) ->
 176;
yeccgoto(fsyntax, 177) ->
 180;
yeccgoto(fsyntax, 196) ->
 197;
yeccgoto(implies, 4) ->
 5;
yeccgoto(implies, 62) ->
 360;
yeccgoto(implies, 86) ->
 95;
yeccgoto(implies, 118) ->
 124;
yeccgoto(implies, 262) ->
 95;
yeccgoto(implies, 289) ->
 95;
yeccgoto(implies, 302) ->
 95;
yeccgoto(implies, 304) ->
 95;
yeccgoto(implies, 311) ->
 95;
yeccgoto(implies, 320) ->
 95;
yeccgoto(implies, 327) ->
 95;
yeccgoto(implies, 374) ->
 375;
yeccgoto(implies, 397) ->
 95;
yeccgoto(import, 10) ->
 11;
yeccgoto(import_stuff, 12) ->
 16;
yeccgoto(import_stuff, 14) ->
 16;
yeccgoto(import_stuff, 52) ->
 55;
yeccgoto(imports, 12) ->
 15;
yeccgoto(imports, 14) ->
 51;
yeccgoto(imports_from_one_mib, 12) ->
 14;
yeccgoto(imports_from_one_mib, 14) ->
 14;
yeccgoto(index, 248) ->
 252;
yeccgoto(index, 253) ->
 254;
yeccgoto(index, 255) ->
 252;
yeccgoto(index, 390) ->
 393;
yeccgoto(index, 394) ->
 393;
yeccgoto(indexpartv1, 387) ->
 388;
yeccgoto(indexpartv2, 244) ->
 245;
yeccgoto(indextypesv1, 390) ->
 392;
yeccgoto(indextypesv2, 248) ->
 251;
yeccgoto(indextypev1, 390) ->
 391;
yeccgoto(indextypev1, 394) ->
 396;
yeccgoto(indextypev2, 248) ->
 250;
yeccgoto(indextypev2, 255) ->
 257;
yeccgoto(last_updated, 73) ->
 74;
yeccgoto(listofdefinitions, 11) ->
 66;
yeccgoto(listofdefinitionsv2, 64) ->
 121;
yeccgoto(listofimports, 12) ->
 13;
yeccgoto(listofimports, 14) ->
 13;
yeccgoto(mandatorypart, 330) ->
 332;
yeccgoto(mib, 0) ->
 2;
yeccgoto(mibid, 11) ->
 65;
yeccgoto(mibname, 0) ->
 1;
yeccgoto(mibname, 329) ->
 331;
yeccgoto(module, 325) ->
 328;
yeccgoto(module, 326) ->
 359;
yeccgoto(modulecompliance, 64) ->
 120;
yeccgoto(modulecompliance, 121) ->
 120;
yeccgoto(moduleidentity, 11) ->
 64;
yeccgoto(modulenamepart, 329) ->
 330;
yeccgoto(modulepart, 325) ->
 327;
yeccgoto(modules, 325) ->
 326;
yeccgoto(nameassign, 86) ->
 94;
yeccgoto(nameassign, 262) ->
 284;
yeccgoto(nameassign, 289) ->
 290;
yeccgoto(nameassign, 302) ->
 303;
yeccgoto(nameassign, 304) ->
 305;
yeccgoto(nameassign, 311) ->
 312;
yeccgoto(nameassign, 320) ->
 321;
yeccgoto(nameassign, 327) ->
 358;
yeccgoto(nameassign, 397) ->
 398;
yeccgoto(namedbits, 182) ->
 183;
yeccgoto(namedbits, 200) ->
 201;
yeccgoto(newtype, 11) ->
 63;
yeccgoto(newtype, 64) ->
 119;
yeccgoto(newtype, 66) ->
 63;
yeccgoto(newtype, 121) ->
 119;
yeccgoto(newtypename, 11) ->
 62;
yeccgoto(newtypename, 64) ->
 118;
yeccgoto(newtypename, 66) ->
 62;
yeccgoto(newtypename, 121) ->
 118;
yeccgoto(notification, 64) ->
 117;
yeccgoto(notification, 121) ->
 117;
yeccgoto(notificationgroup, 64) ->
 116;
yeccgoto(notificationgroup, 121) ->
 116;
yeccgoto(object, 332) ->
 337;
yeccgoto(object, 338) ->
 337;
yeccgoto(objectgroup, 64) ->
 115;
yeccgoto(objectgroup, 121) ->
 115;
yeccgoto(objectidentifier, 11) ->
 61;
yeccgoto(objectidentifier, 64) ->
 114;
yeccgoto(objectidentifier, 66) ->
 61;
yeccgoto(objectidentifier, 121) ->
 114;
yeccgoto(objectidentity, 64) ->
 113;
yeccgoto(objectidentity, 121) ->
 113;
yeccgoto(objectname, 11) ->
 60;
yeccgoto(objectname, 64) ->
 112;
yeccgoto(objectname, 66) ->
 60;
yeccgoto(objectname, 96) ->
 97;
yeccgoto(objectname, 121) ->
 112;
yeccgoto(objectname, 248) ->
 249;
yeccgoto(objectname, 253) ->
 249;
yeccgoto(objectname, 255) ->
 249;
yeccgoto(objectname, 258) ->
 259;
yeccgoto(objectname, 293) ->
 295;
yeccgoto(objectname, 296) ->
 298;
yeccgoto(objectname, 314) ->
 295;
yeccgoto(objectname, 334) ->
 295;
yeccgoto(objectname, 342) ->
 355;
yeccgoto(objectname, 343) ->
 344;
yeccgoto(objectname, 363) ->
 364;
yeccgoto(objectname, 367) ->
 369;
yeccgoto(objectname, 370) ->
 372;
yeccgoto(objectname, 390) ->
 249;
yeccgoto(objectname, 394) ->
 249;
yeccgoto(objects, 293) ->
 294;
yeccgoto(objects, 314) ->
 315;
yeccgoto(objects, 334) ->
 335;
yeccgoto(objectspart, 227) ->
 306;
yeccgoto(objectspart, 229) ->
 291;
yeccgoto(objecttypev1, 11) ->
 59;
yeccgoto(objecttypev1, 66) ->
 59;
yeccgoto(objecttypev2, 64) ->
 111;
yeccgoto(objecttypev2, 121) ->
 111;
yeccgoto(oranization, 76) ->
 77;
yeccgoto(parentintegers, 98) ->
 99;
yeccgoto(parentintegers, 101) ->
 102;
yeccgoto(parentintegers, 105) ->
 106;
yeccgoto(range_num, 205) ->
 207;
yeccgoto(range_num, 213) ->
 207;
yeccgoto(range_num, 216) ->
 207;
yeccgoto(range_num, 220) ->
 221;
yeccgoto(range_num, 221) ->
 207;
yeccgoto(referpart, 164) ->
 167;
yeccgoto(referpart, 243) ->
 244;
yeccgoto(referpart, 288) ->
 289;
yeccgoto(referpart, 301) ->
 302;
yeccgoto(referpart, 310) ->
 311;
yeccgoto(referpart, 319) ->
 320;
yeccgoto(referpart, 324) ->
 325;
yeccgoto(referpart, 373) ->
 374;
yeccgoto(referpart, 386) ->
 387;
yeccgoto(revision, 83) ->
 87;
yeccgoto(revision, 85) ->
 108;
yeccgoto(revision_desc, 91) ->
 92;
yeccgoto(revision_string, 88) ->
 89;
yeccgoto(revisionpart, 83) ->
 86;
yeccgoto(revisions, 83) ->
 85;
yeccgoto(size, 125) ->
 224;
yeccgoto(size, 126) ->
 204;
yeccgoto(sizedescr, 205) ->
 206;
yeccgoto(sizedescr, 213) ->
 214;
yeccgoto(sizedescr, 216) ->
 217;
yeccgoto(sizedescr, 221) ->
 222;
yeccgoto(statusv1, 382) ->
 383;
yeccgoto(statusv2, 161) ->
 162;
yeccgoto(statusv2, 240) ->
 241;
yeccgoto(statusv2, 285) ->
 286;
yeccgoto(statusv2, 299) ->
 300;
yeccgoto(statusv2, 307) ->
 308;
yeccgoto(statusv2, 317) ->
 318;
yeccgoto(statusv2, 322) ->
 323;
yeccgoto(syntax, 124) ->
 127;
yeccgoto(syntax, 170) ->
 171;
yeccgoto(syntax, 177) ->
 179;
yeccgoto(syntax, 196) ->
 179;
yeccgoto(syntax, 232) ->
 233;
yeccgoto(syntax, 346) ->
 347;
yeccgoto(syntax, 349) ->
 350;
yeccgoto(syntax, 360) ->
 127;
yeccgoto(syntax, 377) ->
 378;
yeccgoto(syntaxpart, 344) ->
 345;
yeccgoto(tableentrydefinition, 11) ->
 58;
yeccgoto(tableentrydefinition, 64) ->
 110;
yeccgoto(tableentrydefinition, 66) ->
 58;
yeccgoto(tableentrydefinition, 121) ->
 110;
yeccgoto(textualconvention, 64) ->
 109;
yeccgoto(textualconvention, 121) ->
 109;
yeccgoto(traptype, 11) ->
 57;
yeccgoto(traptype, 66) ->
 57;
yeccgoto(type, 124) ->
 126;
yeccgoto(type, 170) ->
 126;
yeccgoto(type, 177) ->
 126;
yeccgoto(type, 196) ->
 126;
yeccgoto(type, 232) ->
 126;
yeccgoto(type, 346) ->
 126;
yeccgoto(type, 349) ->
 126;
yeccgoto(type, 360) ->
 126;
yeccgoto(type, 377) ->
 126;
yeccgoto(unitspart, 233) ->
 234;
yeccgoto(usertype, 124) ->
 125;
yeccgoto(usertype, 170) ->
 125;
yeccgoto(usertype, 173) ->
 174;
yeccgoto(usertype, 177) ->
 125;
yeccgoto(usertype, 196) ->
 125;
yeccgoto(usertype, 232) ->
 125;
yeccgoto(usertype, 346) ->
 125;
yeccgoto(usertype, 349) ->
 125;
yeccgoto(usertype, 360) ->
 125;
yeccgoto(usertype, 377) ->
 125;
yeccgoto(v1orv2, 11) ->
 56;
yeccgoto(variables, 367) ->
 368;
yeccgoto(varpart, 364) ->
 365;
yeccgoto(writesyntaxpart, 345) ->
 348;
yeccgoto(__Symbol, __State) ->
 erlang:error({yecc_bug,"1.1",{__Symbol, __State, missing_in_goto_table}}).

-compile({inline,{yeccpars2_3_,1}}).
-file("snmpc_mib_gram.yrl", 453).
yeccpars2_3_([__1 | __Stack]) ->
 [begin
   val ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_9_,1}}).
-file("snmpc_mib_gram.yrl", 407).
yeccpars2_9_([__3,__2,__1 | __Stack]) ->
 [begin
   w ( "Sloppy asignment on line ~p" , [ line_of ( __1 ) ] ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_10_,1}}).
-file("snmpc_mib_gram.yrl", 225).
yeccpars2_10_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_14_,1}}).
-file("snmpc_mib_gram.yrl", 228).
yeccpars2_14_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_16_,1}}).
-file("snmpc_mib_gram.yrl", 234).
yeccpars2_16_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_17_,1}}).
-file("snmpc_mib_gram.yrl", 273).
yeccpars2_17_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'AutonomousType' }
  end | __Stack].

-compile({inline,{yeccpars2_18_,1}}).
-file("snmpc_mib_gram.yrl", 242).
yeccpars2_18_([__1 | __Stack]) ->
 [begin
   { builtin , 'Counter' }
  end | __Stack].

-compile({inline,{yeccpars2_19_,1}}).
-file("snmpc_mib_gram.yrl", 287).
yeccpars2_19_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'DateAndTime' }
  end | __Stack].

-compile({inline,{yeccpars2_20_,1}}).
-file("snmpc_mib_gram.yrl", 263).
yeccpars2_20_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'DisplayString' }
  end | __Stack].

-compile({inline,{yeccpars2_21_,1}}).
-file("snmpc_mib_gram.yrl", 243).
yeccpars2_21_([__1 | __Stack]) ->
 [begin
   { builtin , 'Gauge' }
  end | __Stack].

-compile({inline,{yeccpars2_22_,1}}).
-file("snmpc_mib_gram.yrl", 275).
yeccpars2_22_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'InstancePointer' }
  end | __Stack].

-compile({inline,{yeccpars2_23_,1}}).
-file("snmpc_mib_gram.yrl", 241).
yeccpars2_23_([__1 | __Stack]) ->
 [begin
   { builtin , 'IpAddress' }
  end | __Stack].

-compile({inline,{yeccpars2_24_,1}}).
-file("snmpc_mib_gram.yrl", 253).
yeccpars2_24_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'MODULE-COMPLIANCE' }
  end | __Stack].

-compile({inline,{yeccpars2_25_,1}}).
-file("snmpc_mib_gram.yrl", 249).
yeccpars2_25_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'MODULE-IDENTITY' }
  end | __Stack].

-compile({inline,{yeccpars2_26_,1}}).
-file("snmpc_mib_gram.yrl", 267).
yeccpars2_26_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'MacAddress' }
  end | __Stack].

-compile({inline,{yeccpars2_27_,1}}).
-file("snmpc_mib_gram.yrl", 255).
yeccpars2_27_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'NOTIFICATION-GROUP' }
  end | __Stack].

-compile({inline,{yeccpars2_28_,1}}).
-file("snmpc_mib_gram.yrl", 251).
yeccpars2_28_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'NOTIFICATION-TYPE' }
  end | __Stack].

-compile({inline,{yeccpars2_29_,1}}).
-file("snmpc_mib_gram.yrl", 239).
yeccpars2_29_([__1 | __Stack]) ->
 [begin
   { builtin , 'NetworkAddress' }
  end | __Stack].

-compile({inline,{yeccpars2_30_,1}}).
-file("snmpc_mib_gram.yrl", 257).
yeccpars2_30_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'OBJECT-GROUP' }
  end | __Stack].

-compile({inline,{yeccpars2_31_,1}}).
-file("snmpc_mib_gram.yrl", 259).
yeccpars2_31_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'OBJECT-IDENTITY' }
  end | __Stack].

-compile({inline,{yeccpars2_32_,1}}).
-file("snmpc_mib_gram.yrl", 237).
yeccpars2_32_([__1 | __Stack]) ->
 [begin
   { builtin , 'OBJECT-TYPE' }
  end | __Stack].

-compile({inline,{yeccpars2_33_,1}}).
-file("snmpc_mib_gram.yrl", 244).
yeccpars2_33_([__1 | __Stack]) ->
 [begin
   { builtin , 'Opaque' }
  end | __Stack].

-compile({inline,{yeccpars2_34_,1}}).
-file("snmpc_mib_gram.yrl", 265).
yeccpars2_34_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'PhysAddress' }
  end | __Stack].

-compile({inline,{yeccpars2_35_,1}}).
-file("snmpc_mib_gram.yrl", 279).
yeccpars2_35_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'RowPointer' }
  end | __Stack].

-compile({inline,{yeccpars2_36_,1}}).
-file("snmpc_mib_gram.yrl", 281).
yeccpars2_36_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'RowStatus' }
  end | __Stack].

-compile({inline,{yeccpars2_37_,1}}).
-file("snmpc_mib_gram.yrl", 289).
yeccpars2_37_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'StorageType' }
  end | __Stack].

-compile({inline,{yeccpars2_38_,1}}).
-file("snmpc_mib_gram.yrl", 293).
yeccpars2_38_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'TAddress' }
  end | __Stack].

-compile({inline,{yeccpars2_39_,1}}).
-file("snmpc_mib_gram.yrl", 291).
yeccpars2_39_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'TDomain' }
  end | __Stack].

-compile({inline,{yeccpars2_40_,1}}).
-file("snmpc_mib_gram.yrl", 261).
yeccpars2_40_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'TEXTUAL-CONVENTION' }
  end | __Stack].

-compile({inline,{yeccpars2_41_,1}}).
-file("snmpc_mib_gram.yrl", 238).
yeccpars2_41_([__1 | __Stack]) ->
 [begin
   { builtin , 'TRAP-TYPE' }
  end | __Stack].

-compile({inline,{yeccpars2_42_,1}}).
-file("snmpc_mib_gram.yrl", 271).
yeccpars2_42_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'TestAndIncr' }
  end | __Stack].

-compile({inline,{yeccpars2_43_,1}}).
-file("snmpc_mib_gram.yrl", 285).
yeccpars2_43_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'TimeInterval' }
  end | __Stack].

-compile({inline,{yeccpars2_44_,1}}).
-file("snmpc_mib_gram.yrl", 283).
yeccpars2_44_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'TimeStamp' }
  end | __Stack].

-compile({inline,{yeccpars2_45_,1}}).
-file("snmpc_mib_gram.yrl", 240).
yeccpars2_45_([__1 | __Stack]) ->
 [begin
   { builtin , 'TimeTicks' }
  end | __Stack].

-compile({inline,{yeccpars2_46_,1}}).
-file("snmpc_mib_gram.yrl", 269).
yeccpars2_46_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'TruthValue' }
  end | __Stack].

-compile({inline,{yeccpars2_47_,1}}).
-file("snmpc_mib_gram.yrl", 277).
yeccpars2_47_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , { builtin , 'VariablePointer' }
  end | __Stack].

-compile({inline,{yeccpars2_48_,1}}).
-file("snmpc_mib_gram.yrl", 246).
yeccpars2_48_([__1 | __Stack]) ->
 [begin
   { node , val ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_49_,1}}).
-file("snmpc_mib_gram.yrl", 245).
yeccpars2_49_([__1 | __Stack]) ->
 [begin
   filter_v2imports ( get ( snmp_version ) , val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_50_,1}}).
-file("snmpc_mib_gram.yrl", 226).
yeccpars2_50_([__3,__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_51_,1}}).
-file("snmpc_mib_gram.yrl", 229).
yeccpars2_51_([__2,__1 | __Stack]) ->
 [begin
   [ __1 | __2 ]
  end | __Stack].

-compile({inline,{yeccpars2_54_,1}}).
-file("snmpc_mib_gram.yrl", 232).
yeccpars2_54_([__3,__2,__1 | __Stack]) ->
 [begin
   { { val ( __3 ) , lists : reverse ( __1 ) } , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_55_,1}}).
-file("snmpc_mib_gram.yrl", 235).
yeccpars2_55_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_66_,1}}).
-file("snmpc_mib_gram.yrl", 214).
yeccpars2_66_([__1 | __Stack]) ->
 [begin
   { v1_mib , lists : reverse ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_67_,1}}).
-file("snmpc_mib_gram.yrl", 222).
yeccpars2_67_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{'yeccpars2_68_MODULE-IDENTITY',1}}).
-file("snmpc_mib_gram.yrl", 478).
'yeccpars2_68_MODULE-IDENTITY'([__1 | __Stack]) ->
 [begin
   val ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_68_,1}}).
-file("snmpc_mib_gram.yrl", 452).
yeccpars2_68_([__1 | __Stack]) ->
 [begin
   val ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_69_,1}}).
-file("snmpc_mib_gram.yrl", 455).
yeccpars2_69_([__1 | __Stack]) ->
 [begin
   val ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_70_,1}}).
-file("snmpc_mib_gram.yrl", 223).
yeccpars2_70_([__2,__1 | __Stack]) ->
 [begin
   [ __2 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_71_,1}}).
-file("snmpc_mib_gram.yrl", 452).
yeccpars2_71_([__1 | __Stack]) ->
 [begin
   val ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_75_,1}}).
-file("snmpc_mib_gram.yrl", 479).
yeccpars2_75_([__1 | __Stack]) ->
 [begin
   lists : reverse ( val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_78_,1}}).
-file("snmpc_mib_gram.yrl", 480).
yeccpars2_78_([__1 | __Stack]) ->
 [begin
   lists : reverse ( val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_81_,1}}).
-file("snmpc_mib_gram.yrl", 481).
yeccpars2_81_([__1 | __Stack]) ->
 [begin
   lists : reverse ( val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_82_,1}}).
-file("snmpc_mib_gram.yrl", 409).
yeccpars2_82_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_83_,1}}).
-file("snmpc_mib_gram.yrl", 483).
yeccpars2_83_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_84_,1}}).
-file("snmpc_mib_gram.yrl", 408).
yeccpars2_84_([__1 | __Stack]) ->
 [begin
   lists : reverse ( val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_85_,1}}).
-file("snmpc_mib_gram.yrl", 484).
yeccpars2_85_([__1 | __Stack]) ->
 [begin
   lists : reverse ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_87_,1}}).
-file("snmpc_mib_gram.yrl", 486).
yeccpars2_87_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_90_,1}}).
-file("snmpc_mib_gram.yrl", 491).
yeccpars2_90_([__1 | __Stack]) ->
 [begin
   lists : reverse ( val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_92_,1}}).
-file("snmpc_mib_gram.yrl", 489).
yeccpars2_92_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   make_revision ( __2 , __4 )
  end | __Stack].

-compile({inline,{yeccpars2_93_,1}}).
-file("snmpc_mib_gram.yrl", 492).
yeccpars2_93_([__1 | __Stack]) ->
 [begin
   lists : reverse ( val ( __1 ) )
  end | __Stack].

-compile({inline,{yeccpars2_94_,1}}).
-file("snmpc_mib_gram.yrl", 474).
yeccpars2_94_([__12,__11,__10,__9,__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   MI = make_module_identity ( __1 , __4 , __6 , __8 ,
    __10 , __11 , __12 ) ,
    { MI , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_101_,1}}).
-file("snmpc_mib_gram.yrl", 428).
yeccpars2_101_([__1 | __Stack]) ->
 [begin
   [ val ( __1 ) ]
  end | __Stack].

-compile({inline,{yeccpars2_102_,1}}).
-file("snmpc_mib_gram.yrl", 430).
yeccpars2_102_([__2,__1 | __Stack]) ->
 [begin
   [ val ( __1 ) | __2 ]
  end | __Stack].

-compile({inline,{yeccpars2_105_,1}}).
-file("snmpc_mib_gram.yrl", 429).
yeccpars2_105_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   [ val ( __3 ) ]
  end | __Stack].

-compile({inline,{yeccpars2_106_,1}}).
-file("snmpc_mib_gram.yrl", 431).
yeccpars2_106_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   [ val ( __3 ) | __5 ]
  end | __Stack].

-compile({inline,{yeccpars2_107_,1}}).
-file("snmpc_mib_gram.yrl", 399).
yeccpars2_107_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { __3 , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_108_,1}}).
-file("snmpc_mib_gram.yrl", 487).
yeccpars2_108_([__2,__1 | __Stack]) ->
 [begin
   [ __2 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_121_,1}}).
-file("snmpc_mib_gram.yrl", 213).
yeccpars2_121_([__2,__1 | __Stack]) ->
 [begin
   { v2_mib , [ __1 | lists : reverse ( __2 ) ] }
  end | __Stack].

-compile({inline,{yeccpars2_122_,1}}).
-file("snmpc_mib_gram.yrl", 505).
yeccpars2_122_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_123_,1}}).
-file("snmpc_mib_gram.yrl", 506).
yeccpars2_123_([__2,__1 | __Stack]) ->
 [begin
   [ __2 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_125_,1}}).
-file("snmpc_mib_gram.yrl", 339).
yeccpars2_125_([__1 | __Stack]) ->
 [begin
   { { type , val ( __1 ) } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_126_,1}}).
-file("snmpc_mib_gram.yrl", 340).
yeccpars2_126_([__1 | __Stack]) ->
 [begin
   { { type , cat ( __1 ) } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_127_,1}}).
-file("snmpc_mib_gram.yrl", 321).
yeccpars2_127_([__3,__2,__1 | __Stack]) ->
 [begin
   NT = make_new_type ( __1 , dummy , __3 ) ,
    { NT , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_128_,1}}).
-file("snmpc_mib_gram.yrl", 385).
yeccpars2_128_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_131_,1}}).
-file("snmpc_mib_gram.yrl", 376).
yeccpars2_131_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 1 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_132_,1}}).
-file("snmpc_mib_gram.yrl", 392).
yeccpars2_132_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_133_,1}}).
-file("snmpc_mib_gram.yrl", 380).
yeccpars2_133_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_134_,1}}).
-file("snmpc_mib_gram.yrl", 377).
yeccpars2_134_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 1 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_136_,1}}).
-file("snmpc_mib_gram.yrl", 386).
yeccpars2_136_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_138_,1}}).
-file("snmpc_mib_gram.yrl", 382).
yeccpars2_138_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_143_,1}}).
-file("snmpc_mib_gram.yrl", 381).
yeccpars2_143_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_144_,1}}).
-file("snmpc_mib_gram.yrl", 388).
yeccpars2_144_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_145_,1}}).
-file("snmpc_mib_gram.yrl", 389).
yeccpars2_145_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_147_,1}}).
-file("snmpc_mib_gram.yrl", 393).
yeccpars2_147_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_148_,1}}).
-file("snmpc_mib_gram.yrl", 395).
yeccpars2_148_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_149_,1}}).
-file("snmpc_mib_gram.yrl", 394).
yeccpars2_149_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_150_,1}}).
-file("snmpc_mib_gram.yrl", 414).
yeccpars2_150_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_151_,1}}).
-file("snmpc_mib_gram.yrl", 384).
yeccpars2_151_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_152_,1}}).
-file("snmpc_mib_gram.yrl", 391).
yeccpars2_152_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_153_,1}}).
-file("snmpc_mib_gram.yrl", 390).
yeccpars2_153_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_155_,1}}).
-file("snmpc_mib_gram.yrl", 383).
yeccpars2_155_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_156_,1}}).
-file("snmpc_mib_gram.yrl", 387).
yeccpars2_156_([__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) , __1
  end | __Stack].

-compile({inline,{yeccpars2_160_,1}}).
-file("snmpc_mib_gram.yrl", 413).
yeccpars2_160_([__2,__1 | __Stack]) ->
 [begin
   display_hint ( __2 )
  end | __Stack].

-compile({inline,{yeccpars2_162_,1}}).
-file("snmpc_mib_gram.yrl", 411).
yeccpars2_162_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_163_,1}}).
-file("snmpc_mib_gram.yrl", 603).
yeccpars2_163_([__1 | __Stack]) ->
 [begin
   statusv2 ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_164_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_164_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_166_,1}}).
-file("snmpc_mib_gram.yrl", 410).
yeccpars2_166_([__2,__1 | __Stack]) ->
 [begin
   lists : reverse ( val ( __2 ) )
  end | __Stack].

-compile({inline,{yeccpars2_169_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_169_([__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_171_,1}}).
-file("snmpc_mib_gram.yrl", 510).
yeccpars2_171_([__10,__9,__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   NT = make_new_type ( __1 , 'TEXTUAL-CONVENTION' , __4 ,
    __6 , __7 , __8 , __10 ) ,
    { NT , line_of ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_174_,1}}).
-file("snmpc_mib_gram.yrl", 349).
yeccpars2_174_([__3,__2,__1 | __Stack]) ->
 [begin
   { { sequence_of , val ( __3 ) } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_180_,1}}).
-file("snmpc_mib_gram.yrl", 330).
yeccpars2_180_([__2,__1 | __Stack]) ->
 [begin
   [ { val ( __1 ) , __2 } ]
  end | __Stack].

-compile({inline,{yeccpars2_181_,1}}).
-file("snmpc_mib_gram.yrl", 334).
yeccpars2_181_([__1 | __Stack]) ->
 [begin
   { { bits , [ { dummy , 0 } ] } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_187_,1}}).
-file("snmpc_mib_gram.yrl", 364).
yeccpars2_187_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   [ { val ( __1 ) , val ( __3 ) } ]
  end | __Stack].

-compile({inline,{yeccpars2_189_,1}}).
-file("snmpc_mib_gram.yrl", 346).
yeccpars2_189_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   ensure_ver ( 2 , __1 ) ,
    { { bits , __3 } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_193_,1}}).
-file("snmpc_mib_gram.yrl", 366).
yeccpars2_193_([__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   [ { val ( __3 ) , val ( __5 ) } | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_195_,1}}).
-file("snmpc_mib_gram.yrl", 325).
yeccpars2_195_([__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   Seq = make_sequence ( __1 , lists : reverse ( __5 ) ) ,
    { Seq , line_of ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_197_,1}}).
-file("snmpc_mib_gram.yrl", 332).
yeccpars2_197_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   [ { val ( __3 ) , __4 } | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_198_,1}}).
-file("snmpc_mib_gram.yrl", 370).
yeccpars2_198_([__2,__1 | __Stack]) ->
 [begin
   { 'OCTET STRING' , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_199_,1}}).
-file("snmpc_mib_gram.yrl", 372).
yeccpars2_199_([__2,__1 | __Stack]) ->
 [begin
   { 'OBJECT IDENTIFIER' , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_202_,1}}).
-file("snmpc_mib_gram.yrl", 344).
yeccpars2_202_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { { integer_with_enum , 'INTEGER' , __3 } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_203_,1}}).
-file("snmpc_mib_gram.yrl", 371).
yeccpars2_203_([__2,__1 | __Stack]) ->
 [begin
   { 'BIT STRING' , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_204_,1}}).
-file("snmpc_mib_gram.yrl", 341).
yeccpars2_204_([__2,__1 | __Stack]) ->
 [begin
   { { type_with_size , cat ( __1 ) , __2 } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_207_,1}}).
-file("snmpc_mib_gram.yrl", 357).
yeccpars2_207_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_209_,1}}).
-file("snmpc_mib_gram.yrl", 360).
yeccpars2_209_([__1 | __Stack]) ->
 [begin
   val ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_211_,1}}).
-file("snmpc_mib_gram.yrl", 361).
yeccpars2_211_([__2,__1 | __Stack]) ->
 [begin
   make_range_integer ( val ( __1 ) , val ( __2 ) )
  end | __Stack].

-compile({inline,{yeccpars2_212_,1}}).
-file("snmpc_mib_gram.yrl", 362).
yeccpars2_212_([__2,__1 | __Stack]) ->
 [begin
   make_range_integer ( val ( __1 ) , val ( __2 ) )
  end | __Stack].

-compile({inline,{yeccpars2_217_,1}}).
-file("snmpc_mib_gram.yrl", 358).
yeccpars2_217_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 , __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_218_,1}}).
-file("snmpc_mib_gram.yrl", 352).
yeccpars2_218_([__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   make_range ( __4 )
  end | __Stack].

-compile({inline,{yeccpars2_221_,1}}).
-file("snmpc_mib_gram.yrl", 355).
yeccpars2_221_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 , __4 ]
  end | __Stack].

-compile({inline,{yeccpars2_222_,1}}).
-file("snmpc_mib_gram.yrl", 356).
yeccpars2_222_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 , __4 | __5 ]
  end | __Stack].

-compile({inline,{yeccpars2_223_,1}}).
-file("snmpc_mib_gram.yrl", 351).
yeccpars2_223_([__3,__2,__1 | __Stack]) ->
 [begin
   make_range ( __2 )
  end | __Stack].

-compile({inline,{yeccpars2_224_,1}}).
-file("snmpc_mib_gram.yrl", 342).
yeccpars2_224_([__2,__1 | __Stack]) ->
 [begin
   { { type_with_size , val ( __1 ) , __2 } , line_of ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_227_,1}}).
-file("snmpc_mib_gram.yrl", 614).
yeccpars2_227_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_229_,1}}).
-file("snmpc_mib_gram.yrl", 614).
yeccpars2_229_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_233_,1}}).
-file("snmpc_mib_gram.yrl", 600).
yeccpars2_233_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_236_,1}}).
-file("snmpc_mib_gram.yrl", 601).
yeccpars2_236_([__2,__1 | __Stack]) ->
 [begin
   units ( __2 )
  end | __Stack].

-compile({inline,{yeccpars2_239_,1}}).
-file("snmpc_mib_gram.yrl", 605).
yeccpars2_239_([__1 | __Stack]) ->
 [begin
   accessv2 ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_242_,1}}).
-file("snmpc_mib_gram.yrl", 409).
yeccpars2_242_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_243_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_243_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_244_,1}}).
-file("snmpc_mib_gram.yrl", 590).
yeccpars2_244_(__Stack) ->
 [begin
   { indexes , undefined }
  end | __Stack].

-compile({inline,{yeccpars2_245_,1}}).
-file("snmpc_mib_gram.yrl", 444).
yeccpars2_245_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_250_,1}}).
-file("snmpc_mib_gram.yrl", 592).
yeccpars2_250_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_254_,1}}).
-file("snmpc_mib_gram.yrl", 595).
yeccpars2_254_([__2,__1 | __Stack]) ->
 [begin
   { implied , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_256_,1}}).
-file("snmpc_mib_gram.yrl", 588).
yeccpars2_256_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { indexes , lists : reverse ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_257_,1}}).
-file("snmpc_mib_gram.yrl", 593).
yeccpars2_257_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_261_,1}}).
-file("snmpc_mib_gram.yrl", 589).
yeccpars2_261_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { augments , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_269_,1}}).
-file("snmpc_mib_gram.yrl", 447).
yeccpars2_269_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_272_,1}}).
-file("snmpc_mib_gram.yrl", 449).
yeccpars2_272_([__1 | __Stack]) ->
 [begin
   [ val ( __1 ) ]
  end | __Stack].

-compile({inline,{yeccpars2_274_,1}}).
-file("snmpc_mib_gram.yrl", 450).
yeccpars2_274_([__3,__2,__1 | __Stack]) ->
 [begin
   [ val ( __3 ) | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_276_,1}}).
-file("snmpc_mib_gram.yrl", 435).
yeccpars2_276_([__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { defval , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_277_,1}}).
-file("snmpc_mib_gram.yrl", 443).
yeccpars2_277_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { defval , lists : reverse ( val ( __3 ) ) }
  end | __Stack].

-compile({inline,{yeccpars2_280_,1}}).
-file("snmpc_mib_gram.yrl", 440).
yeccpars2_280_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { defval , make_defval_for_string ( line_of ( __1 ) , lists : reverse ( val ( __3 ) ) ,
    val ( __4 ) ) }
  end | __Stack].

-compile({inline,{yeccpars2_281_,1}}).
-file("snmpc_mib_gram.yrl", 437).
yeccpars2_281_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { defval , make_defval_for_string ( line_of ( __1 ) , lists : reverse ( val ( __3 ) ) ,
    val ( __4 ) ) }
  end | __Stack].

-compile({inline,{yeccpars2_282_,1}}).
-file("snmpc_mib_gram.yrl", 433).
yeccpars2_282_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { defval , val ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_283_,1}}).
-file("snmpc_mib_gram.yrl", 434).
yeccpars2_283_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { defval , val ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_284_,1}}).
-file("snmpc_mib_gram.yrl", 583).
yeccpars2_284_([__15,__14,__13,__12,__11,__10,__9,__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   Kind = kind ( __14 , __13 ) ,
    OT = make_object_type ( __1 , __4 , __5 , __7 , __9 ,
    __11 , __12 , Kind , __15 ) ,
    { OT , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_288_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_288_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_290_,1}}).
-file("snmpc_mib_gram.yrl", 516).
yeccpars2_290_([__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { Parent , SubIndex } = __8 ,
    Int = make_internal ( __1 , 'OBJECT-IDENTITY' ,
    Parent , SubIndex ) ,
    { Int , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_295_,1}}).
-file("snmpc_mib_gram.yrl", 616).
yeccpars2_295_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_297_,1}}).
-file("snmpc_mib_gram.yrl", 613).
yeccpars2_297_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   lists : reverse ( __3 )
  end | __Stack].

-compile({inline,{yeccpars2_298_,1}}).
-file("snmpc_mib_gram.yrl", 617).
yeccpars2_298_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_300_,1}}).
-file("snmpc_mib_gram.yrl", 411).
yeccpars2_300_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_301_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_301_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_303_,1}}).
-file("snmpc_mib_gram.yrl", 523).
yeccpars2_303_([__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   OG = make_object_group ( __1 , __3 , __5 , __6 , __7 , __8 ) ,
    { OG , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_305_,1}}).
-file("snmpc_mib_gram.yrl", 303).
yeccpars2_305_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { Parent , SubIndex } = __4 ,
    Int = make_internal ( __1 , dummy , Parent , SubIndex ) ,
    { Int , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_309_,1}}).
-file("snmpc_mib_gram.yrl", 409).
yeccpars2_309_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_310_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_310_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_312_,1}}).
-file("snmpc_mib_gram.yrl", 610).
yeccpars2_312_([__9,__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   Not = make_notification ( __1 , __3 , __5 , __7 , __8 , __9 ) ,
    { Not , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_318_,1}}).
-file("snmpc_mib_gram.yrl", 411).
yeccpars2_318_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_319_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_319_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_321_,1}}).
-file("snmpc_mib_gram.yrl", 529).
yeccpars2_321_([__11,__10,__9,__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   NG = make_notification_group ( __1 , __5 , __8 , __9 ,
    __10 , __11 ) ,
    { NG , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_323_,1}}).
-file("snmpc_mib_gram.yrl", 411).
yeccpars2_323_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_324_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_324_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_325_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_325_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_326_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_326_([__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_328_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_328_([__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_329_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_329_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_330_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_330_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_331_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_331_([__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_332_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_332_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_336_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_336_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_337_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_337_([__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_338_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_338_([__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_339_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_339_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_340_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_340_([__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_341_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_341_([__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_344_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_344_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_345_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_345_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_347_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_347_([__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_348_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_348_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_350_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_350_([__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_351_,1}}).
-file("snmpc_mib_gram.yrl", 411).
yeccpars2_351_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_353_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_353_([__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_354_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_354_([__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_355_,1}}).
-file("snmpc_mib_gram.yrl", 411).
yeccpars2_355_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_356_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_356_([__3,__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_357_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_357_([__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_358_,1}}).
-file("snmpc_mib_gram.yrl", 535).
yeccpars2_358_([__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   MC = make_module_compliance ( __1 , __4 , __5 , __6 ,
    __7 , __8 ) ,
    { MC , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_359_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_359_([__2,__1 | __Stack]) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_364_,1}}).
-file("snmpc_mib_gram.yrl", 401).
yeccpars2_364_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_365_,1}}).
-file("snmpc_mib_gram.yrl", 411).
yeccpars2_365_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_369_,1}}).
-file("snmpc_mib_gram.yrl", 403).
yeccpars2_369_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_371_,1}}).
-file("snmpc_mib_gram.yrl", 402).
yeccpars2_371_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   __3
  end | __Stack].

-compile({inline,{yeccpars2_372_,1}}).
-file("snmpc_mib_gram.yrl", 404).
yeccpars2_372_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_373_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_373_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_376_,1}}).
-file("snmpc_mib_gram.yrl", 297).
yeccpars2_376_([__9,__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   Trap = make_trap ( __1 , __4 , lists : reverse ( __5 ) ,
    __6 , __7 , val ( __9 ) ) ,
    { Trap , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_381_,1}}).
-file("snmpc_mib_gram.yrl", 457).
yeccpars2_381_([__1 | __Stack]) ->
 [begin
   accessv1 ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_384_,1}}).
-file("snmpc_mib_gram.yrl", 459).
yeccpars2_384_([__1 | __Stack]) ->
 [begin
   statusv1 ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_385_,1}}).
-file("snmpc_mib_gram.yrl", 409).
yeccpars2_385_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_386_,1}}).
-file("snmpc_mib_gram.yrl", 0).
yeccpars2_386_(__Stack) ->
 [begin
   '$undefined'
  end | __Stack].

-compile({inline,{yeccpars2_387_,1}}).
-file("snmpc_mib_gram.yrl", 419).
yeccpars2_387_(__Stack) ->
 [begin
   { indexes , undefined }
  end | __Stack].

-compile({inline,{yeccpars2_388_,1}}).
-file("snmpc_mib_gram.yrl", 444).
yeccpars2_388_(__Stack) ->
 [begin
   undefined
  end | __Stack].

-compile({inline,{yeccpars2_391_,1}}).
-file("snmpc_mib_gram.yrl", 421).
yeccpars2_391_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_395_,1}}).
-file("snmpc_mib_gram.yrl", 418).
yeccpars2_395_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { indexes , lists : reverse ( __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_396_,1}}).
-file("snmpc_mib_gram.yrl", 422).
yeccpars2_396_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __3 | __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_398_,1}}).
-file("snmpc_mib_gram.yrl", 315).
yeccpars2_398_([__14,__13,__12,__11,__10,__9,__8,__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   Kind = kind ( __13 , __12 ) ,
    OT = make_object_type ( __1 , __4 , __6 , __8 , __10 ,
    __11 , Kind , __14 ) ,
    { OT , line_of ( __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_399_,1}}).
-file("snmpc_mib_gram.yrl", 209).
yeccpars2_399_([__7,__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { Version , Defs } = __6 ,
    { Version , __1 , __5 , Defs }
  end | __Stack].


-file("snmpc_mib_gram.yrl", 971).
