<?php

require_once 'Horde/Kolab.php';
require_once 'Horde/Identity.php';

/**
 * Horde Kronolith driver for the Kolab IMAP Server.
 * Copyright 2004-2005 Horde Project (http://horde.org/)
 *
 * $Horde: kronolith/lib/Driver/kolab.php,v 1.16.2.4 2005/06/23 21:22:19 selsky Exp $
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Stuart Binge <omicron@mighty.co.za>
 * @package Kronolith
 */
class Kronolith_Driver_kolab extends Kronolith_Driver {

    /**
     * Our Kolab server connection.
     *
     * @var Kolab
     */
    var $_kolab = null;

    function open($calendar)
    {
        $this->_calendar = $calendar;

        $result = $this->_connect();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        return true;
    }

    function close()
    {
        return $this->_disconnect();
    }

    function _connect()
    {
        if (isset($this->_kolab)) {
            return true;
        }

        $this->_kolab = new Kolab();

        return $this->_kolab->open($this->_calendar);
    }

    function _disconnect()
    {
        if (empty($this->_kolab)) {
            return true;
        }

        $this->_kolab->close();
        $this->_kolab = null;

        return true;
    }

    function listAlarms($date)
    {
        $allevents = $this->listEvents($date, $date, true);
        $events = array();

        foreach ($allevents as $eventId) {
            $event = &$this->getEvent($eventId);

            if ($event->getRecurType() == KRONOLITH_RECUR_NONE) {
                $start = &new Horde_Date($event->start);
                $start->min -= $event->getAlarm();
                $start->correct();
                if ($start->compareDateTime($date) <= 0 &&
                    $date->compareDateTime($event->end) <= -1) {
                    $events[] = $eventId;
                }
            } else {
                if ($next = $this->nextRecurrence($eventId, $date)) {
                    $start = &new Horde_Date($next);
                    $start->min -= $event->getAlarm();
                    $start->correct();
                    if ($start->compareDateTime($date) <= 0 &&
                        $date->compareDateTime(new Horde_Date(array('year' => $next->year,
                                                                    'month' => $next->month,
                                                                    'mday' => $next->mday,
                                                                    'hour' => $event->end->hour,
                                                                    'min' => $event->end->min,
                                                                    'sec' => $event->end->sec)) <= -1)) {
                        $events[] = $eventId;
                    }
                }
            }
        }

        return is_array($events) ? $events : array();
    }

    function listEvents($startDate = null, $endDate = null)
    {
        // We don't perform any checking on $startDate and $endDate, as that
        // has the potential to leave out recurring event instances.

        $events = array();

        $msg_list = $this->_kolab->listObjects();
        if (is_a($msg_list, 'PEAR_Error')) {
            return $msg_list;
        }

        if (empty($msg_list)) {
            return $events;
        }

        foreach ($msg_list as $msg) {
            $result = $this->_kolab->loadObject($msg, true);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }

            $events[] = $this->_kolab->getUID();
        }

        return $events;
    }

    function &getEvent($eventID = null)
    {
        if (is_null($eventID)) {
            return $event = &new Kronolith_Event_kolab($this);
        }

        $result = $this->_kolab->loadObject($eventID);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        return $event = &new Kronolith_Event_kolab($this, true);
    }

    function &getByUID($uid)
    {
        return PEAR::raiseError('Not supported');
    }

    function saveEvent($event)
    {
        $k = &$this->_kolab;

        if ($event->isStored()) {
            $uid = $event->getID();

            $result = $k->loadObject($uid);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
        } else {
            if ($event->getID()) {
                $uid = $event->getID();
            } else {
                $uid = md5(uniqid(mt_rand(), true));
            }

            $k->newObject($uid);
        }

        $xml_hash = &$this->_kolab->getCurrentObject();

        $k->setStr('summary', $event->getTitle());
        $k->setStr('body', $event->getDescription());
        $k->setStr('categories', $event->getCategory());
        $k->setStr('location', $event->getLocation());

        $organizer = &$k->initRootElem('organizer');
        $k->setElemStr($organizer, 'smtp-address', $event->getCreatorID());

        $k->setVal('alarm', $event->getAlarm());
        $k->setVal('start-date', Kolab::encodeDateTime($event->start->timestamp()));
        $k->setVal('end-date', Kolab::encodeDateTime($event->end->timestamp()));

        switch ($event->status) {
        case KRONOLITH_STATUS_CANCELLED:
            $k->setVal('show-time-as', 'free');
            break;

        case KRONOLITH_STATUS_TENTATIVE:
            $k->setVal('show-time-as', 'tentative');
            break;

        case KRONOLITH_STATUS_CONFIRMED:
        default:
            $k->setVal('show-time-as', 'busy');
            break;
        }

        $k->delAllRootElems('attendee');
        foreach ($event->attendees as $email => $status) {
            $attendee = &$k->appendRootElem('attendee');
            $k->setElemVal($attendee, 'smtp-address', $email);

            switch ($status['response']) {
            case KRONOLITH_RESPONSE_ACCEPTED:
                $k->setElemVal($attendee, 'status', 'accepted');
                break;

            case KRONOLITH_RESPONSE_DECLINED:
                $k->setElemVal($attendee, 'status', 'declined');
                break;

            case KRONOLITH_RESPONSE_TENTATIVE:
                $k->setElemVal($attendee, 'status', 'tentative');
                break;

            case KRONOLITH_RESPONSE_NONE:
            default:
                $k->setElemVal($attendee, 'status', 'none');
            }

            switch ($status['attendance']) {
            case KRONOLITH_PART_OPTIONAL:
                $k->setElemVal($attendee, 'role', 'optional');
                break;

            case KRONOLITH_PART_NONE:
                $k->setElemVal($attendee, 'role', 'resource');
                break;

            case KRONOLITH_PART_REQUIRED:
            default:
                $k->setElemVal($attendee, 'role', 'required');
            }
        }

        $k->delAllRootElems('recurrence');

        $range_type = 'none';
        $range = 0;
        if (!empty($event->recurEnd)) {
            $range_type = 'date';
            $range = Kolab::encodeDate($event->recurEnd->timestamp());
        }

        if ($event->recurType != KRONOLITH_RECUR_NONE) {
            $recurrence = &$k->initRootElem('recurrence');
            $k->setElemVal($recurrence, 'interval', $event->recurInterval);
            $range = &$k->setElemVal($recurrence, 'range', $range);
            $range->set_attribute('type', $range_type);

            switch ($event->recurType) {
                case KRONOLITH_RECUR_DAILY:
                    $recurrence->set_attribute('cycle', 'daily');
                    break;

                case KRONOLITH_RECUR_WEEKLY:
                    $recurrence->set_attribute('cycle', 'weekly');

                    $days = array('sunday', 'monday', 'tuesday', 'wednesday',
                                  'thursday', 'friday', 'saturday');

                    for ($i = 0; $i <= 7 ; $i++) {
                        if ($event->recurOnDay(pow(2, $i))) {
                            $day = &$k->appendElem('day', $recurrence);
                            $day->set_content($days[$i]);
                        }
                    }
                    break;

                case KRONOLITH_RECUR_DAY_OF_MONTH:
                    $recurrence->set_attribute('cycle', 'monthly');
                    $recurrence->set_attribute('type', 'daynumber');
                    $k->setElemVal($recurrence, 'date', $event->start->mday);
                    break;

                case KRONOLITH_RECUR_WEEK_OF_MONTH:
                    $recurrence->set_attribute('cycle', 'monthly');
                    $recurrence->set_attribute('type', 'weekday');
                    $k->setElemVal($recurrence, 'daynumber', 1);
                    $start = &new Horde_Date($event->start);
                    $days = array('sunday', 'monday', 'tuesday', 'wednesday',
                                  'thursday', 'friday', 'saturday');
                    $k->setElemVal($recurrence, 'day', $days[$start->dayOfWeek()]);
                    break;

                case KRONOLITH_RECUR_YEARLY:
                    $recurrence->set_attribute('cycle', 'yearly');
                    $recurrence->set_attribute('type', 'monthday');

                    $months = array('january', 'february', 'march', 'april',
                                    'may', 'june', 'july', 'august', 'september',
                                    'october', 'november', 'december');

                    $k->setElemVal($recurrence, 'month', $months[$event->start->month]);
                    $k->setElemVal($recurrence, 'date', $event->start->mday);
                    break;
            }
        }

        $result = $k->saveObject();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        return $uid;
    }

    /**
     * Move an event to a new calendar.
     *
     * @param string $eventId      The event to move.
     * @param string $newCalendar  The new calendar.
     */
    function move($eventID, $newCalendar)
    {
        return $this->_kolab->moveObject($eventID, $newCalendar);
    }

    /**
     * Delete a calendar and all its events.
     *
     * @param string $calendar  The name of the calendar to delete.
     *
     * @return mixed  True or a PEAR_Error on failure.
     */
    function delete($calendar)
    {
        // This is handled by the share hooks.
        return true;
    }

    /**
     * Rename a calendar.
     *
     * @param string $from  The current name of the calendar.
     * @param string $to    The new name of the calendar.
     *
     * @return mixed  True or a PEAR_Error on failure.
     */
    function rename($from, $to)
    {
        // This is handled by the share hooks.
        return true;
    }

    /**
     * Delete an event.
     *
     * @param string $eventId  The ID of the event to delete.
     *
     * @return mixed  True or a PEAR_Error on failure.
     */
    function deleteEvent($eventID)
    {
        return $this->_kolab->removeObjects($eventID);
    }

}

class Kronolith_Event_kolab extends Kronolith_Event {

    function fromDriver($dummy)
    {
        $driver = &$this->getDriver();
        $k = &$driver->_kolab;

        $this->eventID = $k->getUID();
        $this->setUID($k->getUID());
        $this->title = $k->getStr('summary');
        $this->description = $k->getStr('body');
        $this->location = $k->getStr('location');
        $this->category = $k->getStr('categories');

        $organizer = &$k->getRootElem('organizer');
        $this->creatorID = $k->getElemStr($organizer, 'smtp-address');

        $this->alarm = $k->getVal('alarm');
        $this->start = &new Horde_Date(Kolab::decodeDateOrDateTime($k->getVal('start-date')));
        $this->end = &new Horde_Date(Kolab::decodeDateOrDateTime($k->getVal('end-date')));
        $this->durMin = ($this->end->timestamp() - $this->start->timestamp()) / 60;

        $status = $k->getVal('show-time-as');
        switch ($status) {
        case 'free':
            $this->status = KRONOLITH_STATUS_CANCELLED;
            break;

        case 'tentative':
            $this->status = KRONOLITH_STATUS_TENTATIVE;
            break;

        case 'busy':
        case 'outofoffice':
        default:
            $this->status = KRONOLITH_STATUS_CONFIRMED;
        }

        $attendees = &$k->getAllRootElems('attendee');
        for ($i = 0, $j = count($attendees); $i < $j; $i++) {
            $attendee = &$attendees[$i];

            $email = $k->getElemStr($attendee, 'smtp-address');
            if (empty($email)) {
                continue;
            }

            $role = $k->getElemVal($attendee, 'role');
            switch ($role) {
            case 'optional':
                $role = KRONOLITH_PART_OPTIONAL;
                break;

            case 'resource':
                $role = KRONOLITH_PART_NONE;
                break;

            case 'required':
            default:
                $role = KRONOLITH_PART_REQUIRED;
                break;
            }

            $status = $k->getElemVal($attendee, 'status');
            switch ($status) {
            case 'accepted':
                $status = KRONOLITH_RESPONSE_ACCEPTED;
                break;

            case 'declined':
                $status = KRONOLITH_RESPONSE_DECLINED;
                break;

            case 'tentative':
                $status = KRONOLITH_RESPONSE_TENTATIVE;
                break;

            case 'none':
            default:
                $status = KRONOLITH_RESPONSE_NONE;
                break;
            }

            $this->addAttendee($email, $role, $status);
        }

        $this->recurEnd = null;
        $this->recurType = KRONOLITH_RECUR_NONE;

        $recurrence = &$k->getRootElem('recurrence');
        if ($recurrence !== false) {
            $cycle = $recurrence->get_attribute('cycle');
            $this->recurInterval = $k->getElemVal($recurrence, 'interval');

            switch ($cycle) {
            case 'daily':
                $this->recurType = KRONOLITH_RECUR_DAILY;
                break;

            case 'weekly':
                $this->recurType = KRONOLITH_RECUR_WEEKLY;

                $mask = 0;
                $bits = array(
                    'monday' => HORDE_DATE_MASK_MONDAY,
                    'tuesday' => HORDE_DATE_MASK_TUESDAY,
                    'wednesday' => HORDE_DATE_MASK_WEDNESDAY,
                    'thursday' => HORDE_DATE_MASK_THURSDAY,
                    'friday' => HORDE_DATE_MASK_FRIDAY,
                    'saturday' => HORDE_DATE_MASK_SATURDAY,
                    'sunday' => HORDE_DATE_MASK_SUNDAY,
                );

                $days = $k->getAllElems('day', $recurrence);
                foreach ($days as $day) {
                    $day_str = $day->get_content();

                    if (empty($day_str) || !isset($bits[$day_str])) {
                        continue;
                    }

                    $mask |= $bits[$day_str];
                }

                $this->setRecurOnDay($mask);
                break;

            case 'monthly':
                $type = $recurrence->get_attribute('type');
                switch ($type) {
                case 'daynumber':
                    $this->recurType = KRONOLITH_RECUR_DAY_OF_MONTH;
                    break;

                case 'weekday':
                    $this->recurType = KRONOLITH_RECUR_DAY_OF_MONTH;
                    break;
                }
                break;

            case 'yearly':
                $this->recurType = KRONOLITH_RECUR_YEARLY;
                break;
            }

            $range = &$k->getElem('range', $recurrence);
            $range_type = $range->get_attribute('type');
            $range_val = $k->getElemVal($recurrence, 'range');

            switch ($range_type) {
            case 'number':
                // Kronolith can't handle recurrence intervals by
                // number of instances yet.
                break;

            case 'date':
                $this->recurEnd = &new Horde_Date(Kolab::decodeDate($range_val));
                break;
            }
        }

        $this->initialized = true;
        $this->stored = true;
    }

    function toDriver()
    {
    }

}
