-module(erl_parse).
-export([parse/1, parse_and_scan/1, format_error/1]).
-file("erl_parse.yrl", 379).

%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id$
%%

-export([parse_form/1,parse_exprs/1,parse_term/1]).
-export([normalise/1,abstract/1,tokens/1,tokens/2]).
-export([abstract/2, package_segments/1]).
-export([inop_prec/1,preop_prec/1,func_prec/0,max_prec/0]).

%% The following directive is needed for (significantly) faster compilation
%% of the generated .erl file by the HiPE compiler.  Please do not remove.
-compile([inline,{hipe,[{regalloc,linear_scan}]}]).


%% mkop(Op, Arg) -> {op,Line,Op,Arg}.
%% mkop(Left, Op, Right) -> {op,Line,Op,Left,Right}.

mkop(L, {Op,Pos}, R) -> {op,Pos,Op,L,R}.

mkop({Op,Pos}, A) -> {op,Pos,Op,A}.

%% keep track of line info in tokens
line(Tup) -> element(2, Tup).

%% Entry points compatible to old erl_parse.
%% These really suck and are only here until Calle gets multiple
%% entry points working.

parse_form(Tokens) ->
    parse(Tokens).

parse_exprs(Tokens) ->
    case parse([{atom,0,f},{'(',0},{')',0},{'->',0}|Tokens]) of
	{ok,{function,_Lf,f,0,[{clause,_Lc,[],[],Exprs}]}} ->
	    {ok,Exprs};
	{error,E} -> {error,E}
    end.

parse_term(Tokens) ->
    case parse([{atom,0,f},{'(',0},{')',0},{'->',0}|Tokens]) of
	{ok,{function,_Lf,f,0,[{clause,_Lc,[],[],[Expr]}]}} ->
	    case catch normalise(Expr) of
		{'EXIT',_R} ->
		    {error,{line(Expr),?MODULE,"bad term"}};
		Term -> {ok,Term}
	    end;
	{ok,{function,_Lf,f,0,[{clause,_Lc,[],[],[_E1,E2|_Es]}]}} ->
	    {error,{line(E2),?MODULE,"bad term"}};
	{error,E} -> {error,E}
    end.

%% build_attribute(AttrName, AttrValue) ->
%%	{attribute,Line,module,Module}
%%	{attribute,Line,export,Exports}
%%	{attribute,Line,import,Imports}
%%	{attribute,Line,record,{Name,Inits}}
%%	{attribute,Line,file,{Name,Line}}
%%	{attribute,Line,Name,Val}

build_attribute({atom,La,module}, Val) ->
    case Val of
	[{atom,_Lm,Module}] ->
	    {attribute,La,module,Module};
	[{atom,_Lm,Module},ExpList] ->
	    {attribute,La,module,{Module,var_list(ExpList)}};
	[Name] ->
	    case package_segments(Name) of
		error ->
		    error_bad_decl(La, module);
		Module ->
		    {attribute,La,module,Module}
	    end;
	[Name,ExpList] ->
	    case package_segments(Name) of
		error ->
		    error_bad_decl(La, module);
		Module ->
		    {attribute,La,module,{Module,var_list(ExpList)}}
	    end;
	_Other ->
	    error_bad_decl(La, module)
    end;
build_attribute({atom,La,export}, Val) ->
    case Val of
	[ExpList] ->
	    {attribute,La,export,farity_list(ExpList)};
	_Other -> error_bad_decl(La, export)
    end;
build_attribute({atom,La,import}, Val) ->
    case Val of
	[Name] ->
	    case package_segments(Name) of
		error ->
		    error_bad_decl(La, import);
		Module ->
		    {attribute,La,import,Module}
	    end;
	[{atom,_Lm,Mod},ImpList] ->
	    {attribute,La,import,{Mod,farity_list(ImpList)}};
	[Name, ImpList] ->
	    case package_segments(Name) of
		error ->
		    error_bad_decl(La, import);
		Module ->
		    {attribute,La,import,{Module,farity_list(ImpList)}}
	    end;
	_Other -> error_bad_decl(La, import)
    end;
build_attribute({atom,La,record}, Val) ->
    case Val of
	[{atom,_Ln,Record},RecTuple] ->
	    {attribute,La,record,{Record,record_tuple(RecTuple)}};
	_Other -> error_bad_decl(La, record)
    end;
build_attribute({atom,La,file}, Val) ->
    case Val of
	[{string,_Ln,Name},{integer,_Ll,Line}] ->
	    {attribute,La,file,{Name,Line}};
	_Other -> error_bad_decl(La, file)
    end;
build_attribute({atom,La,Attr}, Val) ->
    case Val of
	[Expr] ->
	    {attribute,La,Attr,term(Expr)};
	_Other -> return_error(La, "bad attribute")
    end.

var_list({cons,_Lc,{var,_,V},Tail}) ->
    [V|var_list(Tail)];
var_list({nil,_Ln}) -> [];
var_list(Other) ->
    return_error(line(Other), "bad variable list").

error_bad_decl(L, S) ->
    return_error(L, io_lib:format("bad ~w declaration", [S])).

farity_list({cons,_Lc,{op,_Lo,'/',{atom,_La,A},{integer,_Li,I}},Tail}) ->
    [{A,I}|farity_list(Tail)];
farity_list({nil,_Ln}) -> [];
farity_list(Other) ->
    return_error(line(Other), "bad function arity").

record_tuple({tuple,_Lt,Fields}) ->
    record_fields(Fields);
record_tuple(Other) ->
    return_error(line(Other), "bad record declaration").

record_fields([{atom,La,A}|Fields]) ->
    [{record_field,La,{atom,La,A}}|record_fields(Fields)];
record_fields([{match,_Lm,{atom,La,A},Expr}|Fields]) ->
    [{record_field,La,{atom,La,A},Expr}|record_fields(Fields)];
record_fields([Other|_Fields]) ->
    return_error(line(Other), "bad record field");
record_fields([]) -> [].

term(Expr) ->
    case catch normalise(Expr) of
	{'EXIT',_R} -> return_error(line(Expr), "bad attribute");
	Term -> Term
    end.

package_segments(Name) ->
    package_segments(Name, [], []).

package_segments({record_field, _, F1, F2}, Fs, As) ->
    package_segments(F1, [F2 | Fs], As);
package_segments({atom, _, A}, [F | Fs], As) ->
    package_segments(F, Fs, [A | As]);
package_segments({atom, _, A}, [], As) ->
    lists:reverse([A | As]);
package_segments(_, _, _) ->
    error.

%% build_function([Clause]) -> {function,Line,Name,Arity,[Clause]}

build_function(Cs) ->
    Name = element(3, hd(Cs)),
    Arity = length(element(4, hd(Cs))),
    {function,line(hd(Cs)),Name,Arity,check_clauses(Cs, Name, Arity)}.

%% build_rule([Clause]) -> {rule,Line,Name,Arity,[Clause]'}

build_rule(Cs) ->
    Name = element(3, hd(Cs)),
    Arity = length(element(4, hd(Cs))),
    {rule,line(hd(Cs)),Name,Arity,check_clauses(Cs, Name, Arity)}.

%% build_fun(Line, [Clause]) -> {'fun',Line,{clauses,[Clause]}}.

build_fun(Line, Cs) ->
    Arity = length(element(4, hd(Cs))),
    {'fun',Line,{clauses,check_clauses(Cs, 'fun', Arity)}}.

check_clauses(Cs, Name, Arity) ->
     mapl(fun ({clause,L,N,As,G,B}) when N == Name, length(As) == Arity ->
		 {clause,L,As,G,B};
	     ({clause,L,_N,_As,_G,_B}) ->
		 return_error(L, "head mismatch") end, Cs).

build_try(L,Es,Scs,{Ccs,As}) ->
    {'try',L,Es,Scs,Ccs,As}.

%% mapl(F,List)
%% an alternative map which always maps from left to right
%% and makes it possible to interrupt the mapping with throw on
%% the first occurence from left as expected.
%% can be removed when the jam machine (and all other machines)
%% uses the standardized (Erlang 5.0) evaluation order (from left to right)
mapl(F, [H|T]) ->
	V = F(H),
	[V | mapl(F,T)];
mapl(_, []) ->
	[].

%% normalise(AbsTerm)
%% abstract(Term)
%%  Convert between the abstract form of a term and a term.

normalise({char,_,C}) -> C;
normalise({integer,_,I}) -> I;
normalise({float,_,F}) -> F;
normalise({atom,_,A}) -> A;
normalise({string,_,S}) -> S;
normalise({nil,_}) -> [];
normalise({bin,_,Fs}) ->
    {value, B, _} =
	eval_bits:expr_grp(Fs, [],
			   fun(E, _) ->
				   {value, normalise(E), []}
			   end, [], true),
    B;
normalise({cons,_,Head,Tail}) ->
    [normalise(Head)|normalise(Tail)];
normalise({tuple,_,Args}) ->
    list_to_tuple(normalise_list(Args));
%% Atom dot-notation, as in 'foo.bar.baz'
normalise({record_field,_,_,_}=A) ->
    case package_segments(A) of
	error -> erlang:fault({badarg, A});
	As -> list_to_atom(packages:concat(As))
    end;
%% Special case for unary +/-.
normalise({op,_,'+',{char,_,I}}) -> I;
normalise({op,_,'+',{integer,_,I}}) -> I;
normalise({op,_,'+',{float,_,F}}) -> F;
normalise({op,_,'-',{char,_,I}}) -> -I;		%Weird, but compatible!
normalise({op,_,'-',{integer,_,I}}) -> -I;
normalise({op,_,'-',{float,_,F}}) -> -F;
normalise(X) -> erlang:fault({badarg, X}).

normalise_list([H|T]) ->
    [normalise(H)|normalise_list(T)];
normalise_list([]) ->
    [].

abstract(T) when integer(T) -> {integer,0,T};
abstract(T) when float(T) -> {float,0,T};
abstract(T) when atom(T) -> {atom,0,T};
abstract([]) -> {nil,0};
abstract(B) when binary(B) ->
    {bin, 0, lists:map(fun(Byte) ->
			       {bin_element, 0,
				{integer, 0, Byte}, default, default}
		       end,
		       binary_to_list(B))};
abstract([C|T]) when integer(C), 0 =< C, C < 256 ->
    abstract_string(T, [C]);
abstract([H|T]) ->
    {cons,0,abstract(H),abstract(T)};
abstract(Tuple) when tuple(Tuple) ->
    {tuple,0,abstract_list(tuple_to_list(Tuple))}.

abstract_string([C|T], String) when integer(C), 0 =< C, C < 256 ->
    abstract_string(T, [C|String]);
abstract_string([], String) ->
    {string, 0, lists:reverse(String)};
abstract_string(T, String) ->
    not_string(String, abstract(T)).

not_string([C|T], Result) ->
    not_string(T, {cons, 0, {integer, 0, C}, Result});
not_string([], Result) ->
    Result.

abstract_list([H|T]) ->
    [abstract(H)|abstract_list(T)];
abstract_list([]) ->
    [].

%%% abstract/2 keeps the line number
abstract(T, Line) when integer(T) -> {integer,Line,T};
abstract(T, Line) when float(T) -> {float,Line,T};
abstract(T, Line) when atom(T) -> {atom,Line,T};
abstract([], Line) -> {nil,Line};
abstract(B, Line) when binary(B) ->
    {bin, Line, lists:map(fun(Byte) ->
			       {bin_element, Line,
				{integer, Line, Byte}, default, default}
		       end,
		       binary_to_list(B))};
abstract([C|T], Line) when integer(C), 0 =< C, C < 256 ->
    abstract_string(T, [C], Line);
abstract([H|T], Line) ->
    {cons,Line,abstract(H, Line),abstract(T, Line)};
abstract(Tuple, Line) when tuple(Tuple) ->
    {tuple,Line,abstract_list(tuple_to_list(Tuple), Line)}.

abstract_string([C|T], String, Line) when integer(C), 0 =< C, C < 256 ->
    abstract_string(T, [C|String], Line);
abstract_string([], String, Line) ->
    {string, Line, lists:reverse(String)};
abstract_string(T, String, Line) ->
    not_string(String, abstract(T, Line), Line).

not_string([C|T], Result, Line) ->
    not_string(T, {cons, Line, {integer, Line, C}, Result}, Line);
not_string([], Result, _Line) ->
    Result.

abstract_list([H|T], Line) ->
    [abstract(H, Line)|abstract_list(T, Line)];
abstract_list([], _Line) ->
    [].

%% tokens(AbsTerm) -> [Token]
%% tokens(AbsTerm, More) -> [Token]
%%  Generate a list of tokens representing the abstract term.

tokens(Abs) ->
    tokens(Abs, []).

tokens({char,L,C}, More) -> [{char,L,C}|More];
tokens({integer,L,N}, More) -> [{integer,L,N}|More];
tokens({float,L,F}, More) -> [{float,L,F}|More];
tokens({atom,L,A}, More) -> [{atom,L,A}|More];
tokens({var,L,V}, More) -> [{var,L,V}|More];
tokens({string,L,S}, More) -> [{string,L,S}|More];
tokens({nil,L}, More) -> [{'[',L},{']',L}|More];
tokens({cons,L,Head,Tail}, More) ->
    [{'[',L}|tokens(Head, tokens_tail(Tail, More))];
tokens({tuple,L,[]}, More) ->
    [{'{',L},{'}',L}|More];
tokens({tuple,L,[E|Es]}, More) ->
    [{'{',L}|tokens(E, tokens_tuple(Es, line(E), More))].

tokens_tail({cons,L,Head,Tail}, More) ->
    [{',',L}|tokens(Head, tokens_tail(Tail, More))];
tokens_tail({nil,L}, More) ->
    [{']',L}|More];
tokens_tail(Other, More) ->
    L = line(Other),
    [{'|',L}|tokens(Other, [{']',L}|More])].

tokens_tuple([E|Es], Line, More) ->
    [{',',Line}|tokens(E, tokens_tuple(Es, line(E), More))];
tokens_tuple([], Line, More) ->
    [{'}',Line}|More].

%% Give the relative precedences of operators.

inop_prec('=') -> {150,100,100};
inop_prec('!') -> {150,100,100};
inop_prec('orelse') -> {160,150,150};
inop_prec('andalso') -> {200,160,160};
inop_prec('==') -> {300,200,300};
inop_prec('/=') -> {300,200,300};
inop_prec('=<') -> {300,200,300};
inop_prec('<') -> {300,200,300};
inop_prec('>=') -> {300,200,300};
inop_prec('>') -> {300,200,300};
inop_prec('=:=') -> {300,200,300};
inop_prec('=/=') -> {300,200,300};
inop_prec('++') -> {300,300,400};
inop_prec('--') -> {300,300,400};
inop_prec('+') -> {400,400,500};
inop_prec('-') -> {400,400,500};
inop_prec('bor') -> {400,400,500};
inop_prec('bxor') -> {400,400,500};
inop_prec('bsl') -> {400,400,500};
inop_prec('bsr') -> {400,400,500};
inop_prec('or') -> {400,400,500};
inop_prec('xor') -> {400,400,500};
inop_prec('*') -> {500,500,600};
inop_prec('/') -> {500,500,600};
inop_prec('div') -> {500,500,600};
inop_prec('rem') -> {500,500,600};
inop_prec('band') -> {500,500,600};
inop_prec('and') -> {500,500,600};
inop_prec('#') -> {800,700,800};
inop_prec(':') -> {900,800,900};
inop_prec('.') -> {900,900,1000}.

preop_prec('catch') -> {0,100};
preop_prec('+') -> {600,700};
preop_prec('-') -> {600,700};
preop_prec('bnot') -> {600,700};
preop_prec('not') -> {600,700};
preop_prec('#') -> {700,800}.

func_prec() -> {800,700}.

max_prec() -> 1000.

-file("/usr/obj/ports/erlang-11b.0/otp_src_R11B-0/bootstrap/lib/parsetools/include/yeccpre.hrl", 0).
%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id $
%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The parser generator will insert appropriate declarations before this line.%

parse(Tokens) ->
    yeccpars0(Tokens, false).

parse_and_scan({F, A}) -> % Fun or {M, F}
    yeccpars0([], {F, A});
parse_and_scan({M, F, A}) ->
    yeccpars0([], {{M, F}, A}).

format_error(Message) ->
    case io_lib:deep_char_list(Message) of
	true ->
	    Message;
	_ ->
	    io_lib:write(Message)
    end.

% To be used in grammar files to throw an error message to the parser
% toplevel. Doesn't have to be exported!
-compile({nowarn_unused_function,{return_error,2}}).
return_error(Line, Message) ->
    throw({error, {Line, ?MODULE, Message}}).

yeccpars0(Tokens, MFA) ->
    try yeccpars1(Tokens, MFA, 0, [], [])
    catch 
        throw: {error, {_Line, ?MODULE, _M}} = Error -> 
                   Error % probably from return_error/1
    end.

% Don't change yeccpars1/6 too much, it is called recursively by yeccpars2/8!
yeccpars1([Token | Tokens], Tokenizer, State, States, Vstack) ->
    yeccpars2(State, element(1, Token), States, Vstack, Token, Tokens,
	      Tokenizer);
yeccpars1([], {F, A}, State, States, Vstack) ->
    case apply(F, A) of
        {ok, Tokens, _Endline} ->
	    yeccpars1(Tokens, {F, A}, State, States, Vstack);
        {eof, _Endline} ->
            yeccpars1([], false, State, States, Vstack);
        {error, Descriptor, _Endline} ->
            {error, Descriptor}
    end;
yeccpars1([], false, State, States, Vstack) ->
    yeccpars2(State, '$end', States, Vstack, {'$end', 999999}, [], false).

% For internal use only.
yeccerror(Token) ->
    {error,
     {element(2, Token), ?MODULE,
      ["syntax error before: ", yecctoken2string(Token)]}}.

yecctoken2string({atom, _, A}) -> io_lib:write(A);
yecctoken2string({integer,_,N}) -> io_lib:write(N);
yecctoken2string({float,_,F}) -> io_lib:write(F);
yecctoken2string({char,_,C}) -> io_lib:write_char(C);
yecctoken2string({var,_,V}) -> io_lib:format('~s', [V]);
yecctoken2string({string,_,S}) -> io_lib:write_string(S);
yecctoken2string({reserved_symbol, _, A}) -> io_lib:format('~w', [A]);
yecctoken2string({_Cat, _, Val}) -> io_lib:format('~w', [Val]);
yecctoken2string({'dot', _}) -> io_lib:format('~w', ['.']);
yecctoken2string({'$end', _}) ->
    [];
yecctoken2string({Other, _}) when is_atom(Other) ->
    io_lib:format('~w', [Other]);
yecctoken2string(Other) ->
    io_lib:write(Other).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



-file("erl_parse.erl", 515).

yeccpars2(0, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 9, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 10, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(1, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_1_(__Stack),
 yeccpars2(yeccgoto(rule, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(2, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 292, [2 | __Ss], [__T | __Stack]);
yeccpars2(2, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_2_(__Stack),
 yeccpars2(yeccgoto(rule_clauses, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(3, dot, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 291, [3 | __Ss], [__T | __Stack]);
yeccpars2(3, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(4, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_4_(__Stack),
 yeccpars2(yeccgoto(function, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(5, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 286, [5 | __Ss], [__T | __Stack]);
yeccpars2(5, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_5_(__Stack),
 yeccpars2(yeccgoto(function_clauses, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(6, dot, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 285, [6 | __Ss], [__T | __Stack]);
yeccpars2(6, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(7, '$end', _, __Stack, _, _, _) ->
 {ok, hd(__Stack)};
yeccpars2(7, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(8, dot, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 284, [8 | __Ss], [__T | __Stack]);
yeccpars2(8, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(9, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 279, [9 | __Ss], [__T | __Stack]);
yeccpars2(9, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(10, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [10 | __Ss], [__T | __Stack]);
yeccpars2(10, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(11, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [11 | __Ss], [__T | __Stack]);
yeccpars2(11, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_11_(__Stack),
 yeccpars2(274, __Cat, [11 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(12, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_12_(__Stack),
 yeccpars2(yeccgoto(clause_args, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(13, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(14, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(15, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(16, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(17, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_700, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(18, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(19, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(20, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(21, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(22, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(23, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(24, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_700, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(25, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(26, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 272, [26 | __Ss], [__T | __Stack]);
yeccpars2(26, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(27, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 267, [27 | __Ss], [__T | __Stack]);
yeccpars2(27, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_900, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(28, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 263, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 264, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_800, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(29, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [29 | __Ss], [__T | __Stack]);
yeccpars2(29, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_700, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(30, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_600, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(31, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_500, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(32, '*', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 254, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 255, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'and', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 256, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'band', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 257, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'div', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 258, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'rem', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 259, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_400, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(33, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 242, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '++', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 243, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 244, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, '--', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 245, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, 'bor', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 246, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, 'bsl', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 247, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, 'bsr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 248, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, 'bxor', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 249, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, 'or', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 250, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, 'xor', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 251, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_300, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(34, '/=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 232, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '=/=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 233, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '=:=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 234, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '=<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 235, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '==', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 236, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '>', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 237, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, '>=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 238, [34 | __Ss], [__T | __Stack]);
yeccpars2(34, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_200, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(35, 'andalso', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 228, [35 | __Ss], [__T | __Stack]);
yeccpars2(35, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_160, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(36, 'orelse', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 226, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_150, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(37, '!', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 222, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_100, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(38, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(39, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 220, [39 | __Ss], [__T | __Stack]);
yeccpars2(39, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_39_(__Stack),
 yeccpars2(yeccgoto(exprs, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(40, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(41, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(42, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(43, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 203, [43 | __Ss], [__T | __Stack]);
yeccpars2(43, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(44, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [44 | __Ss], [__T | __Stack]);
yeccpars2(44, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(45, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_45_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(argument_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(46, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(47, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(48, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 200, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(49, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, '>>', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 182, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [49 | __Ss], [__T | __Stack]);
yeccpars2(49, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(50, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [50 | __Ss], [__T | __Stack]);
yeccpars2(50, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(51, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(52, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [52 | __Ss], [__T | __Stack]);
yeccpars2(52, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(53, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(54, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(55, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(56, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(57, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(58, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [58 | __Ss], [__T | __Stack]);
yeccpars2(58, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [58 | __Ss], [__T | __Stack]);
yeccpars2(58, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(59, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [59 | __Ss], [__T | __Stack]);
yeccpars2(59, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(60, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(61, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(62, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 126, [62 | __Ss], [__T | __Stack]);
yeccpars2(62, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(63, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'after', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 116, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [63 | __Ss], [__T | __Stack]);
yeccpars2(63, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(64, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(strings, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(65, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(66, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(67, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 69, [67 | __Ss], [__T | __Stack]);
yeccpars2(67, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(68, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 70, [68 | __Ss], [__T | __Stack]);
yeccpars2(68, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(69, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_69_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(tuple, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(70, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_70_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(tuple, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(71, 'after', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 73, [71 | __Ss], [__T | __Stack]);
yeccpars2(71, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [71 | __Ss], [__T | __Stack]);
yeccpars2(71, 'of', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 75, [71 | __Ss], [__T | __Stack]);
yeccpars2(71, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(72, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_72_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(try_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(73, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(74, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 94, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 95, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(75, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [75 | __Ss], [__T | __Stack]);
yeccpars2(75, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(76, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [76 | __Ss], [__T | __Stack]);
yeccpars2(76, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_76_(__Stack),
 yeccpars2(82, __Cat, [76 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(77, 'after', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 73, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(78, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 79, [78 | __Ss], [__T | __Stack]);
yeccpars2(78, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_78_(__Stack),
 yeccpars2(yeccgoto(cr_clauses, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(79, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(80, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_80_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(cr_clauses, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(81, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_81_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(try_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(82, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(83, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [83 | __Ss], [__T | __Stack]);
yeccpars2(83, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(84, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_84_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(clause_guard, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(85, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 86, [85 | __Ss], [__T | __Stack]);
yeccpars2(85, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_85_(__Stack),
 yeccpars2(yeccgoto(guard, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(86, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [86 | __Ss], [__T | __Stack]);
yeccpars2(86, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(87, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_87_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(guard, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(88, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_88_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(cr_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(89, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [89 | __Ss], [__T | __Stack]);
yeccpars2(89, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(90, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_90_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(clause_body, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(91, 'after', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 108, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 109, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(92, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 106, [92 | __Ss], [__T | __Stack]);
yeccpars2(92, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_92_(__Stack),
 yeccpars2(yeccgoto(try_clauses, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(93, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [93 | __Ss], [__T | __Stack]);
yeccpars2(93, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_93_(__Stack),
 yeccpars2(104, __Cat, [93 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(94, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 100, [94 | __Ss], [__T | __Stack]);
yeccpars2(94, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(95, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 96, [95 | __Ss], [__T | __Stack]);
yeccpars2(95, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(96, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(97, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [97 | __Ss], [__T | __Stack]);
yeccpars2(97, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_97_(__Stack),
 yeccpars2(98, __Cat, [97 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(98, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(99, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_99_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(try_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(100, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [100 | __Ss], [__T | __Stack]);
yeccpars2(100, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(101, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [101 | __Ss], [__T | __Stack]);
yeccpars2(101, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_101_(__Stack),
 yeccpars2(102, __Cat, [101 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(102, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [102 | __Ss], [__T | __Stack]);
yeccpars2(102, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(103, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_103_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(try_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(104, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [104 | __Ss], [__T | __Stack]);
yeccpars2(104, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(105, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_105_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(try_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(106, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 94, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 95, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [106 | __Ss], [__T | __Stack]);
yeccpars2(106, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(107, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_107_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(try_clauses, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(108, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(109, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_109_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(try_catch, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(110, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 111, [110 | __Ss], [__T | __Stack]);
yeccpars2(110, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(111, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_111_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(try_catch, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(112, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 113, [112 | __Ss], [__T | __Stack]);
yeccpars2(112, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(113, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_113_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(try_catch, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(114, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_114_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(strings, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(115, 'after', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 120, [115 | __Ss], [__T | __Stack]);
yeccpars2(115, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 121, [115 | __Ss], [__T | __Stack]);
yeccpars2(115, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(116, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [116 | __Ss], [__T | __Stack]);
yeccpars2(116, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(117, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [117 | __Ss], [__T | __Stack]);
yeccpars2(117, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(118, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 119, [118 | __Ss], [__T | __Stack]);
yeccpars2(118, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(119, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_119_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(receive_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(120, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [120 | __Ss], [__T | __Stack]);
yeccpars2(120, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(121, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_121_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(receive_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(122, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [122 | __Ss], [__T | __Stack]);
yeccpars2(122, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(123, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 124, [123 | __Ss], [__T | __Stack]);
yeccpars2(123, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(124, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_124_(__Stack),
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(receive_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(125, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 137, [125 | __Ss], [__T | __Stack]);
yeccpars2(125, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(126, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(127, '||', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [127 | __Ss], [__T | __Stack]);
yeccpars2(127, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(128, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(129, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [129 | __Ss], [__T | __Stack]);
yeccpars2(129, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(130, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [130 | __Ss], [__T | __Stack]);
yeccpars2(130, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_130_(__Stack),
 yeccpars2(yeccgoto(lc_exprs, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(131, '<-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [131 | __Ss], [__T | __Stack]);
yeccpars2(131, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(lc_expr, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(132, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [132 | __Ss], [__T | __Stack]);
yeccpars2(132, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(133, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_133_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(lc_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(134, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(135, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_135_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(lc_exprs, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(136, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_136_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(list_comprehension, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(137, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_137_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(query_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(138, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 144, [138 | __Ss], [__T | __Stack]);
yeccpars2(138, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(139, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [139 | __Ss], [__T | __Stack]);
yeccpars2(139, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_139_(__Stack),
 yeccpars2(yeccgoto(if_clauses, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(140, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [140 | __Ss], [__T | __Stack]);
yeccpars2(140, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(141, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_141_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(if_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(142, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [142 | __Ss], [__T | __Stack]);
yeccpars2(142, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(143, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_143_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(if_clauses, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(144, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_144_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(if_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(145, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 159, [145 | __Ss], [__T | __Stack]);
yeccpars2(145, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(146, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [146 | __Ss], [__T | __Stack]);
yeccpars2(146, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_146_(__Stack),
 yeccpars2(yeccgoto(fun_clauses, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(147, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [147 | __Ss], [__T | __Stack]);
yeccpars2(147, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_147_(__Stack),
 yeccpars2(155, __Cat, [147 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(148, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [148 | __Ss], [__T | __Stack]);
yeccpars2(148, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 150, [148 | __Ss], [__T | __Stack]);
yeccpars2(148, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(149, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [149 | __Ss], [__T | __Stack]);
yeccpars2(149, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(150, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 151, [150 | __Ss], [__T | __Stack]);
yeccpars2(150, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(151, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(152, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [152 | __Ss], [__T | __Stack]);
yeccpars2(152, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(153, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_153_(__Stack),
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(fun_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(154, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_154_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(fun_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(155, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [155 | __Ss], [__T | __Stack]);
yeccpars2(155, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(156, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_156_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fun_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(157, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [157 | __Ss], [__T | __Stack]);
yeccpars2(157, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(158, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_158_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fun_clauses, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(159, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_159_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fun_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(160, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_160_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(161, 'of', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 162, [161 | __Ss], [__T | __Stack]);
yeccpars2(161, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(162, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [162 | __Ss], [__T | __Stack]);
yeccpars2(162, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(163, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 164, [163 | __Ss], [__T | __Stack]);
yeccpars2(163, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(164, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_164_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(case_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(165, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 166, [165 | __Ss], [__T | __Stack]);
yeccpars2(165, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(166, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_166_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_max, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(167, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 170, [167 | __Ss], [__T | __Stack]);
yeccpars2(167, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 171, [167 | __Ss], [__T | __Stack]);
yeccpars2(167, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [167 | __Ss], [__T | __Stack]);
yeccpars2(167, '||', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [167 | __Ss], [__T | __Stack]);
yeccpars2(167, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(168, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_168_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(169, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_169_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(170, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [170 | __Ss], [__T | __Stack]);
yeccpars2(170, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(171, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_171_(__Stack),
 yeccpars2(yeccgoto(tail, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(172, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(173, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 174, [173 | __Ss], [__T | __Stack]);
yeccpars2(173, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(174, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_174_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(tail, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(175, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 170, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 171, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(176, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_176_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(tail, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(177, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [177 | __Ss], [__T | __Stack]);
yeccpars2(177, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(178, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(bit_expr, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(179, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 187, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_179_(__Stack),
 yeccpars2(186, __Cat, [179 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(180, '>>', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 185, [180 | __Ss], [__T | __Stack]);
yeccpars2(180, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(181, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 183, [181 | __Ss], [__T | __Stack]);
yeccpars2(181, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_181_(__Stack),
 yeccpars2(yeccgoto(bin_elements, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(182, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_182_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(binary, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(183, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [183 | __Ss], [__T | __Stack]);
yeccpars2(183, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(184, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_184_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(bin_elements, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(185, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_185_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(binary, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(186, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 191, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_186_(__Stack),
 yeccpars2(190, __Cat, [186 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(187, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [187 | __Ss], [__T | __Stack]);
yeccpars2(187, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(188, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(bit_size_expr, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(189, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_189_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(opt_bit_size_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(190, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_190_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(bin_element, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(191, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 194, [191 | __Ss], [__T | __Stack]);
yeccpars2(191, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(192, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_192_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(opt_bit_type_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(193, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 197, [193 | __Ss], [__T | __Stack]);
yeccpars2(193, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_193_(__Stack),
 yeccpars2(yeccgoto(bit_type_list, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(194, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 195, [194 | __Ss], [__T | __Stack]);
yeccpars2(194, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_194_(__Stack),
 yeccpars2(yeccgoto(bit_type, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(195, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 196, [195 | __Ss], [__T | __Stack]);
yeccpars2(195, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(196, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_196_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(bit_type, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(197, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 194, [197 | __Ss], [__T | __Stack]);
yeccpars2(197, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(198, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_198_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(bit_type_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(199, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_199_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(bit_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(200, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_200_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(expr_900, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(201, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 202, [201 | __Ss], [__T | __Stack]);
yeccpars2(201, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(202, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_202_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_max, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(203, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 205, [203 | __Ss], [__T | __Stack]);
yeccpars2(203, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 206, [203 | __Ss], [__T | __Stack]);
yeccpars2(203, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(204, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_204_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(205, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 219, [205 | __Ss], [__T | __Stack]);
yeccpars2(205, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(206, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 209, [206 | __Ss], [__T | __Stack]);
yeccpars2(206, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 210, [206 | __Ss], [__T | __Stack]);
yeccpars2(206, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 211, [206 | __Ss], [__T | __Stack]);
yeccpars2(206, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(207, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 218, [207 | __Ss], [__T | __Stack]);
yeccpars2(207, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(208, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 216, [208 | __Ss], [__T | __Stack]);
yeccpars2(208, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_208_(__Stack),
 yeccpars2(yeccgoto(record_fields, hd(__Ss)), __Cat, __Ss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(209, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 214, [209 | __Ss], [__T | __Stack]);
yeccpars2(209, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(210, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 212, [210 | __Ss], [__T | __Stack]);
yeccpars2(210, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(211, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_211_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(record_tuple, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(212, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [212 | __Ss], [__T | __Stack]);
yeccpars2(212, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(213, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_213_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_field, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(214, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [214 | __Ss], [__T | __Stack]);
yeccpars2(214, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(215, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_215_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_field, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(216, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 209, [216 | __Ss], [__T | __Stack]);
yeccpars2(216, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 210, [216 | __Ss], [__T | __Stack]);
yeccpars2(216, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(217, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_217_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_fields, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(218, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_218_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_tuple, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(219, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_219_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(220, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(221, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_221_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(exprs, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(222, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(223, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(224, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_224_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_100, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(225, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_225_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_100, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(226, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [226 | __Ss], [__T | __Stack]);
yeccpars2(226, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(227, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_227_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_150, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(228, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(229, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_229_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_160, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(230, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [230 | __Ss], [__T | __Stack]);
yeccpars2(230, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(231, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(232, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(233, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(234, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(235, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(236, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(237, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(238, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(239, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_239_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_200, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(240, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(241, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [241 | __Ss], [__T | __Stack]);
yeccpars2(241, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(242, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(243, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(list_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(244, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(245, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(list_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(246, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(247, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(248, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(249, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(250, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(251, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(252, '*', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 254, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 255, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, 'and', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 256, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, 'band', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 257, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, 'div', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 258, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, 'rem', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 259, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_252_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_400, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(253, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [253 | __Ss], [__T | __Stack]);
yeccpars2(253, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(254, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(255, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(256, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(257, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(258, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(259, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(260, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_260_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_500, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(261, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_261_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_300, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(262, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_262_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(function_call, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(263, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 266, [263 | __Ss], [__T | __Stack]);
yeccpars2(263, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(264, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(265, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_265_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_800, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(266, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_266_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_900, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(267, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 268, [267 | __Ss], [__T | __Stack]);
yeccpars2(267, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(268, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 270, [268 | __Ss], [__T | __Stack]);
yeccpars2(268, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 206, [268 | __Ss], [__T | __Stack]);
yeccpars2(268, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(269, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_269_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(270, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 271, [270 | __Ss], [__T | __Stack]);
yeccpars2(270, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(271, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_271_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(272, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_272_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(argument_list, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(273, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_273_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(expr_600, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(274, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [274 | __Ss], [__T | __Stack]);
yeccpars2(274, ':-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 277, [274 | __Ss], [__T | __Stack]);
yeccpars2(274, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(275, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_275_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(rule_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(276, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_276_(__Stack),
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(function_clause, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(277, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(278, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_278_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(rule_body, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(279, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 280, [279 | __Ss], [__T | __Stack]);
yeccpars2(279, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(280, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 43, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 46, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 47, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, '<<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 49, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 51, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 52, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 54, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, char, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 56, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 58, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 59, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 61, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 62, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, 'try', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 65, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [280 | __Ss], [__T | __Stack]);
yeccpars2(280, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(281, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars2(yeccgoto(attr_val, hd(__Ss)), __Cat, __Ss, __Stack, __T, __Ts, __Tzr);
yeccpars2(282, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 283, [282 | __Ss], [__T | __Stack]);
yeccpars2(282, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(283, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_283_(__Stack),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(attribute, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(284, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_284_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(form, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(285, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_285_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(form, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(286, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 288, [286 | __Ss], [__T | __Stack]);
yeccpars2(286, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(287, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_287_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(function_clauses, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(288, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [288 | __Ss], [__T | __Stack]);
yeccpars2(288, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(289, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [289 | __Ss], [__T | __Stack]);
yeccpars2(289, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_289_(__Stack),
 yeccpars2(290, __Cat, [289 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(290, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [290 | __Ss], [__T | __Stack]);
yeccpars2(290, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(291, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_291_(__Stack),
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(form, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(292, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 294, [292 | __Ss], [__T | __Stack]);
yeccpars2(292, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(293, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_293_(__Stack),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(rule_clauses, hd(__Nss)), __Cat, __Nss, __NewStack, __T, __Ts, __Tzr);
yeccpars2(294, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [294 | __Ss], [__T | __Stack]);
yeccpars2(294, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(295, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 83, [295 | __Ss], [__T | __Stack]);
yeccpars2(295, __Cat, __Ss, __Stack, __T, __Ts, __Tzr) ->
 __NewStack = yeccpars2_295_(__Stack),
 yeccpars2(296, __Cat, [295 | __Ss], __NewStack, __T, __Ts, __Tzr);
yeccpars2(296, ':-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 277, [296 | __Ss], [__T | __Stack]);
yeccpars2(296, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(__Other, _, _, _, _, _, _) ->
 erlang:error({yecc_bug,"1.1",{missing_state_in_action_table, __Other}}).

yeccgoto(add_op, 33) ->
 241;
yeccgoto(argument_list, 10) ->
 12;
yeccgoto(argument_list, 29) ->
 262;
yeccgoto(argument_list, 58) ->
 147;
yeccgoto(argument_list, 157) ->
 147;
yeccgoto(argument_list, 288) ->
 12;
yeccgoto(argument_list, 294) ->
 12;
yeccgoto(atomic, 13) ->
 42;
yeccgoto(atomic, 20) ->
 42;
yeccgoto(atomic, 44) ->
 42;
yeccgoto(atomic, 49) ->
 42;
yeccgoto(atomic, 50) ->
 42;
yeccgoto(atomic, 52) ->
 42;
yeccgoto(atomic, 54) ->
 42;
yeccgoto(atomic, 55) ->
 42;
yeccgoto(atomic, 59) ->
 42;
yeccgoto(atomic, 63) ->
 42;
yeccgoto(atomic, 65) ->
 42;
yeccgoto(atomic, 67) ->
 42;
yeccgoto(atomic, 73) ->
 42;
yeccgoto(atomic, 74) ->
 42;
yeccgoto(atomic, 75) ->
 42;
yeccgoto(atomic, 79) ->
 42;
yeccgoto(atomic, 83) ->
 42;
yeccgoto(atomic, 86) ->
 42;
yeccgoto(atomic, 89) ->
 42;
yeccgoto(atomic, 96) ->
 42;
yeccgoto(atomic, 100) ->
 42;
yeccgoto(atomic, 106) ->
 42;
yeccgoto(atomic, 108) ->
 42;
yeccgoto(atomic, 116) ->
 42;
yeccgoto(atomic, 120) ->
 42;
yeccgoto(atomic, 126) ->
 42;
yeccgoto(atomic, 128) ->
 42;
yeccgoto(atomic, 132) ->
 42;
yeccgoto(atomic, 134) ->
 42;
yeccgoto(atomic, 142) ->
 42;
yeccgoto(atomic, 162) ->
 42;
yeccgoto(atomic, 170) ->
 42;
yeccgoto(atomic, 172) ->
 42;
yeccgoto(atomic, 177) ->
 42;
yeccgoto(atomic, 183) ->
 42;
yeccgoto(atomic, 187) ->
 42;
yeccgoto(atomic, 212) ->
 42;
yeccgoto(atomic, 214) ->
 42;
yeccgoto(atomic, 220) ->
 42;
yeccgoto(atomic, 222) ->
 42;
yeccgoto(atomic, 223) ->
 42;
yeccgoto(atomic, 226) ->
 42;
yeccgoto(atomic, 228) ->
 42;
yeccgoto(atomic, 230) ->
 42;
yeccgoto(atomic, 240) ->
 42;
yeccgoto(atomic, 241) ->
 42;
yeccgoto(atomic, 253) ->
 42;
yeccgoto(atomic, 264) ->
 42;
yeccgoto(atomic, 277) ->
 42;
yeccgoto(atomic, 280) ->
 42;
yeccgoto(attr_val, 280) ->
 282;
yeccgoto(attribute, 0) ->
 8;
yeccgoto(bin_element, 49) ->
 181;
yeccgoto(bin_element, 183) ->
 181;
yeccgoto(bin_elements, 49) ->
 180;
yeccgoto(bin_elements, 183) ->
 184;
yeccgoto(binary, 13) ->
 41;
yeccgoto(binary, 20) ->
 41;
yeccgoto(binary, 44) ->
 41;
yeccgoto(binary, 49) ->
 41;
yeccgoto(binary, 50) ->
 41;
yeccgoto(binary, 52) ->
 41;
yeccgoto(binary, 54) ->
 41;
yeccgoto(binary, 55) ->
 41;
yeccgoto(binary, 59) ->
 41;
yeccgoto(binary, 63) ->
 41;
yeccgoto(binary, 65) ->
 41;
yeccgoto(binary, 67) ->
 41;
yeccgoto(binary, 73) ->
 41;
yeccgoto(binary, 74) ->
 41;
yeccgoto(binary, 75) ->
 41;
yeccgoto(binary, 79) ->
 41;
yeccgoto(binary, 83) ->
 41;
yeccgoto(binary, 86) ->
 41;
yeccgoto(binary, 89) ->
 41;
yeccgoto(binary, 96) ->
 41;
yeccgoto(binary, 100) ->
 41;
yeccgoto(binary, 106) ->
 41;
yeccgoto(binary, 108) ->
 41;
yeccgoto(binary, 116) ->
 41;
yeccgoto(binary, 120) ->
 41;
yeccgoto(binary, 126) ->
 41;
yeccgoto(binary, 128) ->
 41;
yeccgoto(binary, 132) ->
 41;
yeccgoto(binary, 134) ->
 41;
yeccgoto(binary, 142) ->
 41;
yeccgoto(binary, 162) ->
 41;
yeccgoto(binary, 170) ->
 41;
yeccgoto(binary, 172) ->
 41;
yeccgoto(binary, 177) ->
 41;
yeccgoto(binary, 183) ->
 41;
yeccgoto(binary, 187) ->
 41;
yeccgoto(binary, 212) ->
 41;
yeccgoto(binary, 214) ->
 41;
yeccgoto(binary, 220) ->
 41;
yeccgoto(binary, 222) ->
 41;
yeccgoto(binary, 223) ->
 41;
yeccgoto(binary, 226) ->
 41;
yeccgoto(binary, 228) ->
 41;
yeccgoto(binary, 230) ->
 41;
yeccgoto(binary, 240) ->
 41;
yeccgoto(binary, 241) ->
 41;
yeccgoto(binary, 253) ->
 41;
yeccgoto(binary, 264) ->
 41;
yeccgoto(binary, 277) ->
 41;
yeccgoto(binary, 280) ->
 41;
yeccgoto(bit_expr, 49) ->
 179;
yeccgoto(bit_expr, 183) ->
 179;
yeccgoto(bit_size_expr, 187) ->
 189;
yeccgoto(bit_type, 191) ->
 193;
yeccgoto(bit_type, 197) ->
 193;
yeccgoto(bit_type_list, 191) ->
 192;
yeccgoto(bit_type_list, 197) ->
 198;
yeccgoto(case_expr, 13) ->
 40;
yeccgoto(case_expr, 20) ->
 40;
yeccgoto(case_expr, 44) ->
 40;
yeccgoto(case_expr, 49) ->
 40;
yeccgoto(case_expr, 50) ->
 40;
yeccgoto(case_expr, 52) ->
 40;
yeccgoto(case_expr, 54) ->
 40;
yeccgoto(case_expr, 55) ->
 40;
yeccgoto(case_expr, 59) ->
 40;
yeccgoto(case_expr, 63) ->
 40;
yeccgoto(case_expr, 65) ->
 40;
yeccgoto(case_expr, 67) ->
 40;
yeccgoto(case_expr, 73) ->
 40;
yeccgoto(case_expr, 74) ->
 40;
yeccgoto(case_expr, 75) ->
 40;
yeccgoto(case_expr, 79) ->
 40;
yeccgoto(case_expr, 83) ->
 40;
yeccgoto(case_expr, 86) ->
 40;
yeccgoto(case_expr, 89) ->
 40;
yeccgoto(case_expr, 96) ->
 40;
yeccgoto(case_expr, 100) ->
 40;
yeccgoto(case_expr, 106) ->
 40;
yeccgoto(case_expr, 108) ->
 40;
yeccgoto(case_expr, 116) ->
 40;
yeccgoto(case_expr, 120) ->
 40;
yeccgoto(case_expr, 126) ->
 40;
yeccgoto(case_expr, 128) ->
 40;
yeccgoto(case_expr, 132) ->
 40;
yeccgoto(case_expr, 134) ->
 40;
yeccgoto(case_expr, 142) ->
 40;
yeccgoto(case_expr, 162) ->
 40;
yeccgoto(case_expr, 170) ->
 40;
yeccgoto(case_expr, 172) ->
 40;
yeccgoto(case_expr, 177) ->
 40;
yeccgoto(case_expr, 183) ->
 40;
yeccgoto(case_expr, 187) ->
 40;
yeccgoto(case_expr, 212) ->
 40;
yeccgoto(case_expr, 214) ->
 40;
yeccgoto(case_expr, 220) ->
 40;
yeccgoto(case_expr, 222) ->
 40;
yeccgoto(case_expr, 223) ->
 40;
yeccgoto(case_expr, 226) ->
 40;
yeccgoto(case_expr, 228) ->
 40;
yeccgoto(case_expr, 230) ->
 40;
yeccgoto(case_expr, 240) ->
 40;
yeccgoto(case_expr, 241) ->
 40;
yeccgoto(case_expr, 253) ->
 40;
yeccgoto(case_expr, 264) ->
 40;
yeccgoto(case_expr, 277) ->
 40;
yeccgoto(case_expr, 280) ->
 40;
yeccgoto(clause_args, 10) ->
 11;
yeccgoto(clause_args, 288) ->
 289;
yeccgoto(clause_args, 294) ->
 295;
yeccgoto(clause_body, 82) ->
 88;
yeccgoto(clause_body, 98) ->
 99;
yeccgoto(clause_body, 102) ->
 103;
yeccgoto(clause_body, 104) ->
 105;
yeccgoto(clause_body, 117) ->
 118;
yeccgoto(clause_body, 122) ->
 123;
yeccgoto(clause_body, 140) ->
 141;
yeccgoto(clause_body, 155) ->
 156;
yeccgoto(clause_body, 274) ->
 276;
yeccgoto(clause_body, 290) ->
 276;
yeccgoto(clause_guard, 11) ->
 274;
yeccgoto(clause_guard, 76) ->
 82;
yeccgoto(clause_guard, 93) ->
 104;
yeccgoto(clause_guard, 97) ->
 98;
yeccgoto(clause_guard, 101) ->
 102;
yeccgoto(clause_guard, 147) ->
 155;
yeccgoto(clause_guard, 289) ->
 290;
yeccgoto(clause_guard, 295) ->
 296;
yeccgoto(comp_op, 34) ->
 230;
yeccgoto(cr_clause, 63) ->
 78;
yeccgoto(cr_clause, 75) ->
 78;
yeccgoto(cr_clause, 79) ->
 78;
yeccgoto(cr_clause, 162) ->
 78;
yeccgoto(cr_clauses, 63) ->
 115;
yeccgoto(cr_clauses, 75) ->
 77;
yeccgoto(cr_clauses, 79) ->
 80;
yeccgoto(cr_clauses, 162) ->
 163;
yeccgoto(expr, 13) ->
 39;
yeccgoto(expr, 44) ->
 201;
yeccgoto(expr, 50) ->
 167;
yeccgoto(expr, 52) ->
 39;
yeccgoto(expr, 54) ->
 161;
yeccgoto(expr, 55) ->
 160;
yeccgoto(expr, 59) ->
 39;
yeccgoto(expr, 63) ->
 76;
yeccgoto(expr, 65) ->
 39;
yeccgoto(expr, 67) ->
 39;
yeccgoto(expr, 73) ->
 39;
yeccgoto(expr, 74) ->
 93;
yeccgoto(expr, 75) ->
 76;
yeccgoto(expr, 79) ->
 76;
yeccgoto(expr, 83) ->
 39;
yeccgoto(expr, 86) ->
 39;
yeccgoto(expr, 89) ->
 39;
yeccgoto(expr, 96) ->
 97;
yeccgoto(expr, 100) ->
 101;
yeccgoto(expr, 106) ->
 93;
yeccgoto(expr, 108) ->
 39;
yeccgoto(expr, 116) ->
 117;
yeccgoto(expr, 120) ->
 122;
yeccgoto(expr, 126) ->
 127;
yeccgoto(expr, 128) ->
 131;
yeccgoto(expr, 132) ->
 133;
yeccgoto(expr, 134) ->
 131;
yeccgoto(expr, 142) ->
 39;
yeccgoto(expr, 162) ->
 76;
yeccgoto(expr, 170) ->
 175;
yeccgoto(expr, 172) ->
 173;
yeccgoto(expr, 212) ->
 213;
yeccgoto(expr, 214) ->
 215;
yeccgoto(expr, 220) ->
 39;
yeccgoto(expr, 277) ->
 131;
yeccgoto(expr, 280) ->
 39;
yeccgoto(expr_100, 13) ->
 38;
yeccgoto(expr_100, 44) ->
 38;
yeccgoto(expr_100, 50) ->
 38;
yeccgoto(expr_100, 52) ->
 38;
yeccgoto(expr_100, 54) ->
 38;
yeccgoto(expr_100, 55) ->
 38;
yeccgoto(expr_100, 59) ->
 38;
yeccgoto(expr_100, 63) ->
 38;
yeccgoto(expr_100, 65) ->
 38;
yeccgoto(expr_100, 67) ->
 38;
yeccgoto(expr_100, 73) ->
 38;
yeccgoto(expr_100, 74) ->
 38;
yeccgoto(expr_100, 75) ->
 38;
yeccgoto(expr_100, 79) ->
 38;
yeccgoto(expr_100, 83) ->
 38;
yeccgoto(expr_100, 86) ->
 38;
yeccgoto(expr_100, 89) ->
 38;
yeccgoto(expr_100, 96) ->
 38;
yeccgoto(expr_100, 100) ->
 38;
yeccgoto(expr_100, 106) ->
 38;
yeccgoto(expr_100, 108) ->
 38;
yeccgoto(expr_100, 116) ->
 38;
yeccgoto(expr_100, 120) ->
 38;
yeccgoto(expr_100, 126) ->
 38;
yeccgoto(expr_100, 128) ->
 38;
yeccgoto(expr_100, 132) ->
 38;
yeccgoto(expr_100, 134) ->
 38;
yeccgoto(expr_100, 142) ->
 38;
yeccgoto(expr_100, 162) ->
 38;
yeccgoto(expr_100, 170) ->
 38;
yeccgoto(expr_100, 172) ->
 38;
yeccgoto(expr_100, 212) ->
 38;
yeccgoto(expr_100, 214) ->
 38;
yeccgoto(expr_100, 220) ->
 38;
yeccgoto(expr_100, 222) ->
 225;
yeccgoto(expr_100, 223) ->
 224;
yeccgoto(expr_100, 277) ->
 38;
yeccgoto(expr_100, 280) ->
 38;
yeccgoto(expr_150, 13) ->
 37;
yeccgoto(expr_150, 44) ->
 37;
yeccgoto(expr_150, 50) ->
 37;
yeccgoto(expr_150, 52) ->
 37;
yeccgoto(expr_150, 54) ->
 37;
yeccgoto(expr_150, 55) ->
 37;
yeccgoto(expr_150, 59) ->
 37;
yeccgoto(expr_150, 63) ->
 37;
yeccgoto(expr_150, 65) ->
 37;
yeccgoto(expr_150, 67) ->
 37;
yeccgoto(expr_150, 73) ->
 37;
yeccgoto(expr_150, 74) ->
 37;
yeccgoto(expr_150, 75) ->
 37;
yeccgoto(expr_150, 79) ->
 37;
yeccgoto(expr_150, 83) ->
 37;
yeccgoto(expr_150, 86) ->
 37;
yeccgoto(expr_150, 89) ->
 37;
yeccgoto(expr_150, 96) ->
 37;
yeccgoto(expr_150, 100) ->
 37;
yeccgoto(expr_150, 106) ->
 37;
yeccgoto(expr_150, 108) ->
 37;
yeccgoto(expr_150, 116) ->
 37;
yeccgoto(expr_150, 120) ->
 37;
yeccgoto(expr_150, 126) ->
 37;
yeccgoto(expr_150, 128) ->
 37;
yeccgoto(expr_150, 132) ->
 37;
yeccgoto(expr_150, 134) ->
 37;
yeccgoto(expr_150, 142) ->
 37;
yeccgoto(expr_150, 162) ->
 37;
yeccgoto(expr_150, 170) ->
 37;
yeccgoto(expr_150, 172) ->
 37;
yeccgoto(expr_150, 212) ->
 37;
yeccgoto(expr_150, 214) ->
 37;
yeccgoto(expr_150, 220) ->
 37;
yeccgoto(expr_150, 222) ->
 37;
yeccgoto(expr_150, 223) ->
 37;
yeccgoto(expr_150, 226) ->
 227;
yeccgoto(expr_150, 277) ->
 37;
yeccgoto(expr_150, 280) ->
 37;
yeccgoto(expr_160, 13) ->
 36;
yeccgoto(expr_160, 44) ->
 36;
yeccgoto(expr_160, 50) ->
 36;
yeccgoto(expr_160, 52) ->
 36;
yeccgoto(expr_160, 54) ->
 36;
yeccgoto(expr_160, 55) ->
 36;
yeccgoto(expr_160, 59) ->
 36;
yeccgoto(expr_160, 63) ->
 36;
yeccgoto(expr_160, 65) ->
 36;
yeccgoto(expr_160, 67) ->
 36;
yeccgoto(expr_160, 73) ->
 36;
yeccgoto(expr_160, 74) ->
 36;
yeccgoto(expr_160, 75) ->
 36;
yeccgoto(expr_160, 79) ->
 36;
yeccgoto(expr_160, 83) ->
 36;
yeccgoto(expr_160, 86) ->
 36;
yeccgoto(expr_160, 89) ->
 36;
yeccgoto(expr_160, 96) ->
 36;
yeccgoto(expr_160, 100) ->
 36;
yeccgoto(expr_160, 106) ->
 36;
yeccgoto(expr_160, 108) ->
 36;
yeccgoto(expr_160, 116) ->
 36;
yeccgoto(expr_160, 120) ->
 36;
yeccgoto(expr_160, 126) ->
 36;
yeccgoto(expr_160, 128) ->
 36;
yeccgoto(expr_160, 132) ->
 36;
yeccgoto(expr_160, 134) ->
 36;
yeccgoto(expr_160, 142) ->
 36;
yeccgoto(expr_160, 162) ->
 36;
yeccgoto(expr_160, 170) ->
 36;
yeccgoto(expr_160, 172) ->
 36;
yeccgoto(expr_160, 212) ->
 36;
yeccgoto(expr_160, 214) ->
 36;
yeccgoto(expr_160, 220) ->
 36;
yeccgoto(expr_160, 222) ->
 36;
yeccgoto(expr_160, 223) ->
 36;
yeccgoto(expr_160, 226) ->
 36;
yeccgoto(expr_160, 228) ->
 229;
yeccgoto(expr_160, 277) ->
 36;
yeccgoto(expr_160, 280) ->
 36;
yeccgoto(expr_200, 13) ->
 35;
yeccgoto(expr_200, 44) ->
 35;
yeccgoto(expr_200, 50) ->
 35;
yeccgoto(expr_200, 52) ->
 35;
yeccgoto(expr_200, 54) ->
 35;
yeccgoto(expr_200, 55) ->
 35;
yeccgoto(expr_200, 59) ->
 35;
yeccgoto(expr_200, 63) ->
 35;
yeccgoto(expr_200, 65) ->
 35;
yeccgoto(expr_200, 67) ->
 35;
yeccgoto(expr_200, 73) ->
 35;
yeccgoto(expr_200, 74) ->
 35;
yeccgoto(expr_200, 75) ->
 35;
yeccgoto(expr_200, 79) ->
 35;
yeccgoto(expr_200, 83) ->
 35;
yeccgoto(expr_200, 86) ->
 35;
yeccgoto(expr_200, 89) ->
 35;
yeccgoto(expr_200, 96) ->
 35;
yeccgoto(expr_200, 100) ->
 35;
yeccgoto(expr_200, 106) ->
 35;
yeccgoto(expr_200, 108) ->
 35;
yeccgoto(expr_200, 116) ->
 35;
yeccgoto(expr_200, 120) ->
 35;
yeccgoto(expr_200, 126) ->
 35;
yeccgoto(expr_200, 128) ->
 35;
yeccgoto(expr_200, 132) ->
 35;
yeccgoto(expr_200, 134) ->
 35;
yeccgoto(expr_200, 142) ->
 35;
yeccgoto(expr_200, 162) ->
 35;
yeccgoto(expr_200, 170) ->
 35;
yeccgoto(expr_200, 172) ->
 35;
yeccgoto(expr_200, 212) ->
 35;
yeccgoto(expr_200, 214) ->
 35;
yeccgoto(expr_200, 220) ->
 35;
yeccgoto(expr_200, 222) ->
 35;
yeccgoto(expr_200, 223) ->
 35;
yeccgoto(expr_200, 226) ->
 35;
yeccgoto(expr_200, 228) ->
 35;
yeccgoto(expr_200, 277) ->
 35;
yeccgoto(expr_200, 280) ->
 35;
yeccgoto(expr_300, 13) ->
 34;
yeccgoto(expr_300, 44) ->
 34;
yeccgoto(expr_300, 50) ->
 34;
yeccgoto(expr_300, 52) ->
 34;
yeccgoto(expr_300, 54) ->
 34;
yeccgoto(expr_300, 55) ->
 34;
yeccgoto(expr_300, 59) ->
 34;
yeccgoto(expr_300, 63) ->
 34;
yeccgoto(expr_300, 65) ->
 34;
yeccgoto(expr_300, 67) ->
 34;
yeccgoto(expr_300, 73) ->
 34;
yeccgoto(expr_300, 74) ->
 34;
yeccgoto(expr_300, 75) ->
 34;
yeccgoto(expr_300, 79) ->
 34;
yeccgoto(expr_300, 83) ->
 34;
yeccgoto(expr_300, 86) ->
 34;
yeccgoto(expr_300, 89) ->
 34;
yeccgoto(expr_300, 96) ->
 34;
yeccgoto(expr_300, 100) ->
 34;
yeccgoto(expr_300, 106) ->
 34;
yeccgoto(expr_300, 108) ->
 34;
yeccgoto(expr_300, 116) ->
 34;
yeccgoto(expr_300, 120) ->
 34;
yeccgoto(expr_300, 126) ->
 34;
yeccgoto(expr_300, 128) ->
 34;
yeccgoto(expr_300, 132) ->
 34;
yeccgoto(expr_300, 134) ->
 34;
yeccgoto(expr_300, 142) ->
 34;
yeccgoto(expr_300, 162) ->
 34;
yeccgoto(expr_300, 170) ->
 34;
yeccgoto(expr_300, 172) ->
 34;
yeccgoto(expr_300, 212) ->
 34;
yeccgoto(expr_300, 214) ->
 34;
yeccgoto(expr_300, 220) ->
 34;
yeccgoto(expr_300, 222) ->
 34;
yeccgoto(expr_300, 223) ->
 34;
yeccgoto(expr_300, 226) ->
 34;
yeccgoto(expr_300, 228) ->
 34;
yeccgoto(expr_300, 230) ->
 239;
yeccgoto(expr_300, 240) ->
 261;
yeccgoto(expr_300, 277) ->
 34;
yeccgoto(expr_300, 280) ->
 34;
yeccgoto(expr_400, 13) ->
 33;
yeccgoto(expr_400, 44) ->
 33;
yeccgoto(expr_400, 50) ->
 33;
yeccgoto(expr_400, 52) ->
 33;
yeccgoto(expr_400, 54) ->
 33;
yeccgoto(expr_400, 55) ->
 33;
yeccgoto(expr_400, 59) ->
 33;
yeccgoto(expr_400, 63) ->
 33;
yeccgoto(expr_400, 65) ->
 33;
yeccgoto(expr_400, 67) ->
 33;
yeccgoto(expr_400, 73) ->
 33;
yeccgoto(expr_400, 74) ->
 33;
yeccgoto(expr_400, 75) ->
 33;
yeccgoto(expr_400, 79) ->
 33;
yeccgoto(expr_400, 83) ->
 33;
yeccgoto(expr_400, 86) ->
 33;
yeccgoto(expr_400, 89) ->
 33;
yeccgoto(expr_400, 96) ->
 33;
yeccgoto(expr_400, 100) ->
 33;
yeccgoto(expr_400, 106) ->
 33;
yeccgoto(expr_400, 108) ->
 33;
yeccgoto(expr_400, 116) ->
 33;
yeccgoto(expr_400, 120) ->
 33;
yeccgoto(expr_400, 126) ->
 33;
yeccgoto(expr_400, 128) ->
 33;
yeccgoto(expr_400, 132) ->
 33;
yeccgoto(expr_400, 134) ->
 33;
yeccgoto(expr_400, 142) ->
 33;
yeccgoto(expr_400, 162) ->
 33;
yeccgoto(expr_400, 170) ->
 33;
yeccgoto(expr_400, 172) ->
 33;
yeccgoto(expr_400, 212) ->
 33;
yeccgoto(expr_400, 214) ->
 33;
yeccgoto(expr_400, 220) ->
 33;
yeccgoto(expr_400, 222) ->
 33;
yeccgoto(expr_400, 223) ->
 33;
yeccgoto(expr_400, 226) ->
 33;
yeccgoto(expr_400, 228) ->
 33;
yeccgoto(expr_400, 230) ->
 33;
yeccgoto(expr_400, 240) ->
 33;
yeccgoto(expr_400, 277) ->
 33;
yeccgoto(expr_400, 280) ->
 33;
yeccgoto(expr_500, 13) ->
 32;
yeccgoto(expr_500, 44) ->
 32;
yeccgoto(expr_500, 50) ->
 32;
yeccgoto(expr_500, 52) ->
 32;
yeccgoto(expr_500, 54) ->
 32;
yeccgoto(expr_500, 55) ->
 32;
yeccgoto(expr_500, 59) ->
 32;
yeccgoto(expr_500, 63) ->
 32;
yeccgoto(expr_500, 65) ->
 32;
yeccgoto(expr_500, 67) ->
 32;
yeccgoto(expr_500, 73) ->
 32;
yeccgoto(expr_500, 74) ->
 32;
yeccgoto(expr_500, 75) ->
 32;
yeccgoto(expr_500, 79) ->
 32;
yeccgoto(expr_500, 83) ->
 32;
yeccgoto(expr_500, 86) ->
 32;
yeccgoto(expr_500, 89) ->
 32;
yeccgoto(expr_500, 96) ->
 32;
yeccgoto(expr_500, 100) ->
 32;
yeccgoto(expr_500, 106) ->
 32;
yeccgoto(expr_500, 108) ->
 32;
yeccgoto(expr_500, 116) ->
 32;
yeccgoto(expr_500, 120) ->
 32;
yeccgoto(expr_500, 126) ->
 32;
yeccgoto(expr_500, 128) ->
 32;
yeccgoto(expr_500, 132) ->
 32;
yeccgoto(expr_500, 134) ->
 32;
yeccgoto(expr_500, 142) ->
 32;
yeccgoto(expr_500, 162) ->
 32;
yeccgoto(expr_500, 170) ->
 32;
yeccgoto(expr_500, 172) ->
 32;
yeccgoto(expr_500, 212) ->
 32;
yeccgoto(expr_500, 214) ->
 32;
yeccgoto(expr_500, 220) ->
 32;
yeccgoto(expr_500, 222) ->
 32;
yeccgoto(expr_500, 223) ->
 32;
yeccgoto(expr_500, 226) ->
 32;
yeccgoto(expr_500, 228) ->
 32;
yeccgoto(expr_500, 230) ->
 32;
yeccgoto(expr_500, 240) ->
 32;
yeccgoto(expr_500, 241) ->
 252;
yeccgoto(expr_500, 277) ->
 32;
yeccgoto(expr_500, 280) ->
 32;
yeccgoto(expr_600, 13) ->
 31;
yeccgoto(expr_600, 44) ->
 31;
yeccgoto(expr_600, 50) ->
 31;
yeccgoto(expr_600, 52) ->
 31;
yeccgoto(expr_600, 54) ->
 31;
yeccgoto(expr_600, 55) ->
 31;
yeccgoto(expr_600, 59) ->
 31;
yeccgoto(expr_600, 63) ->
 31;
yeccgoto(expr_600, 65) ->
 31;
yeccgoto(expr_600, 67) ->
 31;
yeccgoto(expr_600, 73) ->
 31;
yeccgoto(expr_600, 74) ->
 31;
yeccgoto(expr_600, 75) ->
 31;
yeccgoto(expr_600, 79) ->
 31;
yeccgoto(expr_600, 83) ->
 31;
yeccgoto(expr_600, 86) ->
 31;
yeccgoto(expr_600, 89) ->
 31;
yeccgoto(expr_600, 96) ->
 31;
yeccgoto(expr_600, 100) ->
 31;
yeccgoto(expr_600, 106) ->
 31;
yeccgoto(expr_600, 108) ->
 31;
yeccgoto(expr_600, 116) ->
 31;
yeccgoto(expr_600, 120) ->
 31;
yeccgoto(expr_600, 126) ->
 31;
yeccgoto(expr_600, 128) ->
 31;
yeccgoto(expr_600, 132) ->
 31;
yeccgoto(expr_600, 134) ->
 31;
yeccgoto(expr_600, 142) ->
 31;
yeccgoto(expr_600, 162) ->
 31;
yeccgoto(expr_600, 170) ->
 31;
yeccgoto(expr_600, 172) ->
 31;
yeccgoto(expr_600, 212) ->
 31;
yeccgoto(expr_600, 214) ->
 31;
yeccgoto(expr_600, 220) ->
 31;
yeccgoto(expr_600, 222) ->
 31;
yeccgoto(expr_600, 223) ->
 31;
yeccgoto(expr_600, 226) ->
 31;
yeccgoto(expr_600, 228) ->
 31;
yeccgoto(expr_600, 230) ->
 31;
yeccgoto(expr_600, 240) ->
 31;
yeccgoto(expr_600, 241) ->
 31;
yeccgoto(expr_600, 253) ->
 260;
yeccgoto(expr_600, 277) ->
 31;
yeccgoto(expr_600, 280) ->
 31;
yeccgoto(expr_700, 13) ->
 30;
yeccgoto(expr_700, 20) ->
 273;
yeccgoto(expr_700, 44) ->
 30;
yeccgoto(expr_700, 50) ->
 30;
yeccgoto(expr_700, 52) ->
 30;
yeccgoto(expr_700, 54) ->
 30;
yeccgoto(expr_700, 55) ->
 30;
yeccgoto(expr_700, 59) ->
 30;
yeccgoto(expr_700, 63) ->
 30;
yeccgoto(expr_700, 65) ->
 30;
yeccgoto(expr_700, 67) ->
 30;
yeccgoto(expr_700, 73) ->
 30;
yeccgoto(expr_700, 74) ->
 30;
yeccgoto(expr_700, 75) ->
 30;
yeccgoto(expr_700, 79) ->
 30;
yeccgoto(expr_700, 83) ->
 30;
yeccgoto(expr_700, 86) ->
 30;
yeccgoto(expr_700, 89) ->
 30;
yeccgoto(expr_700, 96) ->
 30;
yeccgoto(expr_700, 100) ->
 30;
yeccgoto(expr_700, 106) ->
 30;
yeccgoto(expr_700, 108) ->
 30;
yeccgoto(expr_700, 116) ->
 30;
yeccgoto(expr_700, 120) ->
 30;
yeccgoto(expr_700, 126) ->
 30;
yeccgoto(expr_700, 128) ->
 30;
yeccgoto(expr_700, 132) ->
 30;
yeccgoto(expr_700, 134) ->
 30;
yeccgoto(expr_700, 142) ->
 30;
yeccgoto(expr_700, 162) ->
 30;
yeccgoto(expr_700, 170) ->
 30;
yeccgoto(expr_700, 172) ->
 30;
yeccgoto(expr_700, 212) ->
 30;
yeccgoto(expr_700, 214) ->
 30;
yeccgoto(expr_700, 220) ->
 30;
yeccgoto(expr_700, 222) ->
 30;
yeccgoto(expr_700, 223) ->
 30;
yeccgoto(expr_700, 226) ->
 30;
yeccgoto(expr_700, 228) ->
 30;
yeccgoto(expr_700, 230) ->
 30;
yeccgoto(expr_700, 240) ->
 30;
yeccgoto(expr_700, 241) ->
 30;
yeccgoto(expr_700, 253) ->
 30;
yeccgoto(expr_700, 277) ->
 30;
yeccgoto(expr_700, 280) ->
 30;
yeccgoto(expr_800, 13) ->
 29;
yeccgoto(expr_800, 20) ->
 29;
yeccgoto(expr_800, 44) ->
 29;
yeccgoto(expr_800, 50) ->
 29;
yeccgoto(expr_800, 52) ->
 29;
yeccgoto(expr_800, 54) ->
 29;
yeccgoto(expr_800, 55) ->
 29;
yeccgoto(expr_800, 59) ->
 29;
yeccgoto(expr_800, 63) ->
 29;
yeccgoto(expr_800, 65) ->
 29;
yeccgoto(expr_800, 67) ->
 29;
yeccgoto(expr_800, 73) ->
 29;
yeccgoto(expr_800, 74) ->
 29;
yeccgoto(expr_800, 75) ->
 29;
yeccgoto(expr_800, 79) ->
 29;
yeccgoto(expr_800, 83) ->
 29;
yeccgoto(expr_800, 86) ->
 29;
yeccgoto(expr_800, 89) ->
 29;
yeccgoto(expr_800, 96) ->
 29;
yeccgoto(expr_800, 100) ->
 29;
yeccgoto(expr_800, 106) ->
 29;
yeccgoto(expr_800, 108) ->
 29;
yeccgoto(expr_800, 116) ->
 29;
yeccgoto(expr_800, 120) ->
 29;
yeccgoto(expr_800, 126) ->
 29;
yeccgoto(expr_800, 128) ->
 29;
yeccgoto(expr_800, 132) ->
 29;
yeccgoto(expr_800, 134) ->
 29;
yeccgoto(expr_800, 142) ->
 29;
yeccgoto(expr_800, 162) ->
 29;
yeccgoto(expr_800, 170) ->
 29;
yeccgoto(expr_800, 172) ->
 29;
yeccgoto(expr_800, 212) ->
 29;
yeccgoto(expr_800, 214) ->
 29;
yeccgoto(expr_800, 220) ->
 29;
yeccgoto(expr_800, 222) ->
 29;
yeccgoto(expr_800, 223) ->
 29;
yeccgoto(expr_800, 226) ->
 29;
yeccgoto(expr_800, 228) ->
 29;
yeccgoto(expr_800, 230) ->
 29;
yeccgoto(expr_800, 240) ->
 29;
yeccgoto(expr_800, 241) ->
 29;
yeccgoto(expr_800, 253) ->
 29;
yeccgoto(expr_800, 277) ->
 29;
yeccgoto(expr_800, 280) ->
 29;
yeccgoto(expr_900, 13) ->
 28;
yeccgoto(expr_900, 20) ->
 28;
yeccgoto(expr_900, 44) ->
 28;
yeccgoto(expr_900, 50) ->
 28;
yeccgoto(expr_900, 52) ->
 28;
yeccgoto(expr_900, 54) ->
 28;
yeccgoto(expr_900, 55) ->
 28;
yeccgoto(expr_900, 59) ->
 28;
yeccgoto(expr_900, 63) ->
 28;
yeccgoto(expr_900, 65) ->
 28;
yeccgoto(expr_900, 67) ->
 28;
yeccgoto(expr_900, 73) ->
 28;
yeccgoto(expr_900, 74) ->
 28;
yeccgoto(expr_900, 75) ->
 28;
yeccgoto(expr_900, 79) ->
 28;
yeccgoto(expr_900, 83) ->
 28;
yeccgoto(expr_900, 86) ->
 28;
yeccgoto(expr_900, 89) ->
 28;
yeccgoto(expr_900, 96) ->
 28;
yeccgoto(expr_900, 100) ->
 28;
yeccgoto(expr_900, 106) ->
 28;
yeccgoto(expr_900, 108) ->
 28;
yeccgoto(expr_900, 116) ->
 28;
yeccgoto(expr_900, 120) ->
 28;
yeccgoto(expr_900, 126) ->
 28;
yeccgoto(expr_900, 128) ->
 28;
yeccgoto(expr_900, 132) ->
 28;
yeccgoto(expr_900, 134) ->
 28;
yeccgoto(expr_900, 142) ->
 28;
yeccgoto(expr_900, 162) ->
 28;
yeccgoto(expr_900, 170) ->
 28;
yeccgoto(expr_900, 172) ->
 28;
yeccgoto(expr_900, 212) ->
 28;
yeccgoto(expr_900, 214) ->
 28;
yeccgoto(expr_900, 220) ->
 28;
yeccgoto(expr_900, 222) ->
 28;
yeccgoto(expr_900, 223) ->
 28;
yeccgoto(expr_900, 226) ->
 28;
yeccgoto(expr_900, 228) ->
 28;
yeccgoto(expr_900, 230) ->
 28;
yeccgoto(expr_900, 240) ->
 28;
yeccgoto(expr_900, 241) ->
 28;
yeccgoto(expr_900, 253) ->
 28;
yeccgoto(expr_900, 277) ->
 28;
yeccgoto(expr_900, 280) ->
 28;
yeccgoto(expr_max, 13) ->
 27;
yeccgoto(expr_max, 20) ->
 27;
yeccgoto(expr_max, 44) ->
 27;
yeccgoto(expr_max, 49) ->
 178;
yeccgoto(expr_max, 50) ->
 27;
yeccgoto(expr_max, 52) ->
 27;
yeccgoto(expr_max, 54) ->
 27;
yeccgoto(expr_max, 55) ->
 27;
yeccgoto(expr_max, 59) ->
 27;
yeccgoto(expr_max, 63) ->
 27;
yeccgoto(expr_max, 65) ->
 27;
yeccgoto(expr_max, 67) ->
 27;
yeccgoto(expr_max, 73) ->
 27;
yeccgoto(expr_max, 74) ->
 27;
yeccgoto(expr_max, 75) ->
 27;
yeccgoto(expr_max, 79) ->
 27;
yeccgoto(expr_max, 83) ->
 27;
yeccgoto(expr_max, 86) ->
 27;
yeccgoto(expr_max, 89) ->
 27;
yeccgoto(expr_max, 96) ->
 27;
yeccgoto(expr_max, 100) ->
 27;
yeccgoto(expr_max, 106) ->
 27;
yeccgoto(expr_max, 108) ->
 27;
yeccgoto(expr_max, 116) ->
 27;
yeccgoto(expr_max, 120) ->
 27;
yeccgoto(expr_max, 126) ->
 27;
yeccgoto(expr_max, 128) ->
 27;
yeccgoto(expr_max, 132) ->
 27;
yeccgoto(expr_max, 134) ->
 27;
yeccgoto(expr_max, 142) ->
 27;
yeccgoto(expr_max, 162) ->
 27;
yeccgoto(expr_max, 170) ->
 27;
yeccgoto(expr_max, 172) ->
 27;
yeccgoto(expr_max, 177) ->
 199;
yeccgoto(expr_max, 183) ->
 178;
yeccgoto(expr_max, 187) ->
 188;
yeccgoto(expr_max, 212) ->
 27;
yeccgoto(expr_max, 214) ->
 27;
yeccgoto(expr_max, 220) ->
 27;
yeccgoto(expr_max, 222) ->
 27;
yeccgoto(expr_max, 223) ->
 27;
yeccgoto(expr_max, 226) ->
 27;
yeccgoto(expr_max, 228) ->
 27;
yeccgoto(expr_max, 230) ->
 27;
yeccgoto(expr_max, 240) ->
 27;
yeccgoto(expr_max, 241) ->
 27;
yeccgoto(expr_max, 253) ->
 27;
yeccgoto(expr_max, 264) ->
 265;
yeccgoto(expr_max, 277) ->
 27;
yeccgoto(expr_max, 280) ->
 27;
yeccgoto(exprs, 13) ->
 26;
yeccgoto(exprs, 52) ->
 165;
yeccgoto(exprs, 59) ->
 85;
yeccgoto(exprs, 65) ->
 71;
yeccgoto(exprs, 67) ->
 68;
yeccgoto(exprs, 73) ->
 112;
yeccgoto(exprs, 83) ->
 85;
yeccgoto(exprs, 86) ->
 85;
yeccgoto(exprs, 89) ->
 90;
yeccgoto(exprs, 108) ->
 110;
yeccgoto(exprs, 142) ->
 85;
yeccgoto(exprs, 220) ->
 221;
yeccgoto(exprs, 280) ->
 281;
yeccgoto(form, 0) ->
 7;
yeccgoto(fun_clause, 58) ->
 146;
yeccgoto(fun_clause, 157) ->
 146;
yeccgoto(fun_clauses, 58) ->
 145;
yeccgoto(fun_clauses, 157) ->
 158;
yeccgoto(fun_expr, 13) ->
 25;
yeccgoto(fun_expr, 20) ->
 25;
yeccgoto(fun_expr, 44) ->
 25;
yeccgoto(fun_expr, 49) ->
 25;
yeccgoto(fun_expr, 50) ->
 25;
yeccgoto(fun_expr, 52) ->
 25;
yeccgoto(fun_expr, 54) ->
 25;
yeccgoto(fun_expr, 55) ->
 25;
yeccgoto(fun_expr, 59) ->
 25;
yeccgoto(fun_expr, 63) ->
 25;
yeccgoto(fun_expr, 65) ->
 25;
yeccgoto(fun_expr, 67) ->
 25;
yeccgoto(fun_expr, 73) ->
 25;
yeccgoto(fun_expr, 74) ->
 25;
yeccgoto(fun_expr, 75) ->
 25;
yeccgoto(fun_expr, 79) ->
 25;
yeccgoto(fun_expr, 83) ->
 25;
yeccgoto(fun_expr, 86) ->
 25;
yeccgoto(fun_expr, 89) ->
 25;
yeccgoto(fun_expr, 96) ->
 25;
yeccgoto(fun_expr, 100) ->
 25;
yeccgoto(fun_expr, 106) ->
 25;
yeccgoto(fun_expr, 108) ->
 25;
yeccgoto(fun_expr, 116) ->
 25;
yeccgoto(fun_expr, 120) ->
 25;
yeccgoto(fun_expr, 126) ->
 25;
yeccgoto(fun_expr, 128) ->
 25;
yeccgoto(fun_expr, 132) ->
 25;
yeccgoto(fun_expr, 134) ->
 25;
yeccgoto(fun_expr, 142) ->
 25;
yeccgoto(fun_expr, 162) ->
 25;
yeccgoto(fun_expr, 170) ->
 25;
yeccgoto(fun_expr, 172) ->
 25;
yeccgoto(fun_expr, 177) ->
 25;
yeccgoto(fun_expr, 183) ->
 25;
yeccgoto(fun_expr, 187) ->
 25;
yeccgoto(fun_expr, 212) ->
 25;
yeccgoto(fun_expr, 214) ->
 25;
yeccgoto(fun_expr, 220) ->
 25;
yeccgoto(fun_expr, 222) ->
 25;
yeccgoto(fun_expr, 223) ->
 25;
yeccgoto(fun_expr, 226) ->
 25;
yeccgoto(fun_expr, 228) ->
 25;
yeccgoto(fun_expr, 230) ->
 25;
yeccgoto(fun_expr, 240) ->
 25;
yeccgoto(fun_expr, 241) ->
 25;
yeccgoto(fun_expr, 253) ->
 25;
yeccgoto(fun_expr, 264) ->
 25;
yeccgoto(fun_expr, 277) ->
 25;
yeccgoto(fun_expr, 280) ->
 25;
yeccgoto(function, 0) ->
 6;
yeccgoto(function_call, 13) ->
 24;
yeccgoto(function_call, 20) ->
 24;
yeccgoto(function_call, 44) ->
 24;
yeccgoto(function_call, 50) ->
 24;
yeccgoto(function_call, 52) ->
 24;
yeccgoto(function_call, 54) ->
 24;
yeccgoto(function_call, 55) ->
 24;
yeccgoto(function_call, 59) ->
 24;
yeccgoto(function_call, 63) ->
 24;
yeccgoto(function_call, 65) ->
 24;
yeccgoto(function_call, 67) ->
 24;
yeccgoto(function_call, 73) ->
 24;
yeccgoto(function_call, 74) ->
 24;
yeccgoto(function_call, 75) ->
 24;
yeccgoto(function_call, 79) ->
 24;
yeccgoto(function_call, 83) ->
 24;
yeccgoto(function_call, 86) ->
 24;
yeccgoto(function_call, 89) ->
 24;
yeccgoto(function_call, 96) ->
 24;
yeccgoto(function_call, 100) ->
 24;
yeccgoto(function_call, 106) ->
 24;
yeccgoto(function_call, 108) ->
 24;
yeccgoto(function_call, 116) ->
 24;
yeccgoto(function_call, 120) ->
 24;
yeccgoto(function_call, 126) ->
 24;
yeccgoto(function_call, 128) ->
 24;
yeccgoto(function_call, 132) ->
 24;
yeccgoto(function_call, 134) ->
 24;
yeccgoto(function_call, 142) ->
 24;
yeccgoto(function_call, 162) ->
 24;
yeccgoto(function_call, 170) ->
 24;
yeccgoto(function_call, 172) ->
 24;
yeccgoto(function_call, 212) ->
 24;
yeccgoto(function_call, 214) ->
 24;
yeccgoto(function_call, 220) ->
 24;
yeccgoto(function_call, 222) ->
 24;
yeccgoto(function_call, 223) ->
 24;
yeccgoto(function_call, 226) ->
 24;
yeccgoto(function_call, 228) ->
 24;
yeccgoto(function_call, 230) ->
 24;
yeccgoto(function_call, 240) ->
 24;
yeccgoto(function_call, 241) ->
 24;
yeccgoto(function_call, 253) ->
 24;
yeccgoto(function_call, 277) ->
 24;
yeccgoto(function_call, 280) ->
 24;
yeccgoto(function_clause, 0) ->
 5;
yeccgoto(function_clause, 286) ->
 5;
yeccgoto(function_clauses, 0) ->
 4;
yeccgoto(function_clauses, 286) ->
 287;
yeccgoto(guard, 59) ->
 140;
yeccgoto(guard, 83) ->
 84;
yeccgoto(guard, 86) ->
 87;
yeccgoto(guard, 142) ->
 140;
yeccgoto(if_clause, 59) ->
 139;
yeccgoto(if_clause, 142) ->
 139;
yeccgoto(if_clauses, 59) ->
 138;
yeccgoto(if_clauses, 142) ->
 143;
yeccgoto(if_expr, 13) ->
 23;
yeccgoto(if_expr, 20) ->
 23;
yeccgoto(if_expr, 44) ->
 23;
yeccgoto(if_expr, 49) ->
 23;
yeccgoto(if_expr, 50) ->
 23;
yeccgoto(if_expr, 52) ->
 23;
yeccgoto(if_expr, 54) ->
 23;
yeccgoto(if_expr, 55) ->
 23;
yeccgoto(if_expr, 59) ->
 23;
yeccgoto(if_expr, 63) ->
 23;
yeccgoto(if_expr, 65) ->
 23;
yeccgoto(if_expr, 67) ->
 23;
yeccgoto(if_expr, 73) ->
 23;
yeccgoto(if_expr, 74) ->
 23;
yeccgoto(if_expr, 75) ->
 23;
yeccgoto(if_expr, 79) ->
 23;
yeccgoto(if_expr, 83) ->
 23;
yeccgoto(if_expr, 86) ->
 23;
yeccgoto(if_expr, 89) ->
 23;
yeccgoto(if_expr, 96) ->
 23;
yeccgoto(if_expr, 100) ->
 23;
yeccgoto(if_expr, 106) ->
 23;
yeccgoto(if_expr, 108) ->
 23;
yeccgoto(if_expr, 116) ->
 23;
yeccgoto(if_expr, 120) ->
 23;
yeccgoto(if_expr, 126) ->
 23;
yeccgoto(if_expr, 128) ->
 23;
yeccgoto(if_expr, 132) ->
 23;
yeccgoto(if_expr, 134) ->
 23;
yeccgoto(if_expr, 142) ->
 23;
yeccgoto(if_expr, 162) ->
 23;
yeccgoto(if_expr, 170) ->
 23;
yeccgoto(if_expr, 172) ->
 23;
yeccgoto(if_expr, 177) ->
 23;
yeccgoto(if_expr, 183) ->
 23;
yeccgoto(if_expr, 187) ->
 23;
yeccgoto(if_expr, 212) ->
 23;
yeccgoto(if_expr, 214) ->
 23;
yeccgoto(if_expr, 220) ->
 23;
yeccgoto(if_expr, 222) ->
 23;
yeccgoto(if_expr, 223) ->
 23;
yeccgoto(if_expr, 226) ->
 23;
yeccgoto(if_expr, 228) ->
 23;
yeccgoto(if_expr, 230) ->
 23;
yeccgoto(if_expr, 240) ->
 23;
yeccgoto(if_expr, 241) ->
 23;
yeccgoto(if_expr, 253) ->
 23;
yeccgoto(if_expr, 264) ->
 23;
yeccgoto(if_expr, 277) ->
 23;
yeccgoto(if_expr, 280) ->
 23;
yeccgoto(lc_expr, 128) ->
 130;
yeccgoto(lc_expr, 134) ->
 130;
yeccgoto(lc_expr, 277) ->
 130;
yeccgoto(lc_exprs, 128) ->
 129;
yeccgoto(lc_exprs, 134) ->
 135;
yeccgoto(lc_exprs, 277) ->
 278;
yeccgoto(list, 13) ->
 22;
yeccgoto(list, 20) ->
 22;
yeccgoto(list, 44) ->
 22;
yeccgoto(list, 49) ->
 22;
yeccgoto(list, 50) ->
 22;
yeccgoto(list, 52) ->
 22;
yeccgoto(list, 54) ->
 22;
yeccgoto(list, 55) ->
 22;
yeccgoto(list, 59) ->
 22;
yeccgoto(list, 63) ->
 22;
yeccgoto(list, 65) ->
 22;
yeccgoto(list, 67) ->
 22;
yeccgoto(list, 73) ->
 22;
yeccgoto(list, 74) ->
 22;
yeccgoto(list, 75) ->
 22;
yeccgoto(list, 79) ->
 22;
yeccgoto(list, 83) ->
 22;
yeccgoto(list, 86) ->
 22;
yeccgoto(list, 89) ->
 22;
yeccgoto(list, 96) ->
 22;
yeccgoto(list, 100) ->
 22;
yeccgoto(list, 106) ->
 22;
yeccgoto(list, 108) ->
 22;
yeccgoto(list, 116) ->
 22;
yeccgoto(list, 120) ->
 22;
yeccgoto(list, 126) ->
 22;
yeccgoto(list, 128) ->
 22;
yeccgoto(list, 132) ->
 22;
yeccgoto(list, 134) ->
 22;
yeccgoto(list, 142) ->
 22;
yeccgoto(list, 162) ->
 22;
yeccgoto(list, 170) ->
 22;
yeccgoto(list, 172) ->
 22;
yeccgoto(list, 177) ->
 22;
yeccgoto(list, 183) ->
 22;
yeccgoto(list, 187) ->
 22;
yeccgoto(list, 212) ->
 22;
yeccgoto(list, 214) ->
 22;
yeccgoto(list, 220) ->
 22;
yeccgoto(list, 222) ->
 22;
yeccgoto(list, 223) ->
 22;
yeccgoto(list, 226) ->
 22;
yeccgoto(list, 228) ->
 22;
yeccgoto(list, 230) ->
 22;
yeccgoto(list, 240) ->
 22;
yeccgoto(list, 241) ->
 22;
yeccgoto(list, 253) ->
 22;
yeccgoto(list, 264) ->
 22;
yeccgoto(list, 277) ->
 22;
yeccgoto(list, 280) ->
 22;
yeccgoto(list_comprehension, 13) ->
 21;
yeccgoto(list_comprehension, 20) ->
 21;
yeccgoto(list_comprehension, 44) ->
 21;
yeccgoto(list_comprehension, 49) ->
 21;
yeccgoto(list_comprehension, 50) ->
 21;
yeccgoto(list_comprehension, 52) ->
 21;
yeccgoto(list_comprehension, 54) ->
 21;
yeccgoto(list_comprehension, 55) ->
 21;
yeccgoto(list_comprehension, 59) ->
 21;
yeccgoto(list_comprehension, 62) ->
 125;
yeccgoto(list_comprehension, 63) ->
 21;
yeccgoto(list_comprehension, 65) ->
 21;
yeccgoto(list_comprehension, 67) ->
 21;
yeccgoto(list_comprehension, 73) ->
 21;
yeccgoto(list_comprehension, 74) ->
 21;
yeccgoto(list_comprehension, 75) ->
 21;
yeccgoto(list_comprehension, 79) ->
 21;
yeccgoto(list_comprehension, 83) ->
 21;
yeccgoto(list_comprehension, 86) ->
 21;
yeccgoto(list_comprehension, 89) ->
 21;
yeccgoto(list_comprehension, 96) ->
 21;
yeccgoto(list_comprehension, 100) ->
 21;
yeccgoto(list_comprehension, 106) ->
 21;
yeccgoto(list_comprehension, 108) ->
 21;
yeccgoto(list_comprehension, 116) ->
 21;
yeccgoto(list_comprehension, 120) ->
 21;
yeccgoto(list_comprehension, 126) ->
 21;
yeccgoto(list_comprehension, 128) ->
 21;
yeccgoto(list_comprehension, 132) ->
 21;
yeccgoto(list_comprehension, 134) ->
 21;
yeccgoto(list_comprehension, 142) ->
 21;
yeccgoto(list_comprehension, 162) ->
 21;
yeccgoto(list_comprehension, 170) ->
 21;
yeccgoto(list_comprehension, 172) ->
 21;
yeccgoto(list_comprehension, 177) ->
 21;
yeccgoto(list_comprehension, 183) ->
 21;
yeccgoto(list_comprehension, 187) ->
 21;
yeccgoto(list_comprehension, 212) ->
 21;
yeccgoto(list_comprehension, 214) ->
 21;
yeccgoto(list_comprehension, 220) ->
 21;
yeccgoto(list_comprehension, 222) ->
 21;
yeccgoto(list_comprehension, 223) ->
 21;
yeccgoto(list_comprehension, 226) ->
 21;
yeccgoto(list_comprehension, 228) ->
 21;
yeccgoto(list_comprehension, 230) ->
 21;
yeccgoto(list_comprehension, 240) ->
 21;
yeccgoto(list_comprehension, 241) ->
 21;
yeccgoto(list_comprehension, 253) ->
 21;
yeccgoto(list_comprehension, 264) ->
 21;
yeccgoto(list_comprehension, 277) ->
 21;
yeccgoto(list_comprehension, 280) ->
 21;
yeccgoto(list_op, 33) ->
 240;
yeccgoto(mult_op, 32) ->
 253;
yeccgoto(mult_op, 252) ->
 253;
yeccgoto(opt_bit_size_expr, 179) ->
 186;
yeccgoto(opt_bit_type_list, 186) ->
 190;
yeccgoto(prefix_op, 13) ->
 20;
yeccgoto(prefix_op, 44) ->
 20;
yeccgoto(prefix_op, 49) ->
 177;
yeccgoto(prefix_op, 50) ->
 20;
yeccgoto(prefix_op, 52) ->
 20;
yeccgoto(prefix_op, 54) ->
 20;
yeccgoto(prefix_op, 55) ->
 20;
yeccgoto(prefix_op, 59) ->
 20;
yeccgoto(prefix_op, 63) ->
 20;
yeccgoto(prefix_op, 65) ->
 20;
yeccgoto(prefix_op, 67) ->
 20;
yeccgoto(prefix_op, 73) ->
 20;
yeccgoto(prefix_op, 74) ->
 20;
yeccgoto(prefix_op, 75) ->
 20;
yeccgoto(prefix_op, 79) ->
 20;
yeccgoto(prefix_op, 83) ->
 20;
yeccgoto(prefix_op, 86) ->
 20;
yeccgoto(prefix_op, 89) ->
 20;
yeccgoto(prefix_op, 96) ->
 20;
yeccgoto(prefix_op, 100) ->
 20;
yeccgoto(prefix_op, 106) ->
 20;
yeccgoto(prefix_op, 108) ->
 20;
yeccgoto(prefix_op, 116) ->
 20;
yeccgoto(prefix_op, 120) ->
 20;
yeccgoto(prefix_op, 126) ->
 20;
yeccgoto(prefix_op, 128) ->
 20;
yeccgoto(prefix_op, 132) ->
 20;
yeccgoto(prefix_op, 134) ->
 20;
yeccgoto(prefix_op, 142) ->
 20;
yeccgoto(prefix_op, 162) ->
 20;
yeccgoto(prefix_op, 170) ->
 20;
yeccgoto(prefix_op, 172) ->
 20;
yeccgoto(prefix_op, 183) ->
 177;
yeccgoto(prefix_op, 212) ->
 20;
yeccgoto(prefix_op, 214) ->
 20;
yeccgoto(prefix_op, 220) ->
 20;
yeccgoto(prefix_op, 222) ->
 20;
yeccgoto(prefix_op, 223) ->
 20;
yeccgoto(prefix_op, 226) ->
 20;
yeccgoto(prefix_op, 228) ->
 20;
yeccgoto(prefix_op, 230) ->
 20;
yeccgoto(prefix_op, 240) ->
 20;
yeccgoto(prefix_op, 241) ->
 20;
yeccgoto(prefix_op, 253) ->
 20;
yeccgoto(prefix_op, 277) ->
 20;
yeccgoto(prefix_op, 280) ->
 20;
yeccgoto(query_expr, 13) ->
 19;
yeccgoto(query_expr, 20) ->
 19;
yeccgoto(query_expr, 44) ->
 19;
yeccgoto(query_expr, 49) ->
 19;
yeccgoto(query_expr, 50) ->
 19;
yeccgoto(query_expr, 52) ->
 19;
yeccgoto(query_expr, 54) ->
 19;
yeccgoto(query_expr, 55) ->
 19;
yeccgoto(query_expr, 59) ->
 19;
yeccgoto(query_expr, 63) ->
 19;
yeccgoto(query_expr, 65) ->
 19;
yeccgoto(query_expr, 67) ->
 19;
yeccgoto(query_expr, 73) ->
 19;
yeccgoto(query_expr, 74) ->
 19;
yeccgoto(query_expr, 75) ->
 19;
yeccgoto(query_expr, 79) ->
 19;
yeccgoto(query_expr, 83) ->
 19;
yeccgoto(query_expr, 86) ->
 19;
yeccgoto(query_expr, 89) ->
 19;
yeccgoto(query_expr, 96) ->
 19;
yeccgoto(query_expr, 100) ->
 19;
yeccgoto(query_expr, 106) ->
 19;
yeccgoto(query_expr, 108) ->
 19;
yeccgoto(query_expr, 116) ->
 19;
yeccgoto(query_expr, 120) ->
 19;
yeccgoto(query_expr, 126) ->
 19;
yeccgoto(query_expr, 128) ->
 19;
yeccgoto(query_expr, 132) ->
 19;
yeccgoto(query_expr, 134) ->
 19;
yeccgoto(query_expr, 142) ->
 19;
yeccgoto(query_expr, 162) ->
 19;
yeccgoto(query_expr, 170) ->
 19;
yeccgoto(query_expr, 172) ->
 19;
yeccgoto(query_expr, 177) ->
 19;
yeccgoto(query_expr, 183) ->
 19;
yeccgoto(query_expr, 187) ->
 19;
yeccgoto(query_expr, 212) ->
 19;
yeccgoto(query_expr, 214) ->
 19;
yeccgoto(query_expr, 220) ->
 19;
yeccgoto(query_expr, 222) ->
 19;
yeccgoto(query_expr, 223) ->
 19;
yeccgoto(query_expr, 226) ->
 19;
yeccgoto(query_expr, 228) ->
 19;
yeccgoto(query_expr, 230) ->
 19;
yeccgoto(query_expr, 240) ->
 19;
yeccgoto(query_expr, 241) ->
 19;
yeccgoto(query_expr, 253) ->
 19;
yeccgoto(query_expr, 264) ->
 19;
yeccgoto(query_expr, 277) ->
 19;
yeccgoto(query_expr, 280) ->
 19;
yeccgoto(receive_expr, 13) ->
 18;
yeccgoto(receive_expr, 20) ->
 18;
yeccgoto(receive_expr, 44) ->
 18;
yeccgoto(receive_expr, 49) ->
 18;
yeccgoto(receive_expr, 50) ->
 18;
yeccgoto(receive_expr, 52) ->
 18;
yeccgoto(receive_expr, 54) ->
 18;
yeccgoto(receive_expr, 55) ->
 18;
yeccgoto(receive_expr, 59) ->
 18;
yeccgoto(receive_expr, 63) ->
 18;
yeccgoto(receive_expr, 65) ->
 18;
yeccgoto(receive_expr, 67) ->
 18;
yeccgoto(receive_expr, 73) ->
 18;
yeccgoto(receive_expr, 74) ->
 18;
yeccgoto(receive_expr, 75) ->
 18;
yeccgoto(receive_expr, 79) ->
 18;
yeccgoto(receive_expr, 83) ->
 18;
yeccgoto(receive_expr, 86) ->
 18;
yeccgoto(receive_expr, 89) ->
 18;
yeccgoto(receive_expr, 96) ->
 18;
yeccgoto(receive_expr, 100) ->
 18;
yeccgoto(receive_expr, 106) ->
 18;
yeccgoto(receive_expr, 108) ->
 18;
yeccgoto(receive_expr, 116) ->
 18;
yeccgoto(receive_expr, 120) ->
 18;
yeccgoto(receive_expr, 126) ->
 18;
yeccgoto(receive_expr, 128) ->
 18;
yeccgoto(receive_expr, 132) ->
 18;
yeccgoto(receive_expr, 134) ->
 18;
yeccgoto(receive_expr, 142) ->
 18;
yeccgoto(receive_expr, 162) ->
 18;
yeccgoto(receive_expr, 170) ->
 18;
yeccgoto(receive_expr, 172) ->
 18;
yeccgoto(receive_expr, 177) ->
 18;
yeccgoto(receive_expr, 183) ->
 18;
yeccgoto(receive_expr, 187) ->
 18;
yeccgoto(receive_expr, 212) ->
 18;
yeccgoto(receive_expr, 214) ->
 18;
yeccgoto(receive_expr, 220) ->
 18;
yeccgoto(receive_expr, 222) ->
 18;
yeccgoto(receive_expr, 223) ->
 18;
yeccgoto(receive_expr, 226) ->
 18;
yeccgoto(receive_expr, 228) ->
 18;
yeccgoto(receive_expr, 230) ->
 18;
yeccgoto(receive_expr, 240) ->
 18;
yeccgoto(receive_expr, 241) ->
 18;
yeccgoto(receive_expr, 253) ->
 18;
yeccgoto(receive_expr, 264) ->
 18;
yeccgoto(receive_expr, 277) ->
 18;
yeccgoto(receive_expr, 280) ->
 18;
yeccgoto(record_expr, 13) ->
 17;
yeccgoto(record_expr, 20) ->
 17;
yeccgoto(record_expr, 44) ->
 17;
yeccgoto(record_expr, 50) ->
 17;
yeccgoto(record_expr, 52) ->
 17;
yeccgoto(record_expr, 54) ->
 17;
yeccgoto(record_expr, 55) ->
 17;
yeccgoto(record_expr, 59) ->
 17;
yeccgoto(record_expr, 63) ->
 17;
yeccgoto(record_expr, 65) ->
 17;
yeccgoto(record_expr, 67) ->
 17;
yeccgoto(record_expr, 73) ->
 17;
yeccgoto(record_expr, 74) ->
 17;
yeccgoto(record_expr, 75) ->
 17;
yeccgoto(record_expr, 79) ->
 17;
yeccgoto(record_expr, 83) ->
 17;
yeccgoto(record_expr, 86) ->
 17;
yeccgoto(record_expr, 89) ->
 17;
yeccgoto(record_expr, 96) ->
 17;
yeccgoto(record_expr, 100) ->
 17;
yeccgoto(record_expr, 106) ->
 17;
yeccgoto(record_expr, 108) ->
 17;
yeccgoto(record_expr, 116) ->
 17;
yeccgoto(record_expr, 120) ->
 17;
yeccgoto(record_expr, 126) ->
 17;
yeccgoto(record_expr, 128) ->
 17;
yeccgoto(record_expr, 132) ->
 17;
yeccgoto(record_expr, 134) ->
 17;
yeccgoto(record_expr, 142) ->
 17;
yeccgoto(record_expr, 162) ->
 17;
yeccgoto(record_expr, 170) ->
 17;
yeccgoto(record_expr, 172) ->
 17;
yeccgoto(record_expr, 212) ->
 17;
yeccgoto(record_expr, 214) ->
 17;
yeccgoto(record_expr, 220) ->
 17;
yeccgoto(record_expr, 222) ->
 17;
yeccgoto(record_expr, 223) ->
 17;
yeccgoto(record_expr, 226) ->
 17;
yeccgoto(record_expr, 228) ->
 17;
yeccgoto(record_expr, 230) ->
 17;
yeccgoto(record_expr, 240) ->
 17;
yeccgoto(record_expr, 241) ->
 17;
yeccgoto(record_expr, 253) ->
 17;
yeccgoto(record_expr, 277) ->
 17;
yeccgoto(record_expr, 280) ->
 17;
yeccgoto(record_field, 206) ->
 208;
yeccgoto(record_field, 216) ->
 208;
yeccgoto(record_fields, 206) ->
 207;
yeccgoto(record_fields, 216) ->
 217;
yeccgoto(record_tuple, 203) ->
 204;
yeccgoto(record_tuple, 268) ->
 269;
yeccgoto(rule, 0) ->
 3;
yeccgoto(rule_body, 274) ->
 275;
yeccgoto(rule_body, 296) ->
 275;
yeccgoto(rule_clause, 0) ->
 2;
yeccgoto(rule_clause, 292) ->
 2;
yeccgoto(rule_clauses, 0) ->
 1;
yeccgoto(rule_clauses, 292) ->
 293;
yeccgoto(strings, 13) ->
 16;
yeccgoto(strings, 20) ->
 16;
yeccgoto(strings, 44) ->
 16;
yeccgoto(strings, 49) ->
 16;
yeccgoto(strings, 50) ->
 16;
yeccgoto(strings, 52) ->
 16;
yeccgoto(strings, 54) ->
 16;
yeccgoto(strings, 55) ->
 16;
yeccgoto(strings, 59) ->
 16;
yeccgoto(strings, 63) ->
 16;
yeccgoto(strings, 64) ->
 114;
yeccgoto(strings, 65) ->
 16;
yeccgoto(strings, 67) ->
 16;
yeccgoto(strings, 73) ->
 16;
yeccgoto(strings, 74) ->
 16;
yeccgoto(strings, 75) ->
 16;
yeccgoto(strings, 79) ->
 16;
yeccgoto(strings, 83) ->
 16;
yeccgoto(strings, 86) ->
 16;
yeccgoto(strings, 89) ->
 16;
yeccgoto(strings, 96) ->
 16;
yeccgoto(strings, 100) ->
 16;
yeccgoto(strings, 106) ->
 16;
yeccgoto(strings, 108) ->
 16;
yeccgoto(strings, 116) ->
 16;
yeccgoto(strings, 120) ->
 16;
yeccgoto(strings, 126) ->
 16;
yeccgoto(strings, 128) ->
 16;
yeccgoto(strings, 132) ->
 16;
yeccgoto(strings, 134) ->
 16;
yeccgoto(strings, 142) ->
 16;
yeccgoto(strings, 162) ->
 16;
yeccgoto(strings, 170) ->
 16;
yeccgoto(strings, 172) ->
 16;
yeccgoto(strings, 177) ->
 16;
yeccgoto(strings, 183) ->
 16;
yeccgoto(strings, 187) ->
 16;
yeccgoto(strings, 212) ->
 16;
yeccgoto(strings, 214) ->
 16;
yeccgoto(strings, 220) ->
 16;
yeccgoto(strings, 222) ->
 16;
yeccgoto(strings, 223) ->
 16;
yeccgoto(strings, 226) ->
 16;
yeccgoto(strings, 228) ->
 16;
yeccgoto(strings, 230) ->
 16;
yeccgoto(strings, 240) ->
 16;
yeccgoto(strings, 241) ->
 16;
yeccgoto(strings, 253) ->
 16;
yeccgoto(strings, 264) ->
 16;
yeccgoto(strings, 277) ->
 16;
yeccgoto(strings, 280) ->
 16;
yeccgoto(tail, 167) ->
 169;
yeccgoto(tail, 175) ->
 176;
yeccgoto(try_catch, 71) ->
 72;
yeccgoto(try_catch, 77) ->
 81;
yeccgoto(try_clause, 74) ->
 92;
yeccgoto(try_clause, 106) ->
 92;
yeccgoto(try_clauses, 74) ->
 91;
yeccgoto(try_clauses, 106) ->
 107;
yeccgoto(try_expr, 13) ->
 15;
yeccgoto(try_expr, 20) ->
 15;
yeccgoto(try_expr, 44) ->
 15;
yeccgoto(try_expr, 49) ->
 15;
yeccgoto(try_expr, 50) ->
 15;
yeccgoto(try_expr, 52) ->
 15;
yeccgoto(try_expr, 54) ->
 15;
yeccgoto(try_expr, 55) ->
 15;
yeccgoto(try_expr, 59) ->
 15;
yeccgoto(try_expr, 63) ->
 15;
yeccgoto(try_expr, 65) ->
 15;
yeccgoto(try_expr, 67) ->
 15;
yeccgoto(try_expr, 73) ->
 15;
yeccgoto(try_expr, 74) ->
 15;
yeccgoto(try_expr, 75) ->
 15;
yeccgoto(try_expr, 79) ->
 15;
yeccgoto(try_expr, 83) ->
 15;
yeccgoto(try_expr, 86) ->
 15;
yeccgoto(try_expr, 89) ->
 15;
yeccgoto(try_expr, 96) ->
 15;
yeccgoto(try_expr, 100) ->
 15;
yeccgoto(try_expr, 106) ->
 15;
yeccgoto(try_expr, 108) ->
 15;
yeccgoto(try_expr, 116) ->
 15;
yeccgoto(try_expr, 120) ->
 15;
yeccgoto(try_expr, 126) ->
 15;
yeccgoto(try_expr, 128) ->
 15;
yeccgoto(try_expr, 132) ->
 15;
yeccgoto(try_expr, 134) ->
 15;
yeccgoto(try_expr, 142) ->
 15;
yeccgoto(try_expr, 162) ->
 15;
yeccgoto(try_expr, 170) ->
 15;
yeccgoto(try_expr, 172) ->
 15;
yeccgoto(try_expr, 177) ->
 15;
yeccgoto(try_expr, 183) ->
 15;
yeccgoto(try_expr, 187) ->
 15;
yeccgoto(try_expr, 212) ->
 15;
yeccgoto(try_expr, 214) ->
 15;
yeccgoto(try_expr, 220) ->
 15;
yeccgoto(try_expr, 222) ->
 15;
yeccgoto(try_expr, 223) ->
 15;
yeccgoto(try_expr, 226) ->
 15;
yeccgoto(try_expr, 228) ->
 15;
yeccgoto(try_expr, 230) ->
 15;
yeccgoto(try_expr, 240) ->
 15;
yeccgoto(try_expr, 241) ->
 15;
yeccgoto(try_expr, 253) ->
 15;
yeccgoto(try_expr, 264) ->
 15;
yeccgoto(try_expr, 277) ->
 15;
yeccgoto(try_expr, 280) ->
 15;
yeccgoto(tuple, 13) ->
 14;
yeccgoto(tuple, 20) ->
 14;
yeccgoto(tuple, 44) ->
 14;
yeccgoto(tuple, 49) ->
 14;
yeccgoto(tuple, 50) ->
 14;
yeccgoto(tuple, 52) ->
 14;
yeccgoto(tuple, 54) ->
 14;
yeccgoto(tuple, 55) ->
 14;
yeccgoto(tuple, 59) ->
 14;
yeccgoto(tuple, 63) ->
 14;
yeccgoto(tuple, 65) ->
 14;
yeccgoto(tuple, 67) ->
 14;
yeccgoto(tuple, 73) ->
 14;
yeccgoto(tuple, 74) ->
 14;
yeccgoto(tuple, 75) ->
 14;
yeccgoto(tuple, 79) ->
 14;
yeccgoto(tuple, 83) ->
 14;
yeccgoto(tuple, 86) ->
 14;
yeccgoto(tuple, 89) ->
 14;
yeccgoto(tuple, 96) ->
 14;
yeccgoto(tuple, 100) ->
 14;
yeccgoto(tuple, 106) ->
 14;
yeccgoto(tuple, 108) ->
 14;
yeccgoto(tuple, 116) ->
 14;
yeccgoto(tuple, 120) ->
 14;
yeccgoto(tuple, 126) ->
 14;
yeccgoto(tuple, 128) ->
 14;
yeccgoto(tuple, 132) ->
 14;
yeccgoto(tuple, 134) ->
 14;
yeccgoto(tuple, 142) ->
 14;
yeccgoto(tuple, 162) ->
 14;
yeccgoto(tuple, 170) ->
 14;
yeccgoto(tuple, 172) ->
 14;
yeccgoto(tuple, 177) ->
 14;
yeccgoto(tuple, 183) ->
 14;
yeccgoto(tuple, 187) ->
 14;
yeccgoto(tuple, 212) ->
 14;
yeccgoto(tuple, 214) ->
 14;
yeccgoto(tuple, 220) ->
 14;
yeccgoto(tuple, 222) ->
 14;
yeccgoto(tuple, 223) ->
 14;
yeccgoto(tuple, 226) ->
 14;
yeccgoto(tuple, 228) ->
 14;
yeccgoto(tuple, 230) ->
 14;
yeccgoto(tuple, 240) ->
 14;
yeccgoto(tuple, 241) ->
 14;
yeccgoto(tuple, 253) ->
 14;
yeccgoto(tuple, 264) ->
 14;
yeccgoto(tuple, 277) ->
 14;
yeccgoto(tuple, 280) ->
 14;
yeccgoto(__Symbol, __State) ->
 erlang:error({yecc_bug,"1.1",{__Symbol, __State, missing_in_goto_table}}).

-compile({inline,{yeccpars2_1_,1}}).
-file("erl_parse.yrl", 365).
yeccpars2_1_([__1 | __Stack]) ->
 [begin
   build_rule ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_2_,1}}).
-file("erl_parse.yrl", 367).
yeccpars2_2_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_4_,1}}).
-file("erl_parse.yrl", 72).
yeccpars2_4_([__1 | __Stack]) ->
 [begin
   build_function ( __1 )
  end | __Stack].

-compile({inline,{yeccpars2_5_,1}}).
-file("erl_parse.yrl", 74).
yeccpars2_5_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_11_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_11_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_12_,1}}).
-file("erl_parse.yrl", 81).
yeccpars2_12_([__1 | __Stack]) ->
 [begin
   element ( 1 , __1 )
  end | __Stack].

-compile({inline,{yeccpars2_39_,1}}).
-file("erl_parse.yrl", 316).
yeccpars2_39_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_45_,1}}).
-file("erl_parse.yrl", 312).
yeccpars2_45_([__2,__1 | __Stack]) ->
 [begin
   { [ ] , line ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_69_,1}}).
-file("erl_parse.yrl", 199).
yeccpars2_69_([__2,__1 | __Stack]) ->
 [begin
   { tuple , line ( __1 ) , [ ] }
  end | __Stack].

-compile({inline,{yeccpars2_70_,1}}).
-file("erl_parse.yrl", 200).
yeccpars2_70_([__3,__2,__1 | __Stack]) ->
 [begin
   { tuple , line ( __1 ) , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_72_,1}}).
-file("erl_parse.yrl", 278).
yeccpars2_72_([__3,__2,__1 | __Stack]) ->
 [begin
   build_try ( line ( __1 ) , __2 , [ ] , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_76_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_76_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_78_,1}}).
-file("erl_parse.yrl", 247).
yeccpars2_78_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_80_,1}}).
-file("erl_parse.yrl", 248).
yeccpars2_80_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_81_,1}}).
-file("erl_parse.yrl", 276).
yeccpars2_81_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   build_try ( line ( __1 ) , __2 , __4 , __5 )
  end | __Stack].

-compile({inline,{yeccpars2_84_,1}}).
-file("erl_parse.yrl", 83).
yeccpars2_84_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_85_,1}}).
-file("erl_parse.yrl", 319).
yeccpars2_85_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_87_,1}}).
-file("erl_parse.yrl", 320).
yeccpars2_87_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_88_,1}}).
-file("erl_parse.yrl", 251).
yeccpars2_88_([__3,__2,__1 | __Stack]) ->
 [begin
   { clause , line ( __1 ) , [ __1 ] , __2 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_90_,1}}).
-file("erl_parse.yrl", 86).
yeccpars2_90_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_92_,1}}).
-file("erl_parse.yrl", 287).
yeccpars2_92_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_93_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_93_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_97_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_97_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_99_,1}}).
-file("erl_parse.yrl", 297).
yeccpars2_99_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   L = line ( __1 ) ,
    { clause , L , [ { tuple , L , [ __1 , __3 , { var , L , '_' } ] } ] , __4 , __5 }
  end | __Stack].

-compile({inline,{yeccpars2_101_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_101_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_103_,1}}).
-file("erl_parse.yrl", 294).
yeccpars2_103_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   L = line ( __1 ) ,
    { clause , L , [ { tuple , L , [ __1 , __3 , { var , L , '_' } ] } ] , __4 , __5 }
  end | __Stack].

-compile({inline,{yeccpars2_105_,1}}).
-file("erl_parse.yrl", 291).
yeccpars2_105_([__3,__2,__1 | __Stack]) ->
 [begin
   L = line ( __1 ) ,
    { clause , L , [ { tuple , L , [ { atom , L , throw } , __1 , { var , L , '_' } ] } ] , __2 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_107_,1}}).
-file("erl_parse.yrl", 288).
yeccpars2_107_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_109_,1}}).
-file("erl_parse.yrl", 281).
yeccpars2_109_([__3,__2,__1 | __Stack]) ->
 [begin
   { __2 , [ ] }
  end | __Stack].

-compile({inline,{yeccpars2_111_,1}}).
-file("erl_parse.yrl", 283).
yeccpars2_111_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { __2 , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_113_,1}}).
-file("erl_parse.yrl", 285).
yeccpars2_113_([__3,__2,__1 | __Stack]) ->
 [begin
   { [ ] , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_114_,1}}).
-file("erl_parse.yrl", 330).
yeccpars2_114_([__2,__1 | __Stack]) ->
 [begin
   { string , line ( __1 ) , element ( 3 , __1 ) ++ element ( 3 , __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_119_,1}}).
-file("erl_parse.yrl", 256).
yeccpars2_119_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { 'receive' , line ( __1 ) , [ ] , __3 , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_121_,1}}).
-file("erl_parse.yrl", 254).
yeccpars2_121_([__3,__2,__1 | __Stack]) ->
 [begin
   { 'receive' , line ( __1 ) , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_124_,1}}).
-file("erl_parse.yrl", 258).
yeccpars2_124_([__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { 'receive' , line ( __1 ) , __2 , __4 , __5 }
  end | __Stack].

-compile({inline,{yeccpars2_130_,1}}).
-file("erl_parse.yrl", 192).
yeccpars2_130_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_133_,1}}).
-file("erl_parse.yrl", 196).
yeccpars2_133_([__3,__2,__1 | __Stack]) ->
 [begin
   { generate , line ( __2 ) , __1 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_135_,1}}).
-file("erl_parse.yrl", 193).
yeccpars2_135_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_136_,1}}).
-file("erl_parse.yrl", 190).
yeccpars2_136_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { lc , line ( __1 ) , __2 , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_137_,1}}).
-file("erl_parse.yrl", 309).
yeccpars2_137_([__3,__2,__1 | __Stack]) ->
 [begin
   { 'query' , line ( __1 ) , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_139_,1}}).
-file("erl_parse.yrl", 237).
yeccpars2_139_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_141_,1}}).
-file("erl_parse.yrl", 241).
yeccpars2_141_([__2,__1 | __Stack]) ->
 [begin
   { clause , line ( hd ( hd ( __1 ) ) ) , [ ] , __1 , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_143_,1}}).
-file("erl_parse.yrl", 238).
yeccpars2_143_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_144_,1}}).
-file("erl_parse.yrl", 235).
yeccpars2_144_([__3,__2,__1 | __Stack]) ->
 [begin
   { 'if' , line ( __1 ) , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_146_,1}}).
-file("erl_parse.yrl", 268).
yeccpars2_146_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_147_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_147_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_153_,1}}).
-file("erl_parse.yrl", 264).
yeccpars2_153_([__6,__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { 'fun' , line ( __1 ) , { function , element ( 3 , __2 ) , element ( 3 , __4 ) , element ( 3 , __6 ) } }
  end | __Stack].

-compile({inline,{yeccpars2_154_,1}}).
-file("erl_parse.yrl", 262).
yeccpars2_154_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { 'fun' , line ( __1 ) , { function , element ( 3 , __2 ) , element ( 3 , __4 ) } }
  end | __Stack].

-compile({inline,{yeccpars2_156_,1}}).
-file("erl_parse.yrl", 272).
yeccpars2_156_([__3,__2,__1 | __Stack]) ->
 [begin
   { Args , Pos } = __1 ,
    { clause , Pos , 'fun' , Args , __2 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_158_,1}}).
-file("erl_parse.yrl", 269).
yeccpars2_158_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_159_,1}}).
-file("erl_parse.yrl", 266).
yeccpars2_159_([__3,__2,__1 | __Stack]) ->
 [begin
   build_fun ( line ( __1 ) , __2 )
  end | __Stack].

-compile({inline,{yeccpars2_160_,1}}).
-file("erl_parse.yrl", 89).
yeccpars2_160_([__2,__1 | __Stack]) ->
 [begin
   { 'catch' , line ( __1 ) , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_164_,1}}).
-file("erl_parse.yrl", 245).
yeccpars2_164_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { 'case' , line ( __1 ) , __2 , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_166_,1}}).
-file("erl_parse.yrl", 144).
yeccpars2_166_([__3,__2,__1 | __Stack]) ->
 [begin
   { block , line ( __1 ) , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_168_,1}}).
-file("erl_parse.yrl", 154).
yeccpars2_168_([__2,__1 | __Stack]) ->
 [begin
   { nil , line ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_169_,1}}).
-file("erl_parse.yrl", 155).
yeccpars2_169_([__3,__2,__1 | __Stack]) ->
 [begin
   { cons , line ( __1 ) , __2 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_171_,1}}).
-file("erl_parse.yrl", 157).
yeccpars2_171_([__1 | __Stack]) ->
 [begin
   { nil , line ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_174_,1}}).
-file("erl_parse.yrl", 158).
yeccpars2_174_([__3,__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_176_,1}}).
-file("erl_parse.yrl", 159).
yeccpars2_176_([__3,__2,__1 | __Stack]) ->
 [begin
   { cons , line ( __2 ) , __2 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_179_,1}}).
-file("erl_parse.yrl", 175).
yeccpars2_179_(__Stack) ->
 [begin
   default
  end | __Stack].

-compile({inline,{yeccpars2_181_,1}}).
-file("erl_parse.yrl", 165).
yeccpars2_181_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_182_,1}}).
-file("erl_parse.yrl", 162).
yeccpars2_182_([__2,__1 | __Stack]) ->
 [begin
   { bin , line ( __1 ) , [ ] }
  end | __Stack].

-compile({inline,{yeccpars2_184_,1}}).
-file("erl_parse.yrl", 166).
yeccpars2_184_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_185_,1}}).
-file("erl_parse.yrl", 163).
yeccpars2_185_([__3,__2,__1 | __Stack]) ->
 [begin
   { bin , line ( __1 ) , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_186_,1}}).
-file("erl_parse.yrl", 178).
yeccpars2_186_(__Stack) ->
 [begin
   default
  end | __Stack].

-compile({inline,{yeccpars2_189_,1}}).
-file("erl_parse.yrl", 174).
yeccpars2_189_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_190_,1}}).
-file("erl_parse.yrl", 169).
yeccpars2_190_([__3,__2,__1 | __Stack]) ->
 [begin
   { bin_element , line ( __1 ) , __1 , __2 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_192_,1}}).
-file("erl_parse.yrl", 177).
yeccpars2_192_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_193_,1}}).
-file("erl_parse.yrl", 181).
yeccpars2_193_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_194_,1}}).
-file("erl_parse.yrl", 183).
yeccpars2_194_([__1 | __Stack]) ->
 [begin
   element ( 3 , __1 )
  end | __Stack].

-compile({inline,{yeccpars2_196_,1}}).
-file("erl_parse.yrl", 184).
yeccpars2_196_([__3,__2,__1 | __Stack]) ->
 [begin
   { element ( 3 , __1 ) , element ( 3 , __3 ) }
  end | __Stack].

-compile({inline,{yeccpars2_198_,1}}).
-file("erl_parse.yrl", 180).
yeccpars2_198_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_199_,1}}).
-file("erl_parse.yrl", 171).
yeccpars2_199_([__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 )
  end | __Stack].

-compile({inline,{yeccpars2_200_,1}}).
-file("erl_parse.yrl", 131).
yeccpars2_200_([__2,__1 | __Stack]) ->
 [begin
   { record_field , line ( __1 ) , { atom , line ( __1 ) , '' } , __2 }
  end | __Stack].

-compile({inline,{yeccpars2_202_,1}}).
-file("erl_parse.yrl", 143).
yeccpars2_202_([__3,__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_204_,1}}).
-file("erl_parse.yrl", 214).
yeccpars2_204_([__3,__2,__1 | __Stack]) ->
 [begin
   { record , line ( __1 ) , element ( 3 , __2 ) , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_208_,1}}).
-file("erl_parse.yrl", 223).
yeccpars2_208_([__1 | __Stack]) ->
 [begin
   [ __1 ]
  end | __Stack].

-compile({inline,{yeccpars2_211_,1}}).
-file("erl_parse.yrl", 220).
yeccpars2_211_([__2,__1 | __Stack]) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_213_,1}}).
-file("erl_parse.yrl", 226).
yeccpars2_213_([__3,__2,__1 | __Stack]) ->
 [begin
   { record_field , line ( __1 ) , __1 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_215_,1}}).
-file("erl_parse.yrl", 227).
yeccpars2_215_([__3,__2,__1 | __Stack]) ->
 [begin
   { record_field , line ( __1 ) , __1 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_217_,1}}).
-file("erl_parse.yrl", 224).
yeccpars2_217_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_218_,1}}).
-file("erl_parse.yrl", 221).
yeccpars2_218_([__3,__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_219_,1}}).
-file("erl_parse.yrl", 212).
yeccpars2_219_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { record_index , line ( __1 ) , element ( 3 , __2 ) , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_221_,1}}).
-file("erl_parse.yrl", 317).
yeccpars2_221_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_224_,1}}).
-file("erl_parse.yrl", 92).
yeccpars2_224_([__3,__2,__1 | __Stack]) ->
 [begin
   { match , line ( __2 ) , __1 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_225_,1}}).
-file("erl_parse.yrl", 93).
yeccpars2_225_([__3,__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_227_,1}}).
-file("erl_parse.yrl", 96).
yeccpars2_227_([__3,__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_229_,1}}).
-file("erl_parse.yrl", 99).
yeccpars2_229_([__3,__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_239_,1}}).
-file("erl_parse.yrl", 103).
yeccpars2_239_([__3,__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_252_,1}}).
-file("erl_parse.yrl", 111).
yeccpars2_252_([__3,__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_260_,1}}).
-file("erl_parse.yrl", 115).
yeccpars2_260_([__3,__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_261_,1}}).
-file("erl_parse.yrl", 107).
yeccpars2_261_([__3,__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 , __3 )
  end | __Stack].

-compile({inline,{yeccpars2_262_,1}}).
-file("erl_parse.yrl", 232).
yeccpars2_262_([__2,__1 | __Stack]) ->
 [begin
   { call , line ( __1 ) , __1 , element ( 1 , __2 ) }
  end | __Stack].

-compile({inline,{yeccpars2_265_,1}}).
-file("erl_parse.yrl", 127).
yeccpars2_265_([__3,__2,__1 | __Stack]) ->
 [begin
   { remote , line ( __2 ) , __1 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_266_,1}}).
-file("erl_parse.yrl", 133).
yeccpars2_266_([__3,__2,__1 | __Stack]) ->
 [begin
   { record_field , line ( __2 ) , __1 , __3 }
  end | __Stack].

-compile({inline,{yeccpars2_269_,1}}).
-file("erl_parse.yrl", 218).
yeccpars2_269_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { record , line ( __2 ) , __1 , element ( 3 , __3 ) , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_271_,1}}).
-file("erl_parse.yrl", 216).
yeccpars2_271_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   { record_field , line ( __2 ) , __1 , element ( 3 , __3 ) , __5 }
  end | __Stack].

-compile({inline,{yeccpars2_272_,1}}).
-file("erl_parse.yrl", 313).
yeccpars2_272_([__3,__2,__1 | __Stack]) ->
 [begin
   { __2 , line ( __1 ) }
  end | __Stack].

-compile({inline,{yeccpars2_273_,1}}).
-file("erl_parse.yrl", 119).
yeccpars2_273_([__2,__1 | __Stack]) ->
 [begin
   mkop ( __1 , __2 )
  end | __Stack].

-compile({inline,{yeccpars2_275_,1}}).
-file("erl_parse.yrl", 371).
yeccpars2_275_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { clause , line ( __1 ) , element ( 3 , __1 ) , __2 , __3 , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_276_,1}}).
-file("erl_parse.yrl", 78).
yeccpars2_276_([__4,__3,__2,__1 | __Stack]) ->
 [begin
   { clause , line ( __1 ) , element ( 3 , __1 ) , __2 , __3 , __4 }
  end | __Stack].

-compile({inline,{yeccpars2_278_,1}}).
-file("erl_parse.yrl", 373).
yeccpars2_278_([__2,__1 | __Stack]) ->
 [begin
   __2
  end | __Stack].

-compile({inline,{yeccpars2_283_,1}}).
-file("erl_parse.yrl", 68).
yeccpars2_283_([__5,__4,__3,__2,__1 | __Stack]) ->
 [begin
   build_attribute ( __2 , __4 )
  end | __Stack].

-compile({inline,{yeccpars2_284_,1}}).
-file("erl_parse.yrl", 64).
yeccpars2_284_([__2,__1 | __Stack]) ->
 [begin
   __1
  end | __Stack].

-compile({inline,{yeccpars2_285_,1}}).
-file("erl_parse.yrl", 65).
yeccpars2_285_([__2,__1 | __Stack]) ->
 [begin
   __1
  end | __Stack].

-compile({inline,{yeccpars2_287_,1}}).
-file("erl_parse.yrl", 75).
yeccpars2_287_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_289_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_289_(__Stack) ->
 [begin
   [ ]
  end | __Stack].

-compile({inline,{yeccpars2_291_,1}}).
-file("erl_parse.yrl", 66).
yeccpars2_291_([__2,__1 | __Stack]) ->
 [begin
   __1
  end | __Stack].

-compile({inline,{yeccpars2_293_,1}}).
-file("erl_parse.yrl", 368).
yeccpars2_293_([__3,__2,__1 | __Stack]) ->
 [begin
   [ __1 | __3 ]
  end | __Stack].

-compile({inline,{yeccpars2_295_,1}}).
-file("erl_parse.yrl", 84).
yeccpars2_295_(__Stack) ->
 [begin
   [ ]
  end | __Stack].


-file("erl_parse.yrl", 797).
