/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.ui.fix;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;

import org.eclipse.jface.dialogs.IDialogSettings;

import org.eclipse.jdt.core.dom.CompilationUnit;

import org.eclipse.jdt.internal.corext.fix.IFix;
import org.eclipse.jdt.internal.corext.fix.VariableDeclarationFix;

import org.eclipse.jdt.ui.text.java.IProblemLocation;

public class VariableDeclarationCleanUp extends AbstractCleanUp {

	/**
	 * Add a final modifier to private fields where possible
	 * i.e.:<pre><code>
	 * private int field= 0; -> private final int field= 0;</code></pre>
	 */
	public static final int ADD_FINAL_MODIFIER_FIELDS= 1;
	
	/**
	 * Add a final modifier to method parameters where possible
	 * i.e.:<pre><code>
	 * void foo(int i) {} -> void foo(final int i) {}</code></pre>
	 */
	public static final int ADD_FINAL_MODIFIER_PARAMETERS= 2;
	
	/**
	 * Add a final modifier to local variables where possible
	 * i.e.:<pre><code>
	 * int i= 0; -> final int i= 0;</code></pre>
	 */
	public static final int ADD_FINAL_MODIFIER_LOCAL_VARIABLES= 4;
	

	private static final int DEFAULT_FLAG= 0;
	private static final String SECTION_NAME= "CleanUp_VariableDeclarations"; //$NON-NLS-1$

	public VariableDeclarationCleanUp(int flag) {
		super(flag);
	}

	public VariableDeclarationCleanUp(IDialogSettings settings) {
		super(getSection(settings, SECTION_NAME), DEFAULT_FLAG);
	}

	/**
	 * {@inheritDoc}
	 */
	public IFix createFix(CompilationUnit compilationUnit) throws CoreException {
		if (compilationUnit == null)
			return null;
		
		return VariableDeclarationFix.createCleanUp(compilationUnit, 
				isFlag(ADD_FINAL_MODIFIER_FIELDS),
				isFlag(ADD_FINAL_MODIFIER_PARAMETERS),
				isFlag(ADD_FINAL_MODIFIER_LOCAL_VARIABLES));
	}

	/**
	 * {@inheritDoc}
	 */
	public IFix createFix(CompilationUnit compilationUnit, IProblemLocation[] problems) throws CoreException {
		//No warnings generated by the compiler
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	public Map getRequiredOptions() {
		return null;
	}
	
	public void saveSettings(IDialogSettings settings) {
		super.saveSettings(getSection(settings, SECTION_NAME));
	}

	/**
	 * {@inheritDoc}
	 */
	public String[] getDescriptions() {
		List result= new ArrayList();
		if (isFlag(ADD_FINAL_MODIFIER_FIELDS))
			result.add(MultiFixMessages.VariableDeclarationCleanUp_AddFinalField_description);
		if (isFlag(ADD_FINAL_MODIFIER_PARAMETERS))
			result.add(MultiFixMessages.VariableDeclarationCleanUp_AddFinalParameters_description);
		if (isFlag(ADD_FINAL_MODIFIER_LOCAL_VARIABLES))
			result.add(MultiFixMessages.VariableDeclarationCleanUp_AddFinalLocals_description);
		
		return (String[])result.toArray(new String[result.size()]);
	}
	
	public String getPreview() {
		StringBuffer buf= new StringBuffer();
		
		if (isFlag(ADD_FINAL_MODIFIER_FIELDS)) {
			buf.append("private final int i= 0;\n"); //$NON-NLS-1$
		} else {
			buf.append("private int i= 0;\n"); //$NON-NLS-1$
		}
		if (isFlag(ADD_FINAL_MODIFIER_PARAMETERS)) {
			buf.append("public void foo(final int j) {\n"); //$NON-NLS-1$
		} else {
			buf.append("public void foo(int j) {\n"); //$NON-NLS-1$
		}
		if (isFlag(ADD_FINAL_MODIFIER_LOCAL_VARIABLES)) {
			buf.append("    final int k;\n"); //$NON-NLS-1$
			buf.append("    int h;\n"); //$NON-NLS-1$
			buf.append("    h= 0;\n"); //$NON-NLS-1$
		} else {
			buf.append("    int k, h;\n"); //$NON-NLS-1$
			buf.append("    h= 0;\n"); //$NON-NLS-1$
		}
		buf.append("}\n"); //$NON-NLS-1$
		
		return buf.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean canFix(CompilationUnit compilationUnit, IProblemLocation problem) throws CoreException {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public int maximalNumberOfFixes(CompilationUnit compilationUnit) {
		return -1;
	}

	/**
	 * {@inheritDoc}
	 */
	public int getDefaultFlag() {
		return DEFAULT_FLAG;
	}

}
