/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_plugin_h
#define __libnd_plugin_h

#include <ltdl.h>
#include <libnd_trace.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Netdude plugins are dynamically loaded modules that can act as filters
 * on packets of a tracefile, or can basically do anything you want them
 * to. They can potentially access any public functions in Netdude and
 * can act as if they were part of the main application.
 *
 * Possible applications are anonymizers, checksummers etc. 
 */
typedef struct lnd_plugin LND_Plugin;

/* For plugins run from the command line, it is useful to wrap command
 * line arguments into single structure that is passed to the run()
 * hook of the plugin.
 */
typedef struct lnd_plugin_args
{
  int          argc;
  char       **argv;
} LND_PluginArgs;

/**
 * LND_PluginFunc - signature of plugin iteration callbacks.
 * @plugin: iterated plugin.
 * @user_data: arbitrary user data.
 *
 * This is the signature of functions that can be passed to
 * libnd_plugin_foreach().
 */
typedef void (*LND_PluginFunc)(LND_Plugin *plugin, void *user_data);


/**
 * libnd_plugin_init - initializes feature plugins.
 *
 * The function scans the plugin directories, reads in the
 * plugins it finds and hooks them into the main window.
 */
void          libnd_plugin_init(void);


/**
 * libnd_plugin_find - looks up a plugin by name.
 * @name: name of plugin to look up.
 *
 * The function scans the list of loaded plugins and
 * returns the plugin of name @name, or %NULL if no
 * such plugin exists. NOTE: name lookups are case-insensitive.
 *
 * Returns: plugin with name @name, or %NULL on failure.
 */
LND_Plugin   *libnd_plugin_find(const char *name);


/**
 * libnd_plugin_run - runs a plugin.
 * @plugin: plugin to run.
 * @trace: trace to run the plugin on.
 * @user_data: arbitrary data to pass to plugin implementation.
 * 
 * The function runs the given @plugin on the  current
 * trace. If there is no current trace, it just returns.
 */
void          libnd_plugin_run(LND_Plugin *plugin,
			       LND_Trace *trace,
			       void *user_data);

/**
 * libnd_plugin_get_name - returns plugin name.
 * @plugin: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string describing the plugin.
 *
 * Returns: name string.
 */
const char   *libnd_plugin_get_name(LND_Plugin *plugin);


/**
 * libnd_plugin_get_author - returns plugin author.
 * @plugin: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string with the author's name.
 *
 * Returns: author name string.
 */
const char   *libnd_plugin_get_author(LND_Plugin *plugin);


/**
 * libnd_plugin_get_version - returns plugin version string.
 * @plugin: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string describing the plugin version.
 *
 * Returns: version string.
 */
const char   *libnd_plugin_get_version(LND_Plugin *plugin);


/**
 * libnd_plugin_foreach - plugin iteration.
 * @callback: callback to call for each plugin.
 * @user_data: arbitrary user data passed through to @callback.
 *
 * The function iterates over the registered plugins and applies
 * a callback function to each of them.
 */
void           libnd_plugin_foreach(LND_PluginFunc callback,
				    void *user_data);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
