%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%%%----------------------------------------------------------------------
%%% File    : code_server_int.erl
%%% Author  : Peter Olin <olin@smaug>
%%% Purpose : Holds the interpreter related code of the code_server.
%%% Created : 10 Feb 1998 by Peter Olin <olin@smaug>
%%%----------------------------------------------------------------------

-module(code_server_int).
-author('olin@smaug').

%%-compile(export_all).

-export([del_interpret/2,
	 delete_interpret/2,
	 load_interpret/4,
	 add_interpret/3,
	 ints/1]).


ints([Mod|Mods]) -> [{Mod,interpreted}|ints(Mods)];
ints([])         -> [].


del_interpret({module,Mod},Int) -> delete_interpret(Mod,Int);
del_interpret(_,Int)            -> Int.




delete_interpret(Module,Int) ->
    M = code_aux:to_atom(Module),
    case lists:member(M,Int) of
	true ->
	    NewModules = lists:delete(M,Int),
	    tell_interpreter({delete,M}),
	    NewModules;
	_ ->
	    Int
    end.



add_interpret(Module,Int,Db) when atom(Module) ->
    case code_aux:sticky(Module,Db) of
	true ->     %% Sticky file reject the load (interpretation)
	    error_logger:error_msg("Can't interpret module that reside in sticky dir\n",[]),
	    {{error, sticky_directory}, Int};
	false ->
	    case lists:member(Module,Int) of
		false ->
		    case tell_interpreter({interpret,Module}) of
			ok -> 
			    {{module,Module},[Module|Int]};
			Error ->
			    {Error,Int}
		    end;
		true ->
		    {{module,Module},Int}
	    end
    end;
add_interpret(Module,Int,Db) when list(Module) ->
    add_interpret(list_to_atom(Module),Int,Db);
add_interpret(_,Int,_) ->
    {{error,badarg},Int}.

%% tell_interpreter/1 - Sends a message to the interpreter server
%%
%%
%% (???) Currently returns 'ok' even if the message handling were 
%%       not 'successful'

tell_interpreter(Mess) ->
    case whereis(interpret) of
	Pid when pid(Pid) ->
	    Pid ! {self(),Mess},
	    ok;					%(???) FIXME: Kludge fix already made. /olin
	_ ->
	    {error, no_interpreter}
    end.


load_interpret(ReplyTo,File,Module,Binary) ->
    M = code_aux:to_atom(Module),
    delete_modules([M]),
    tell_interpreter({load,ReplyTo,M,File,Binary}).

delete_modules([Mod|Modules]) ->
    code_aux:do_purge(Mod),
    erlang:delete_module(Mod),
    delete_modules(Modules);
delete_modules([]) ->
    true.
