;;; Basic read and write
;;; Copyright (c) 1993 by Olin Shivers.

;;; Note: read ops should check to see if their string args are mutable.

(define (bogus-substring-spec? s start end)
  (or (< start 0)
      (< (string-length s) end)
      (< end start)))


;;; Best-effort/forward-progress reading 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (read-string!/partial s . args)
  (let-optionals args ((fd/port (current-input-port))
		       (start   0)
		       (end     (string-length s)))
    (cond ((integer? fd/port)
	   (let ((port (fdes->inport fd/port)))
	     (set-port-buffering port bufpol/block (max (- end start) 0))
	     (read-string!/partial port start end)))

          (else ; no differnce between fd/ports and s48 ports
	   (let* ((buffer (make-string (- end start)))
		  (needed (if (> (byte-vector-length (port-buffer fd/port)) 1)
			      'any 'immediate))   ;bufpol/none may return with 0
		  (len (read-block buffer 0 needed fd/port)))
	     (if (eof-object? len)
		 #f
		 (begin 
		   (copy-bytes! buffer 0 s start len)
		   len)))))))


(define (read-string/partial len . maybe-fd/port) 
  (let* ((s (make-string len))
	 (fd/port (:optional maybe-fd/port (current-input-port)))
	 (nread (read-string!/partial s fd/port 0 len)))
    (cond ((not nread) #f) ; EOF
	  ((= nread len) s)
	  (else (substring s 0 nread)))))


;;; Persistent reading
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Operation on ports is easy, since we can use read-block

(define (read-string! s . args)
  (let-optionals args ((fd/port (current-input-port))
		       (start   0)
		       (end     (string-length s)))
    (cond ((integer? fd/port)
	   (let ((port (fdes->inport fd/port)))
	     (set-port-buffering port bufpol/block (max (- end start) 0))
	     (read-string! port start end)))

	  (else ; no differnce between fd/port and s48 ports
	   (let ((nbytes/eof (read-block s start (- end start) fd/port)))
	     (if (eof-object? nbytes/eof)
		 #f
		 nbytes/eof))))))

(define (read-string len . maybe-fd/port) 
  (let* ((s (make-string len))
	 (fd/port (:optional maybe-fd/port (current-input-port)))
	 (nread (read-string! s fd/port 0 len)))
    (cond ((not nread) #f) ; EOF
	  ((= nread len) s)
	  (else (substring s 0 nread)))))


;;; Best-effort/forward-progress writing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Non-blocking output to a buffered port is not defined.

(define (write-string/partial s . args)
  (let-optionals args ((fd/port (current-output-port))
		       (start 0)
		       (end (string-length s)))
    (cond ((integer? fd/port)
	   (let ((port (fdes->outport fd/port)))
	     (set-port-buffering port bufpol/block (max (- end start) 0))
	     (write-string/partial s port start end)))
	  (else 
	   ;; the only way to implement this, would be to use 
	   ;; channel-maybe-write. But this is an VM-instruction which is not
	   ;; exported. Since we now have threads this shouldn;t matter.
	   (error "write-string/parital is currently dereleased.
See the RELEASE file for details")))))


;;; Persistent writing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (write-string s . args)
  (let-optionals args ((fd/port (current-output-port))
		       (start   0)
		       (end     (string-length s)))
    (cond ((integer? fd/port)
	   (let ((port (fdes->outport fd/port)))
	     (set-port-buffering port bufpol/block (max (- end start) 0))
	     (write-string s port start end)))
	  (else (write-block s start (- end start) fd/port)))))
