/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_trace_registry_h
#define __nd_trace_registry_h

#include <nd_trace.h>

/**
 * ND_TraceFunc - function prototype used by the trace registry.
 * @trace: the trace iterated.
 * @user_data: arbitrary data.
 *
 * The trace registry uses functions of this signature as
 * callbacks for iterating over the registered traces.
 * See also nd_trace_registry_foreach().
 */
typedef void (*ND_TraceFunc)(LND_Trace *trace, void *user_data);

/**
 * nd_trace_registry_init - initializes the trace registry.
 *
 * The trace registry is Netdude's central repository for currently
 * edited traces. This function sets up the necessary data structures
 * and is called when Netdude is launched.
 */
void       nd_trace_registry_init(void);

/**
 * nd_trace_registry_set_current - declares a trace as the current one
 * @trace: new current trace.
 *
 * There is always one trace in the registry that is the current one,
 * i.e. the one the user has selected right now. This function sets it
 * and updates the GUI accordingly (i.e. adjusts window title etc).
 */
void       nd_trace_registry_set_current(LND_Trace *trace);

/**
 * nd_trace_registry_get_current - returns current trace.
 *
 * This function returns the current trace. Use this to obtain the
 * trace when you have no other means, like a #ND_Packet, for example.
 *
 * Returns: current trace.
 */
LND_Trace *nd_trace_registry_get_current(void);

/**
 * nd_trace_registry_add - adds a trace to the registry.
 * @trace: the added trace.
 *
 * In order to register a trace in the registry, use this function.
 * It checks if the trace is already registered and in that case,
 * simply returns. Takes care of adding the trace in the GUI etc.
 */
void       nd_trace_registry_add(LND_Trace *trace);

/**
 * nd_trace_registry_remove - removes a trace from the registry.
 * @trace: trace to remove.
 *
 * Removes a trace from the registry and takes care of updating
 * the gui etc.
 */
void       nd_trace_registry_remove(LND_Trace *trace);

/**
 * nd_trace_registry_foreach - calls a function for each registered trace.
 * @callback: a function of type #ND_TraceFunc
 * @user_data: arbitrary data passed through to the callback
 *
 * The function calls @callback for every trace currently registered
 * and passes it as first argument. The second argument is the user data,
 * which is simply passed through.
 */
void       nd_trace_registry_foreach(ND_TraceFunc callback, void *user_data);

#endif
