/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_tcpfilter_h
#define __libnd_tcpfilter_h

#include <libnd.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

/* We need the TCP Connection State Tracker and trace sets */
#include <plugins/libnd_conntrack.h>
#include <plugins/libnd_traceset.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct lnd_tcpfilter
{
  LND_TraceSet      *traces;
  LND_ConnTable     *tcp_conns;

} LND_TCPFilter;


/**
 * libnd_tcpfilter_new - creates a new filter handle.
 *
 * Returns: new filter handle, or %NULL when out of memory.
 */
LND_TCPFilter *libnd_tcpfilter_new(void);

/**
 * libnd_tcpfilter_free - releases memory associated with a filter handle.
 * @filter: filter to release.
 *
 * The function releases all memory associated with @filter.
 */
void           libnd_tcpfilter_free(LND_TCPFilter *filter);

/**
 * libnd_tcpfilter_set_traceset - sets the input traces of a filter according to a traceset.
 * @filter: filter to set traces for.
 * @set: traceset to take traces from.
 *
 * The function sets the traces that filter @filter will work on by
 * duplicating the traceset pointed to by @set. You still need to
 * get rid of @set yourself.
 *
 * Returns: %TRUE if successful, %FALSE otherwise.
 */
gboolean       libnd_tcpfilter_set_traceset(LND_TCPFilter *filter, const LND_TraceSet *set);

gboolean       libnd_tcpfilter_add_trace(LND_TCPFilter *filter, LND_Trace *trace);
gboolean       libnd_tcpfilter_add_trace_name(LND_TCPFilter *filter, const char *file_name);
gboolean       libnd_tcpfilter_add_trace_name_list(LND_TCPFilter *filter, const char *file_name);

/**
 * libnd_tcpfilter_run - starts the filtering process.
 * @filter: filter handle.
 *
 * The function will scan the traces added to @filter in the sequence
 * they were provided, and remove incomplete TCP flows from these traces
 * in the resulting output files. "incomplete" here means connections
 * that did not see the full three-way handshake, and that timed out,
 * were cut off in the trace, or that otherwise did not get terminated
 * via a proper FIN teardown or RST. The connection state is not flushed
 * between successive traces, so flows spanning adjacent files will still
 * remain in the trace if the entire flow is a complete TCP connection.
 * Each input trace's filtered version has the suffix ".lnd_tcpflows".
 */
void           libnd_tcpfilter_run(LND_TCPFilter *filter);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
