/*
 * $Header: /cvshome/build/org.osgi.service.upnp/src/org/osgi/service/upnp/UPnPAction.java,v 1.7 2005/05/13 20:33:16 hargrave Exp $
 *
 * Copyright (c) OSGi Alliance (2002, 2005). All Rights Reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html.
 */
package org.osgi.service.upnp;

import java.util.Dictionary;

/**
 * A UPnP action.
 * 
 * Each UPnP service contains zero or more actions. Each action may have zero or
 * more UPnP state variables as arguments.
 *  
 */
public interface UPnPAction {
	/**
	 * Returns the action name.
	 * 
	 * The action name corresponds to the <code>name</code> field in the
	 * <code>actionList</code> of the service description.
	 * <ul>
	 * <li>For standard actions defined by a UPnP Forum working committee,
	 * action names must not begin with <code>X_ </code> nor <code> A_</code>.</li>
	 * <li>For non-standard actions specified by a UPnP vendor and added to a
	 * standard service, action names must begin with <code>X_</code>.</li>
	 * </ul>
	 * 
	 * @return Name of action, must not contain a hyphen character or a hash
	 *         character
	 */
	String getName();

	/**
	 * Returns the name of the designated return argument.
	 * <p>
	 * One of the output arguments can be flagged as a designated return
	 * argument.
	 * 
	 * @return The name of the designated return argument or <code>null</code> if
	 *         none is marked.
	 */
	String getReturnArgumentName();

	/**
	 * Lists all input arguments for this action.
	 * <p>
	 * Each action may have zero or more input arguments.
	 * 
	 * @return Array of input argument names or <code>null</code> if no input
	 *         arguments.
	 * 
	 * @see UPnPStateVariable
	 */
	String[] getInputArgumentNames();

	/**
	 * List all output arguments for this action.
	 * 
	 * @return Array of output argument names or <code>null</code> if there are no
	 *         output arguments.
	 * 
	 * @see UPnPStateVariable
	 */
	String[] getOutputArgumentNames();

	/**
	 * Finds the state variable associated with an argument name.
	 * 
	 * Helps to resolve the association of state variables with argument names
	 * in UPnP actions.
	 * 
	 * @param argumentName The name of the UPnP action argument.
	 * @return State variable associated with the named argument or
	 *         <code>null</code> if there is no such argument.
	 * 
	 * @see UPnPStateVariable
	 */
	UPnPStateVariable getStateVariable(String argumentName);

	/**
	 * Invokes the action.
	 * 
	 * The input and output arguments are both passed as <code>Dictionary</code>
	 * objects. Each entry in the <code>Dictionary</code> object has a
	 * <code>String</code> object as key representing the argument name and the
	 * value is the argument itself. The class of an argument value must be
	 * assignable from the class of the associated UPnP state variable.
	 * 
	 * The input argument <code>Dictionary</code> object must contain exactly
	 * those arguments listed by <code>getInputArguments</code> method. The output
	 * argument <code>Dictionary</code> object will contain exactly those
	 * arguments listed by <code>getOutputArguments</code> method.
	 *
	 * @param args A <code>Dictionary</code> of arguments. Must contain the correct set and
	 * type of arguments for this action. May be <code>null</code> if no
	 * input arguments exist.
	 *
	 * @return A <code>Dictionary</code> with the output arguments.
	 *         <code>null</code> if the action has no output arguments.
	 *
	 * @throws Exception The execution fails for some reason.
	 * @throws UPnPException  A UPnP error has occured.
	 *
	 * @see UPnPStateVariable
	 */
	Dictionary invoke(Dictionary args) throws Exception, UPnPException;
}
