#
# This script was written by George A. Theall, <theall@tifaware.com>.
#
# See the Nessus Scripts License for details.
#


# NB: I define the script description here so I can later modify
#     it with the version number and install directory.
desc["english"] = "
This script detects whether the target is running osTicket and extracts
version numbers and locations of any instances found.

osTicket is a PHP-based open source support ticket system. See
http://www.osticket.com/ for more information. 

Risk factor : None";


if (description) {
  script_id(13858);
  script_version("$Revision: 1.2 $");
 
  name["english"] = "osTicket Detection";
  script_name(english:name["english"]);
 
  script_description(english:desc["english"]);
 
  summary["english"] = "Checks for the presence of osTicket";
  script_summary(english:summary["english"]);
 
  script_category(ACT_GATHER_INFO);
  script_copyright(english:"This script is Copyright (C) 2004 George A. Theall");

  family["english"] = "General";
  script_family(english:family["english"]);

  script_dependencie("global_settings.nasl", "http_version.nasl");
  script_require_ports("Services/www", 80);

  exit(0);
}

include("global_settings.inc");
include("http_func.inc");
include("http_keepalive.inc");

host = get_host_name();
port = get_http_port(default:80);
if (debug_level) display("debug: looking for osTicket on ", host, ":", port, ".\n");

if (!get_port_state(port)) exit(0);
if (!can_host_php(port:port)) exit(0);

# Search for osTicket in a couple of different locations.
#
# NB: Directories beyond cgi_dirs() come from a Google search - 
#     "inurl:open.php osticket" - and represent the more popular
#     installation paths currently. Still, cgi_dirs() should catch
#     the directory if its referenced elsewhere on the target.
dirs = make_list(
  "",
  "/osticket",
  "/helpdesk",
  "/soporte",
  "/support",
  "/ticket",
  cgi_dirs()
);
installs = 0;
foreach dir (dirs) {
  # Get osTicket's open.php.
  url = string(dir, "/open.php");
  if (debug_level) display("debug: checking ", url, ".\n");
  req = http_get(item:url, port:port);
  res = http_keepalive_send_recv(port:port, data:req);
  if (res == NULL) exit(0);           # can't connect
  if (debug_level) display("debug: res =>>", res, "<<\n");

  # Make sure the page is from osTicket.
  if (egrep(pattern:'alt="osTicket"', string:res, icase:TRUE)) {
    # nb: I only know about versions 1.1b, 1.2.4, 1.2.5, and 1.2.7 and only
    #     have access to 1.2.5 and 1.2.7 so I don't know how accurate
    #     version numbers will be.
    pat = "alt=.osTicket STS v(.+) *$";
    if (debug_level) display("debug: grepping results for =>>", pat, "<<\n");
    matches = egrep(pattern:pat, string:res);
    foreach match (split(matches)) {
      match = chomp(match);
      if (debug_level) display("debug: grepping >>", match, "<< for =>>", pat, "<<\n");
      ver = eregmatch(pattern:pat, string:match);
      if (ver == NULL) break;
      ver = ver[1];

      # Distinguish between 1.2.5 and 1.2.7.
      if (ver == "1.2") {
        # nb: 1.2.5 has an attachments dir whereas 1.2.7 has attachments.php
        url = string(dir, "/attachments.php");
        if (debug_level) display("debug: checking ", url, ".\n");
        req = http_get(item:url, port:port);
        res = http_keepalive_send_recv(port:port, data:req);
        if (res == NULL) exit(0);           # can't connect
        if (debug_level) display("debug: res =>>", res, "<<\n");

        #     thus if attachments.php exists it's 1.2.7...
        if (egrep(pattern:"You do not have access to attachments", string:res)) {
          ver = "1.2.7";
        }
        #     otherwise it's 1.2.5.
        else {
          ver = "1.2.5";
        }
      }
      if (debug_level) display("debug: osTicket version =>>", ver, "<<\n");

      # Success!
      set_kb_item(
        name:string("www/", port, "/osticket"), 
        value:string(ver, " under ", dir)
      );
      installations[dir] = ver;
      ++installs;

      # nb: only worried about the first match.
      break;
    }
  }
  # Scan for multiple installations only if "Thorough Tests" is checked.
  if (installs && !thorough_tests) break;
}

# Report any instances found unless Report verbosity is "Quiet".
if (installs && report_verbosity > 0) {
  if (installs == 1) {
    foreach dir (keys(installations)) {
      # empty - just need to set 'dir'.
    }
    info = string("osTicket ", ver, " was detected on the remote host under the path ", dir, ".");
  }
  else {
    info = string(
      "Multiple instances of osTicket were detected on the remote host:\n",
      "\n"
    );
    foreach dir (keys(installations)) {
      info = info + string("    ", installations[dir], ", installed under ", dir, "\n");
    }
    info = chomp(info);
  }

  desc = ereg_replace(
    string:desc["english"],
    pattern:"This script[^\.]+\.",
    replace:info
  );
  security_note(port:port, data:desc);
}
