\name{Lattice}
\alias{Lattice}
\title{Lattice Graphics}
\description{
  Implementation of Trellis Graphics in R
}
\details{
  Trellis Graphics is a framework for data visualization developed at
  the Bell Labs by Rick Becker, Bill Cleveland et al, extending ideas
  presented in Bill Cleveland's 1993 book \emph{Visualizing Data}.

  Lattice is best thought of as an implementation of Trellis Graphics
  for R. Its interface is based on the implementation in S-PLUS,
  but there are several differences. To the extent possible, care has
  been taken to ensure that existing Trellis code written for S-PLUS
  works unchanged (or with minimal change) in Lattice. If you are having
  problems porting S-PLUS code, read the entry for \code{panel} in
  the documentation for \code{\link{xyplot}}. The Lattice website
  referenced below has some useful material as well.

  Most high level Trellis functions in S-PLUS are implemented, with the
  exception of \code{piechart}.

  Lattice is built upon the Grid Graphics engine for R being developed
  by Paul Murrell and requires the \code{grid} add-on package.

  Type \code{library(help = lattice)} to see a list of (public)
  Lattice graphics functions for which further documentation is
  available. Apart from the documentation accompanying this package, 
  several documents outlining the use of Trellis graphics is available
  from Bell Lab's website that might provide a holistic introduction to
  the Trellis paradigm. Lattice also has a website with some tips.
}
\note{
  High level Lattice functions (like \code{\link{xyplot}}) are different
  from conventional S graphics functions because they don't actually
  draw anything. Instead, they return an object of class ``trellis''
  which has to be then \code{print}ed. This often causes confusion when
  the high level functions are called inside another function (most
  often \code{source}) and hence don't produce any output.

  Lattice plots are highly customizable via user-modifiable settings.
  However, these are completely unrelated to base graphics settings; in
  particular, changing \code{par()} settings have no effect on lattice
  plots.
}
\seealso{
  To get a general idea of what arguments are generally accepted by
  Lattice functions, see \code{\link{xyplot}}.

  To learn how to customise the Graphical parameters used by the Lattice
  functions, see \code{\link{trellis.par.set}} and \code{\link{lset}}.

  To learn how to initialise new devices or change the settings of the
  current device, see \code{\link{trellis.device}}

  To learn about sophisticated (non-default) printing capabilities, see
  \code{\link{print.trellis}}. Also described there are the
  possibilities to enhance the plots thus produced (using \code{locator}
  like functionality).

  Here is a list of `high level' functions in the Lattice library with a
  brief description of what they do:

  \bold{Univariate:}  

  \code{\link{barchart}} bar plots

  \code{\link{bwplot}} box and whisker plots

  \code{\link{densityplot}} kernel density plots  

  \code{\link{dotplot}} dot plots

  \code{\link{histogram}} histograms

  \code{\link{qqmath}} quantile plots against mathematical distributions

  \code{\link{stripplot}} 1-dimensional scatterplot

  \bold{Bivariate:}

  \code{\link{qq}} q-q plot for comparing two distributions
  
  \code{\link{xyplot}} scatter plot (and possibly a lot more)

  \bold{Trivariate:}

  \code{\link{levelplot}} level plots (similar to image plots in R)

  \code{\link{contourplot}} contour plots

  \code{\link{cloud}} 3-D scatter plots

  \code{\link{wireframe}} 3-D surfaces (similar to persp plots in R)

  \bold{Hypervariate:}
  
  \code{\link{splom}} scatterplot matrix
  
  \code{\link{parallel}} parallel coordinate plots
  
  \bold{Miscellaneous:}

  \code{\link{rfs}} residual and fitted value plot (also see
  \code{\link{oneway}})

  \code{\link{tmd}} Tukey Mean-Difference plot

  Lattice also has a collection of convenience functions that correspond
  to the base graphics primitives \code{\link{lines}},
  \code{\link{points}}, etc. They are implemented using Grid graphics,
  but try to be as close to the base versions as possible in terms of
  their argument list. These functions have imaginative names like
  \code{\link{llines}} are are often useful when writing (or porting
  from S-PLUS code) nontrivial panel functions.
}

\references{
  Bell Lab's Trellis Page:
  \url{http://cm.bell-labs.com/cm/ms/departments/sia/project/trellis/}

  Cleveland, W.S. (1993) \emph{Visualizing Data}.

  Becker, R.A., Cleveland, W.S. and Shyu, M.
  ``The Visual Design and Control of Trellis Display'',
  \emph{Journal of Computational and Graphical Statistics}

  Lattice Webpage:
  \url{http://packages.r-project.org/lattice/}
}
\author{Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{dplot}

\eof
\name{Rows}
\alias{Rows}
\title{Extract `rows' from a list}
\description{
  Convenience function to extract subset of a list. Usually used in
  creating keys.
}
\usage{
Rows(x, which)
}
\arguments{
  \item{x}{ list with each member a vector of the same length }
  \item{which}{ index for members of \code{x}}
}
\value {
  A list similar to \code{x}, with each x[[i]] replaced by x[[i]][which]
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{xyplot}}, \code{\link{Lattice}}}
\keyword{utilities}

















\eof
\name{banking}
\alias{banking}
\title{Banking}
\description{
  Calculates `banking' slope
}
\usage{
banking(dx, dy)
}
\arguments{
  \item{dx, dy}{ vector of consecutive x, y differences. }
}
\details{
  \code{banking} is a crude banking function used when aspect="xy" in
  high level Trellis functions. Its usually not very menaingful except
  for \code{xyplot}. It just orders the absolute slopes and returns a value
  which when adjusted by the panel scale limits will make the median of
  the above absolute slopes a 45 degree line.
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{Lattice}}, \code{\link{xyplot}}}
\keyword{dplot}

















\eof
\name{cloud}
\title{3d Scatter Plot}
\synopsis{
cloud(formula,
             data = parent.frame(),
             allow.multiple = is.null(groups) || outer,
             outer = FALSE,
             auto.key = FALSE,
             aspect = c(1,1),
             panel = "panel.cloud",
             prepanel = NULL,
             scales = NULL,
             strip = TRUE,
             groups = NULL,
             xlab,
             xlim = if (is.factor(x)) levels(x) else range(x, na.rm = TRUE),
             ylab,
             ylim = if (is.factor(y)) levels(y) else range(y, na.rm = TRUE),
             zlab,
             zlim = if (is.factor(z)) levels(z) else range(z, na.rm = TRUE),
             distance = .2,
             perspective = TRUE,
             R.mat = diag(4),
             screen = list(z = 40, x = -60),
             zoom = .8,
             at,
             pretty = FALSE,
             drape = FALSE,
             drop.unused.levels,
             ...,
             colorkey = any(drape),
             col.regions, cuts = 70,
             subset = TRUE,
             axs.default = "r")
wireframe(formula,
             data = parent.frame(),
             panel = "panel.wireframe",
             prepanel = NULL,
             strip = TRUE,
             groups = NULL,
             cuts = 70,
             pretty = FALSE,
             drape = FALSE,
             ...,
             col.regions = trellis.par.get("regions")$col,
             colorkey = any(drape),
             subset = TRUE)
}
\usage{
cloud(formula,
      data,
      aspect = c(1, 1),
      scales = list(distance = rep(1, 3), arrows = TRUE),
      zlab,
      zlim, 
      distance = 0.2,
      par.box,
      perspective = TRUE,
      R.mat = diag(4), 
      screen = list(z = 40, x = -60),
      zoom = .9,
      \dots)
wireframe(formula, data,
          at = pretty(z, cuts),
          col.regions,
          drape = FALSE,
          shade = FALSE,
          pretty = FALSE,
          colorkey = any(drape),
          cuts = 70,
          distance = 0.2,
          par.box,
          screen = list(z = 40, x = -60),
          zoom,
          scales),
          \dots)
}
\description{
  Draws 3d scatter plots and surfaces.
}
\alias{cloud}
\alias{wireframe}
\arguments{
  \item{formula}{a formula of the form \code{z ~ x * y | g1 * g2 * ...},
    where \code{z} is a numeric response, and \code{x, y} are numeric
    values. \code{g1,g2,...}, if present, are conditioning variables
    used for conditioning, and must be either factors or shingles. In
    the case of \code{wireframe}, calculations are based on the
    assumption the \code{x} and \code{y} values are evaluated on a
    rectangular grid (defined by their unique values). The grid points
    need not be equally spaced.

    For \code{wireframe}, \code{x}, \code{y} and \code{z} may also be
    matrices (of the same dimension), in which case they are taken to
    represent a 3-D surface parametrized on a 2-D grid (e.g., a sphere).
    Conditioning is not possible with this feature. See details below.

    Missing values are allowed, either as \code{NA} values in the
    \code{z} vector, or missing rows in the data frame (note however
    that in that case the X and Y grids will be determined only by the
    available values). For a grouped display (producing multiple
    surfaces), missing rows are not allowed, but \code{NA}'s in \code{z}
    are.

    As an extension to partially support the form used in
    \code{filled.contour} and \code{image}, \code{formula} can be a
    matrix.
  }
  \item{data}{data frame in which variables are evaluated}
  \item{aspect}{ vector of length 2, giving the relative aspects of the
    y-size/x-size and z-size/x-size of the enclosing rectangle.}
  \item{scales}{ describes scales. Can contain lists named x, y and
    z. Arrows are drawn if \code{arrows=TRUE}, otherwise tick marks with
    labels are drawn. Both can be suppressed by \code{draw=FALSE}.
    Several other components that work in the usual \code{scales}
    argument also work here (see \code{\link{xyplot}}).
  }
  \item{zlab}{z label}
  \item{zlim}{z limits}
  \item{distance}{
    numeric, between 0 and 1, controls amount of perspective. See
    details below
  }
  \item{par.box}{
    graphical parameters for box, namely, col, lty and lwd. By default
    obtained from the parameter \code{box.3d}
  }
  \item{perspective}{
    logical, whether to plot a perspective view. Setting this to
    \code{FALSE} is equivalent to setting \code{distance=0}
  }
  \item{R.mat}{
    initial rotation matrix in homogeneous coordinates. (Untested)
  }
  \item{screen}{
    A list determining the sequence of rotations to be applied to the
    data before being plotted. The initial position starts with the
    viewing point along the positive z-axis, and the x and y axes in the
    usual position. Each component of the list should be named one of
    \code{x, y, z} (repititions are allowed), with their values
    indicating the amount of rotation about that axis in degrees.
  }
  \item{zoom}{
    factor by which to scale the picture. Useful to get the variable
    names into the plot
  }
  \item{drape}{ logical, whether the wireframe is to be draped in color}
  \item{shade}{
    logical, whether the wireframe is to be illuminated from a light
    source. See \code{\link{panel.3dwire}} for details
  }
  \item{at, col.regions}{
    these arguments are analogous to those in
    \code{\link{levelplot}}. if \code{drape=TRUE}, \code{at} gives the
    vector of values where the colors change, and \code{col.regions} the
    vector of colors to be used in that case
  }
  \item{cuts}{ the default number of cutpoints if \code{drape=TRUE}}
  \item{pretty}{ whether the cutpoints should be pretty}
  \item{colorkey}{
    whether a color key should be drawn alongside. See
    \code{\link{levelplot}} for details
  }
  \item{\dots}{ other arguments, passed to the panel function }
}
\details{
  These functions produce three dimensional plots in each panel. The
  orientation is obtained as follows: the data are scaled to fall within
  a bounding box that is contained in the [-0.5, 0.5] cube (even smaller
  for non-default values of \code{aspect}).  The viewing direction is
  given by a sequence of rotations specified by the \code{screen}
  argument, starting from the positive Z-axis. The viewing point
  (camera) is located at a distance of \code{1/distance} from the
  origin. If \code{perspective=FALSE}, \code{distance} is set to 0
  (i.e., the viewing point is at an infinite distance).

  \code{cloud} draws a 3-D Scatter Plot, while \code{wireframe} draws a
  3-D surface (usually evaluated on a grid). Multiple surfaces can be
  drawn by \code{wireframe} using the \code{groups} argument (although
  this is of limited use because the display is messed up when the
  surfaces intersect). Specifying \code{groups} with \code{cloud}
  results in a \code{panel.superpose}-like effect (via
  \code{\link{panel.3dscatter}}).

  \code{wireframe} can optionally render the surface as being
  illuminated by a light source (no shadows though). Details can be
  found in the help page for \code{\link{panel.3dwire}}. Note that
  although arguments controlling these are actually arguments for the
  panel function, they can be supplied to \code{cloud} and
  \code{wireframe} directly.

  For single panel plots, \code{wireframe} can also plot parametrized
  3-D surfaces (i.e., functions of the form f(u,v) = (x(u,v), y(u,v),
  z(u,v)), where values of (u,v) lie on a rectangle. The simplest
  example of this sort of surface is a sphere parametrized by latitude
  and longitude. This can be achieved by calling \code{wireframe} with a
  \code{formula} of the form \code{z~x*y}, where \code{x}, \code{y} and
  \code{z} are all matrices of the same dimension, representing the
  values of x(u,v), y(u,v) and z(u,v) evaluated on a discrete
  rectangular grid (the actual values of (u,v) are unimportant). 

  When this feature is used, the heights used to calculate \code{drape}
  colors or shading colors are no longer the \code{z} values, but the
  distances of \code{(x,y,z)} from the origin.

  Note that this feature does not work with \code{groups},
  \code{subscripts}, \code{subset}, etc. Conditioning variables are also
  not supported in this case.


  The algorithm for identifying which edges of the bounding box are
  `behind' the points doesn't work in some extreme situations. Also,
  \code{panel.cloud} tries to figure out the optimal location of the
  arrows and axis labels automatically, but can fail on occasion
  (especially when the view is from ``below'' the data). This can be
  manually controlled by the \code{scpos} argument in
  \code{\link{panel.cloud}}.

  These and all other high level Trellis functions have several other
  arguments in common. These are extensively documented only in the
  help page for \code{xyplot}, which should be consulted to learn more
  detailed usage. 
}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\seealso{
  \code{\link{xyplot}}, \code{\link{levelplot}},
  \code{\link{panel.cloud}},
  \code{\link{Lattice}}  
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\examples{
data(volcano)  ## 87 x 61 matrix
wireframe(volcano, shade = TRUE,
          aspect = c(61/87, 0.4),
          light.source = c(10,0,10))

g <- expand.grid(x = 1:10, y = 5:15, gr = 1:2)
g$z <- log((g$x^g$g + g$y^2) * g$gr)
wireframe(z ~ x * y, data = g, groups = gr,
          scales = list(arrows = FALSE),
          drape = TRUE,
          screen = list(z = 30, x = -60))

data(iris)
cloud(Sepal.Length ~ Petal.Length * Petal.Width | Species, data = iris,
      screen = list(x = -90, y = 70), distance = .4, zoom = .6)

par.set <- list(axis.line = list(col = "transparent"), clip = list(panel = FALSE))
print(cloud(Sepal.Length ~ Petal.Length * Petal.Width, 
            data = iris, cex = .8, 
            groups = Species, 
            subpanel = panel.superpose,
            main = "Stereo",
            screen = list(z = 20, x = -70, y = 3),
            par.settings = par.set),
      split = c(1,1,2,1), more = TRUE)
print(cloud(Sepal.Length ~ Petal.Length * Petal.Width,
            data = iris, cex = .8, 
            groups = Species,
            subpanel = panel.superpose,
            main = "Stereo",
            screen = list(z = 20, x = -70, y = 0),
            par.settings = par.set),
      split = c(2,1,2,1))

}
\keyword{hplot}

\eof
\name{draw.colorkey}
\alias{draw.colorkey}
\title{Produce a Colorkey for levelplot}
\description{
  Produces (and possibly draws) a Grid frame grob which is a colorkey
  that can be placed in other Grid plots. Used in levelplot
}
\usage{
draw.colorkey(key, draw=FALSE, vp=NULL)
}
\arguments{
  \item{key}{ A list determining the key. See documentation for
    \code{levelplot}, in particular the section describing the
    `colorkey' argument for details.
  }
  \item{draw}{ logical, whether the grob is to be drawn. }
  \item{vp}{ viewport }
}
\details{
  Maybe later :)
}
\value{
  A Grid frame object (that inherits from ``grob'')
}
\author{Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{xyplot}}}
\keyword{dplot}


\eof
\name{draw.key}
\alias{draw.key}
\title{Produce a Legend or Key}
\description{
  Produces (and possibly draws) a Grid frame grob which is a legend (aka 
  key) that can be placed in other Grid plots.
}
\usage{
draw.key(key, draw=FALSE, vp=NULL)
}
\arguments{
  \item{key}{ A list determining the key. See documentation for
    \code{xyplot}, in particular the section describing the `key'
    argument for details.
  }
  \item{draw}{ logical, whether the grob is to be drawn. }
  \item{vp}{ viewport }
}
\details{
  Maybe later :)
}
\value{
  A Grid frame object (that inherits from ``grob'')
}
\author{Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{xyplot}}}
\keyword{dplot}


\eof
\name{histogram}
\alias{histogram}
\alias{densityplot}
\title{Histograms and Kernel Density Plots}
\synopsis{
histogram(formula,
             data = parent.frame(),
             allow.multiple = is.null(groups) || outer,
             outer = FALSE,
             auto.key = FALSE,
             aspect = "fill",
             panel = "panel.histogram",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab,
             xlim,
             ylab,
             ylim,
             type = c("percent", "count", "density"),
             nint = if (is.factor(x)) length(levels(x))
             else round(log2(length(x)) + 1),
             endpoints = extend.limits(range(x[!is.na(x)]), prop = 0.04),
             breaks = if (is.factor(x)) seq(0.5, length = length(levels(x))+1)
             else do.breaks(endpoints, nint),
             equal.widths = TRUE,
             drop.unused.levels,
             ...,
             subscripts = !is.null(groups),
             subset = TRUE)
densityplot(formula,
             data = parent.frame(),
             allow.multiple = is.null(groups) || outer,
             outer = FALSE,
             auto.key = FALSE,
             aspect = "fill",
             panel = if (is.null(groups)) "panel.densityplot" else "panel.superpose",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab,
             xlim,
             ylab,
             ylim,
             bw = NULL,
             adjust = NULL,
             kernel = NULL,
             window = NULL,
             width = NULL,
             give.Rkern = FALSE,
             n = 50,
             from = NULL,
             to = NULL,
             cut = NULL,
             na.rm = NULL,
             drop.unused.levels,
             ...,
             panel.groups = "panel.densityplot",
             subscripts = !is.null(groups),
             subset = TRUE)
}
\usage{
histogram(formula,
          data, 
          type = c("percent", "count", "density"),
          nint = if(is.factor(x)) length(levels(x))
                 else round(log2(length(x))+1),
          endpoints = range(x[!na.x]),
          breaks = if(is.factor(x)) seq(0.5, length = length(levels(x))
          + 1) else do.breaks(endpoints, nint),
          equal.widths = FALSE, 
          \dots)

densityplot(formula, data, n = 50, plot.points = TRUE, ref = FALSE,
            \dots)
}
\description{
  Draw Histograms and Kernel Density Plots
}
\arguments{
  \item{formula}{A formula of the form \code{~ x | g1 * g2 * \dots}
    indicates that histograms or Kernel Density estimates of \code{x}
    should be produced conditioned on the levels of the (optional)
    variables \code{g1,g2,\dots}. When the conditioning variables
    \code{g1,g2,\dots} are missing, the leading \code{~} can be dropped.
    
    \code{x} can be numeric (or factor for \code{histogram}), and each
    of \code{g1,g2,\dots} must be either factors or shingles.

    As a special case, the right hand side of the formula (\code{x}) can
    actually contain more than one variable separated by a '+'
    sign. What happens in this case is described in details in the
    documentation for \code{\link{xyplot}}.
  }
  \item{data}{
    optional data frame in which variables are to be evaluated
  }
  \item{type}{ Character string indicating type of histogram to be
    drawn. ``percent'' and ``count'' give relative frequency and
    frequency histograms, and can be misleading when breakpoints are not
    equally spaced. ``density'' produces a density scale histogram.

    \code{type} defaults to ``percent'', except when the breakpoints
    are unequally spaced or \code{breaks = NULL}, when it defaults to
    ``density''.
  }
  \item{nint}{ Number of bins. Applies only when \code{breaks} is
    unspecified in the call.
  }
  \item{endpoints}{ vector of length 2 indicating the range of x-values
    that is to be covered by the  histogram. Again, applies only when
    \code{breaks} is unspecified.
  }
  \item{breaks}{ numeric vector of length = (number of bins + 1)
    defining the breakpoints of the bins. Note that when breakpoints are
    not equally spaced, the only value of \code{type} that makes sense
    is density.

    Usually all panels use the same breakpoints. This can be changed by
    specifying \code{breaks = NULL}. This has the effect of letting each
    panel choose its own breakpoints. The choice of these breakpoints
    are made as follows: The number of bins is calculated by the formula
    for \code{nint} above, and then breakpoints are chosen according to
    the value of \code{equal.widths}.
  }
  \item{equal.widths}{ logical, relevant only when \code{breaks=NULL}.
    If \code{TRUE}, equally spaced bins will be selected, otherwise, 
    approximately equal area bins will be selected (this would mean that
    the breakpoints will \bold{not} be equally spaced).
  }
  \item{n}{number of points at which density is to be evaluated}
  \item{plot.points}{ logical specifying whether the \code{x} values
    should be plotted.
  }
  \item{ref}{ logical specifying whether a reference x-axis should be
    drawn.
  }
  \item{\dots}{ other arguments, passed along to the panel function. In
    the case of \code{densityplot}, if the default panel function is
    used, then arguments appropriate to \code{density} can be
    included. This can control the details of how the Kernel Density
    Estimates are calculated. See documentation for
    \code{\link{density}} for details.
  }
}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\details{
  \code{histogram} draws Conditional Histograms, while
  \code{densityplot} draws Conditional Kernel Density Plots.  The
  density estimate in \code{densityplot} is actually calculated using
  the function \code{density}, and all arguments accepted by it can be
  passed (as \code{\dots}) in the call to \code{densityplot} to control
  the output. See documentation of \code{density} for details. (Note: The
  default value of the argument \code{n} of \code{density} is changed to
  50.)
  
  These and all other high level Trellis functions have several
  arguments in common. These are extensively documented only in the
  help page for \code{xyplot}, which should be consulted to learn more
  detailed usage. 
}
\note{
  The form of the arguments accepted by the default
  panel function \code{panel.histogram} is different from that in
  S-PLUS. Whereas S-PLUS calculates the heights inside \code{histogram}
  and passes only the breakpoints and the heights to the panel function,
  here the original variable \code{x} is passed along with the
  breakpoints. This allows plots as in the second example below.
}
\seealso{
  \code{\link{xyplot}},
  \code{\link{panel.histogram}},
  \code{\link{density}},
  \code{\link{panel.densityplot}},
  \code{\link{panel.mathdensity}},
  \code{\link{Lattice}} 
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\examples{
require(stats)
data(singer)

histogram( ~ height | voice.part, data = singer, nint = 17,
          endpoints = c(59.5, 76.5), layout = c(2,4), aspect = 1,
          xlab = "Height (inches)")

histogram( ~ height | voice.part, data = singer,
          xlab = "Height (inches)", type = "density",
          panel = function(x, ...) {
              panel.histogram(x, ...)
              panel.mathdensity(dmath = dnorm, col = "black",
                                args = list(mean=mean(x),sd=sd(x)))
          } )

densityplot( ~ height | voice.part, data = singer, layout = c(2, 4),  
            xlab = "Height (inches)", bw = 5)
}
\keyword{hplot}

\eof
\name{latticeParseFormula}
\alias{latticeParseFormula}
\title{Parse Trellis formula}
\usage{
latticeParseFormula(model, data, dimension = 2,
                    subset = TRUE, groups = NULL,
                    multiple, outer,
                    subscripts,
                    drop)
}
\description{
  this function is used by high level Lattice functions like
  \code{xyplot} to parse the formula argument and evaluate various
  components of the data. Might be useful by itself. \code{y} and
  \code{g_1,\dots,g_n} may be missing.
}

\arguments{
  \item{model}{
    the model/formula to be parsed.  This can be in either of two
    possible forms, one for 2d and one for 3d formulas, determined by
    the \code{dimension} argument. The 2d formulas are of the form
    \code{y~x|g_1*\dots*g_n}, and the 3d formulas are of the form
    \code{z~x*y|g_1*\dots*g_n}. In the first form, \code{y} may be
    omitted. The conditioning variables \code{g_1,
      \dots,g_n} can be omitted in either case.
  }
  \item{data}{ the environment/dataset where the variables in the 
    formula are evaluated. By default, the environment where the
    function was called from.
  }
  \item{dimension}{ dimension of the model, see above}
  \item{subset}{ index for choosing a subset of the data frame }
  \item{groups}{ the grouping variable, if present}
  \item{multiple, outer}{
    logicals, determining how a `+' in the y and x components of the
    formula are processed. See \code{\link{xyplot}} for details
  }
  \item{subscripts}{ logical, whether subscripts are to be calculated }
  \item{drop}{
    logical or list, similar to the \code{drop.unused.levels} argument
    in \code{\link{xyplot}}, indicating whether unused levels of
    conditioning factors and data variables that are factors are to be
    dropped.
  }
}

\value{ returns a list with several components, including \code{left,
    right, left.name, right.name, condition} for 2-D, and \code{left,
    right.x, right.y, left.name, right.x.name, right.y.name, condition}
  for 3-D. Other possible components are groups, subscr

}

\seealso{
  \code{\link{xyplot}}, \code{\link{Lattice}}
}
\author{Saikat DebRoy, Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{utilities}

\eof
\name{levelplot}
\title{Level Plots}
\synopsis{
levelplot(formula,
             data = parent.frame(),
             allow.multiple = is.null(groups) || outer,
             outer = TRUE,
             aspect = "fill",
             panel = "panel.levelplot",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab,
             xlim,
             ylab,
             ylim,
             at,
             contour = FALSE,
             cuts = 15,
             labels = FALSE,
             pretty = FALSE,
             region = TRUE,
             drop.unused.levels,
             ...,
             colorkey = region,
             col.regions = trellis.par.get("regions")$col,
             subset = TRUE)
contourplot(formula,
             data = parent.frame(),
             panel = "panel.levelplot",
             prepanel = NULL,
             strip = TRUE,
             groups = NULL,
             cuts = 7,
             labels = TRUE,
             contour = TRUE,
             pretty = TRUE,
             region = FALSE,
             ...,
             subset = TRUE)
}
\usage{
levelplot(formula, data,
          at,
          contour = FALSE,
          cuts = 15,
          pretty = FALSE,
          region = TRUE,
          ...,
          col.regions = trellis.par.get("regions")$col,
          colorkey = region)
contourplot(formula, data, at,
            contour = TRUE,
            labels = format(at),
            cuts = 7,
            pretty = TRUE,
            ...)

}
\description{
  Draw Level Plots and Contour plots.
}
\alias{levelplot}
\alias{contourplot}
\arguments{
  \item{formula}{a formula of the form \code{z ~ x * y | g1 * g2 * ...},
    where \code{z} is a numeric response, and \code{x, y} are numeric
    values evaluated on a rectangular grid.  \code{g1,g2,...} are
    optional conditional variables, and must be either factors or
    shingles if present.

    Calculations are based on the assumption that all x and y values are
    evaluated on a grid (defined by their unique values). The function
    will not return an error if this is not true, but the display might
    not be meaningful.  However, the x and y values need not be equally
    spaced.

    As an extension to partially support the form used in
    \code{filled.contour} and \code{image}, \code{formula} can be a
    matrix.
  }
  \item{data}{
    optional data frame in which variables are to be evaluated 
  }
  \item{at}{
    numeric vector giving breaks along the range of \code{z}. Contours
    (if any) will be drawn at these heights, and the regions in between
    would be colored using \code{col.regions}.
  }
  \item{col.regions}{ color vector to be used if regions is TRUE. The
    general idea is that this should be a color vector of moderately
    large length (longer than the number of regions. By default this is
    100). It is expected that this vector would be gradually varying in
    color (so that nearby colors would be similar). When the colors are
    actually chosen, they are chosen to be equally spaced along this
    vector. When there are more regions than col.regions, the colors are
    recycled.
  }
  \item{colorkey}{logical specifying whether a color key is to be drawn
    alongside the plot, or a list describing the color key. The list may
    contain the following components:

    \code{space} location of the colorkey, can be one of ``left'',
    ``right'', ``top'' and ``bottom''. Defaults to ``right''.
    
    \code{x,y} location, currently unused

    \code{col} vector of colors

    \code{at} numeric vector specifying where the colors change. must be
    of length 1 more than the col vector.

    \code{labels} a character vector for labelling the \code{at} values,
    or more commonly, a list of components \code{labels, at, cex, col,
      font} describing label positions.

    \code{tick.number} approximate number of ticks.

    \code{corner} interacts with x, y; unimplemented

    \code{width} width of the key

    \code{height} length of key w.r.t side of plot.
  }
  \item{contour}{
    logical, whether to draw contour lines.
  }
  \item{cuts}{
    number of levels the range of \code{z} would be divided into
  }
  \item{labels}{
    logical specifying whether contour lines should be labelled, or
    character vector of labels for contour lines. The type of labelling
    can be controlled by the \code{label.style} argument, which is
    passed on to \code{\link{panel.levelplot}}
  }
  \item{pretty}{
    logical, whether to use pretty cut locations and labels
  }
  \item{region}{
    logical, whether regions between contour lines should be filled
  }
  \item{\dots}{other arguments}
}
\details{
  These and all other high level Trellis functions have several
  arguments in common. These are extensively documented only in the
  help page for \code{xyplot}, which should be consulted to learn more
  detailed usage.

  Other useful arguments are mentioned in the help page for the default
  panel function \code{\link{panel.levelplot}} (these are formally
  arguments to the panel function, but can be specified in the high
  level calls directly).
}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\seealso{
  \code{\link{xyplot}}, \code{\link{Lattice}},
  \code{\link{panel.levelplot}}
}
\author{Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\examples{
x <- seq(pi/4, 5 * pi, length = 100)
y <- seq(pi/4, 5 * pi, length = 100)
r <- as.vector(sqrt(outer(x^2, y^2, "+")))
grid <- expand.grid(x=x, y=y)
grid$z <- cos(r^2) * exp(-r/(pi^3))
levelplot(z~x*y, grid, cuts = 50, scales=list(log="e"), xlab="",
          ylab="", main="Weird Function", sub="with log scales",
          colorkey = FALSE, region = TRUE)

#S-PLUS example
require(stats)
data(environmental)
attach(environmental)
ozo.m <- loess((ozone^(1/3)) ~ wind * temperature * radiation,
       parametric = c("radiation", "wind"), span = 1, degree = 2)
w.marginal <- seq(min(wind), max(wind), length = 50)
t.marginal <- seq(min(temperature), max(temperature), length = 50)
r.marginal <- seq(min(radiation), max(radiation), length = 4)
wtr.marginal <- list(wind = w.marginal, temperature = t.marginal,
        radiation = r.marginal)
grid <- expand.grid(wtr.marginal)
grid[, "fit"] <- c(predict(ozo.m, grid))
contourplot(fit ~ wind * temperature | radiation, data = grid,
          cuts = 10, region = TRUE,
          xlab = "Wind Speed (mph)",
          ylab = "Temperature (F)",
          main = "Cube Root Ozone (cube root ppb)",
          col.regions = trellis.par.get("regions")$col)
detach()
}
\keyword{hplot}




\eof
\name{llines}
\alias{larrows}
\alias{llines}
\alias{lplot.xy}
\alias{lpoints}
\alias{lsegments}
\alias{ltext}
\title{Lattice Replacements of base graphics functions}
\description{
  These functions are intended to replace some commonly used base R
  graphics functions in panel functions.
}
\synopsis{
larrows(x0, y0, x1, y1, x2, y2, angle = 30, code = 2, length = NULL, 
    proportion = 0.05, \dots)
llines(x, y = NULL, type = "l", col = plot.line$col, lty = plot.line$lty, 
    lwd = plot.line$lwd, \dots)
lplot.xy(xy, type, pch = 1, lty = 1, col = 1, cex = 1, lwd = 1,
         font = 1, fontfamily = NULL, fontface = NULL,
         col.line = col, \dots)  
lpoints(x, y = NULL, type = "p", col = plot.symbol$col,
        pch = plot.symbol$pch, 
        font = plot.symbol$font,
        fontfamily  = plot.symbol$fontfamily,
        fontface = plot.symbol$fontface,
        cex = plot.symbol$cex, \dots)  
lsegments(x0, y0, x1, y1, x2, y2, col = add.line$col, lty = add.line$lty, 
    lwd = add.line$lwd, \dots)
ltext(x, y = NULL, labels = seq(along = x), col = add.text$col,
    cex = add.text$cex, srt = 0,
    font, fontfamily = add.text$fontfamily,
    fontface = add.text$fontface, adj = c(0.5, 0.5), pos = NULL, \dots)  
}
\usage{
larrows(x0, y0, x1, y1, proportion, \dots)
llines(x, y, \dots)
lplot.xy(xy, \dots)
lpoints(x, \dots)
lsegments(x0, y0, x1, y1, \dots)
lsegments(x1, y1, x2, y2, \dots)
ltext(x, \dots)
}
\arguments{
  \item{ x, y, x0, y0, x1, y1, x2, y2, xy}{ locations. x2, y2 present
    for S compatibility
  }
  \item{proportion}{ The proportion of the arrow edges compared to
  the whole line. This is different from \code{arrows}}
  \item{\dots}{ other arguments}
}
\details{
  These functions are meant to be grid replacements of the corresponding
  base R graphics functions, to allow existing Trellis code to be used
  with minimal modification. 

  See the documentation of the base functions for usage. Not all
  arguments are always supported. All these correspond to the default
  methods only. For \code{ltext}, only values 0, .5 and 1 for \code{adj}
  have any effect.
}
\seealso{\code{\link{points}}, \code{\link{lines}},
  \code{\link{text}}, \code{\link{segments}},
  \code{\link{Lattice}}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{dplot}

\eof
\name{lset}
\alias{lset}
\alias{col.whitebg}
\title{Simple Interfaces to modify Trellis Settings}
\description{
  A (hopefully) simpler alternative interface to
  \code{trellis.par.get/set}. NOT S compatible
}
\usage{
lset(theme = col.whitebg())
col.whitebg()
}
\arguments{
  \item{theme}{ a list decribing how to change the settings of the
    current active device. Valid components are those in the list
    returned by \code{trellis.par.get()}. Each component must itself be
    a list, with one or more of the appropriate components (need not
    have all components). Changes are made to the settings for the
    currently active device only.
  }
}
\details{
  \code{lset} is a wrapper to \code{trellis.par.set} that allows setting
  more than one parameter at a time, and only the components that are to
  be changed need be specified.
  
  This function is expected to be used in conjunction with functions
  that return `themes', i.e., lists which when supplied to \code{lset}
  change the overall look and feel of the display.

  \code{col.whitebg} is one such function, which returns a list which
  when supplied to \code{lset} changes the background to transparent and
  the other colors to something suitable. Contributions of more such
  `themes' are welcome. 

  \code{lset} is used to modify the settings on a one-time basis. It is
  also possible to make persistent changes in the default theme used via
  a user-settable option, see \code{\link{trellis.device}} for details.

}

\seealso{

  \code{\link{lattice.theme}}, \code{\link{trellis.device}},
  \code{\link{show.settings}}, \code{\link{Lattice}}

}


\examples{
show.settings(col.whitebg())
# The following reproduces S-PLUS Unix black and white pch:
#lset(list(superpose.symbol=list(pch=c("o","+",">","s","w","#","\{"))))
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{utilities}

\eof
\name{oneway}
\alias{oneway}
\title{ Fit One-way Model}
\description{
  Fits a One-way model to univariate data grouped by a factor, the
  result often being displayed using \code{rfs} 
}
\usage{
oneway(formula, data, location=mean, spread=function(x) sqrt(var(x)))
}
\arguments{
  \item{formula}{ formula of the form \code{y ~ x} where \code{y} is the
    numeric response and \code{x} is the grouping factor
  }
  \item{data}{ data frame in which the model is to be evaluated }
  \item{location}{ function or numeric giving the location statistic to
    be used for centering the observations, e.g. \code{median}, 0 (to
    avoid centering).
  }
  \item{spread}{ function or numeric giving the spread statistic to
    be used for scaling the observations, e.g. \code{sd}, 1 (to
    avoid scaling).
  }
}
\value{
  A list with components
  \item{location}{ vector of locations for each group.}
  \item{spread}{ vector of spreads for each group.}
  \item{fitted.values}{ vector of locations for each observation.}
  \item{residuals}{ residuals (\code{y - fitted.values}).}
  \item{scaled.residuals}{residuals scaled by \code{spread} for their
    group} 
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{rfs}}, \code{\link{Lattice}} }
\keyword{dplot}

\eof
\name{panel.barchart}
\alias{panel.barchart}
\title{ Default Panel Function for barchart}
\description{
  Default panel function for \code{barchart}.
}
\usage{
panel.barchart(x, y, box.ratio = 1,
               horizontal = TRUE,
               origin = NULL, reference = TRUE,
               stack = FALSE,
               groups = NULL, 
               col = if (is.null(groups)) bar.fill$col else
               regions$col,
               \dots)
}
\arguments{
  \item{x}{ Extent of Bars. By default, bars start at left of panel,
    unless \code{origin} is specified, in which case they start there
  }
  \item{y}{ Horizontal location of bars, possibly factor}
  \item{box.ratio}{ratio of bar width to inter-bar space}
  \item{horizontal}{ logical. If FALSE, the plot is `transposed' in the
    sense that the behaviours of x and y are switched. x is now
    the `factor'.  Interpretation of other arguments change
    accordingly. See documentation of \code{bwplot} for a fuller
    explanation.
  }
  \item{origin}{ origin of the bars. For grouped displays with
    \code{stack = TRUE}, this argument is ignored and the origin set to
    0. Otherwise, defaults to \code{NULL}, in which case bars start at
    leftmost end of panel.
  }
  \item{reference}{ logical, whether a reference line is to be drawn at
    the origin
  }
  \item{stack}{ logical, relevant when groups is non-null. If
    \code{FALSE} (the default), bars for different values of the
    grouping variable are drawn side by side, otherwise they are
    stacked.
  }
  \item{groups}{ optional grouping variable}
  \item{col}{ Color of the bars. By default, the trellis parameter
    \code{bar.fill} is used if there is no grouping variable, otherwise
    \code{regions} is used. 
  }
  \item{\dots}{ extra arguments will be accepted but ignored }
}

\details{
  A barchart is drawn in the panel. Note that most arguments controlling
  the display can be supplied to the high-level \code{barchart} call
  directly.
}

\seealso{
  \code{\link{barchart}}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{dplot}


\eof
\name{panel.bwplot}
\alias{panel.bwplot}
\title{ Default Panel Function for bwplot }
\description{
  This is the default panel function for \code{bwplot}.
}
\synopsis{
panel.bwplot(x, y, box.ratio=1, horizontal = TRUE, pch=box.dot$pch,
             col=box.dot$col, cex = box.dot$cex,
             font = box.dot$font, fontfamily = box.dot$fontfamily,
             fontface = box.dot$fontface,
             fill = box.rectangle$fill, varwidth = FALSE,
             levels.fos = NULL, coef = 1.5, ...)
}
\usage{
panel.bwplot(x, y, box.ratio = 1, horizontal, pch, col, cex,
            fill = "transparent", varwidth = FALSE, coef = 1.5, ...)
}
\arguments{
  \item{x, y}{ numeric vector or factor. Boxplots drawn for each unique value of
    \code{y} (\code{x}) if \code{horizontal} is \code{TRUE} (\code{FALSE})}
  \item{box.ratio}{ ratio of box height to inter box space}
  \item{horizontal}{ logical. If FALSE, the plot is `transposed' in the
    sense that the behaviours of x and y are switched. x is now
    the `factor'.  Interpretation of other arguments change
    accordingly. See documentation of \code{bwplot} for a fuller
    explanation.
  }
  \item{pch, col, cex}{ graphical parameters controlling the dot}
  \item{fill}{ color to fill the boxplot}
  \item{varwidth}{ logical. If TRUE, widths of boxplots are proportional
    to the number of points used in creating it.}
  \item{coef}{ passed to \code{boxplot.stats}}
  \item{\dots}{ graphical parameters can be supplied}
}
\details{
  Creates Box and Whisker plot of \code{x} for every level of
  \code{y}. The actual boxplot statistics are calculated using
  \code{boxplot.stats}. Note that most arguments controlling the display
  can be supplied to the high-level \code{bwplot} call directly.
}

\seealso{
  \code{\link{bwplot}}, \code{\link{boxplot.stats}}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{dplot}

















\eof
\name{panel.cloud}
\alias{panel.cloud}
\alias{panel.wireframe}
\alias{panel.3dscatter}
\alias{panel.3dwire}
\title{ Default Panel Function for cloud }
\description{
  These are default panel functions controlling \code{cloud} and
  \code{wireframe} displays.
}

\usage{
panel.cloud(
x, y, z, subscripts,
            groups = NULL,
            distance, xlim, ylim, zlim,
            panel.3d.cloud = "panel.3dscatter",
            panel.3d.wireframe = "panel.3dwire",
            rot.mat, aspect,
            par.box = NULL,
            xlab, ylab, zlab,
            xlab.default, 
            ylab.default,
            zlab.default,
            scales.3d,
            proportion = 0.6, wireframe = FALSE,
            scpos,
            \dots,
            col.at,
            col.regions)
panel.wireframe(\dots)
panel.3dscatter(x, y, z, rot.mat, distance,
                groups, type = 'p',
                xlim.scaled, ylim.scaled, zlim.scaled,
                zero.scaled,
                col, col.point, col.line,
                lty, lwd, cex, pch,
                cross, \dots, subscripts)
panel.3dwire(x, y, z, rot.mat, distance,
             col.at, col.regions,
             shade = FALSE,
             shade.colors = trellis.par.get("shade.colors")$palette,
             light.source = c(0, 0, 1000),
             xlim.scaled,
             ylim.scaled,
             zlim.scaled,
             col,
             col.groups = superpose.line$col,
             polynum = 100,
             \dots)
}
\arguments{
  \item{x, y, z}{
    numeric (or possibly factors) vectors representing the data to be
    displayed. The interpretation depends on the context. For
    \code{panel.cloud} these are essentially the same as the data passed
    to the high level plot (except if \code{formula} was a matrix, the
    appropriate \code{x, y} vectors are generated). By the time they are
    passed to \code{panel.3dscatter} and \code{panel.3dwire}, they have
    been scaled (to lie inside a bounding box, usually the [-0.5, 0.5]
    cube).

    Further, for \code{panel.3dwire}, \code{x} and \code{y} are shorter
    than \code{z} and represent the sorted locations defining a
    rectangular grid. Also in this case, \code{z} may be a matrix if the
    display is grouped, with each column representing one surface.

    In \code{panel.cloud} (called from \code{wireframe}) and
    \code{panel.3dwire}, \code{x}, \code{y} and \code{z} could also be
    matrices (of the same dimension) when they represent a 3-D surface
    parametrized on a 2-D grid.
  }
  \item{subscripts}{
    index specifying which points to draw. The same \code{x, y, z} values
    (representing the whole data) are passed to \code{panel.cloud} for
    each panel. \code{subscripts} specifies the subset of rows to be
    used for the particular panel.
  }
  \item{distance, par.box}{ described in \code{\link{cloud}} }
  \item{xlim, ylim, zlim}{
    limits for the respective axes. As with other lattice functions,
    these could each be a numeric 2-vector or a character vector
    indicating levels of a factor.
  }
  \item{panel.3d.cloud, panel.3d.wireframe}{
    functions that draw the data-driven part of the plot (as opposed to
    the bounding box and scales) in \code{cloud} and
    \code{wireframe}. This function is called after the `back' of the
    bounding box is drawn, but before the `front' is drawn.

    Any user-defined custom display would probably want to change these
    functions. The intention is to pass as much information to this
    function as might be useful (not all of which are used by the
    defaults). In particular, these functions can expect arguments
    called \code{xlim}, \code{ylim}, \code{zlim} which give the bounding
    box ranges in the original data scale and \code{xlim.scaled},
    \code{ylim.scaled}, \code{zlim.scaled} which give the bounding box
    ranges in the transformed scale. More arguments can be considered on
    request.
  }
  \item{aspect}{ aspect as in \code{cloud} }
  \item{xlab, ylab, zlab}{
    Labels, have to be lists. Typically the user will not manipulate
    these, but instead control this via arguments to \code{cloud}
    directly.
  }
  \item{xlab.default}{ for internal use  }
  \item{ylab.default}{ for internal use  }
  \item{zlab.default}{ for internal use  }
  \item{scales.3d}{ list defining the scales }
  \item{proportion}{
    numeric scalar, gives the length of arrows as a proportion of the
    sides
  }
  \item{scpos}{
    A list with three components x, y and z (each a scalar integer),
    describing which of the 12 sides of the cube the scales should be
    drawn. The defaults should be OK. Valid values are x: 1, 3, 9, 11;
    y: 8, 5, 7, 6 and z: 4, 2, 10, 12. (See comments in the source code
    of \code{panel.cloud} to see the details of this enumeration.)
  }
  \item{wireframe}{logical, indicating whether this is a wireframe plot}
  \item{col.at, col.regions}{
    deals with specification of colors when \code{drape = TRUE} in
    \code{\link{wireframe}}
  }
  \item{rot.mat}{
    4x4 transformation matrix in homogeneous coordinates. This gives the
    rotation matrix combining the \code{screen} and \code{R.mat}
    arguments to \code{\link{cloud}}
  }
  \item{type}{
    character vector, specifying type of cloud plot. Can include one or
    more of ``p'', ``l'', ``h'' or ``b''. ``p'' and ``l'' mean `points'
    and `lines' respectively, and ``b'' means `both'. ``h'' stands for
    `histogram', and causes a line to be drawn from each point to
    the X-Y plane (i.e., the plane representing \code{z = 0}), or the
    lower (or upper) bounding box face, whichever is closer.
  }
  \item{xlim.scaled, ylim.scaled, zlim.scaled}{
     axis limits (after being scaled to the bounding box) 
  }
  \item{zero.scaled}{
    z-axis location (after being scaled to the bounding box) of the X-Y
    plane in the original data scale, to which lines will be dropped (if
    within range) from each point when \code{type = 'h'}
  }
  \item{cross}{
    logical, defaults to \code{TRUE} if \code{pch='+'}.
    \code{panel.3dscatter} can represent each point by a 3d `cross' of
    sorts (it's much easier to understand looking at an example than
    from a description). This is different from the usual \code{pch}
    argument, and reflects the depth of the points and the orientation
    of the axes. This argument indicates whether this feature will be
    used.

    This is useful for two reasons. It can be set to \code{FALSE} to use
    `+' as the plotting character in the regular sense. It can also be
    used to force this feature in grouped displays.
  }
  \item{groups}{
    specification of a grouping variable, passed down from the high
    level functions.
  }
  \item{shade}{
    logical, indicating whether the surface is to be colored using an
    illumination model with a single light source
  }
  \item{shade.colors}{
    a function (or the name of one) that is supposed to calculate the
    color of a facet when shading is being used. Three pieces of
    information is available to the function: first, the cosine of the
    angle between the incident light ray and the normal to the surface
    (representing foreshortening); second, the cosine of half the angle
    between the reflected ray and the viewing direction (useful for
    non-lambertian surfaces); and third, the scaled (average) height of
    that particular facet with respect to the total plot z-axis limits.

    All three numbers should be between 0 and 1. The \code{shade.colors}
    function should return a valid color. The default function is
    obtained from the Lattice settings.
  }
  \item{light.source}{
    a 3-vector representing (in cartesian coordinates) the light
    source. This is relative to the viewing point being (0, 0,
    1/distance) (along the positive z-axis), keeping in mind that all
    observations are bounded within the [-0.5, 0.5] cube
  }
  \item{polynum}{
    quadrilateral faces are drawn in batches of \code{polynum} at a
    time. Drawing too few at a time increases the total number of calls
    to the underlying \code{grid.polygon} function, which affects
    speed. Trying to draw too many at once may be unnecessarily memory
    intensive. This argument controls the trade-off.
  }
  \item{col.groups}{colors for different groups}
  \item{col, col.point, col.line, lty, lwd, cex, pch}{graphical parameters}
  \item{\dots}{ other parameters, passed down when appropriate}
}
\details{
  These functions together are responsible for the content drawn inside
  each panel in \code{cloud} and
  \code{wireframe}. \code{panel.wireframe} is a wrapper to
  \code{panel.cloud}, which does the actual work.

  \code{panel.cloud} is responsible for drawing the content that does
  not depend on the data, namely, the bounding box, the arrows/scales,
  etc. At some point, depending on whether \code{wireframe} is TRUE, it
  calls either \code{panel.3d.wireframe} or \code{panel.3d.cloud}, which
  draws the data-driven part of the plot.

  The arguments accepted by these two functions are different, since
  they have essentially different purposes. For cloud, the data is
  unstructured, and \code{x, y, z} are all passed to the
  \code{panel.3d.cloud} function. For wireframe, on the other hand,
  \code{x} and \code{y} are increasing vectors with unique values,
  defining a rectangular grid. \code{z} must be a matrix with
  \code{length(x) * length(y)} rows, and as many columns as the number
  of groups.

  \code{panel.3dscatter} is the default \code{panel.3d.cloud} function.
  It has a \code{type} argument similar to \code{\link{panel.xyplot}},
  and supports grouped displays. It tries to honour depth ordering,
  i.e., points and lines closer to the camera are drawn later,
  overplotting more distant ones. (Of course there is no absolute
  ordering for line segments, so an ad hoc ordering is used. There is no
  hidden point removal.)

  \code{panel.3dwire} is the default \code{panel.3d.wireframe}
  function. It calculates polygons corresponding to the facets one by
  one, but waits till it has collected information about \code{polynum}
  facets, and draws them all at once. This avoids the overhead of
  drawing \code{grid.polygon} repeatedly, speeding up the rendering
  considerably. If \code{shade = TRUE}, these attempt to color the
  surface as being illuminated from a light source at
  \code{light.source}. \code{palette.shade} is a simple function that
  provides the deafult shading colors

  Multiple surfaces are drawn if \code{groups} is non-null in the call
  to \code{wireframe}, however, the algorithm is not sophisticated
  enough to render intersecting surfaces correctly.

}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{cloud}}}
\keyword{dplot}

\eof
\name{panel.densityplot}
\alias{panel.densityplot}
\title{ Default Panel Function for densityplot }
\description{
  This is the default panel function for \code{densityplot}.
}
\usage{
panel.densityplot(x, darg, plot.points = TRUE, ref = FALSE,
                  col, col.line, ...)
}
\arguments{
  \item{x}{ data points for which density is to be estimated}
  \item{darg}{list of arguments to be passed to the \code{density}
    function. Typically, this should be a list with zero or more of the
    following components : \code{bw, adjust, kernel, window, width,
    give.Rkern, n, from, to, cut, na.rm} (see \code{density} for
    details)
} 
  \item{plot.points}{ logical, specifying whether or not the data 
    points should be plotted along with the estimated density.
  }
  \item{ref}{ logical, whether to draw x-axis }
  \item{col}{ color for points and line}
  \item{col.line}{ color for line, overrides col }
  \item{\dots}{ extra grahical parameters }
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{densityplot}}}
\keyword{dplot}

















\eof
\name{panel.dotplot}
\alias{panel.dotplot}
\title{ Default Panel Function for dotplot}
\description{
  Default panel function for \code{dotplot}.
}
\usage{
panel.dotplot(x, y, horizontal = TRUE,
              pch, col, lty, lwd, col.line,
              levels.fos = NULL, groups = NULL,
              \dots)
}
\arguments{
  \item{x,y}{ variables to be plotted in the panel. Typically y is the
    'factor' }
  \item{horizontal}{ logical. If FALSE, the plot is `transposed' in the
    sense that the behaviours of x and y are switched. x is now
    the `factor'.  Interpretation of other arguments change
    accordingly. See documentation of \code{bwplot} for a fuller
    explanation.
  }
  \item{pch, col, lty, lwd, col.line}{ graphical parameters}
  \item{levels.fos}{ number of levels of \code{y}}
  \item{groups}{ grouping variable (affects graphical parameters)}
  \item{\dots}{ extra parameters, ignored}
}
\details{ Creates (possibly grouped) Dotplot of \code{x} against
  \code{y}
}
\seealso{\code{\link{dotplot}}}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{dplot}

















\eof
\name{panel.functions}
\alias{panel.functions}
\alias{panel.abline}
\alias{panel.curve}
\alias{panel.rug}
\alias{panel.fill}
\alias{panel.grid}
\alias{panel.lmline}
\alias{panel.loess}
\alias{panel.linejoin}
\alias{panel.axis}
\alias{panel.mathdensity}
\title{Useful Panel Functions}
\description{
  These are predefined panel functions available in lattice for use in
  constructing new panel functions (usually on-the-fly).
}
\synopsis{
panel.abline(a, b = NULL, h = numeric(0), v = numeric(0),
    col,
    col.line = add.line$col,
    lty = add.line$lty,
    lwd = add.line$lwd, 
    ...)
panel.curve(expr, from, to, n = 101,
            curve.type = "l",
            col = add.line$col,
            lty = add.line$lty,
            lwd = add.line$lwd,
            type = NULL,
            ...)
panel.rug(x = NULL, y = NULL,
          regular = TRUE, 
          start = if (regular) 0 else 0.97,
          end = if (regular) 0.03 else 1,
          col, ...)
panel.linejoin(x, y, fun = mean, horizontal = TRUE, lwd = reference.line$lwd, 
    lty = reference.line$lty, col, col.line = reference.line$col, 
    ...)
panel.axis(side = 1:4,
             at,
             labels = TRUE,
             tick = TRUE,
             half = TRUE,
             which.half = c("lower", "upper", "upper","lower"),

             tck = as.numeric(tick),
             rot = if (is.logical(labels)) 0 else c(90, 0),

             text.col,
             text.cex,

             text.font,
             text.fontfamily,
             text.fontface,

             line.col,
             line.lty,
             line.lwd)
}
\usage{
panel.abline(a, b, ...)
panel.abline(coef, ...)
panel.abline(reg, ...)
panel.abline(h= ,v= , ...)
panel.curve(expr, from, to, n = 101,
            curve.type = "l", ...)
panel.rug(x = NULL, y = NULL,
          regular = TRUE, start, end, \dots)
panel.fill(col="grey", ...)
panel.grid(h=3, v=3, ...)
panel.lmline(x,y,...) = panel.abline(lm(y~x),...)
panel.loess(x, y, span = 2/3, degree = 1,
            family = c("symmetric", "gaussian"),
            evaluation = 50, \dots)
panel.linejoin(x, y, fun = mean, horizontal = TRUE, ...)
panel.mathdensity(dmath = dnorm, args = list(mean=0, sd=1))
panel.axis(side = 1:4, at,
           labels = TRUE, tick = TRUE,
           half = TRUE, which.half = c("lower", "upper", "upper","lower"),
           tck = 1, rot)
}
\arguments{
  \item{x, y}{ variables defining the contents of the panel}
  \item{a, b}{ Coefficients of the line to be added}
  \item{coef}{ Coefficients of the line to be added as a length 2 vector}
  \item{reg}{ A regression object. The corresponding fitted line will
    be drawn }
  \item{h,v}{ For panel.abline, numerical vectors giving y and x
    locations respectively of horizontal and vertical lines to be
    added to the plot. For panel.grid, number of horizontal and vertical
    reference lines to be added to the plot; h=-1 and v=-1 make the
    grids aligned with the axis labels (this doesn't always work).
  }
  \item{expr}{ expression as a function of x or a function to plot as a
    curve
  }
  \item{regular}{ logical indicating whether `rug' to be drawn on the
    regular side (left / bottom) or not (right / top)
  }
  \item{start, end}{ endpoints of rug segments, in normalized parent
    coordinates (between 0 and 1). Defaults depend on value of regular,
    and cover 3\% of the panel width and height
  }
  \item{from, to}{ optional lower and upper x-limits of curve. If
    missing, limits of current panel are used
  }
  \item{n}{ number of points to use for drawing the curve }
  \item{curve.type}{ type of curve ('p' for points, etc), passed to
    \code{llines}
  }
  \item{col}{color}
  \item{span, degree, family, evaluation}{ arguments to 
    \code{loess.smooth} in the \code{modreg} library. \code{panel.loess}
    is essentially a wrapper on \code{loess.smooth} 
  }
  \item{fun}{ the function that will be applied to the subset of x(y)
    determined by the unique values of y(x)}
  \item{horizontal}{ logical. If FALSE, the plot is `transposed' in the
    sense that the behaviours of x and y are switched. x is now
    the `factor'.  Interpretation of other arguments change
    accordingly. See documentation of \code{bwplot} for a fuller
    explanation.
  }
  \item{dmath}{ A vectorized function that produces density values
    given a numeric vector named \code{x}, e.g., \code{dnorm}}
  \item{args}{ list giving additional arguments to be passed to dmath}

  \item{side}{ subset of 1:4, indicating which side which }
  \item{at}{ location of labels. No default.}
  \item{labels}{ logical indicating whether labels are to be drawn, or
    the labels to go along with \code{at}}
  \item{tick}{ logical, whether to draw ticks }
  \item{half}{
    logical, whether only half of scales will be drawn for each side
  }
  \item{which.half}{
    character vector of length 4 (recycled if necessary), each being
    either ``lower'' or ``upper''. Indicates which half is to be used
    for tick locations if \code{half = TRUE}, for sides 1 to 4
  }
  \item{tck}{ numeric scalar, tick length }
  \item{rot}{
    numeric 4-vector (recycled if necessary), rotation angles for sides
    1 to 4
  }
  \item{\dots}{ graphical parameters can be supplied. see function
    definition for details. Color can usually be specified by \code{col,
      col.line} and \code{col.symbol}, the last two overriding the first
    for lines and points respectively.
  }
}
\details{
  \code{panel.abline} adds a line of the form \code{y=a+bx} or vertical
  and/or horizontal lines. Graphical parameters are obtained from 
  \code{reference.line} for panel.grid, and \code{add.line} for the
  others (can be set using \code{trellis.par.set} )

  \code{panel.curve} adds a curve, similar to what \code{curve} does
  with \code{add = TRUE}. Graphical parameters for the line are obtained
  from the \code{add.line} setting.

  \code{panel.linejoin} treats one of x and y as a factor (according to
  the value of \code{horizontal}, calculates \code{fun} applied to the
  subsets of the other variable determined by each unique value of the
  factor, and joins them by a line. Can be used in conjunction with
  \code{panel.xyplot} and more commonly with panel.superpose to produce
  interaction plots. See \code{\link{xyplot}} documentation for an example.

  \code{panel.mathdensity} plots a (usually theoretical) probability
  density function. Can be useful in conjunction with \code{histogram}
  and \code{densityplot} to visually estimate goodness of fit.

  \code{panel.axis} draws axis tick marks INSIDE a panel. It honours the
  (native) axis scales within the panel, but the locations need to be
  explicitly specified. Used in \code{\link{panel.pairs}} for
  \code{\link{splom}}.
}
\seealso{\code{\link[stats:scatter.smooth]{loess.smooth}},
  \code{\link{trellis.par.get}}}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{dplot}





\eof
\name{panel.histogram}
\alias{panel.histogram}
\title{ Default Panel Function for histogram }
\description{
  This is the default panel function for \code{histogram}.
}
\usage{
panel.histogram(x, breaks, equal.widths, type, col, ...)
}
\arguments{
  \item{x}{ The data points for which the histogram is to be drawn} 
  \item{breaks}{ The breakpoints for the histogram}
  \item{equal.widths}{ logical used when \code{breaks==NULL} }
  \item{type}{ Type of histogram, possible values being ``percent'',
    ``density'' and ``count''}
  \item{col}{ Color of bars (NULL for empty), by default
    \code{bar.fill$col}.}
  \item{\dots}{ other arguments.}
}

\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{histogram}}}
\keyword{dplot}

















\eof
\name{panel.levelplot}
\alias{panel.levelplot}
\title{ Default Panel Function for levelplot }
\description{
  This is the default panel function for \code{levelplot}.
}
\usage{
panel.levelplot(x, y, z, zcol,
                subscripts,
                at = mean(z),
                shrink,
                labels = NULL,
                label.style = c("mixed", "flat", "align"),
                contour = TRUE,
                region = TRUE,
                col = add.line$col,
                lty = add.line$lty,
                lwd = add.line$lwd,
                cex = add.text$cex,
                font = add.text$font,
                fontfamily = add.text$fontfamily,
                fontface = add.text$fontface,
                col.text = add.text$col,
                \dots,
                col.regions)
}
\arguments{
  \item{x, y, z}{ variables defining the plot }
  \item{zcol}{
    color numbers for each x,y combination, used as indices for the
    color vector \code{col.regions}
  }
  \item{subscripts}{ which points of \code{x,y,z} to draw }
  \item{at}{ numeric vector specifying cut points }
  \item{shrink}{
    either a numeric vector of length 2 (meant to work as both x and y
    components), or a list with components x and y which are numeric
    vectors of length 2.  This allows the rectangles to be scaled
    proportional to the z-value. The specification can be made
    separately for widths (x) and heights (y).  The elements of the
    length 2 numeric vector gives the minimum and maximum proportion of
    shrinkage (corresponding to min and max of z).
  }
  \item{labels}{contour labels}
  \item{label.style}{
    controls choice of how label positions are determined.
  }
  \item{contour}{
    logical, specifying whether contour lines should be
    drawn
  }
  \item{region}{
    logical, specifying whether inter-contour regions should be filled
    with the appropriate color
  }
  \item{col, lty, lwd}{ graphical parameters for contour lines}
  \item{cex, col.text, font, fontfamily, fontface}{ graphical parameters for contour labels}
  \item{\dots}{ extra parameters }
  \item{col.regions}{ colors indexed by zcol, used if \code{region=TRUE}
  }
}

\details{

  The same function is used for both \code{levelplot} and
  \code{contourplot} (which differ only in default values of some
  arguments). 

  When \code{contour=TRUE}, the \code{contourLines} function is used to
  calculate the contour lines.
} 

\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{levelplot}}, \code{\link{contourLines}} }
\keyword{dplot}

















\eof
\name{panel.pairs}
\alias{panel.pairs}
\alias{diag.panel.splom}
\title{ Default Superpanel Function for splom}
\description{
  This is the default superpanel function for \code{splom}.
}
\synopsis{
panel.pairs(z,
             panel = "panel.splom",
             lower.panel = panel,
             upper.panel = panel,
             diag.panel = "diag.panel.splom",
             as.matrix = FALSE,

             groups = NULL,
             panel.subscripts,
             subscripts,
             pscales = 5,
             panel.number = 0,
             panel.counter = 0,

             prepanel.limits = function(x) if (is.factor(x)) levels(x) else
             extend.limits(range(as.numeric(x), na.rm = TRUE)),


             varname.col = add.text$col,
             varname.cex = add.text$cex,
             varname.font = add.text$font,
             varname.fontfamily = add.text$fontfamily,
             varname.fontface = add.text$fontface,

             axis.text.col = axis.text$col,
             axis.text.cex = axis.text$cex,
             axis.text.font = axis.text$font,
             axis.text.fontfamily = axis.text$fontfamily,
             axis.text.fontface = axis.text$fontface,

             axis.line.col = axis.line$col,
             axis.line.lty = axis.line$lty,
             axis.line.lwd = axis.line$lwd,

             ...)
diag.panel.splom(varname = NULL, limits, at = NULL, lab = NULL,
             draw = TRUE,

             varname.col = add.text$col,
             varname.cex = add.text$cex,
             varname.font = add.text$font,
             varname.fontfamily = add.text$fontfamily,
             varname.fontface = add.text$fontface,

             axis.text.col = axis.text$col,
             axis.text.cex = axis.text$cex,
             axis.text.font = axis.text$font,
             axis.text.fontfamily = axis.text$fontfamily,
             axis.text.fontface = axis.text$fontface,

             axis.line.col = axis.line$col,
             axis.line.lty = axis.line$lty,
             axis.line.lwd = axis.line$lwd,
             ...)
}
\usage{
panel.pairs(z,
            panel = "panel.splom",
            lower.panel = panel,
            upper.panel = panel,
            diag.panel = "diag.panel.splom",
            as.matrix = FALSE,
            groups = NULL,
            panel.subscripts,
            subscripts,
            pscales = 5,
            prepanel.limits = function(x) if (is.factor(x)) levels(x) else
            extend.limits(range(as.numeric(x), na.rm = TRUE)),
            ...)
diag.panel.splom(varname, limits, at, lab, draw = TRUE, ...)
}
\arguments{
  \item{z}{ The data frame used for the plot.}
  \item{panel, lower.panel, upper.panel}{
    The panel function used to display each pair of variables. If
    specified, \code{lower.panel} and \code{upper.panel} are used for
    panels below and above the diagonal respectively.
  }
  \item{diag.panel}{
    The panel function used for the diagonals. See arguments to
    \code{diag.panel.splom} to know what arguments this function is
    passed when called.
  }
  \item{as.matrix}{
    logical. If \code{TRUE}, the layout of the panels will have origin
    on the top left instead of bottom left (similar to
    \code{pairs}). This is in essence the same functionality as provided
    by \code{as.table} for the panel layout
  }
  \item{groups}{ Grouping variable, if any }
  \item{panel.subscripts}{
    logical specifying whether the panel function accepts an argument
    named \code{subscripts}.
  }
  \item{subscripts}{ The indices of the rows of \code{z} that are to be
    displayed in this (super)panel.
  }
  \item{pscales}{ Controls axis labels, passed down from
    \code{splom}. If pscales is a single number, it indicates the
    approximate number of equally-spaced ticks that should appear on
    each axis. If pscales is a list, it should have p components, each
    of which is itself a list with the following valid components:

    at: a numeric vector specifying tick locations

    labels: character vector labels to go with at

    limits: numeric 2-vector specifying axis limits (should be made more
    flexible at some point to handle factors)

    These are specifications on a per-variable basis, and used on all
    four sides in  the diagonal cells used for labelling.

    Factor variables are labelled with the factor names. Use pscales=0
    to supress the axes entirely.
  }
  \item{prepanel.limits}{ The `regular' high level lattice plots such as
    \code{xyplot} use the \code{prepanel} function for deciding on axis
    limits from data. This function serves a similar function, and works
    on a per-variable basis, by calculating the limits, which can be
    overridden by the corresponding \code{limits} component in the
    \code{pscales} list.
  }
  \item{varname}{ (scalar) character string or expression that is to be
    written centred within the panel
  }
  \item{limits}{
    numeric of length 2, or, vector of characters, specifying the scale
    for that panel (used to calculate tick locations when missing)
  }
  \item{at}{ locations of tick marks}
  \item{lab}{ optional labels for tick marks}
  \item{draw}{
    logical, specifying whether to draw the tick marks and labels. If
    \code{FALSE}, only variable names are written
  }
  \item{\dots}{
    extra arguments, typically graphical parameters to control
    appearance of axes and variable names in the diagonal panels. The
    actual list of possible arguments is rather large, see the formal
    arguments to \code{panel.pairs} and \code{panel.diag.splom} (using
    \code{formals}) for all the possibilities.
  }
}
\details{
  This is the function that is actually passed in as the \code{panel}
  function in a trellis object produced by splom (taking the `panel'
  function as its argument).
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{splom}}}
\keyword{dplot}

















\eof
\name{panel.parallel}
\alias{panel.parallel}
\title{ Default Panel Function for parallel}
\description{
  This is the default panel function for \code{parallel}.
}
\usage{
panel.parallel(z, subscripts, col, lwd, lty, ...)
}
\arguments{
  \item{z}{ The data frame used for the plot.}
  \item{subscripts}{ The indices of the rows of \code{z} that are to be
    displyed in this panel.}
  \item{col, lwd, lty}{graphical parameters (chosen by default as the settings
    for \code{superpose.line}).}
  \item{\dots}{other arguments}
}
\details{difficult to describe. See example for \code{parallel}}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{parallel}}}
\keyword{dplot}

















\eof
\name{panel.qqmathline}
\alias{panel.qqmathline}
\title{Useful panel function with qqmath}
\description{
  Useful panel function with qqmath. Draws a line passing through the
  points determined by the .25 and .75 quantiles of the sample and the
  theoretical distribution.
}
\usage{
panel.qqmathline(y, distribution, ...)
}
\arguments{
  \item{y}{ The original sample, possibly reduced to a fewer number of
    quantiles, as determined by the \code{f.value} argument to
    \code{qqmath}
  }
  \item{distribution}{quantile function for reference theoretical
    distribution.
  }
  \item{\dots}{other arguments.}
}
\author{Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{prepanel.qqmathline}}, \code{\link{qqmath}}}
\keyword{dplot}

\eof
\name{panel.stripplot}
\alias{panel.stripplot}
\title{ Default Panel Function for stripplot }
\description{
  This is the default panel function for \code{stripplot}. Also see
  \code{panel.superpose}
}
\usage{
panel.stripplot(x, y, jitter.data = FALSE, factor = 0.5,
                horizontal = TRUE, groups = NULL,
                \dots)
}
\arguments{
  \item{x,y}{ coordinates of points to be plotted}
  \item{jitter.data}{ whether points should be jittered to avoid
    overplotting.} 
  \item{factor}{factor by which to jitter}
  \item{horizontal}{ logical. If FALSE, the plot is `transposed' in the
    sense that the behaviours of x and y are switched. x is now
    the `factor'.  Interpretation of other arguments change
    accordingly. See documentation of \code{bwplot} for a fuller
    explanation.
  }
  \item{groups}{ optional grouping variable }
  \item{\dots}{ graphical parameters etc can be supplied, passed to
    \code{panel.xyplot} or \code{panel.superpose}, depending on whether
    \code{groups} is present}
}
\details{ Creates stripplot (one dimensional scatterplot) of \code{x}
  for each level of \code{y}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{stripplot}}}
\keyword{dplot}

















\eof
\name{panel.superpose}
\alias{panel.superpose}
\alias{panel.superpose.2}
\title{ Panel Function for Display Marked by groups }
\description{
  This is an useful panel function for trellis displays. Uses a grouping
  variable, plotting the \code{x} (and \code{y} where appropriate)
  variables with different graphical properties for each distinct value
  of the grouping variable. 
}
\synopsis{
panel.superpose(x, y = NULL, subscripts, groups,
             panel.groups = "panel.xyplot",
             col,
             col.line = superpose.line$col,
             col.symbol = superpose.symbol$col,
             pch = superpose.symbol$pch,
             cex = superpose.symbol$cex, 
             font = superpose.symbol$font,
             fontface = superpose.symbol$fontface,
             fontfamily = superpose.symbol$fontfamily,
             lty = superpose.line$lty,
             lwd = superpose.line$lwd,
             ...)
panel.superpose.2(x, y, subscripts, groups, col, col.line = superpose.line$col,
                  col.symbol = superpose.symbol$col, pch = superpose.symbol$pch,
                  cex = superpose.symbol$cex, lty = superpose.line$lty,
                  lwd = superpose.line$lwd, type="p", ...)
}
\usage{
panel.superpose(x, y, panel.groups = "panel.xyplot", subscripts, groups, type="p", ...)
panel.superpose.2(x, y, subscripts, groups, type="p", ...)
}
\arguments{
  \item{x,y}{ coordinates of the points to be displayed}
  \item{panel.groups}{ the panel function to be used for each group of
    points. Defaults to \code{panel.xyplot} (behaviour in S)}
  \item{subscripts}{ subscripts giving indices in original data frame}
  \item{groups}{ a grouping variable. Different graphical parameters
    will be used to plot the subsets of observations given by each
    distinct value of \code{groups}. The default graphical parameters
    are obtained from \code{superpose.symbol} and \code{superpose.line}
    using \code{\link{trellis.par.get}} wherever appropriate
  }
  \item{type}{ character vector specifying what should be drawn for each
    group. See \code{\link{panel.xyplot}} for details of what the values
    can be. This is meaningful for \code{panel.superpose} only when
    \code{panel.groups} is \code{panel.xyplot}.

    The 2 functions \code{panel.superpose} (with \code{panel.xyplot})
    and \code{panel.superpose.2} differ only in the way the \code{type}
    argument is interpreted. For the former, the interpretation is the
    same as for \code{panel.xyplot} for each of the unique groups. For
    the latter, the \code{type} vector is replicated to be as long as
    the number of unique values in \code{groups}, and the points
    corresponding to the different groups are plotted according to the
    corresponding type. 

    See \code{xyplot} documentation for an example.
  }
  \item{\dots}{ other arguments (mostly graphical parameters, replicated
    to be as long as the number of groups), ultimately passed down to
    \code{panel.xyplot}. In particular, colour can be specified using 3
    arguments (\code{col, col.line, col.symbol}), as in
    \code{panel.xyplot}.
  }
}
\seealso{
  Different functions when used as \code{panel.groups} gives different
  types of plots, for example \code{\link{panel.xyplot}},
  \code{\link{panel.dotplot}} and \code{\link{panel.linejoin}} (This can
  be used to produce interaction plots).
}

\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}
  (\code{panel.superpose.2} contributed by Neil Klepeis)
}
\keyword{dplot}

\eof
\name{panel.tmd}
\alias{panel.tmd}
\title{ Panel function for tmd}
\description{
  This is the default panel function for tmd. Essentially panel.xyplot,
  with a horizontal line at y = 0.
}
\usage{
panel.tmd(...)
}
\arguments{
  \item{\dots}{ arguments passed to \code{panel.xyplot}}
}
\author{Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{tmd}} ,\code{\link{Lattice}}}
\keyword{dplot }


\eof
\name{panel.xyplot}
\alias{panel.xyplot}
\alias{panel.splom}
\alias{panel.qqmath}
\alias{panel.qq}
\title{ Default Panel Function for xyplot }
\description{
  This is the default panel function for \code{xyplot}. Also see
  \code{panel.superpose}. The default panel functions for
  \code{splom} and \code{qqmath} are essentially the same function.
}
\usage{
panel.xyplot(x, y, type="p",
             pch = plot.symbol$pch,
             col,
             col.line = plot.line$col,
             col.symbol = plot.symbol$col,
             font = plot.symbol$font,
             fontfamily = plot.symbol$fontfamily,
             fontface = plot.symbol$fontface,
             lty = plot.line$lty,
             cex = plot.symbol$cex,
             lwd = plot.line$lwd, ...)
panel.splom(...)
panel.qqmath(...)
panel.qq(...)
}
\arguments{
  \item{x,y}{ variables to be plotted in the scatterplot}
  \item{type}{ character vector consisting of one or more of the
    following: \code{"p", "l", "h", "b", "o", "s", "S", "r", "smooth"}. 

    The behaviour if any of the first six are included in \code{type} is
    similar to the effect of \code{type} in \code{\link{plot}}. 
    \code{"r"} adds a  regression line (same as
    \code{\link{panel.lmline}}, except for default graphical
    parameters), and \code{"smooth"} adds a lowess fit (same as
    \code{\link{panel.loess}}).

    See example in \code{xyplot} for an example.
  }
  \item{col, col.line, col.symbol}{
    default colours are obtained from \code{plot.symbol} and
    \code{plot.line} using \code{\link{trellis.par.get}}.
  }
  \item{font, fontface, fontfamily}{
    font used when \code{pch} is a character
  }
  \item{pch, lty, cex, lwd}{
    other graphical parameters.
  }
  \item{\dots}{ extra arguments, if any, for \code{panel.xyplot}. In
    most cases \code{panel.xyplot} ignores these. For types "r" and
    "smooth", these are passed on to \code{panel.lmline} and
    \code{panel.loess} respectively.
  }
}
\details{ 

  Creates scatterplot of \code{x} and \code{y}, with various
  modifications possible via the type argument. \code{panel.qq} draws a
  45 degree line before calling \code{panel.xyplot}.

  Note that most of the arguments controlling the display can be
  supplied directly to the high-level \code{xyplot} call.

}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}

\seealso{
  \code{\link{panel.superpose}},
  \code{\link{xyplot}},
  \code{\link{splom}},\code{\link{qqmath}}
}
\keyword{dplot}

\eof
\name{prepanel.functions}
\alias{prepanel.lmline}
\alias{prepanel.loess}
\alias{prepanel.qqmathline}
\title{ Useful Prepanel Function for Lattice}
\description{
  These are predefined prepanel functions available in Lattice.
}
\synopsis{
prepanel.loess(x, y, span, degree, family, evaluation, ...)
}
\usage{
prepanel.lmline(x, y, ...)
prepanel.loess(x, y, ...)
prepanel.qqmathline(y, distribution, f.value = ppoints, ...) 
}
\arguments{
  \item{x, y}{x and y values, numeric or factor}
  \item{distribution}{ quantile function for theoretical
    distribution. This is automatically passed in when this is used as a
    prepanel function in \code{qqmath}.
  }
  \item{f.value}{ as used in \code{qqmath}}
  \item{\dots}{other arguments}
}
\value{
  usually a list with components \code{xlim, ylim, dx, dy}, the first
  two being used to calculate panel axes limits, the last two for
  banking computations. The form of these components are described under
  \code{\link{xyplot}}. There are also several undocumented prepanel
  functions that serve as the default for high level functions, e.g.,
  \code{prepanel.default.xyplot}
}

\seealso{\code{\link{trellis.par.get}}, \code{\link{xyplot}},
  \code{\link{Lattice}}. See \code{\link[modreg]{loess.smooth}} for
  further options to \code{prepanel.loess}
}
\author{ Deepayan Sarkar\email{deepayan@stat.wisc.edu}}
\keyword{dplot}


\eof
\name{print.trellis}
\title{Print Trellis Objects}
\usage{
\method{print}{trellis}(x, position, split,
     more = FALSE, newpage = TRUE,
     panel.height = list(1, "null"),
     panel.width = list(1, "null"),
     ...)
}
\description{
  Print (plot) a trellis object.
}
\alias{print.trellis}
\arguments{
  \item{x}{ The object to be plotted, of class ``trellis''}
  \item{position}{ a vector of 4 numbers, typically c(xmin, ymin, xmax,
    ymax) that give the lower-left and upper-right corners of a
    rectangle in which the Trellis plot of x is to be positioned. The
    coordinate system for this rectangle is [0-1] in both the x and y
    directions.
  }
  \item{split}{ a vector of 4 integers, c(x,y,nx,ny) , that says to
    position the current plot at the x,y position in a regular array of
    nx by ny plots. (Note: this has origin at top left)
  }
  \item{more}{ A logical specifying whether more plots will follow on
    this page.
  }
  \item{newpage}{ A logical specifying whether the plot should be on a
    new page. This option is specific to lattice, and is useful for
    including lattice plots in an arbitrary grid viewport (see the
    details section).
  }
  \item{panel.width, panel.height}{ lists with 2 components, that should
    be valid \code{x} and \code{units} arguments to \code{unit()} (the
    \code{data} argument cannot be specified currently, but can be
    considered for addition if needed). The resulting \code{unit} object
    will be the width/height of each panel in the Lattice plot. These
    arguments can be used to explicitly control the dimensions of the
    panel, rather than letting them expand to maximize available space.

    Note that this option should not be used in conjunction with
    non-default values of the \code{aspect} argument in the original
    high level call (no error will be produced, but the resulting
    behaviour is undefined).
  }
  \item{\dots}{extra arguments, ignored}
}
\details{
  This is the default print method for objects of class
  \code{"trellis"}, produced by calls to functions like \code{xyplot,
    bwplot} etc. It is usually called automatically when a trellis
  object is produced. It can also be called explicitly to control plot
  positioning by means of the arguments \code{split} and
  \code{position}.

  When \code{newpage = FALSE}, the current grid viewport is treated as
  the plotting area, making it possible to embed a Lattice plot inside
  an arbitrary grid viewport.

  The print method uses the information in \code{x} (the object to be
  printed) to produce a display using the Grid graphics engine. At the
  heart of the plot is a grid layout, of which the entries of most
  interest to the user are the ones containing the display panels.

  Unlike in older versions of Lattice (and Grid), the grid display tree
  is retained after the plot is produced, making it possible to access
  individual viewport locations and make additions to the plot. The full
  tree can be viewed using the grid function
  \code{\link[grid:current.viewport]{current.vpTree}}, and individual
  viewports can be `seeked' using
  \code{\link[grid:viewports]{seekViewport}}.  The viewports that
  correspond to panels (which is what the user would be interested in)
  have names of the form \code{panel.n}, where \code{n} is a simple
  incremental counter for the panels. These names can be used in
  \code{seekViewport}. For multi-page displays, only the viewports for
  the last page are available.  This functionality is still very new,
  and details may change in the future.

}

\note{
  Trying to position multipage displays will mess things up.
}

\seealso{
  \code{\link{Lattice}}, \code{\link[grid:unit]{unit}}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\examples{
data(singer)
p11 <- histogram( ~ height | voice.part, data = singer, xlab="Height")
p12 <- densityplot( ~ height | voice.part, data = singer, xlab = "Height")
p2 <- histogram( ~ height, data = singer, xlab = "Height")
data(sunspot)
p3 <- xyplot(sunspot~1:37, aspect="xy", type = "l")
## simple positioning by split
print(p11, split=c(1,1,1,2), more=TRUE)
print(p2, split=c(1,2,1,2))

## Combining split and position:
print(p11, position = c(0,0,.75,.75), split=c(1,1,1,2), more=TRUE)
print(p12, position = c(0,0,.75,.75), split=c(1,2,1,2), more=TRUE)
print(p3, position = c(.5,.75,1,1), more=FALSE)

## Using seekViewport
data(longley)

## repeat same plot, with different polynomial fits in each panel
xyplot(Armed.Forces ~ Year, longley, index.cond = list(rep(1, 6)),
       panel = function(x, y, panel.counter, ...)
       {
           panel.xyplot(x, y, ...)
           fm <- lm(y ~ poly(x, panel.counter))
           llines(x, predict(fm))
       })

\dontrun{
grid::seekViewport("panel.1")
cat("Click somewhere inside the first panel:\n")
ltext(grid::grid.locator(), lab = "linear")
}

grid::seekViewport("panel.1")
grid::grid.text("linear")

grid::seekViewport("panel.2")
grid::grid.text("quadratic")

grid::seekViewport("panel.3")
grid::grid.text("cubic")

grid::seekViewport("panel.4")
grid::grid.text("degree 4")

grid::seekViewport("panel.5")
grid::grid.text("degree 5")

grid::seekViewport("panel.6")
grid::grid.text("degree 6")
}
\keyword{hplot}




\eof
\name{qq}
\alias{qq}
\title{ Quantile-Quantile Plots of Two Samples }
\description{
  Quantile-Quantile plots for comparing two Distributions
}
\synopsis{
qq(formula, data = parent.frame(), aspect = "fill", 
    panel = panel.qq, prepanel = NULL, scales = list(), strip = TRUE, 
    groups = NULL, xlab, xlim, ylab, ylim, f.value = NULL, 
    drop.unused.levels,
    ..., subscripts = !is.null(groups), subset = TRUE)
}
\usage{
qq(formula, f.value = NULL, ...)
}
\arguments{
  \item{formula}{formula of the form \code{y ~ x | g1 * g2 * \ldots},
    where \code{x} must be a numeric, and \code{y} can be a factor,
    shingle, character or numeric vector, with the restriction that there
    must be exactly two levels of \code{y}, which divide the values of
    \code{x} into two groups. Quantiles for these groups will be plotted
    along the two axes.
  }
  \item{f.value}{ optional function of a single integer (representing
    the larger of the two sample sizes), returning a vector of
    probabilities corresponding to which quantiles should be
    plotted. The typical value for this argument is \code{ppoints},
    which is also the S-PLUS default. If specified, the probabilities
    generated by this function is used for the plotted quantiles, using
    the \code{quantile} function.

    \code{f.value} defaults to \code{NULL}, which is equivalent to
    using \code{function(n) ppoints(n, a = 1)}. This has the effect of
    including the minimum and maximum data values in the computed
    quantiles. This is similar to what happens for \code{qqplot} but
    different from the default \code{qq} behaviour in S-PLUS.
  }
  \item{\dots}{Other arguments}
}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\author{Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\details{
  \code{qq} produces a Q-Q plot of two samples.  The default behaviour
  of \code{qq} is different from the corresponding S-PLUS function. See
  the entry for \code{f.value} for specifics.

  This and all other high level Trellis functions have several
  arguments in common. These are extensively documented only in the
  help page for \code{xyplot}, which should be consulted to learn more
  detailed usage.
}

\seealso{\code{\link{xyplot}}, \code{\link{panel.qq}},
  \code{\link{qqmath}}, \code{\link{Lattice}}
}

\examples{
data(singer)
qq(voice.part ~ height, aspect = 1, data = singer,
   subset = (voice.part == "Bass 2" | voice.part == "Tenor 1"))
}
\keyword{dplot}





\eof
\name{qqmath}
\alias{qqmath}
\title{ Q-Q Plot with Theoretical Distribution }
\description{
  Quantile-Quantile plot of a sample and a theoretical distribution
}
\synopsis{
qqmath(formula,
             data = parent.frame(),
             allow.multiple = FALSE,
             outer = TRUE,
             auto.key = FALSE,
             aspect = "fill",
             panel = "panel.qqmath",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab,
             xlim,
             ylab,
             ylim,
             f.value = NULL,
             distribution = qnorm,
             drop.unused.levels,
             ...,
             subscripts = !is.null(groups),
             subset = TRUE)
}
\usage{
qqmath(formula, distribution = qnorm, f.value = NULL, \dots)
}
\arguments{
  \item{formula}{formula of the form \code{~ x | g1 * g2 * \ldots},
    where \code{x} must be a numeric.
  }
  \item{distribution}{ a quantile function that takes a vector of
    probabilities as argument and produces the corresponding
    quantiles. Possible values are \code{qnorm,qunif} etc.
    Distributions with other required arguments need to be passed in as
    user defined functions. This is also passed in as an argument to the
    panel function (though not used by the default panel function), 
    and the prepanel function if specified, for use in the bery commonly
    used \code{(pre)panel.qqmathline}.
  }
  \item{f.value}{ optional function of a single integer (representing
    sample size), returning a vector of probabilities corresponding to
    which quantiles should be plotted. The typical value for this
    argument is \code{ppoints}, which is also the S-PLUS default. If
    specified, the probabilities generated by this function is used for
    the plotted quantiles, using the \code{quantile} function for the
    sample, and the \code{distribution} argument for the theoretical
    distribution.

    \code{f.value} defaults to \code{NULL}, which has the effect of
    using \code{ppoints} for the quantiles of the theoretical
    distribution, but the exact data values for the sample. This is
    similar to what happens for \code{qqnorm}.

  }
  \item{\dots}{Other arguments}
}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\details{
  \code{qqmath} produces a Q-Q plot of the given sample and a
  theoretical distribution.  The default behaviour of \code{qqmath} is
  different from the corresponding S-PLUS function, but is similar to
  \code{qqnorm}. See the entry for \code{f.value} for specifics.

  This and all other high level Trellis functions have several
  arguments in common. These are extensively documented only in the
  help page for \code{xyplot}, which should be consulted to learn more
  detailed usage.
}
\author{Deepayan Sarkar \url{deepayan@stat.wisc.edu}}
\seealso{\code{\link{xyplot}}, \code{\link{panel.qqmath}},
  \code{\link{panel.qqmathline}}, \code{\link{prepanel.qqmathline}},
  \code{\link{Lattice}}, \code{\link{quantile}}
}
\examples{
qqmath(~ rnorm(100), distribution = function(p) qt(p, df = 10))
data(singer)
qqmath(~ height | voice.part, aspect = 1, data = singer,
       prepanel = prepanel.qqmathline,
       panel = function(x, y) {
          panel.qqmathline(y, distribution = qnorm)
          panel.qqmath(x, y)
       })
}
\keyword{dplot}


\eof
\name{rfs}
\alias{rfs}
\title{ Residual and Fit Spread Plots}
\description{
  Plots fitted values and residuals (via qqmath) on a common scale for
  any object that has methods for fitted values and residuals. 
}
\synopsis{
rfs(model, layout=c(2, 1), xlab="f-value", ylab=NULL,
    distribution = qunif,
    panel, prepanel, strip, ...)
}
\usage{
rfs(model, layout=c(2, 1), xlab="f-value", distribution = qunif,...)
}
\arguments{
  \item{model}{a fitted model object with methods \code{fitted.values}
    and \code{residuals}. Can be the value returned by \code{oneway} }
  \item{layout}{default layout is c(2,1)  }
  \item{xlab}{defaults to \code{"f.value"} }
  \item{distribution}{the distribution function to be used for \code{qqmath}}
  \item{\dots}{other arguments}
}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{oneway}}, \code{\link{qqmath}},
  \code{\link{xyplot}}, \code{\link{Lattice}}
}
\examples{
data(singer)
rfs(oneway(height ~ voice.part, data = singer, spread = 1), aspect = 1)
}
\keyword{dplot}

\eof
% $Id: shingles.Rd,v 1.14 2003/03/03 01:03:32 deepayan Exp $
\name{shingles}
\alias{equal.count}
\alias{plot.shingle}
\alias{print.shingle}
\alias{print.shingleLevel}
\alias{summary.shingle}
\alias{as.data.frame.shingle}
\alias{as.factorOrShingle}
\alias{as.shingle}
\alias{[.shingle}
\alias{is.shingle}
\alias{shingle}
\title{shingles}
\usage{
shingle(x, intervals=sort(unique(x)))
equal.count(x, \dots)
as.shingle(x)
is.shingle(x)
\method{plot}{shingle}(x, col, aspect, \dots)
\method{print}{shingle}(x, showValues = TRUE, \dots)
\method{print}{shingleLevel}(x, \dots)
\method{summary}{shingle}(object, \dots)
\method{as.data.frame}{shingle}(x, row.names = NULL, optional = FALSE)
x[subset, drop = FALSE]
as.factorOrShingle(x, subset, drop)
}
\description{
  Functions to handle shingles
}
\arguments{
  \item{x}{ numeric variable or R object, shingle in
    \code{plot.shingle, x[]}. An object (list of intervals) of class
    "shingleLevel" in \code{print.shingleLevel}}
  \item{object}{ shingle object to be summarized}
  \item{showValues}{ logical, whether to print the numeric part. If
    FALSE, only the intervals are printed}
  \item{row.names}{ a character vector giving the row names for the data frame}
  \item{optional}{ logical. If `TRUE', setting row names is optional}
  \item{intervals}{ numeric vector or matrix with 2 columns}
  \item{subset}{logical vector}
  \item{drop}{whether redundant shingle levels are to be dropped}
  \item{col}{ color to fill the rectangles, defaults to
    \code{bar.fill$col}} 
  \item{aspect}{aspect ratio}
  \item{\dots}{ other arguments, passed to \code{co.intervals}}
}
\details{
  A shingle is a data structure used in Trellis, and is meant to be a
  generalization of factors to `continuous' variables. It consists of a
  numeric vector along with some possibly overlapping intervals. These
  intervals are the `levels' of the shingle. The \code{levels} and
  \code{nlevels} functions, usually applicable to factors, are also
  applicable to shingles.

  There are print methods for shingles, as well as for printing the
  result of \code{levels()} applied to a shingle.

  The implementation of shingles is slightly different from S.

  \code{equal.count} converts \code{x} to a shingle. Essentially a
  wrapper around \code{co.intervals}. All arguments are passed to
  \code{co.intervals}

  \code{shingle} creates a shingle using the given \code{intervals}. If
  \code{intervels} is a vector, these are used to form 0 length
  intervals.

  \code{as.shingle} returns \code{shingle(x)} if \code{x} is not a
  shingle.

  \code{is.shingle} tests whether \code{x} is a shingle.

  \code{plot.shingle} displays the ranges of shingles via
  rectangles. \code{print.shingle} and \code{summary.shingle} describe
  the shingle object.
}
\value{
  \code{x$intervals} for \code{levels.shingle(x)}, 
  logical for \code{is.shingle}, an object of class ``trellis'' for
  \code{plot} (printed by default by \code{print.trellis}), and 
  an object of class ``shingle'' for the others.
}
\examples{
z <- equal.count(rnorm(50))
plot(z)
print(z)
print(levels(z))
\testonly{data.frame(x = equal.count(rnorm(100)), y = rnorm(100))}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{
  \code{\link{xyplot}},
  \code{\link{co.intervals}}, \code{\link{Lattice}} 
}
\keyword{print}
\keyword{hplot}
\keyword{methods}
\keyword{classes}




\eof
\name{simpleKey}
\alias{simpleKey}
\title{ Function to generate a simple key }
\description{
  Simple interface to generate a list appropriate for \code{draw.key}
}
\usage{
simpleKey(text, points = TRUE,
          rectangles = FALSE,
          lines = FALSE,
          col, cex, font,
          fontface, fontfamily, \dots)
}
\arguments{
  \item{text}{ character or expression vector, giving levels of grouping
    variable
  }
  \item{points}{logical}
  \item{rectangles}{logical}
  \item{lines}{logical}
  \item{col, cex, font, fontface, fontfamily}{
    Used as top-level components of the list produced, to be used for
    the text labels. Defaults to the values in
    \code{trellis.par.get("add.text")}
  }
  \item{\dots}{ arguments added to the list, finally passed to \code{draw.key}}
}
\details{
  A lattice plot can include a legend (key) if an appropriate list is
  specified as the \code{key} argument to a high level Lattice function
  such as \code{xyplot}. This key can be very flexible, but that
  flexibility comes at the cost of this list being very complicated even
  in simple situations. The \code{simpleKey} function is a shortcut,
  which assumes that the key is being drawn in conjunction with the
  \code{groups} argument, and that the default Trellis settings are
  being used. At most one each of points, rectangles and lines can be
  drawn.

  See also the \code{auto.key} argument for high level plots.
}
\value{
  A list that would work as the \code{key} argument to \code{xyplot}
  etc.
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{draw.key}}, \code{\link{xyplot}}, \code{\link{Lattice}}}

\keyword{dplot}

















\eof
\name{splom}
\title{Scatter Plot Matrices}
\synopsis{
splom(formula,
             data = parent.frame(),
             auto.key = FALSE,
             aspect = 1,
             between = list(x = 0.5, y = 0.5),
             panel = if (is.null(groups)) "panel.splom" else "panel.superpose",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab = "Scatter Plot Matrix",
             xlim,
             ylab = NULL,
             ylim,
             superpanel = "panel.pairs",
             pscales = 5,
             varnames,
             drop.unused.levels,
             ...,
             subset = TRUE)
parallel(formula,
             data = parent.frame(),
             aspect = "fill",
             between = list(x = 0.5, y = 0.5),
             panel = "panel.parallel",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab = NULL,
             xlim,
             ylab = NULL,
             ylim,
             varnames,
             drop.unused.levels,
             ...,
             subset = TRUE)
}
\usage{
splom(formula,
      data,
      aspect = 1,
      between = list(x = 0.5, y = 0.5),
      panel = if (is.null(groups)) "panel.splom" else "panel.superpose",
      superpanel = "panel.pairs",
      pscales = 5,
      varnames, \dots)
parallel(formula,
         data,
         between = list(x = 0.5, y = 0.5),
         panel = "panel.parallel",
         varnames,
         \dots)
}
\description{
  Draw Conditional Scatter Plot Matrices and Parallel Coordinate Plots
}
\alias{splom}
\alias{parallel}
\arguments{
  \item{formula}{ a formula describing the structure of the plot, which
    should be of the form \code{~ x | g1 * g2 * \dots}, where \code{x}
    is a data frame or a matrix. Each of \code{g1,g2,\dots} must be
    either factors or shingles. The conditioning variables
    \code{g1,g2,\dots} may be omitted, in which case the leading
    \code{~} may also be omitted.
  }
  \item{data}{ a data frame containing values for any variables in the
    formula. By default the environment where the function 
    was called from is used.
  }
  \item{aspect}{ 
    aspect ratio of each panel (and subpanel), square by default for
    \code{splom}.
  }
  \item{between}{ 
    to avoid confusion between panels and subpanels, the default is to
    show the panels of a splom plot with space between them.
  }
  \item{panel}{
    Usual interpretation for \code{parallel}, namely the function that
    creates the display within each panel. 

    For \code{splom}, the terminology is slightly complicated. The role
    played by the panel function in most other high-level functions is
    played here by the \code{superpanel} function, which is responsible
    for the display for each conditional data subset. \code{panel} is
    simply an argument to the default \code{superpanel} function
    \code{panel.pairs}, and is passed on to it unchanged. It is used
    there to create each pairwise display. See \code{\link{panel.pairs}}
    for more useful options.

  }
  \item{superpanel}{ 
    function that sets up the splom display, by default as a scatterplot
    matrix.
  }
  \item{pscales}{ 
    a numeric value or a list, meant to be a less functional substitute
    for the \code{scales} argument in \code{xyplot} etc. This argument
    is passed to the \code{superpanel} function, and is handled by the
    default superpanel function \code{panel.pairs}. The help page for
    the latter documents this argument in more detail.
  }
  \item{varnames}{ 
    character vector giving the names of the p variables in x. By
    default, the column names of x.
  }
  \item{\dots}{ other arguments}
}
\details{
  \code{splom} produces Scatter Plot Matrices. The role usually played by
  \code{panel} is taken over by \code{superpanel}, which determines how
  the columns of \code{x} are to be arranged for pairwise plots. The
  only available option currently is \code{panel.pairs}.

  Many of the finer customizations usually done via arguments to high
  level function like \code{xyplot} are instead done by
  \code{panel.pairs} for \code{splom}. These include control of axis
  limits, tick locations and prepanel calcultions. If you are trying to
  fine-tune your \code{splom} plot, definitely look at the
  \code{\link{panel.pairs}} help page. The \code{scales} argument is
  usually not very useful in \code{splom}, and trying to change it may
  have undesired effects.

  \code{parallel} draws Parallel Coordinate Plots. (Difficult to
  describe, see example.)
  
  These and all other high level Trellis functions have several
  arguments in common. These are extensively documented only in the
  help page for \code{xyplot}, which should be consulted to learn more
  detailed usage.
}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\seealso{
  \code{\link{xyplot}}, \code{\link{Lattice}}, \code{\link{panel.pairs}}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\examples{
data(iris)
super.sym <- trellis.par.get("superpose.symbol")
splom(~iris[1:4], groups = Species, data = iris,
      panel = panel.superpose,
      key = list(title = "Three Varieties of Iris",
                 columns = 3, 
                 points = list(pch = super.sym$pch[1:3],
                 col = super.sym$col[1:3]),
                 text = list(c("Setosa", "Versicolor", "Virginica"))))
splom(~iris[1:3]|Species, data = iris, 
      layout=c(2,2), pscales = 0,
      varnames = c("Sepal\nLength", "Sepal\nWidth", "Petal\nLength"),
      page = function(...) {
          ltext(x = seq(.6, .8, len = 4), 
                y = seq(.9, .6, len = 4), 
                lab = c("Three", "Varieties", "of", "Iris"),
                cex = 2)
      })
parallel(~iris[1:4] | Species, iris) 
}
\keyword{hplot}







\eof
\name{strip.default}
\alias{strip.default}
\title{Default Trellis strip function}
\usage{
strip.default(which.given,
              which.panel,
              var.name,
              factor.levels,
              shingle.intervals,
              strip.names = c(FALSE, TRUE),
              style = 1,
              bg = trellis.par.get("strip.background")$col[which.given],
              fg = trellis.par.get("strip.shingle")$col[which.given],
              par.strip.text = trellis.par.get("add.text"))
}
\description{
  function that draws the strips in conditional trellis plots.
}

\arguments{
  \item{which.given}{ integer index specifying which of the conditioning
    variables this strip corresponds to. 
  }
  \item{which.panel}{ vector of integers as long as the number of
    conditioning variables. The contents are indices specifing the
    current levels of each of the conditioning variables (thus, this
    would be unique for each distinct panel).
  }
  \item{var.name}{ vector of character strings or ecpressions as long as
    the number of conditioning variables. The contents are the names of
    the conditioning variables, to be used if the appropriate entry in
    \code{strip.names} (see below) is true.
  }
  \item{factor.levels}{ if the current strip corresponds to a factor,
    this should be a character or expression vector giving the levels of
    the factor.  Otherwise, it should be \code{NULL}
  }
  \item{shingle.intervals}{if the current strip corresponds to a shingle,
    this should be a 2-column matrix giving the levels of the shingle.
    (of the form that would be produced by \bold{printing}
    \code{levels(shingle)}). Otherwise, it should be \code{NULL}
  }
  \item{strip.names}{ a logical vector of length 2, indicating whether
    or not the name of the conditioning variable that corresponds to the
    strip being drawn is to be written on the strip. The two components
    give the values for factors and shingles respectively.

    This argument is ignored for a factor when \code{style} is not one
    of 1 and 3.
  }
  \item{style}{ integer, with values 1,2,3,4,5 and 6 currently supported.
    Applicable only when \code{x} is a factor. Determines how the current
    level of \code{x} is indicated on the strip.

    The best way to find out what effect the value of \code{style} has
    is to try them out. Here is a short description: for a style value
    of 1, the strip is colored in the background color with the strip
    text (as determined by other arguments) centred on it. A value of 3
    is the same, except that a part of the strip is colored in the
    foreground color, indicating the current level of the factor. For
    styles 2 and 4, the part corresponding to the current level remains
    colored in the foreground color, however, for style = 2, the
    remaining part is not colored at all, whereas for 4, it is colored
    with the background color. For both these, the names of all the
    levels of the factor are placed on the strip from left to right.
    Styles 5 and 6 produce the same effect (they are subtly different in
    S, this implementation corresponds to 5), they are similar to style
    1, except that the strip text is not centred, it is instead
    postioned according to the current level.

    Note that unlike S-PLUS, the default value of \code{style} is 1.
  }
  \item{par.strip.text}{ list with parameters controlling the text on
    each strip, with components \code{col, cex, font}
  }
  \item{bg}{ strip background color. }
  \item{fg}{ strip foreground color. }
}
\details{ default strip function for trellis functions. Useful 
  mostly because of the \code{style} argument --- non-default styles
  are often more informative, especially when the names of the levels
  of the factor \code{x} are small. Typical use is as 
  \code{strip = function(\dots) strip.default(style=2,\dots)}.
}
\seealso{
  \code{\link{xyplot}}, \code{\link{Lattice}} 
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\keyword{dplot}



\eof
\name{tmd}
\alias{tmd}
\title{Tukey Mean-Difference Plot}
\description{
  Create Tukey Mean-Difference Plots from a trellis object returned by
  \code{xyplot} and \code{qq}
}
\usage{
tmd(object,
    xlab = "mean",
    ylab = "difference",
    panel = "panel.tmd",
    prepanel = "prepanel.default.tmd",
    ...)
}
\arguments{
  \item{object}{ An object of class ``trellis'' returned by
    \code{xyplot} or \code{qq}
  }
  \item{xlab}{ x label}
  \item{ylab}{ y label}
  \item{panel}{ panel function to be used }
  \item{prepanel}{
    prepanel function used to recalculate limits using the modified
    \code{x, y} values
  }
  \item{\dots}{ other arguments}
}
\details{
  The Tukey Mean-difference plot is produced by modifying the (x,y)
  values of each panel as follows: the new coordinates are given by
  \code{x=(x+y)/2, y=y-x}.

  \code{tmd} uses the \code{update} method for ``trellis'' objects,
  which processes all extra arguments supplied to \code{tmd}.
}

\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{ \code{\link{panel.tmd}}, \code{\link{xyplot}},
  \code{\link{Lattice}}
}
\examples{
data(singer)
tmd(qqmath(~height|voice.part, data = singer))
}
\keyword{dplot}

\eof
\name{trellis.datasets}
\alias{barley}
\alias{singer}
\alias{melanoma}
\alias{ethanol}
\alias{sunspot}
\alias{environmental}
\non_function{}
\title{Data Sets in the Lattice library}
\usage{
data(singer)
data(melanoma)
data(ethanol)
data(sunspot)
data(environmental)
}
\description{
  These data sets are included to facilitate the same examples as S-PLUS
  trellis. (Detailed descriptions of the datasets are omitted for now.)
}
\source{
  These datasets are part of the collection of datasets used in the book
  \emph{Visualizing Data} by Bill Cleveland, and are available at Bill
  Cleveland's Homepage
  \url{http://cm.bell-labs.com/cm/ms/departments/sia/wsc/index.html}
  as well as in Statlib.
}
\keyword{datasets}








\eof
\name{trellis.device}
\alias{trellis.device}
\alias{canonical.theme}
\title{ Initializing Trellis Displays }
\description{
  \code{trellis.device} initializes a Trellis display device with
  appropriate settings of the graphics control parameter
  \code{trellis.settings}.
}
\usage{
trellis.device(device = getOption("device"),
               color = !(dev.name == "postscript"),
               theme = getOption("lattice.theme"),
               bg = NULL,
               new = TRUE,
               retain = FALSE,
               ...)
canonical.theme(name, color)
}
\arguments{
  \item{device}{ name of a function (possibly as a character string)
    that initializes a device. Admissible values include "x11",
    "windows", "mac", "postscript", "pdf", "png", etc.
  }
  \item{color}{ logical, whether the display should be color or black
    and white. \code{FALSE} for postscript devices, \code{TRUE}
    otherwise.
  }
  \item{theme}{ list of components that change the settings of the
    device opened, or, a function that when called produces such a
    list. The function name can be supplied as a quoted string.

    A possible use of this argument is to change the default settings at
    session startup, for example by setting \code{options(lattice.theme
      = "col.whitebg")}. If \code{theme} is a function, it will not be
    supplied any arguments, however, it is guaranteed that a device will
    already be open when it is called, so one may use \code{.Device}
    inside the function to ascertain what device has been opened.
  }
  \item{bg}{ background color for the device.}
  \item{new}{ logical flag indicating whether a new device should be
    started. If \code{FALSE},  the options for the current device are
    changed to the defaults determined by the other arguments.
  }
  \item{retain}{
    logical. If \code{TRUE} and a setting for this device already
    exists, then that is used instead of the defaults for this
    device. By default, pre-existing settings are overwritten (and
    lost).
  }
  \item{name}{ name of the device for which the setting is required, as
    returned by \code{.Device}
  }
  \item{\dots}{ additional parameters to be passed to the \code{device}
    function, most commonly \code{file} for non-screen devices.
  }
}
\details{
  \code{trellis.device} calls the \code{device} function to start a
  display, and sets an appropriate component of the variable
  \code{lattice.theme} (in an unexported environment named
  \code{.LatticeEnv}) to appropriate (device-specific) values.

  The device specific settings are determined by \code{canonical.theme}
  which returns a list (see below).

  The settings can subsequently be handled by the interface functions
  \code{trellis.par.get}, \code{trellis.par.set} and \code{lset}
}

\value{
  \code{canonical.theme} returns a list of components defining graphical
  parameter settings for Lattice displays. It is used internally in
  \code{trellis.device}, and can also be used as an argument to
  \code{lset}, or even as \code{theme} in \code{trellis.device} to use
  the defaults for another device.
}

\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{
  \code{\link{trellis.par.set}}, \code{\link{lset}}, 
  \code{\link{postscript}},\code{\link{x11}}, \code{\link{Lattice}}
}
\keyword{dplot}


\eof
\name{trellis.object}
\alias{trellis.object}
\non_function{}
\title{Lattice Display Object}
\description{
  This class of objects is returned by Lattice display functions, and is
  usually plotted by default by its `print' method.
}
\value{
  A ``trellis'' object is a list with the following components:


  \item{aspect.fill}{ logical specifying whether aspect==``fill''}
  \item{aspect.ratio}{ numeric, aspect ratio to be used IF aspect.fill
    is FALSE}
  \item{as.table }{ logical, whether rows are drawn top to bottom}
  \item{call}{ call that generated the object. Not used}
  \item{condlevels}{ Levels of the conditioning variables}
  \item{fontsize.normal}{ default font size for the plot}
  \item{fontsize.small}{ font size for the axis labels (?)}
  \item{key      }{ list defining the key to be drawn}
  \item{layout   }{ layout of the trellis. Must be of length 3, but
    first element can be 0, in which case the second element is taken to
    be the desired number of panels per page and a `nice' layout is
    calculated based on aspect ratio, screen dimensions etc.}
  \item{page     }{ the page function}
  \item{panel    }{ the panel function}
  \item{panel.args}{ a list as long as the number of panels, each
    element being a list itself, containing the arguments in named form
    to be passed to the panel function in that panel. If \code{skip=T}
    for a particular panel, then the corresponding element is
    \code{FALSE} instead of a list.}
  \item{panel.args.common }{ a list containing the arguments common to
    all the panel functions in name=value form}
  \item{par.strip.text }{ list containing graphical parameters
    \code{cex, col, font} for the strip labels}
  \item{skip     }{ logical vector, not used}
  \item{strip    }{ the function to draw the strip labels}
  \item{main     }{ \code{NULL}, or list with components \code{label,
      cex, col, font}}
  \item{sub      }{ same as main}
  \item{xlab     }{ same as sub}
  \item{ylab     }{ same as xlab}
  \item{x.scales }{ list describing x-scale, can consist of several
    other lists, paralleling panel.args, if x-relation is not ``same''}
  \item{y.scales }{ list describing y-scale, same as x.scales}
  \item{x.between}{ numeric vector of interpanel x-space}
  \item{y.between}{ numeric vector of interpanel y-space}
  \item{x.limits}{ numeric vector of length 2 or list, giving x-axis
    limits} 
  \item{y.limits }{ similar to x.limits}
}
\keyword{classes}


\eof
\name{trellis.par.get}
\alias{trellis.par.get}
\alias{trellis.par.set}
\alias{show.settings}
\title{ Graphical Parameters for Trellis Displays }
\description{
  Functions used to query, display and modify graphical parameters for
  fine control of Trellis displays.
}
\usage{
trellis.par.get(name = NULL)
trellis.par.set(name, value, warn)
show.settings(x = NULL)
}
\arguments{
  \item{name}{ character giving the name of a component. For a list of
    valid values type \code{names(trellis.par.get())}.
  }
  \item{value}{ a list giving the desired value of the component. }
  \item{warn}{ logical, whether a warning should be issued when
    \code{trellis.par.get} is called when no graphics device is open }
  \item{x}{ optional list of components that change the settings (any
    valid value of \code{theme} in \code{lset}). These are used to
    modify the current settings (obtained by \code{trellis.par.get})
    before they are displayed
  }
}
\details{
  The various graphical parameters (color, line type, background etc)
  that control the look and feel of Trellis displays are highly
  customizable. Also, R can produce graphics on a number of devices, and
  it is expected that a different set of parameters would be more suited
  to different devices. These parameters are stored internally in a
  variable named \code{lattice.theme}, which is a list whose components
  define settings for particular devices. The components are idenified
  by the name of the device they represent (as obtained by
  \code{.Device}), and are created as and when new devices are opened
  for the first time using \code{trellis.device} (or Lattice plots are
  drawn on a device for the first time in that session).

  The initial settings for each device defaults to values appropriate
  for that device. In practice, this boils down to three distinct
  settings, one for screen devices like \code{x11} and \code{windows},
  one for black and white plots (mostly useful for \code{postscript})
  and one for color printers (color \code{postcript, pdf}). 

  Once a device is open, it's settings can be modified. When another
  instance of the same device is opened later using
  \code{trellis.device}, the settings for that device are reset to its
  defaults, unless otherwise specified in the call to
  \code{trellis.device}. But settings for different devices are treated
  separately, i.e., opening a postscript device will not alter the x11
  settings, which will remain in effect whenever an x11 device is
  active.

  The functions \code{trellis.par.*} are meant to be interfaces to the
  global settings. They always apply on the settings for the currently
  ACTIVE device.

  \code{trellis.par.get}, called without any arguments, returns the full
  list of settings for the active device. With the \code{name} argument
  present, it returns that component only. \code{trellis.par.get} sets
  the value of the \code{name} component of the current active device
  settings to \code{value}.

  \code{trellis.par.get} is usually used inside trellis functions to get
  graphical parameters before plotting. Modifications by users via
  \code{trellis.par.set} is traditionally done as follows:

  \code{add.line <- trellis.par.get("add.line")}

  \code{add.line$col <- "red"}

  \code{trellis.par.set("add.line", add.line)}

  \code{\link{lset}} is a usually more convenient interface to achieve
  the same end (but is not compatible with S-PLUS), the equivalent usage
  being:

  \code{lset(list(add.line = list(col = "red")))}

  The actual list of the components in \code{trellis.settings} has not
  been finalized, so I'm not attempting to list them here. The current
  value can be obtained by \code{print(trellis.par.get())}. Most names
  should be self-explanatory.

  \code{show.settings} provides a graphical display summarizing some of
  the values in the current setting.
}

\value{
  \code{trellis.par.get} returns a list giving parameters for that
  component. If \code{name} is missing, it returns the full list.
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{
  There are some simpler alternatives that can be used to manipulate the
  settings. Details can be found in the documentation for
  \code{\link{lset}}. Other related pages: \code{\link{trellis.device}},
  \code{\link{trellis.par.get}},\code{\link{Lattice}}
}

\examples{
show.settings()
}
\keyword{dplot}



\eof
\name{update.trellis}
\alias{update.trellis}
\title{Update Trellis Object}
\description{
  Update method for objects of class ``trellis''
}
\synopsis{
\method{update}{trellis}(object,
       panel,
       aspect,
       as.table,
       between,
       key,
       auto.key,
       legend,
       layout,
       main,
       page,
       par.strip.text,
       prepanel,
       scales,
       skip,
       strip,
       sub,
       xlab,
       xlim,
       ylab,
       ylim,
       par.settings,
       index.cond,
       perm.cond,
       \dots)
}
\usage{
\method{update}{trellis}(object, \dots)
}
\arguments{
  \item{object}{ The object to be updated, of class ``trellis''. }
  \item{\dots}{
    arguments that will be used to update \code{object}. See details
    below. 
  }
}


\details{

  All the high level lattice functions such as \code{xyplot} produce an
  object of (S3) class ``trellis'', which is usually printed by its
  \code{print} method. However, the object itself can be manipulated and
  modified to a large extent using the \code{update} method, and then
  re-plotted as needed.

  Most arguments to high level functions can also be supplied to the
  \code{update} method as well, with some exceptions. Generally
  speaking, anything that would needs to change the data within each
  panel is a no-no (this includes the \code{formula, data, groups,
    subscripts} and \code{subset}). Everything else is technically game,
  though might not be implemented yet. If you find something missing
  that you wish to have, feel free to make a request.

  Not all arguments accepted by a Lattice function are processed by
  \code{update}, but the following should work (they are described in
  the help page for \code{\link{xyplot}}):

  \code{panel}, \code{aspect}, \code{as.table}, \code{between},
  \code{key}, \code{auto.key}, \code{legend}, \code{layout},
  \code{main}, \code{page}, \code{par.strip.text}, \code{prepanel},
  \code{scales}, \code{skip}, \code{strip}, \code{sub}, \code{xlab},
  \code{xlim}, \code{ylab}, \code{ylim}, \code{par.settings},
  \code{index.cond}, \code{perm.cond}

  Any other argument is added to the list of arguments to be passed to
  the \code{panel} function. Because of their somewhat special nature,
  updates to objects produced by \code{cloud} and \code{wireframe} do
  not work very well yet.
}

\value{
  An object of class \code{trellis}, by default plotted by
  \code{print.trellis}.
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{
  \code{\link{trellis.object}}, \code{\link{Lattice}},
  \code{\link{xyplot}}
}
\keyword{dplot}

\eof
\name{utilities.3d}
\alias{utilities.3d}
\alias{palette.shade}
\alias{ltransform3dMatrix}
\alias{ltransform3dto3d}
\title{ Default Panel Function for cloud }
\description{
  These are (related to) the default panel functions for \code{cloud}
  and \code{wireframe}.
}
\usage{
ltransform3dMatrix(screen, R.mat)
ltransform3dto3d(x, R.mat, dist)
% palette.shade(irr, ref, height, saturation)
}
\arguments{
  \item{x}{
    \code{x} can be a numeric matrix with 3 rows for
    \code{ltransform3dto3d}
  }
  \item{screen}{list, as described in \code{\link{cloud}}}
  \item{R.mat}{4x4 transformation matrix in homogeneous
    coordinates}
  \item{dist}{controls transformation to account for
    perspective viewing}

%   \item{irr, ref}{
%     numeric, between 0 and 1. See details below. (The names are intended
%     to suggest a link with irradiance and reflectance respectively.)
%   }
%   \item{height}{
%     numeric, denoting height scaled to lie between 0 and 1
%   }
%   \item{saturation}{
%     used to control \code{s} in \code{hsv} inside \code{palette.shade}
%   }
}
\details{
  \code{ltransform3dMatrix} and \code{ltransform3dto3d} are utility
  functions to help in computation of projections. These functions are
  used inside the panel functions for \code{cloud} and
  \code{wireframe}. They may be useful in user-defined panel functions
  as well.

  The first function takes a list of the form of the \code{screen}
  argument in \code{cloud} and \code{wireframe} and a \code{R.mat}, a
  4x4 transformation matrix in homogeneous coordinates, to return a new
  4x4 transformation matrix that is the result of applying \code{R.mat}
  followed by the rotations in \code{screen}. The second function
  applies a 4x4 transformation matrix in homogeneous coordinates to a
  3xn matrix representing points in 3-D space, and optionally does some
  perspective computations. (There has been no testing with non-trivial
  transformation matrices, and my knowledge of the homogeneous
  coordinate system is very limited, so there may be bugs here.)

%   \code{palette.shade} is the function used by default for calculation
%   of colors when `shading' a wireframe. Such a function can expect three
%   arguments in general, namely \code{irr}, \code{ref} and
%   \code{height}.  \code{irr} should represent foreshortening, being the
%   cosine of the angle between the incident light ray and the normal to
%   the surface.  \code{ref} represents the direction of the reflected
%   light, and in fact is the cosine of half the angle between the
%   reflected ray and the viewing direction. \code{height} is the scaled
%   (average) height (of any particular facet).

}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\seealso{\code{\link{cloud}}}
\keyword{dplot}

\eof
\name{xyplot}
\title{Common Bivariate Trellis Plots}
\alias{xyplot}
\alias{barchart}
\alias{bwplot}
\alias{dotplot}
\alias{stripplot}
\synopsis{
xyplot(formula,
             data = parent.frame(),
             allow.multiple = is.null(groups) || outer,
             outer = FALSE,
             auto.key = FALSE,
             aspect = "fill",
             panel = if (is.null(groups)) "panel.xyplot"
             else "panel.superpose",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab,
             xlim,
             ylab,
             ylim,
             drop.unused.levels = TRUE,
             ...,
             subscripts = !is.null(groups),
             subset = TRUE)
bwplot(formula,
             data = parent.frame(),
             allow.multiple = is.null(groups) || outer,
             outer = FALSE,
             auto.key = FALSE,
             aspect = "fill",
             panel = "panel.bwplot",
             prepanel = NULL,
             scales = list(),
             strip = TRUE,
             groups = NULL,
             xlab,
             xlim,
             ylab,
             ylim,
             box.ratio = 1,
             horizontal = NULL,
             drop.unused.levels = TRUE,
             ...,
             subscripts = !is.null(groups),
             subset = TRUE)
}
\usage{
xyplot(formula,
       data = parent.frame(),
       panel = if (is.null(groups)) "panel.xyplot"
               else "panel.superpose",
       allow.multiple,
       outer,
       aspect = "fill",
       as.table = FALSE,
       between,
       groups,
       key,
       auto.key = FALSE,
       legend,
       layout,
       main,
       page,
       par.strip.text,
       prepanel,
       scales,
       skip,
       strip = "strip.default",
       sub,
       xlab,
       xlim,
       ylab,
       ylim,
       drop.unused.levels = TRUE,
       par.settings,
       perm.cond,
       index.cond,
       \dots,
       panel.groups = "panel.xyplot",
       subscripts,
       subset)
dotplot(formula,
        data, 
        panel = "panel.dotplot",
        groups = NULL,
        ...,
        subset = TRUE)
barchart(formula,
         data,
         panel = "panel.barchart",
         box.ratio = 2,
         groups = NULL,
         ...,
         subset = TRUE)
stripplot(formula,
          data,
          panel = "panel.stripplot",
          jitter = FALSE,
          factor = .5,
          box.ratio = if (jitter) 1 else 0,
          groups = NULL,
          ...,
          subset = TRUE)
bwplot(formula,
       data,
       panel = "panel.bwplot",
       box.ratio = 1,
       ...,
       horizontal,
       subset = TRUE)
}

\description{
  These are the most commonly used Trellis functions to look at pairs of
  variables. By far the most common is \code{xyplot}, designed mainly
  for two continuous variates (though factors can be supplied as well,
  in which case they will simply be coerced to numeric), which produces
  Conditional Scatterplots. The others are useful when one of the
  variates is a factor or a shingle. See details below.

  Most of the arguments documented here are also applicable for many of
  the other Trellis functions. These are not described in any detail
  elsewhere, and this should be considered the canonical documentation
  for such arguments.

  Note that any arguments passed to these functions and not recognized
  by them will be passed to the panel function. Most predefined panel
  functions have arguments that customize its output. These arguments
  are described only in the help pages for these panel functions, but
  can usually be supplied as arguments to the high level plot.  
}
\arguments{
  \item{formula}{
    a formula describing the form of conditioning plot. The formula is
    generally of the form \code{y ~ x | g1 * g2 * \dots}, indicating
    that plots of \code{y} (on the y axis) versus \code{x} (on the x
    axis) should be produced conditional on the variables
    \code{g1, g2, \dots}. However, the conditioning variables
    \code{g1,g2,\dots} may be omitted. For S-PLUS compatibility, the
    formula can also be written as \code{y ~ x | g1 + g2 + \dots}.

    For all these functions other than \code{xyplot}, a formula of the
    form \code{ ~ x | g1 * g2 * \dots} is also allowed. In that case,
    \code{y} defaults to \code{as.factor(names(x))} if \code{x} is
    named, and a factor with a single level otherwise.

    Although it is not recommended, usage of the form \code{dotplot(x)}
    (where the formula argument is not a formula at all) is also
    allowed, and is equivalent to \code{dotplot{ ~ x}}.
    
    The conditioning variables \code{g1, g2, \dots} must be either
    factors or shingles (Shingles are a way of processing numeric
    variables for use in conditioning. See documentation of
    \code{\link{shingle}} for details. Like factors, they have a
    `levels' attribute, which is used in producing the conditioning
    plots). For each unique combination of the levels of the
    conditioning variables \code{g1, g2, \dots}, a separate panel is
    produced using the points \code{(x,y)} for the subset of the data
    (also called packet) defined by that combination.

    The order in which the panels are drawn depends on the order in
    which the conditioning variables are specified (\code{g1} varies
    fastest). Within a conditioning variable, the order depends on the
    order of the levels (which for factors is usually in alphabetical
    order). Both of these orders can be modified using the
    \code{order.cond} and \code{perm.cond} arguments, typically in the
    \code{update} method.

    Numeric conditioning variables are converted to shingles by the
    function \code{shingle} (however, using \code{\link{equal.count}}
    might be more appropriate in many cases) and character vectors are
    coerced to factors.

    The formula can involve expressions, e.g. \code{sqrt(),log()}.

    A special case is when the left and/or right sides of the formula
    (before the conditioning variables) contain a `+' sign, e.g.,
    \code{y1+y2 ~ x | a*b}. This formula would be taken to mean that the
    user wants to plot both \code{y1~x | a*b} and \code{y2~x | a*b}, but
    with the \code{y1~x} and \code{y2~x} superposed in each panel (this
    is slightly more complicated in \code{barchart}). The two parts
    would be distinguished by different graphical parameters. This is
    essentially what the \code{groups} argument would produce, if
    \code{y1} and \code{y2} were concatenated to produce a longer
    vector, with the \code{groups} argument being an indicator of which
    rows come from which variable. In fact, this is exactly what is done
    internally using the \code{\link{reshape}} function. This feature
    cannot be used in conjunction with the \code{groups} argument.

    To interpret \code{y1+y2} as a sum, one can either set
    \code{allow.multiple=FALSE} or use \code{I(y1+y2)}.

    A variation on this feature is when the \code{outer} argument is set
    to \code{TRUE} as well as \code{allow.multiple}. In that case, the
    plots are not superposed in each panel, but instead separated into
    different panels (as if a new conditioning variable had been added).
    
    The \code{x} and \code{y} variables should both be numeric in
    \code{xyplot}, and an attempt is made to coerce them if
    not. However, if either is a factor, the levels of that factor are
    used as axis labels. In the other four functions documented here,
    exactly one of \code{x} and \code{y} should be numeric, and the
    other a factor or shingle. Which of these will happen is determined
    by the \code{horizontal} argument --- if \code{horizontal=TRUE},
    then \code{y} will be coerced to be a factor or shingle, otherwise
    \code{x}. The default value of \code{horizontal} is \code{FALSE} if
    \code{x} is a factor or shingle, \code{TRUE} otherwise. (The
    functionality provided by \code{horizontal=FALSE} is not
    S-compatible.)
  
    All points with at least one of its values missing (NA) in any
    of the variates involved are omitted from the plot.
  }
  \item{data}{a data frame containing values for any variables in the
    formula, as well as \code{groups} and \code{subset} if applicable. 
    By default the environment where the function was called from is
    used. 
  }
  \item{allow.multiple, outer}{
    logical flags to control what happens with formulas like \code{y1 +
      y2 ~ x}. See the entry for \code{formula} for details.
    \code{allow.multiple} defaults to \code{TRUE} whenever it makes
    sense, and \code{outer} defaults to \code{FALSE} except when
    \code{groups} is explicitly specified or grouping doesn't make sense
    for the default panel function
  }
  \item{box.ratio}{ applicable to \code{bwplot, barchart} and
    \code{stripplot}, specifies the ratio of the width of the rectangles
    to the inter rectangle space.
  }
  \item{horizontal}{ logical, applicable to \code{bwplot, dotplot,
      barchart} and \code{stripplot}. Determines which of \code{x} and
    \code{y} is to be a factor or shingle (\code{y} if TRUE, \code{x}
    otherwise). Defaults to \code{FALSE} if \code{x} is a factor or
    shingle, \code{TRUE} otherwise. This argument is used to process the  
    arguments to these high level functions, but more importantly, it is
    passed as an argument to the panel function, which is supposed to
    use it as approporiate.

    A potentially useful component of \code{scales} is this case might
    be \code{abbreviate = TRUE}, in which case long labels which would
    usually overlap will be abbreviated. \code{scales} could also
    contain a \code{minlength} argument in this case, which would be
    passed to the \code{abbreviate} function.
  }
  \item{jitter}{ logical specifying whether the values should be jittered
    by adding a random noise in stripplot.}
  \item{factor}{ numeric controlling amount of jitter. Inverse effect
    compared to S ?
  }

  The following arguments are common to all the functions documented
  here, as well as most other high level Trellis functions. These are
  not documented elsewhere, except to override the usage given here.

  \item{panel}{ Once the subset of rows defined by each unique
    combination of the levels of the grouping variables are obtained
    (see above), the corresponding \code{x} and \code{y} variables (or
    some other variables, as appropriate, in the case of other
    functions) are passed on to be plotted in each panel. The actual
    plotting is done by the function specified by the \code{panel}
    argument. Each high level function has its own default panel
    function, which could depend on whether the \code{groups} argument
    was supplied.

    The panel function can be a function object or a character string
    giving the name of a predefined function. (The latter is preferred
    when possible, especially when the trellis object returned by the
    high level function is to be stored and plotted later.)

    Much of the power of Trellis Graphics comes from the ability to
    define customized panel functions. A panel function appropriate
    for the functions described here would usually expect arguments
    named \code{x} and \code{y}, which would be provided by the
    conditioning process. It can also have other arguments. It might be
    useful to know in this context that all arguments passed to a high
    level Trellis function (such as \code{xyplot}) that are not
    recognized by it are passed through to the panel function. It is
    thus generally good practice when defining panel functions to allow
    a \code{\dots} argument. Such extra arguments typically control
    graphical parameters, but other uses are also common. See
    documentation for individual panel functions for specifics.

    Note that unlike in S-PLUS, it is not guaranteed that panel
    functions will be supplied only numeric vectors for the \code{x} and
    \code{y} arguments; they can be factors as well (but not
    shingles). panel functions need to handle this case, which to get
    the old behaviour could simply coerce them to numeric.

    Technically speaking, panel functions must be written using Grid
    graphics functions. However, knowledge of Grid is usually not 
    necessary to construct new custom panel functions, there are several
    predefined panel functions which can help; for example,
    \code{panel.grid}, \code{panel.loess} etc. There are also some
    grid-compatible replacements of base R graphics functions useful for
    this purpose, such as \code{llines}. (Note that the corresponding
    base R graphics functions like \code{lines} would not work.) These
    are usually sufficient to convert existing custom panel functions
    written for S-PLUS.
    
    One case where a bit more is required of the panel function is when
    the \code{groups} argument is not null. In that case, the panel
    function should also accept arguments named \code{groups} and
    \code{subscripts} (see below for details). A very useful panel
    function predefined for use in such cases is \code{panel.superpose},
    which can be combined with different \code{panel.groups}
    functions. See the examples section for an interaction plot
    constructed this way. Several other panel functions can also handle
    the \code{groups} argument, including the default ones for
    \code{barchart, dotplot} and \code{stripplot}.

    Even when \code{groups} is not present, the panel function can have
    \code{subscripts} as a formal argument. In either case, the
    \code{subscripts} argument passed to the panel function are the
    indices of the \code{x} and \code{y} data for that panel in the
    original \code{data}, BEFORE taking into account the effect of
    the \code{subset} argument. Note that \code{groups} remains
    unaffected by any subsetting operations, so
    \code{groups[subscripts]} gives the values of \code{groups} that
    correspond to the data in that panel. (This becomes slightly more
    complicated when \code{allow.multiple} is in effect. Details are
    explained in the source code (function \code{latticeParseFormula}).)

    A panel function can have two other optional arguments for
    convenience, namely \code{panel.number} and
    \code{panel.counter}. Both provide a simple integer index indicating
    which panel is currently being drawn, but differ in how the count is
    calculated. \code{panel.counter} is a simple incremental counter
    that starts with 1 and is incremented each time a panel is
    drawn. \code{panel.number} on the other hand depends only on the
    combination of levels of the conditioning variables that is
    represented by that panel. The two indices coincide unless the order
    of conditioning variables is permuted and/or the plotting order of
    levels within one or more conditioning variables is altered (using
    \code{perm.cond} and \code{index.cond} respectively), in which case
    \code{panel.number} gives the index corresponding to the `natural'
    ordering of that combination of levels of the conditioning
    variables.

    \code{panel.xyplot} has an argument called \code{type} which is
    worth mentioning here because it is quite frequently used (and as
    mentioned above, can be passed to \code{xyplot} directly).  panel
    functions for \code{bwplot} and friends should have an argument
    called \code{horizontal} to account for the cases when \code{x} is
    the factor or shingle.
  }
  \item{panel.groups}{ useful mostly for \code{xyplot} and
    \code{densityplot}. Applies when \code{panel} is
    \code{panel.superpose} (which happens by default in these cases if
    \code{groups} is non-null)
  }
  \item{aspect}{ controls physical aspect ratio of the panels (same for
    all the panels). It can be specified as a ratio (vertical
    size/horizontal size) or as a character string. Legitimate 
    values are "fill" (the default) which tries to make the panels as
    big as possible to fill the available space, and "xy", which
    \bold{tries} to compute the aspect based on the 45 degree banking
    rule (see \emph{Visualizing Data} by William S. Cleveland for
    details). 

    If a \code{prepanel} function is specified, the \code{dx, dy}
    components returned by it are used to compute the aspect, otherwise
    the default prepanel function is used. Currently, only the default
    prepanel function for \code{xyplot} produces sensible banking
    calculations.

    The current implementation of banking is not very sophisticated, but
    is not totally vague either. See \code{\link{banking}} for details.
  }
  \item{as.table}{ logical that controls the order in which panels
    should be plotted: if FALSE, panels are drawn left to right,
    bottom to top (graph), if TRUE, left to right, top to bottom
    (matrix).
  }
  \item{between}{a list with components \code{x} and \code{y} (both
    usually 0 by default), numeric vectors specifying the space between
    the panels (units are character heights). \code{x} and \code{y} are
    repeated to account for all panels in a page and any extra
    components are ignored. The result is used for all pages in a
    multipage display. (In other words, it is not possible to use
    different \code{between} values for different pages).
  }
  \item{groups}{ used typically with \code{panel=panel.superpose}
    to allow display controls (color, lty etc) to vary according
    to a grouping variable. Formally, if groups is specified, then
    \code{groups} along with \code{subscripts} is passed to the panel
    function, which is expected to handle these arguments.

    It is very common to use a key (legend) when a grouping
    variable is specified. See entries for \code{key, auto.key} and
    \code{\link{simpleKey}} for how to draw a key.
  }
  \item{auto.key}{
    A logical (indicating whether a key is to be drawn automatically when
    a grouping variable is present in the plot), or a list of parameters
    that would be valid arguments to \code{\link{simpleKey}}. If
    \code{auto.key} is not \code{FALSE}, \code{groups} is non-null and
    there is no \code{key} or \code{legend} argument specified in the
    call, a key is created with \code{simpleKey} with
    \code{levels(groups)} as the first argument. (Note: this may not
    work in all high level functions, but it does work for the ones
    where grouping makes sense with the default panel function)

    \code{simpleKey} uses the trellis settings to determine the
    graphical parameters in the key, so this will be meaningful only if
    the settings are used in the plot as well.

    One disadvantage to using \code{key} (or even \code{simpleKey})
    directly is that the graphical parameters used in the key are
    absolutely determined at the time when the ``trellis'' object is
    created. Consequently, if a plot once created is re-\code{print}ed
    on another device, the parameter settings for the original device
    will be used. However, with \code{auto.key}, the key is actually
    created at printing time, so the key settings match the device
    settings.
  }
  \item{key}{
    A list of arguments that define a legend to be drawn on the
    plot. This list is used as an argument to the \code{\link{draw.key}}
    function, which produces a grid object eventually plotted by the
    print method for ``trellis'' objects.

    There is also a less flexible but usually sufficient shortcut
    function \code{\link{simpleKey}} that can generate such a list, as
    well as the argument \code{auto.key} that can be convenient in the
    most common situation where legends are used, namely when there is a
    grouping variable. To use more than one legend, or to have arbitrary
    legends not constrained by the structure imposed by \code{key}, use
    the \code{legend} argument.

    The position of the key can be controlled in either of two possible
    ways. If a component called \code{space} is present, the key is
    positioned outside the plot region, in one of the four sides,
    determined by the value of \code{space}, which can be one of
    ``top'', ``bottom'', ``left'' and ``right''. Alternatively, the key
    can be positioned inside the plot region by specifying components
    \code{x,y} and \code{corner}. \code{x} and \code{y} determine the
    location of the corner of the key given by \code{corner}, which can
    be one of \code{c(0,0), c(1,0), c(1,1),c(0,1)}, which denote the
    corners of the unit square. \code{x} and \code{y} must be numbers
    between 0 and 1, giving coordinates with respect to the whole
    display area.

    The key essentially consists of a number of columns, possibly
    divided into blocks, each containing some rows. The contents of the
    key are determined by (possibly repeated) components named
    ``rectangles'', ``lines'', ``points'' or ``text''. Each of these
    must be lists with relevant graphical parameters (see later)
    controlling their appearance. The \code{key} list itself can contain
    graphical parameters, these would be used if relevant graphical
    components are omitted from the other components.

    The length (number of rows) of each such column (except ``text''s)
    is taken to be the largest of the lengths of the graphical
    components, including the ones specified outside (see the entry for
    \code{rep} below for details on this). The ``text'' component has to
    have a character or expression vector as its first component, and
    the length of this vector determines the number of rows.

    The graphical components that can be included in \code{key} (and
    also in the components named ``text'', ``lines'', ``points'' and
    ``rectangles'' when appropriate) are \code{cex=1, col="black",
      lty=1, lwd=1, font=1, pch=8, adj=0, type="l", size=5, angle=0,
      density=-1}. \code{adj, angle, density} are
    unimplemented. \code{size} determines the width of columns of 
    rectangles and lines in character widths. \code{type} is relevant
    for lines; `"l"' denotes a line, `"p"' denotes a point, and `"b"'
    and `"o"' both denote both together.

    Other possible components of \code{key} are:

    \code{between}: numeric vector giving the amount of space (character
    widths) surrounding each column (split equally on both sides),

    \code{title}: string or expression, title of the key,

    \code{rep}: logical, defaults to TRUE. By default, it's assumed that
    all columns in the key (except the ``text''s) will have the same
    number of rows, and all components are replicated to be as long as
    the longest. This can be suppressed by specifying \code{rep =
      FALSE}, in which case the length of each column will be determined
    by components of that colunm alone.

    \code{cex.title}: cex for the title

    \code{background}: defaults to default background

    \code{border}: color of border, black if TRUE, defaults to FALSE (no
    border drawn)

    \code{transparent=FALSE}: logical, whether key area should be cleared

    \code{columns}: the number of columns column-blocks the key is to be
    divided into, which are drawn side by side.

    \code{betwen.columns}: Space between column blocks, in addition to
    \code{between}.

    \code{divide} Number of point symbols to divide each line when
    \code{type} is `"b"' or `"o"' in \code{lines}.
  }
  \item{legend}{
    the legend argument can be useful if one wants to place more than
    one key. It also allows one to use arbitrary ``grob''s (grid
    objects) as legends.
    
    If used, \code{legend} must be a list with an arbitrary number of
    components. Each component must be named one of ``left'', ``right'',
    ``top'', ``bottom'' or ``inside''. The name ``inside'' can be
    repeated, but not the others. This name will be used to determine
    the location for that component, and is similar to the \code{space}
    component of \code{key}.  If \code{key} (or \code{colorkey} for
    \code{\link{levelplot}} and \code{\link{wireframe}}) is specified,
    their \code{space} component must not conflict with the name of any
    component of \code{legend}.

    Each component of \code{legend} must have a component called
    \code{fun}. This can be a ``grob'', or a function or the name of a
    function that produces a ``grob'' when called. If this function
    expects any arguments, they must be supplied as a list in another
    component called \code{args}. For components named ``inside'', there
    can be additional components called \code{x, y} and \code{corner},
    which work in the same way as it does for \code{key}.
  }
  \item{layout}{
    In general, a Trellis conditioning plot consists of several panels
    arranged in a rectangular array, possibly spanning multiple
    pages. \code{layout} determines this arrangement.

    \code{layout} is a numeric vector giving the number of columns, rows
    and pages in a multipanel display. By default, the number of columns
    is determined by the number of levels in the first given variable;
    the number of rows is the number of levels of the second given
    variable. If there is one given variable, the default layout vector
    is c(0,n) , where n is the number of levels of the given vector. Any
    time the first value in the layout vector is 0 , the second value is
    used as the desired number of panels per page and the actual layout
    is computed from this, taking into account the aspect ratio of the
    panels and the device dimensions (via \code{par("din")}). The number
    of pages is by default set to as many as is required to plot all the
    panels. In general, giving a high value of \code{layout[3]} is not
    wasteful because blank pages are never created.
  }
  \item{main}{character string or expression or list describing main
    title to be placed on top of each page. Defaults to \code{NULL}. Can
    be a character string or expression, or a list with components
    \code{label, cex, col, font}. The \code{label} tag can be omitted if
    it is the first element of the list. Expressions are treated as
    specification of LaTeX-like markup as in \code{\link{plotmath}}
  }
  \item{page}{a function of one argument (page number) to be called
    after drawing each page. The function must be `grid-compliant', and
    is called with the whole display area as the default viewport.
  }
  \item{par.strip.text}{ list of graphical parameters to control the
    strip text, possible components are \code{col, cex, font, lines}.
    The first three control graphical parameters while the last is a
    means of altering the height of the strips. This can be useful, for
    example, if the strip labels (derived from factor levels, say) are
    double height (i.e., contains ``\\n''-s) or if the default height
    seems too small or too large.
  }
  \item{prepanel}{ function that takes the same arguments as the
    \code{panel} function and returns a list containing four components
    \code{xlim, ylim, dx, dy}. If \code{xlim} and \code{ylim} are not
    explicitly specified (possibly as components in \code{scales}), then
    the actual limits of the panels are guaranteed to include the limits
    returned by the prepanel function. This happens globally if the
    \code{relation} component of \code{scales} is "same", and on a panel
    by panel basis otherwise. See \code{xlim} to see what forms of the
    components \code{xlim, ylim} are allowed.

    The \code{dx} and \code{dy} components are used for banking
    computations in case \code{aspect} is specified as "xy". See
    documentation for the function \code{banking} for details regarding
    how this is done.

    The return value of the prepanel function need not have all the
    components named above; in case some are missing, they are replaced
    by the usual componentwise defaults.

    The prepanel function is responsible for providing a meaningful
    return value when the \code{x, y} (etc.) variables are zero-length
    vectors. When nothing is appropriate, values of NA should be
    returned for the \code{xlim} and \code{ylim} components.
  }
  \item{scales}{ list determining how the x- and y-axes (tick marks and
    labels) are drawn. The list contains parameters in name=value form,
    and may also contain two other lists called \code{x} and \code{y} of
    the same form (described below). Components of \code{x} and \code{y}
    affect the respective axes only, while those in \code{scales} affect
    both. (When parameters are specified in both lists, the values in
    \code{x} or \code{y} are used.) The components are :

    relation : determines limits of the axis. Possible values are "same"
    (default), "free" and "sliced". For relation="same", the same limits
    (determined by \code{xlim, ylim, scales$limits} etc) are used for
    all the panels. For relation="free", limits for each panel is
    determined by the points in that panel (via the \code{prepanel}
    function). Behaviour for relation = "sliced" is similar, except that
    the length (max - min) of the scales are constrained to remain the
    same across panels (limits specified as character vectors, if any,
    are ignored in these computations). If relation is not "same", the
    value of \code{xlim/ ylim/ scales$limits} is normally ignored,
    except when the latter is a list, when it is treated as if its
    components were the limit values obtained from the prepanel
    calculations for each panel.

    tick.number: Suggested number of ticks (ignored for a factor, shingle
    or character vector, in which case there's no natural rule for
    leaving out some of the labels. But see xlim).
    
    draw = TRUE: logical, whether to draw the axis at all.

    alternating = TRUE/c(1,2): logical specifying whether axes alternate
    from one side of the group of panels to the other. For more accurate
    control, alternating can be a vector (replicated to be as long as
    the number of rows or columns per page) consisting of the possible
    numbers 0=do not draw, 1=bottom/left, 2=top/right and
    3=both. alternating applies only when relation="same".

    limits: same as xlim and ylim.

    at: location of tick marks along the axis (in native coordinates),
    or a list as long as the number of panels describing tick locations
    for each panel.

    labels: Labels (strings or expressions) to go along with
    \code{at}. Can be a list like \code{at} as well.

    cex: factor to control character sizes for axis labels. Can be a
    vector of length 2, to control left/bottom and right/top separately.

    font: font face for axis labels (integer 1-4).
    
    tck: factor to control length of tick marks. Can be a vector of
    length 2, to control left/bottom and right/top separately.

    col: color of ticks and labels.

    rot: Angle by which the axis labels are to be rotated. Can be a
    vector of length 2, to control left/bottom and right/top separately.

    abbreviate: logical, whether to abbreviate the labels using
    \code{abbreviate}. Can be useful for long labels (e.g., in factors),
    especially on the x-axis.

    minlength: argument to \code{abbreviate} if \code{abbreviate=TRUE}.
    
    log: Use a log scale. Defaults to \code{FALSE}, other possible
    values are any number that works as a base for taking logarithm,
    \code{TRUE}, equivalent to 10, and \code{"e"} (for natural
    logarithm). Note that in this case the values passed to the panel
    function are already transformed, so all computations done inside
    the panel funtion will be affected accordingly. For example,
    \code{panel.lmline} will fit a line to the transformed values.

    format: the \code{format} to use for POSIXct variables. See
    \code{\link{strptime}} for description of valid strings.
    
    axs: character, ``r'' or ``i''. In the latter case, the axis limits
    are calculated as the exact data range, instead of being padded on
    either side. (May not always work as expected.) 

    Note: Much of the function of \code{scales} is accomplished by
    \code{pscales} in \code{splom}.
  }
  \item{skip}{ logical vector (default \code{FALSE}), replicated to be
    as long as the number of panels (spanning all pages). If
    \code{TRUE}, nothing is plotted in the corresponding panel. Useful
    for arranging plots in an informative manner.
  }
  \item{strip}{logical flag or function. If \code{FALSE}, strips are
    not drawn. Otherwise, strips are drawn using the \code{strip}
    function, which defaults to \code{strip.default}. See documentation
    of \code{strip.default} to see the form of a strip function.
  }
  \item{sub}{character string or expression for a subtitle to be placed
    at the bottom of each page. See entry for \code{main} for finer
    control options.
  }
  \item{subscripts}{ logical specifying whether or not a vector named
    subscripts should be passed to the panel function. Defaults to
    FALSE, unless \code{groups} is specified, or if the panel function
    accepts an argument named \code{subscripts}. (One should be careful
    when defining the panel function on-the-fly.)
  }
  \item{subset}{ logical or integer indexing vector (can be specified in
    terms of variables in \code{data}). Everything will be done on the
    data points indexed by \code{subset}. In case \code{subscripts}
    is TRUE, the subscripts will provide indices to the rows of data
    AFTER the subsetting is done (unlike S-PLUS).
  }
  \item{xlab}{character string or expression giving label for the
    x-axis. Defaults to the expression for \code{x} in
    \code{formula}. Specify as \code{NULL} to omit the label
    altogether. Fine control is possible, see entry for \code{sub}.
  }
  \item{xlim}{ Normally a numeric vector of length 2 (possibly a
    DateTime object) giving minimum and maximum for the x-axis, or, a
    character vector, expected to denote the levels of \code{x}. The
    latter form is interpreted as a range containing c(1, length(xlim)),
    with the character vector determining labels at tick positions
    \code{1:length(xlim)}

    \code{xlim} could also be a list, with as many components as the
    number of panels (recycled if necessary), with each component as
    described above. This is meaningful only when
    \code{scales$x$relation} is "free" or "sliced", in which case these
    are treated as if they were the corresponding limit components
    returned by prepanel calculations.

  }
  \item{ylab}{ character string or expression giving label for the
    y-axis. Defaults to the expression for \code{y} in
    \code{formula}. Fine control possible, see entry for \code{xlab}.
  }
  \item{ylim}{ same as \code{xlim}, applied to the y-axis. 
  }
  \item{drop.unused.levels}{ 
    logical indicating whether the unused levels of factors will be
    dropped. Defaults to \code{TRUE}, but it is sometimes useful to
    suppress dropping to preserve an useful layout. For finer control,
    this argument could also be list containing components \code{cond}
    and \code{data}, both logical, indicating desired behaviour for
    conditioning variables and data variables respectively.
  }
  \item{par.settings}{
    a list that could be supplied to \code{\link{lset}}. This enables
    the user to attach some display settings to the trellis object
    itself rather than change the settings globally. When the object is
    printed, these settings are temporarily in effect for the duration
    of the plot, after which the settings revert back to whatever it was
    before.
  }
  \item{perm.cond}{
    numeric vector, a permutation \code{1:n}, where \code{n} is the
    number of conditioning variables.  The order in which panels are
    drawn depends on the order of the conditioning variables specified
    in the \code{formula}. \code{perm.cond} can modify this order. If
    the trellis display is thought of as an \code{n}-dimensional array,
    then during printing, its dimensions are permuted using
    \code{perm.cond} as the \code{perm} argument to \code{\link{aperm}}
  }
  \item{index.cond}{
    While \code{perm.cond} permutes the dimensions of the
    multidimensional array of panels, \code{index.cond} can be used to
    subset that array.

    The panel display order within each conditioning variable depends on
    the order of their levels. \code{index.cond} can be used to choose a
    `subset' (in the R sense) of each of these levels.
    \code{index.cond} has to be a list as long as the number of
    conditioning variables, and the \code{i}-th component has to be a
    valid indexing vector for the integer vector \code{1:nlevels(g_i)}
    (which can repeat some of the levels or drop some altogether). The
    result of this indexing determines the order of panels within that
    conditioning variable. (To keep the order of a particular variable
    unchanged, one can simply set the corresponding component to
    \code{TRUE}.)

    Note that the components of \code{index.cond} are in the order of
    the conditioning variables in the original call, and is not affected
    by \code{perm.cond}.

    Although they can be supplied in high level function calls directly,
    it is more typical to use \code{perm.cond} and \code{index.cond} to
    update an existing ``trellis'' object, thus allowing it to be
    displayed in a different arrangement without re-calculating the data
    subsets that go into each panel.
  }  
  \item{\dots}{ other arguments, passed to the panel function.

    The arguments \code{horizontal} and \code{panel.groups} are
    documented here to avoid confusion, but they are actually not
    recognised by these high level functions. Rather, they are passed
    along to the panel function, as are any other unrecognized
    arguments.

  }

}
\value{
  An object of class ``trellis''. The `update' method can be used to
  update components of the object and the `print' method (usually called
  by default) will plot it on an appropriate plotting device.
}
\details{
  These are for the most part decriptions generally applicable to all
  high level Lattice functions, with special emphasis on \code{xyplot,
    bwplot} etc. For other functions, their individual documentation
  should be studied in addition to this. 
}
\seealso{
  \code{\link{Lattice}},
  \code{\link{print.trellis}},
  \code{\link{shingle}},
  \code{\link{banking}},
  \code{\link{reshape}}
  \code{\link{panel.xyplot}},
  \code{\link{panel.bwplot}},
  \code{\link{panel.barchart}},
  \code{\link{panel.dotplot}},
  \code{\link{panel.stripplot}},
  \code{\link{panel.superpose}},
  \code{\link{panel.loess}},
  \code{\link{panel.linejoin}},
  \code{\link{strip.default}},
  \code{\link{simpleKey}}
  \code{\link{lset}}
}
\author{ Deepayan Sarkar \email{deepayan@stat.wisc.edu}}
\examples{
require(stats)
## Tonga Trench Earthquakes
data(quakes)
Depth <- equal.count(quakes$depth, number=8, overlap=.1)
xyplot(lat ~ long | Depth, data = quakes)

## Examples with data from `Visualizing Data' (Cleveland)
## (obtained from Bill Cleveland's Homepage :
## http://cm.bell-labs.com/cm/ms/departments/sia/wsc/, also
## available at statlib)
data(ethanol)
EE <- equal.count(ethanol$E, number=9, overlap=1/4)
## Constructing panel functions on the fly; prepanel
xyplot(NOx ~ C | EE, data = ethanol,
       prepanel = function(x, y) prepanel.loess(x, y, span = 1),
       xlab = "Compression Ratio", ylab = "NOx (micrograms/J)",
       panel = function(x, y) {
           panel.grid(h=-1, v= 2)
           panel.xyplot(x, y)
           panel.loess(x,y, span=1)
       },
       aspect = "xy")



## with and without banking

data(sunspots)
spots <- by(sunspots, gl(235, 12, lab = 1749:1983), mean)
plot <- xyplot(spots ~ 1749:1983, xlab = "", type = "l",
               scales = list(x = list(alternating = 2)),
               main = "Average Yearly Sunspots")
print(plot, position = c(0, .3, 1, .9), more = TRUE)
print(update(plot, aspect = "xy", main = "", xlab = "Year"),
      position = c(0, 0, 1, .3))




## Multiple variables in formula for grouped displays

data(iris)
xyplot(Sepal.Length + Sepal.Width ~ Petal.Length + Petal.Width | Species, 
       data = iris, scales = "free", layout = c(2, 2),
       auto.key = list(x = .6, y = .7, corner = c(0, 0)))


## user defined panel functions

data(state)
states <- data.frame(state.x77,
                     state.name = dimnames(state.x77)[[1]], 
                     state.region = state.region) 
xyplot(Murder ~ Population | state.region, data = states, 
       groups = state.name, 
       panel = function(x, y, subscripts, groups)  
       ltext(x = x, y = y, label = groups[subscripts], cex=1,
             fontfamily = "HersheyPlain"))

data(barley)
barchart(yield ~ variety | site, data = barley,
         groups = year, layout = c(1,6),
         ylab = "Barley Yield (bushels/acre)",
         scales = list(x = list(abbreviate = TRUE,
                       minlength = 5)))
barchart(yield ~ variety | site, data = barley,
         groups = year, layout = c(1,6), stack = TRUE, 
         auto.key = list(points = FALSE, rectangles = TRUE, space = "right"),
         ylab = "Barley Yield (bushels/acre)",
         scales = list(x = list(rot = 45)))

data(singer)
bwplot(voice.part ~ height, data=singer, xlab="Height (inches)")
dotplot(variety ~ yield | year * site, data=barley)

dotplot(variety ~ yield | site, data = barley, groups = year,
        key = simpleKey(levels(barley$year), space = "right"),
        xlab = "Barley Yield (bushels/acre) ",
        aspect=0.5, layout = c(1,6), ylab=NULL)

stripplot(voice.part ~ jitter(height), data = singer, aspect = 1,
          jitter = TRUE, xlab = "Height (inches)")
## Interaction Plot
data(OrchardSprays)
bwplot(decrease ~ treatment, OrchardSprays, groups = rowpos,
       panel = "panel.superpose",
       panel.groups = "panel.linejoin",
       xlab = "treatment",
       key = list(lines = Rows(trellis.par.get("superpose.line"),
                  c(1:7, 1)), 
                  text = list(lab = as.character(unique(OrchardSprays$rowpos))),
                  columns = 4, title = "Row position"))
}
\keyword{hplot}

\eof
