"""
utils module (imdb package).

This module provides basic utilities for the imdb package.

Copyright 2004 Davide Alberani <davide.alberani@erlug.linux.it>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""

import re

from _exceptions import IMDbParserError


# The regular expression for the "long" year format of IMDb, like
# "(1998)" and "1986/II", where the optional roman number (that I call
# "imdbIndex" after the slash is used for movies with the same title
# and year of release.
# XXX: probably X, L, C, D and M are far too much! ;-)
re_year_index = re.compile('\(([0-9\?]{4}(/[IVXLCDM]+)?)\)')


def analyze_name(name):
    """Return a dictionary with the name and the optional imdbIndex
    keys, from the given string.
    
    raise an IMDbParserError exception if the name is not valid.
    """
    original_n = name
    name = name.strip()
    res = {}
    imdbIndex = ''
    opi = name.rfind('(')
    cpi = name.rfind(')')
    if opi != -1 and cpi != -1:
        imdbIndex = name[opi+1:cpi].strip()
        name = name[:opi].strip()
    if not name:
        raise IMDbParserError, 'invalid name: "%s"' % str(original_n)
    res.update({'name': name})
    if imdbIndex:
        res.update({'imdbIndex': imdbIndex})
    return res


def build_name(name_dict):
    """Given a dictionary that represents a "long" IMDb name,
    return a string.
    """
    name = name_dict.get('name', '')
    imdbIndex = name_dict.get('imdbIndex')
    if imdbIndex:
        name += ' (%s)' % imdbIndex
    return name


def analyze_title(title):
    """Analyze the given title and return a dictionary with the
    "stripped" title, the kind of the show ("movie", "tv series", etc.),
    the year of production and the optional imdbIndex (a roman number
    used to distinguish between movies with the same title and year).
    
    raise an IMDbParserError exception if the title is not valid.
    """
    original_t = title
    title = title.strip()
    year = ''
    kind = ''
    imdbIndex = ''
    # First of all, search for the kind of show.
    if title.endswith('(mini)'):
        kind = 'tv mini series'
        title = title[:-6]
    elif title.endswith('(TV)'):
        kind = 'tv movie'
        title = title[:-4]
    elif title.endswith('(V)'):
        kind = 'video movie'
        title = title[:-3]
    elif title.endswith('(VG)'):
        kind = 'video game'
        title = title[:-4]
    title = title.strip()
    # Search for the year and the optional imdbIndex (a roman number).
    yi = re_year_index.findall(title)
    if yi:
        last_yi = yi[-1]
        year = last_yi[0]
        if last_yi[1]:
            imdbIndex = last_yi[1][1:]
            year = year[:-len(imdbIndex)-1]
        i = title.rfind('(%s)' % last_yi[0])
        if i != -1:
            title = title[:i-1]
    title = title.strip()
    # This is a tv (mini) series: strip the '"' at the begin and at the end.
    # XXX: strip('"') is not used for compatibility with Python 2.0.
    if title and title[0] == title[-1] == '"':
        if not kind:
            kind = 'tv series'
        title = title[1:-1]
    title = title.strip()
    if not title:
        raise IMDbParserError, 'invalid title: "%s"' % str(original_t)
    # 'kind' is one in ('movie', 'tv series', 'tv mini series', 'tv movie'
    #                   'video movie', 'video game')
    result = {'title': title, 'kind': kind or 'movie'}
    if year and year != '????':
        result.update({'year': year})
    if imdbIndex:
        result.update({'imdbIndex': imdbIndex})
    return result


def build_title(title_dict):
    """Given a dictionary that represents a "long" IMDb title,
    return a string.
    """
    title = title_dict.get('title', '')
    kind = title_dict.get('kind')
    imdbIndex = title_dict.get('imdbIndex')
    year = title_dict.get('year', '????')
    if kind in ('tv series', 'tv mini series'):
        title = '"%s"' % title
    title += ' (%s' % year
    if imdbIndex:
        title += '/%s' % imdbIndex
    title += ')'
    if kind == 'tv mini series':
        title += ' (mini)'
    elif kind == 'tv movie':
        title += ' (TV)'
    elif kind == 'video movie':
        title += ' (V)'
    elif kind == 'video game':
        title += ' (VG)'
    return title.strip()


