"""
imdb package.

This package can be used to retrieve information about a movie
from the IMDb database.
It can fetch data through different media (e.g.: the IMDb web pages,
the e-mail interface, a local installation, etc.)
So far the only data access system available is through the
IMDb's web server.

Copyright 2004 Davide Alberani <davide.alberani@erlug.linux.it>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""

__all__ = ['IMDb', 'IMDbError']

import types
import Movie, Person
from _exceptions import IMDbError, IMDbDataAccessError


# URLs of the main pages for movies and persons.
imdbURL_movie_main = 'http://akas.imdb.com/title/tt%s/'
imdbURL_person_main = 'http://akas.imdb.com/name/nm%s/'


def IMDb(accessSystem='http', *arguments, **keywords):
    """Return an instance of the appropriate class.
    The accessSystem parameter is used to specify the type of
    the preferred access system.
    """
    if accessSystem in ('http', 'web', 'html'):
        from parser.http import IMDbHTTPAccessSystem
        return IMDbHTTPAccessSystem(*arguments, **keywords)
    else:
        raise IMDbError, 'unknown kind of data access system: "%s"' \
                            % str(accessSystem)


class IMDbBase:
    """The base class used to search for a movie/person and to get a
    Movie/Person object.
    
    This class cannot directly fetch data of any kind and so you
    should search the "real" code into a subclass.
    """
    # The name of the preferred access system (should be overridden
    # in the subclasses).
    accessSystem = 'UNKNOWN'

    def get_movie(self, movieID, info=Movie.Movie.default_info):
        """Return a Movie object for the given movieID.
        
        The movieID is something used to univocally identify a movie;
        it can be the imdbID used by the IMDb web server, a file
        pointer, a line number in a file, an ID in a database, etc.

        info is the list of sets of information to retrieve.
        """
        if info == 'all':
            info = Movie.Movie.all_info
        if type(info) not in (types.ListType, types.TupleType):
            info = (info,)
        res = {}
        movie = Movie.Movie(movieID=movieID, accessSystem=self.accessSystem)
        for i in info:
            try:
                method = getattr(self, 'get_movie_%s' % i.replace(' ', '_'))
            except AttributeError:
                raise IMDbDataAccessError, 'unknown information set: "%s"' \
                                            % str(i)
            res.update(method(movieID))
            movie.add_to_current_info(i)
        movie.set_data(res, override=1)
        return movie

    def _search_movie(title, results):
        """Return a list of tuple (movieID, {movieData})"""
        # XXX: for the real implementation, see the method of the
        #      subclass, somewhere under the imdb.parser package.
        return []

    def search_movie(self, title, results=20):
        """Return a list of Movie objects for a query for the given title.
        The results argument is the maximum number of results to return.
        """
        try:
            results = int(results)
        except (ValueError, OverflowError):
            results = 20
        res = self._search_movie(title, results)
        return [Movie.Movie(movieID=mi, movieData=md,
                accessSystem=self.accessSystem) for mi, md in res][:results]

    def get_person(self, personID, info=Person.Person.default_info):
        """Return a Person object for the given personID.
        
        The personID is something used to univocally identify a person;
        it can be the imdbID used by the IMDb web server, a file
        pointer, a line number in a file, an ID in a database, etc.

        info is the list of sets of information to retrieve.
        """
        if info == 'all':
            info = Person.Person.all_info
        if type(info) not in (types.ListType, types.TupleType):
            info = (info,)
        res = {}
        person = Person.Person(personID=personID,
                                accessSystem=self.accessSystem)
        for i in info:
            try:
                method = getattr(self, 'get_person_%s' % i.replace(' ', '_'))
            except AttributeError:
                raise IMDbDataAccessError, 'unknown information set "%s"' \
                                            % str(i)
            res.update(method(personID))
            person.add_to_current_info(i)
        person.set_data(res, override=1)
        # Return a Person object.
        return person

    def _search_person(name, results):
        """Return a list of tuple (personID, {personData})"""
        # XXX: for the real implementation, see the method of the
        #      subclass, somewhere under the imdb.parser package.
        return []

    def search_person(self, name, results=20):
        """Return a list of Person objects for a query for the given name.

        The results argument is the maximum number of results to return.
        """
        try:
            results = int(results)
        except (ValueError, OverflowError):
            results = 20
        res = self._search_person(name, results)
        return [Person.Person(personID=pi, personData=pd,
                accessSystem=self.accessSystem) for pi, pd in res][:results]

    def new_movie(self, *arguments, **keywords):
        """Return a Movie.Movie object."""
        # XXX: not really useful...
        return Movie.Movie(accessSystem=self.accessSystem,
                            *arguments, **keywords)

    def new_person(self, *arguments, **keywords):
        """Return a Person.Person object."""
        # XXX: not really useful...
        return Person.Person(accessSystem=self.accessSystem,
                                *arguments, **keywords)

    def update(self, mop, info=None, override=0):
        """Given a Movie or Person object with only partial information,
        retrieve every accessible information.

        info is the list of sets of information to retrieve.

        If override is set, the information are retrieved and updated
        even if they're already in the object.
        """
        # XXX: should this be a method of the Movie and Person classes?
        mopID = None
        prefix = ''
        if isinstance(mop, Movie.Movie):
            mopID = mop.movieID
            prefix = 'movie'
        elif isinstance(mop, Person.Person):
            mopID = mop.personID
            prefix = 'person'
        else:
            raise IMDbDataAccessError, 'object ' + repr(mop) + \
                        ' is not a Movie or Person instance'
        if not mopID:
            raise IMDbDataAccessError, \
                    'the supplied object has null movieID or personID'
        if mop.accessSystem == self.accessSystem:
            as = self
        else:
            as = IMDb(mop.accessSystem)
        if info is None:
            info = mop.default_info
        elif info == 'all':
            info = mop.all_info
        if type(info) not in (types.ListType, types.TupleType):
            info = (info,)
        res = {}
        for i in info:
            if mop.has_current_info(i) and not override: continue
            try:
                method = getattr(as, 'get_%s_%s' %
                                    (prefix, i.replace(' ', '_')))
            except AttributeError:
                raise IMDbDataAccessError, 'unknown information set "%s"' \
                                            % str(i)
            res.update(method(mopID))
            mop.add_to_current_info(i)
        mop.set_data(res, override=0)

    def get_imdbMovieID(self, movieID):
        """Translate a movieID in an imdbID (the ID used by the IMDb
        web server; must be overridden by the subclass.
        """
        # XXX: for the real implementation, see the method of the
        #      subclass, somewhere under the imdb.parser package.
        return ''

    def get_imdbPersonID(self, personID):
        """Translate a personID in a imdbID (the ID used by the IMDb
        web server; must be overridden by the subclass.
        """
        # XXX: for the real implementation, see the method of the
        #      subclass, somewhere under the imdb.parser package.
        return ''

    def get_imdbURL(self, mop):
        """Return the main IMDb URL for the given Movie or Person object."""
        url = ''
        if mop.accessSystem == self.accessSystem:
            as = self
        else:
            as = IMDb(mop.accessSystem)
        if isinstance(mop, Movie.Movie):
            url = imdbURL_movie_main % as.get_imdbMovieID(mop.movieID)
        elif isinstance(mop, Person.Person):
            url = imdbURL_person_main % self.get_imdbPersonID(mop.personID)
        else:
            raise IMDbDataAccessError, 'object ' + repr(mop) + \
                        ' is not a Movie or Person instance'
        return url

    def get_special_methods(self):
        """Return the special methods defined by the subclass."""
        sm_dict = {}
        base_methods = []
        for name in dir(IMDbBase):
            member = getattr(IMDbBase, name)
            if type(member) is types.MethodType:
                base_methods.append(name)
        for name in dir(self.__class__):
            if name.startswith('_') or name in base_methods or \
                    name.startswith('get_movie_') or \
                    name.startswith('get_person_'):
                continue
            member = getattr(self.__class__, name)
            if type(member) is types.MethodType:
                sm_dict.update({name: member.__doc__})
        return sm_dict


