;; Copyright (C) 1997, 1998, 1999 Free Software Foundation, Inc.
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this software; see the file COPYING.  If not, write to
;; the Free Software Foundation, 59 Temple Place, Suite 330, Boston, MA 02111, USA.
;;
;; As a special exception, the Free Software Foundation gives permission
;; for additional uses of the text contained in its release of GUILE.
;;
;; The exception is that, if you link the GUILE library with other files
;; to produce an executable, this does not by itself cause the
;; resulting executable to be covered by the GNU General Public License.
;; Your use of that executable is in no way restricted on account of
;; linking the GUILE library code into it.
;;
;; This exception does not however invalidate any other reasons why
;; the executable file might be covered by the GNU General Public License.
;;
;; This exception applies only to the code released by the
;; Free Software Foundation under the name GUILE.  If you copy
;; code from other Free Software Foundation releases into a copy of
;; GUILE, as the General Public License permits, the exception does
;; not apply to the code that you add in this way.  To avoid misleading
;; anyone as to the status of such modified files, you must delete
;; this exception notice from them.
;;
;; If you write modifications of your own for GUILE, it is your choice
;; whether to permit this exception to apply to your modifications.
;; If you do not wish that, delete this exception notice.

;;;; "Macro.scm", Support for syntax-rules macros.
;;; Author: Radey Shouman
;;
;; As in SYNTAX-CASE, the identifier ... may be quoted in a
;; SYNTAX-RULES pattern or template as (... ...).
;;
;; THE-MACRO may be used to define macros, eg
;; (define-syntax foo (the-macro and))
;; defines the syntactic keyword FOO to have the same transformer
;; as the macro AND.

(require 'rev2-procedures) ;append!
(require 'record)

(define macro:compile-syntax-rules
  ;; We keep local copies of these standard special forms, otherwise,
  ;; redefining them before they are memoized below can lead to
  ;; infinite recursion.
  (@let-syntax ((lambda (the-macro lambda))
		(let (the-macro let))
		(cond (the-macro cond))
		(if (the-macro if))
		(and (the-macro and))
		(or (the-macro or)))
    (let ((var-rtd (make-record-type '? '(name rank)))
	  (e-pat-rtd (make-record-type '... '(pattern vars)))
	  (rule-rtd (make-record-type 'rule '(pattern inserted template))))

      (define pattern-variable (record-constructor var-rtd '(name rank)))
      (define pattern-variable? (record-predicate var-rtd))
      (define pattern-variable->name
	(let ((acc (record-accessor var-rtd 'name)))
	  (lambda (x) (identifier->symbol (acc x)))))
      (define pattern-variable->rank (record-accessor var-rtd 'rank))

      ;; An ellipsis-pattern is used both for ellipses in patterns and
      ;; ellipses in templates.  In a pattern, VARS is the list of variables
      ;; bound by the ellipsis pattern.  In a template, VARS is the list of
      ;; variables opened by the ellipsis template.

      (define ellipsis-pattern (record-constructor e-pat-rtd '(pattern vars)))
      (define ellipsis-pattern? (record-predicate e-pat-rtd))
      (define ellipsis-pattern->pattern (record-accessor e-pat-rtd 'pattern))
      (define ellipsis-pattern->vars (record-accessor e-pat-rtd 'vars))

      (define rule (record-constructor rule-rtd '(pattern inserted template)))
      (define rule->pattern (record-accessor rule-rtd 'pattern))
      (define rule->inserted (record-accessor rule-rtd 'inserted))
      (define rule->template (record-accessor rule-rtd 'template))

      (define (append2 x y)
	(if (null? y) x
	    (append x y)))

      (define ellipsis?
	(let (($... (renamed-identifier '... #f)))
	  (lambda (x env)
	    (and
	     (identifier? x)
	     (identifier-equal? x $... env)))))

      ;; Yeah, it's quadratically slow.
      (define (duplicates? vars)
	(if (null? vars)
	    #f
	    (if (memq (car vars) (cdr vars))
		(car vars)
		(duplicates? (cdr vars)))))

      (define (compile-pattern literals rule-exp env-def)
	(let recur ((pat (cdar rule-exp))
		    (vars '())
		    (rank 0)
		    (k (lambda (compiled vars)
			 (let ((dup (duplicates? (map car vars))))
			   (if dup
			       (error
				"syntax-rules: duplicate pattern variable:"
				dup " in rule " rule-exp)))
			 (apply rule
				compiled
				(rewrite-template
				 (cadr rule-exp) vars env-def)))))
	  (cond ((null? pat)
		 (k pat vars))
		((identifier? pat)
		 (let ((lit (memq pat literals)))
		   (if lit
		       (k (renamed-identifier pat env-def) vars)
		       (let ((var (pattern-variable pat rank)))
			 (k var (cons (cons pat var) vars))))))
		((pair? pat)
		 (if (and (pair? (cdr pat))
			  (ellipsis? (cadr pat) env-def)
			  (or (null? (cddr pat))
			      (error "bad ellipsis:" pat)))
		     (if (ellipsis? (car pat) env-def)
			 (k (renamed-identifier (car pat) env-def) vars)
			 (recur (car pat) '() (+ rank 1)
				(lambda (comp1 vars1)
				  (k (list
				      (ellipsis-pattern comp1 (map cdr vars1)))
				     (append2 vars1 vars)))))
		     (recur (car pat) '() rank
			    (lambda (comp1 vars1)
			      (recur (cdr pat) vars rank
				     (lambda (comp2 vars2)
				       (k (cons comp1 comp2)
					  (append2 vars1 vars2))))))))
		((vector? pat)
		 (recur (vector->list pat) vars rank
			(lambda (comp vars)
			  (k (list->vector comp) vars))))
		(else
		 (k pat vars)))))

      (define (rewrite-template template vars env-def)
	(let recur ((tmpl template)
		    (rank 0)
		    (inserted '())
		    (k (lambda (compiled inserted opened)
			 (list inserted compiled))))
	  (cond ((null? tmpl)
		 (k tmpl '() '()))
		((identifier? tmpl)
		 (let ((v (assq tmpl vars)))
		   (if v
		       (cond ((= rank (pattern-variable->rank (cdr v)))
			      (k (cdr v) '() (list (cdr v))))
			     ((> rank (pattern-variable->rank (cdr v)))
			      (k (cdr v) '() '()))
			     (else
			      (error "pattern variable rank mismatch:" tmpl
				     " in " template)))
		       (k tmpl (list tmpl) '()))))
		((pair? tmpl)
		 (if (and (pair? (cdr tmpl))
			  (ellipsis? (cadr tmpl) env-def))
		     (if (and (ellipsis? (car tmpl) env-def)
			      (or (null? (cddr tmpl))
				  (error "bad ellipsis:" tmpl)))
			 ;; (... ...) escape
			 (k (car tmpl) (list (car tmpl)) '())
			 (recur (car tmpl) (+ rank 1) '()
				(lambda (comp1 ins1 op1)
				  (if (null? op1)
				      (error "Bad ellipsis:"
					     tmpl " in template " template))
				  (recur (cddr tmpl) rank inserted
					 (lambda (comp2 ins2 op2)
					   (k (cons
					       (ellipsis-pattern comp1 op1)
					       comp2)
					      (append2 ins1 ins2)
					      (append2 op1 op2)))))))
		     (recur (car tmpl) rank '()
			    (lambda (comp1 ins1 op1)
			      (recur (cdr tmpl) rank inserted
				     (lambda (comp2 ins2 op2)
				       (k (cons comp1 comp2)
					  (append2 ins1 ins2)
					  (append2 op1 op2))))))))
		((vector? tmpl)
		 (recur (vector->list tmpl) rank inserted
			(lambda (compiled inserted opened)
			  (k (list->vector compiled) inserted opened))))
		(else
		 (k tmpl '() '())))))


;;; Match EXP to RULE, returning alist of variable bindings or #f.

      (define (match rule exp env-use)
	(let recur ((r (rule->pattern rule))
		    (x (cdr exp)))
	  (cond ((null? r)
		 (and (null? x) '()))
		((pair? r)
		 (if (ellipsis-pattern? (car r))
		     (and
		      (list? x)
		      (let ((pat (ellipsis-pattern->pattern (car r))))
			(let match1 ((x x)
				     (vals '()))
			  (if (null? x)
			      (if (null? vals)
				  (map list (ellipsis-pattern->vars (car r)))
				  (let ((vars (map car (car vals))))
				    (apply map list vars
					   (map (lambda (al)
						  (map cdr al))
						(reverse vals)))))
			      (let ((val (recur pat (car x))))
				(and val
				     (match1 (cdr x) (cons val vals))))))))
		     (and
		      (pair? x)
		      (let ((v1 (recur (car r) (car x))))
			(and v1
			     (let ((v2 (recur (cdr r) (cdr x))))
			       (and v2 (append2 v1 v2))))))))
		((identifier? r)		;literal
		 (and (identifier? x) (identifier-equal? r x env-use) '()))
		((pattern-variable? r)
		 (list (cons r x)))
		((vector? r)
		 (and (vector? x)
		      (recur (vector->list r) (vector->list x))))
		(else
		 (and (equal? r x) '())))))

      (define (substitute-in-template rule vars env-def)
	(let ((ins (map (lambda (id)
			  (cons id (renamed-identifier id env-def)))
			(rule->inserted rule))))
	  (let recur ((tmpl (rule->template rule))
		      (vars vars))
	    (cond ((null? tmpl)
		   tmpl)
		  ((pair? tmpl)
		   (if (ellipsis-pattern? (car tmpl))
		       (let* ((enames (ellipsis-pattern->vars (car tmpl)))
			      (etmpl (ellipsis-pattern->pattern (car tmpl)))
			      (evals (map (lambda (nam)
					    (cdr (assq nam vars)))
					  enames))
			      (n (length (car evals))))
			 (let check ((es (cdr evals)))
			   (if (pair? es)
			       (if (= n (length (car es)))
				   (check (cdr es))
				   (error "syntax-rules: pattern variable length mismatch:"))))
			 (append!
			  (map (lambda (eval)
				 (recur etmpl
					(append!
					 (map cons enames eval)
					 vars)))
			       (apply map list evals))
			  (recur (cdr tmpl) vars)))
		       (cons (recur (car tmpl) vars)
			     (recur (cdr tmpl) vars))))
		  ((identifier? tmpl)
		   (let ((a (assq tmpl ins)))
		     (if a (cdr a) tmpl)))
		  ((pattern-variable? tmpl)
		   (@copy-tree (cdr (assq tmpl vars))))
		  ((vector? tmpl)
		   (list->vector (recur (vector->list tmpl) vars)))
		  (else
		   tmpl)))))

      ;; MACRO:COMPILE-SYNTAX-RULES
      (lambda (x-def env-def)
	(or (and (list? x-def)
		 (< 2 (length x-def))
		 (list? (cadr x-def)))
	    (error "Malformed syntax-rules:" x-def))
	(let ((literals (cadr x-def)))
	  (for-each (lambda (x)
		      (or (identifier? x)
			  (error "Bad literals list:" x-def)))
		    literals)
	  (let ((rules (map (lambda (rule-expr)
			      (or (and (list? rule-expr)
				       (= 2 (length rule-expr))
				       (pair? (car rule-expr)))
				  (error "Bad rule:" rule-expr))
			      (compile-pattern literals rule-expr env-def))
			    (cddr x-def))))

	    (lambda (x-use env-use)
	      (let loop ((rules rules))
		(cond ((null? rules)
		       (error "macro use does not match definition:"
			      x-use))
		      ((match (car rules) x-use env-use)
		       => (lambda (vars)
			    (substitute-in-template (car rules) vars env-def)))
		      (else
		       (loop (cdr rules))))))))))))

(define syntax-rules
  (procedure->syntax
   (lambda (expr env-def)
     (procedure->memoizing-macro
      (macro:compile-syntax-rules expr env-def)))))

;; Explicit renaming macro facility, as in
;; W. Clinger, "Hygienic Macros Through Explicit Renaming"
(define (macro:renaming-transformer-procedure proc env-def)
  (procedure->memoizing-macro
   (lambda (expr env-use)
     (proc (@copy-tree expr)
	   (let ((al '()))
	     (lambda (id)
	       (cond ((not (identifier? id))
		      (error id "non-identifier passed to rename procedure"
			     expr))
		     ((assq id al) => cdr)
		     (else
		      (let ((r-id (renamed-identifier id env-def)))
			(set! al (cons id r-id))
			r-id)))))
	   (lambda (id1 id2)
	     (or (and (identifier? id1)
		      (identifier? id2)
		 (error (if (identifier? id1) id2 id1)
			"non-identifier passed to compare procedure"
			expr)))
	     (identifier-equal? id1 id2 env-use))))))

(define renaming-transformer
  (let ((?transformer
	 (renamed-identifier 'macro:renaming-transformer-procedure #f))
	(?syntax-quote (renamed-identifier 'syntax-quote #f)))
    (procedure->memoizing-macro
     (lambda (exp env-def)
       `(,?transformer ,(cadr exp) (,?syntax-quote ,env-def))))))

(define define-syntax
  (@procedure->primitive-macro
   (macro:compile-syntax-rules 
    '(syntax-rules ()
       ((define-syntax ?name ?val)
	(define ?name (the-macro ?val))))
    '())))

(define let-syntax
  (@procedure->primitive-macro
   (macro:compile-syntax-rules 
    '(syntax-rules ()
       ((let-syntax ((?name ?val) ...) . ?body)
	(@let-syntax ((?name (the-macro ?val)) ...) . ?body)))
    '())))

(define letrec-syntax
  (@procedure->primitive-macro
   (macro:compile-syntax-rules    
    '(syntax-rules ()
       ((letrec-syntax ((?name ?val) ...) . ?body)
	(@letrec-syntax
	    ((?name (the-macro ?val)) ...) . ?body)))
    '())))



;; MACRO:EXPAND would require substantial work.
(define macro:load load)
(define macro:eval eval)
(define (macro:expand . args)
  (load (in-vicinity (implementation-vicinity) "Macexp"))
  (apply macro:expand args))
(provide 'macro)

;; These are not part of the SYNTAX-RULES implementation, but I see
;; no better place to put them:

;;This is a lighter weight gentemp than the one defined in Init*.scm,
;;which interns a new symbol for each call.
(define gentemp
  (let ((name (string->symbol "scm:G")))
    (lambda ()
      (renamed-identifier name #f))))

;; A debugging utility macro that is easy to grep for.
(define-syntax @print
  (syntax-rules (quote)
    ((_ '?arg)
     (begin (display '?arg)
	    (display " ")))
    ((_ ?arg)
     (begin (display '?arg)
	    (display " => ")
	    (let ((x ?arg))
	      (write x)
	      (newline)
	      x)))
    ((_ ?arg1 ?arg ...)
     (begin 
       (@print ?arg1)
       (begin
	 (display " ")
	 (@print ?arg))
       ...))))

;; Better run time error reporting than the version in Init*.scm,
;; also only takes a given car or cdr once.
(define-syntax destructuring-bind
  (syntax-rules ()
    ((_ "PARSE-LLIST" () ?val ?body ?err)
     (if (null? ?val) ?body (?err '() ?val)))
    ((_ "PARSE-LLIST" (?name1 . ?rest) ?val ?body ?err)
     (if (pair? ?val)
	 (let ((carv (car ?val))
	       (cdrv (cdr ?val)))
	   (destructuring-bind "PARSE-LLIST" ?name1 carv
              (destructuring-bind "PARSE-LLIST" ?rest cdrv ?body ?err)
	      ?err))
	 (?err '(?name1 . ?rest) ?val)))
    ((_ "PARSE-LLIST" ?name ?val ?body ?err)
     (let ((?name ?val)) ?body))
    ((_ ?llist ?val ?body1 ?body ...)
     (let ((err (lambda (pat val)
		  (slib:error 'destructuring-bind '?llist
			      val "does not match" pat)))
	   (val ?val))
       (destructuring-bind "PARSE-LLIST" ?llist val
			   ;;Use LET to allow internal DEFINE in body.
			   (let () ?body1 ?body ...)
			   err)))))
