# Messages mixin 

import sys, os, string, re
import rfc822
from mailbox import UnixMailbox, PortableUnixMailbox
from cStringIO import StringIO
from urllib import quote
from AccessControl import getSecurityManager, ClassSecurityInfo
from DateTime import DateTime
import Permissions
from Utils import DLOG, html_quote
from Regexps import fromlineexpr


class MessagesSupport:
    """
    This mix-in class manipulates rfc2822 messages in the page text.

    This class looks for messages in the page text, in mbox/RFC2822
    format, and provides services for parsing and displaying them.
    Everything above the first message is considered "document", the rest
    of the page is considered "messages".

    So that we can recognize messages without extra markup or too many
    false positives, we require each message to begin with BAW's "strict"
    From line regexp from the python mailbox module.

    """
    security = ClassSecurityInfo()

    security.declareProtected(Permissions.View, 'supportsMessages')
    def supportsMessages(self):
        """does this page parse embedded rfc822-ish messages ?"""
        return re.search(r'(?i)(msg)',self.page_type) is not None

    security.declareProtected(Permissions.View, 'hasMessages')
    def hasMessages(self):
        """does this page have one or more embedded rfc822-ish messages ?"""
        return self.messagesPart() != ''

    security.declareProtected(Permissions.View, 'documentPart')
    def documentPart(self):
        """
        This page's text from beginning up to the first message, if any.
        """
        return re.split(fromlineexpr,self.text(),1)[0]

    security.declareProtected(Permissions.View, 'messagesPart')
    def messagesPart(self):
        """
        This page's text from the first message to the end (or '').
        """
        m = re.search(fromlineexpr,self.text())
        if m:
            return m.group() + re.split(fromlineexpr,self.text(),1)[1]
        else:
            return ''

    def makeMessageFrom(self,From,time=None,subject='',
                        message_id=None,in_reply_to=None,body=''):
        """
        Create a nice RFC2822-format message to store in the page.
        """
        zt = self.ZopeTime()
        if not time:
            time = zt.strftime('%Y/%m/%d %H:%M %Z')
        if not message_id:
            message_id = self.messageIdFromTime(zt) #use same time as Date:
        msg = """\
%s
From: %s
Date: %s
Subject: %s
Message-ID: %s
""" % (self.fromLineFrom(From,time)[:-1],From,time,subject,message_id)
        if in_reply_to:
            msg += 'In-reply-to: %s\n' % in_reply_to
        msg += '\n'+body
        return msg

    def fromLineFrom(self,email,date):
        """
        Generate a conformant mbox From line from email and date strings.

        (unless date is unparseable, in which case we omit that part)
        """
        # "email" is in fact a real name or zwiki username - adapt it
        email = re.sub(r'\s','',email) or 'unknown'
        try:
            d = DateTime(date)
            return 'From %s %s %s %d %02d:%02d:%02d %s %d\n' % (
                email,d.aDay(),d.aMonth(),d.day(),d.hour(),
                d.minute(),d.second(),d.timezone(),d.year())
        except (DateTime.SyntaxError,AttributeError,IndexError):
            return 'From %s\n' % email

    security.declareProtected(Permissions.View, 'upgradeMessages')
    def upgradeMessages(self,REQUEST=None):
        """
        Update the format of any messages on this page.
        """
        # upgrade old-style zwiki comments to mbox-style messages ?
        pass

        # add proper From delimiters
        if re.search(r'\n\nFrom: ',self.text()):
            msgs = ''
            mailbox = PortableUnixMailbox(StringIO(
                re.sub(r'\n\nFrom: ',r'\n\nFrom \nFrom: ',self.text())))
            m = mailbox.next()
            while m is not None:
                msgs += self.fromLineFrom(m.get('from'),m.get('date'))
                msgs += string.join(m.headers,'')
                msgs += '\n'
                msgs += m.fp.read()
                m = mailbox.next()
            new = re.split('\n\nFrom: ',self.text(),1)[0]+'\n\n'+msgs
            self.edit(text=new, REQUEST=REQUEST,log='upgraded messages')
            DLOG('upgraded messages on',self.id())

    security.declareProtected(Permissions.View, 'mailbox')
    def mailbox(self):
        """
        Return the messages on this page as an iterator of rfc822.Message.
        """
        # XXX UnixMailbox doesn't like unicode.. work around for now.
        # NB at present unicode may get added:
        # - via user edit
        # - when a message is posted and the local timezone contains unicode
        # - when rename writes a placeholder page (because of the use of _() !)
        try:
            return UnixMailbox(StringIO(self.text()))
        except TypeError:
            DLOG(self.id(),'contains unicode, could not parse messages')
            #DLOG(repr(self.text()))
            return UnixMailbox(StringIO(''))

    security.declareProtected(Permissions.View, 'formattedMessages')
    def formattedMessages(self):
        """
        Prepare our messages for rendering by the text formatting rules.
        """
        t = ''
        mailbox = self.mailbox()
        m = mailbox.next()
        while m is not None:
            t += self.makeCommentHeading(m.get('subject'),
                                         m.get('from'),
                                         m.get('date'),
                                         m.get('message-id'),
                                         m.get('in-reply-to'))
            body = m.fp.read()
            # format citations
            if self.supportsHtml():
                lines = string.split(body,'\n')
                for i in range(len(lines)):
                    if lines[i] and lines[i][0] == '>':
                        lines[i] = '><i>' + lines[i][1:] + '</i>'
                        if i < len(lines)-1 and lines[i+1] and \
                           lines[i+1][0] == '>':
                            lines[i] += '<br />'
                body = string.join(lines,'\n')
            t += body
            m = mailbox.next()
        if t:
            t = self.discussionSeparator() + t
        return t

    security.declareProtected(Permissions.View, 'makeCommentHeading')
    def makeCommentHeading(self, subject, username, time, 
                           message_id=None,in_reply_to=None):
        """
        Create a formatted comment heading that's likely to be readable.

        # XXX templatize ?
        # XXX better solution for RST ?
        """
        heading = '\n\n'
        if self.supportsHtml():
            if message_id:
                # XXX use the message id for linking, but strip the <>
                # and leave it unquoted, browsers can handle it
                heading += '<a name="msg%s"></a>\n' % \
                           re.sub(r'^<(.*)>$',r'\1',message_id)
        if self.supportsHtml() and self.inCMF():
            heading += \
              '<img src="discussionitem_icon.gif" style="border:none; margin:0" />'
        if self.supportsHtml():
            heading += '<b>%s</b> --' % (subject or '...') #more robust
        else:
            heading += '**%s** --' % (subject or '...')
        if username: heading = heading + '[%s], ' % (username)
        if self.supportsHtml():
            if message_id:
                heading += ' <a href="%s#msg%s">%s</a>' % \
                           (self.page_url(),
                            re.sub(r'^<(.*)>$',r'\1',message_id),
                            html_quote(time))
                inreplytobit = '&in_reply_to='+quote(message_id)
            else:
                heading += html_quote(time)
                inreplytobit = ''
            heading += ' <a href="%s?subject=%s%s#bottom">reply</a>'\
                       % (self.page_url(),quote(subject or ''),inreplytobit)
            heading += '<br />\n'
        else:
            heading += html_quote(time)
            heading += '\n\n'
        return heading

    def discussionSeparator(self):
        if self.supportsHtml():
            #return '\n<a name="messages">\n<hr class="messages">\n\n'
            return '\n<a name="messages"></a>\n\n'
        else:
            return '----\n\n'

