% File:          docbook.sl      -*- SLang -*-
%
% Author:        Guido Gonzato, <ggonza@tin.it>
% 
% Version:       0.9.4. Provides a large subset of Docbook 3.1, as described
%                in the LDP Author Guide 
%                <http://www.linuxdoc.org/LDP/LDP-Author-Guide/> and
%                "Docbook: The Definive Guide" <http://www.docbook.org/tdg/>
%                Complete enough for writing Linux HOWTOs, manuals, and much
%                more...
%
% Installation:  if opening a document with .sgml extension doesn't toggle
%                docbook mode on, then insert these lines in your .jedrc:
%
%                  autoload ("docbook_mode", "docbook");
%                  add_mode_for_extension ("docbook", "sgml");
%                     
% Last updated:	 14 February 2001

custom_variable ("SGML_INDENT", 2);
custom_variable ("SGML_Compile_PS_Cmd", "db2ps");
custom_variable ("SGML_Compile_HTML_Cmd", "db2html");
custom_variable ("SGML_View_PS_Cmd", "gv");
custom_variable ("SGML_View_HTML_Cmd", "konqueror"); % kde's browser

static variable NO_PUSH_SPOT = 0;
static variable PUSH_SPOT    = 1;
static variable NO_POP_SPOT  = 0;
static variable POP_SPOT     = 1;
variable DOC_TYPE  = "DOC_TYPE";  % "a" = article, "b" = book
variable SECT_TYPE = "SECT_TYPE"; % "#" = sect#, "s" = section

% ----- %

!if (is_defined ("tex_ldots")) % read tex_insert_quote
  () = evalfile ("texcom");

#iffalse
define insert_spaces (num_spaces)
{
  variable i;
  for (i = 1; i < num_spaces; i++)
    insert (" ");
}
#endif

static define sgml_insert_pair_around_region (left, right)
{
   check_region (1);
   exchange_point_and_mark ();
   insert (left);
   exchange_point_and_mark ();
   insert (right);
   pop_spot ();
   pop_mark_0 ();
}

define sgml_comment ()		% ^C;
{
  sgml_insert_pair_around_region ("<!-- ", "-->");
}  

define sgml_skip_tag ()		% ^C^F
{
  go_right_1 ();
  () = fsearch_char ('<');
}

define sgml_bskip_tag ()	% ^C^B
{
  () = bsearch_char ('<');
}  

static define sgml_insert_tags (tag1, tag2, do_push_spot, do_pop_spot)
{
  % if a region is defined, insert the tags around it
  if (markp () ) {
    sgml_insert_pair_around_region (tag1, tag2);
    return;
  }
  insert (tag1);
  if (do_push_spot)
    push_spot ();
  insert (tag2);
  if (do_pop_spot)
    pop_spot ();
}

define sgml_insert_tag (tag, do_push_spot, do_pop_spot)
{
  variable tag1, tag2;
  tag1 = sprintf ("<%s>", tag);
  tag2 = sprintf ("</%s>", tag);
  sgml_insert_tags (tag1, tag2, do_push_spot, do_pop_spot);
}

static define sgml_insert_tag_with_newline (tag, do_push_spot, do_pop_spot)
{
  variable tag1, tag2;
  tag1 = sprintf ("<%s>", tag);
  tag2 = sprintf ("</%s>\n", tag);
  sgml_insert_tags (tag1, tag2, do_push_spot, do_pop_spot);
}

static define sgml_paragraph_separator ()
{
  bol_skip_white ();
  eolp () or ffind ("<para>") or ffind ("</para>")
    or ffind ("<Para>") or ffind ("</Para>")
      or ffind ("<PARA>") or ffind ("</PARA>");
}

% let's start

% Section stuff

define sgml_para (do_push_spot) % ^CP
{
  variable col = what_column ();
  if (markp () ) {
    sgml_insert_pair_around_region ("<Para>\n", "</Para>\n");
    return;
  }
  insert ("<Para>\n");
  insert_spaces (col + SGML_INDENT);
  if (do_push_spot)
    push_spot ();
  insert ("\n");
  insert_spaces (col);
  insert ("</Para>");
}

define sgml_title (tag, do_push_spot)
{
  sgml_insert_tag_with_newline (tag, do_push_spot, NO_POP_SPOT);
}

% define sgml_title (tag, do_push_spot)
% {
%   variable tag1, tag2;
%   tag1 = sprintf ("<%s>", tag);
%   tag2 = sprintf ("</%s>\n", tag);
%   if (markp () ) {
%     sgml_insert_pair_around_region (tag1, tag2);
%     return;
%   }
%   insert (tag1);
%   if (do_push_spot)
%     push_spot ();
%   insert (tag2); % pop_spot () elsewhere
% }

static define sgml_section_skel (what_tag, do_push_spot)
{
  variable col = what_column ();
  vinsert ("<%s>\n", what_tag);
  insert_spaces (col);
  sgml_title ("Title", do_push_spot);
  insert ("\n");
  insert_spaces (col);
  sgml_para (NO_PUSH_SPOT);
  insert ("\n\n");
  insert_spaces (col);
  vinsert ("</%s>", what_tag);
  if (do_push_spot)
    pop_spot ();
}

define sgml_chapter (do_push_spot)
{
  sgml_section_skel ("Chapter", do_push_spot);
}

define sgml_appendix (do_push_spot)
{
  sgml_section_skel ("Appendix", do_push_spot);
}

define sgml_section ()
{
  variable col = what_column ();
  if ( ("#" == get_blocal_var (SECT_TYPE) ) and 
      ("a" == get_blocal_var (DOC_TYPE)) ) {
    beep ();
    error ("`section' not allowed in this environment!");
    % !!! what about <section> and <sectX> in Book doctype?
    return ();
  }
  set_blocal_var ("s", SECT_TYPE);
  sgml_section_skel ("Section", PUSH_SPOT);
}

define sgml_simple_section ()
{
  sgml_section_skel ("SimpleSection", PUSH_SPOT);
}

define sgml_sect (do_push_spot, level_str) % ^CS
{
  variable level_int;
  variable ok = 0;

  if ( ("s" == get_blocal_var (SECT_TYPE) ) and
      ("a" == get_blocal_var (DOC_TYPE) ) ) {
    beep ();
    error ("`sect' not allowed in this environment!");
    return ();
  }
  set_blocal_var ("#", SECT_TYPE);
  !if (strcmp ("0", level_str) )
    while (0 == ok) {
      level_str = read_mini ("Section level (1..5)? ", Null_String, "1");
      level_int = level_str [0] - '0';
      if ( (level_int > 0) and (level_int < 6) )
        ok = 1;
      !if (ok) {
        beep ();
        message ("Wrong value! Only 1..5 allowed. ");
      }
    }
  
  insert (sprintf ("<Sect%s id=\"", level_str));
  if (do_push_spot)
    push_spot ();
  insert ("\">\n");
  sgml_title ("Title", NO_PUSH_SPOT);
  insert ("\n");
  sgml_para (NO_PUSH_SPOT);
  insert (sprintf ("\n\n\n</Sect%s>", level_str));
  if (do_push_spot)
    pop_spot ();
}

define sgml_formalpara ()
{
  sgml_section_skel ("FormalPara", PUSH_SPOT);
}

% Header stuff

define sgml_revision ()
{
  variable col = what_column ();
  insert ("<Revision>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<RevNumber>");
  push_spot (); % pop_spot () later
  insert ("</RevNumber>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("Date", NO_PUSH_SPOT, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("AuthorInitial", NO_PUSH_SPOT, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  insert ("<RevMark>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("\n");
  insert_spaces (col + SGML_INDENT);
  insert ("</RevMark>\n");
  insert_spaces (col);
  insert ("</Revision>\n");
}

define sgml_revision_history ()
{
  variable col = what_column ();
  insert ("<RevHistory>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_revision (); % the spot is pushed
  insert_spaces (col);
  insert ("</RevHistory>");
  pop_spot ();
}

% Author stuff

static define sgml_firstname (do_push_spot, do_pop_spot)
{
  sgml_insert_tag_with_newline ("FirstName", do_push_spot, do_pop_spot);
}

static define sgml_surname (do_push_spot, do_pop_spot)
{
  sgml_insert_tag_with_newline ("SurName", do_push_spot, do_pop_spot);
}

static define sgml_othername (do_push_spot, do_pop_spot)
{
  sgml_insert_tag_with_newline ("OtherName", do_push_spot, do_pop_spot);
}

static define sgml_affiliation (do_push_spot, do_pop_spot)
{
  variable col = what_column ();
  insert ("<Affiliation>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("ShortAffil", do_push_spot, do_pop_spot);
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("JobTitle", do_push_spot, do_pop_spot);
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("OrgName", do_push_spot, do_pop_spot);
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("OrgDiv", do_push_spot, do_pop_spot);
  insert_spaces (col);
  insert ("</Affiliation>\n");
}

static define sgml_honorific (do_push_spot, do_pop_spot)
{
  sgml_insert_tag_with_newline ("Honorific", do_push_spot, do_pop_spot);
}

static define sgml_author (do_push_spot, do_pop_spot)
{
  variable col = what_column ();
  insert ("<Author>\n");
  if (do_push_spot)
    push_spot ();
  insert_spaces (col + SGML_INDENT);
  sgml_honorific (do_push_spot, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_firstname (do_push_spot, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_othername (do_push_spot, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_surname (do_push_spot, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_affiliation (do_push_spot, NO_POP_SPOT);
  insert_spaces (col);
  insert ("</Author>\n");
  if (do_pop_spot)
    pop_spot ();
}

define sgml_template ()
{
  variable col, ok, type, is_article = 0;

  col = what_column ();
  ok = 0;
  while (0 == ok) {
    type = 
      read_mini ("Document type (Article, Book)? ", Null_String, "a");
    if ( (type [0] == 'a') or (type [0] == 'b') )
      ok = 1;
    !if (ok) {
      beep ();
      message ("Wrong! 'a' or 'b'! ");
    }
  }
  if (type [0] == 'a') {
    set_blocal_var ("a", DOC_TYPE);
    type = "Article";
    is_article = 1;
  }
  else {
    set_blocal_var ("b", DOC_TYPE);
    type = "Book";
  }
  vinsert 
    ("<!DOCTYPE %s PUBLIC \"-//OASIS//DTD DocBook V3.1//EN\">", type);
  if (is_article)
    vinsert ("\n\n<%s lang=\"en\">\n<%s>\n\n", type, "ArtHeader");
  else
    vinsert ("\n\n<%s lang=\"en\">\n<%s>\n\n", type, "BookInfo");
  
  insert_spaces (col + SGML_INDENT);
  sgml_title ("Title", PUSH_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("SubTitle", NO_PUSH_SPOT, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_author (NO_PUSH_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("Email", NO_PUSH_SPOT, NO_POP_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_insert_tag_with_newline ("PubDate", NO_PUSH_SPOT, NO_POP_SPOT);
  insert ("\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Abstract>\n");
  insert_spaces (col + 2 * SGML_INDENT);
  sgml_para (NO_PUSH_SPOT);
  insert ("\n");
  insert_spaces (col + SGML_INDENT);
  insert ("</Abstract>\n");
  
  if (is_article) {
    vinsert ("\n</%s>\n\n", "ArtHeader");
    insert ("<!-- your sections here... -->\n");
    % sgml_sect (NO_PUSH_SPOT, "1");
  }
  else {
    vinsert ("\n</%s>\n\n\n", "BookInfo");
    insert ("<Preface>\n<Title></Title>\n\n");
    sgml_para (NO_PUSH_SPOT);
    insert ("\n</Preface>\n\n");
    sgml_chapter (NO_PUSH_SPOT);
  }
  vinsert ("\n\n\n</%s>", type);
  pop_spot ();
  vmessage ("Document is %s", type);
}

define sgml_copyright ()
{
  variable col = what_column ();
  insert_spaces (col);
  insert ("<Copyright>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Year>");
  push_spot ();
  insert ("</Year>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Holder></Holder>\n");
  insert_spaces (col);
  insert ("</Copyright>\n");
  pop_spot ();
}

% character styles

define sgml_directory ()
{
  insert ("<FileName id=\"Directory\">");
  push_spot ();
  insert ("</FileName>");
  pop_spot ();
}

% keys

define sgml_keycombo (do_push_spot)
{
  insert ("<KeyCombo><KeyCap>");
  if (do_push_spot)
    push_spot ();
  insert ("</KeyCap><KeyCap></KeyCap></KeyCombo>");
  if (do_push_spot)
    pop_spot ();
}

% gui/menus

define sgml_menuchoice ()
{
  variable col = what_column ();
  insert ("<MenuChoice>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<ShortCut>\n");
  insert_spaces (col + 2 * SGML_INDENT);
  sgml_keycombo (NO_PUSH_SPOT);
  insert ("\n");
  insert_spaces (col + SGML_INDENT);
  insert ("</ShortCut>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<GUIMenu>");
  push_spot ();
  insert ("</GUIMenu>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<GUIMenuItem></GUIMenuItem>\n");
  insert_spaces (col);
  insert ("</MenuChoice>\n");
  pop_spot ();
}

% computer-related

define sgml_arg_plain (do_push_spot, do_pop_spot)
{
  sgml_insert_tag ("Arg", do_push_spot, do_pop_spot);
}

define sgml_arg (do_push_spot, do_pop_spot)
{
  variable ch, rep, choice, tag1, tag2;
  
  choice = "";
  rep = "";
  
  flush ("Choice:  Req  Plain (Enter=nothing) ");
  ch = tolower (getkey ());
  switch (ch)
    {case 'r': choice =  "Choice=req"; }
    {case 'p': choice =  "Req=plain"; }

  flush ("Rep: :  Repeat (Enter=nothing) ");
  ch = tolower (getkey ());
  switch (ch)
    {case 'r': rep =  "rep = repeat"; }
  
  tag1 = sprintf ("<Arg %s %s>", choice, rep );
  tag2 = "</Arg>";
  sgml_insert_tags (tag1, tag2, do_push_spot, do_pop_spot);
}

% static define sgml_replaceable (do_push_spot, do_pop_spot)
% {
%   sgml_insert_tag ("Replaceable", do_push_spot, do_pop_spot);
% }

% define sgml_sbr ()
% {
%   insert ("<Sbr>\n");
% }

define sgml_group ()
{
  variable col = what_column ();
  insert ("<Group>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_arg (PUSH_SPOT, NO_POP_SPOT);
  insert ("\n");
  insert_spaces (col);
  insert ("</Group>\n");
  pop_spot ();
}

define sgml_cmdsynopsis ()
{
  variable col = what_column ();
  insert ("<CmdSynopsis>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Command>");
  push_spot ();
  insert ("</Command>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_arg (NO_PUSH_SPOT);
  insert ("\n");
  insert_spaces (col);
  insert ("</CmdSynopsis>\n");
  pop_spot ();
}

% environments, sort of

define sgml_blockquote ()
{
  variable col = what_column ();
  insert ("<BlockQuote>\n");
  insert_spaces (col);
  insert ("<Attribution>");
  push_spot ();
  insert ("</Attribution>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_para (NO_PUSH_SPOT);
  insert ("\n");
  insert_spaces (col);
  insert ("</BlockQuote>");
  pop_spot ();
}

define sgml_env (what)
{
  variable col = what_column ();
  vinsert ("<%s>\n", what);
  insert_spaces (col);
  !if (strcmp ("Example", what)) {
    sgml_title ("Title", PUSH_SPOT);
    insert_spaces (col + SGML_INDENT);
    sgml_para (NO_PUSH_SPOT);
    insert ("\n");
  }
  else {
    insert_spaces (col + SGML_INDENT);
    sgml_para (PUSH_SPOT);
    insert ("\n");
  }
  insert_spaces (col);
  vinsert ("</%s>\n", what);
  pop_spot ();
}

% % index
% 
% define sgml_index ()
% {
%   insert ("<indexterm>\n");
%   insert ("<primary>");
%   push_spot ();
%   insert ("</primary>\n");
%   insert ("</indexterm>");
%   pop_spot ();
% }

% links

define sgml_id ()
{
  insert ("Id=\"");
  push_spot ();
  insert ("\"");
}

define sgml_anchor ()
{
  insert ("<Anchor ");
  sgml_id (); % spot pushed
  insert (">");
  pop_spot ();
}

define sgml_url ()
{
  insert ("<Ulink URL=\"");
  push_spot ();
  insert ("\"><CiteTitle></CiteTitle></Ulink>");
  pop_spot ();
}

define sgml_xref ()
{
  insert ("<Xref LinkEnd=\"");
  push_spot ();
  insert ("\">");
  pop_spot ();
}

% lists

define sgml_listitem (do_push_spot)
{
  variable col = what_column ();
  insert ("<ListItem>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_para (do_push_spot);
  insert ("\n");
  insert_spaces (col);
  insert ("</ListItem>");
}

define sgml_itemized_list ()
{
  variable col = what_column ();
  insert ("<ItemizedList>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_listitem (PUSH_SPOT);
  insert ("\n");
  insert_spaces (col);
  insert ("</ItemizedList>\n");
  pop_spot ();
}

define sgml_ordered_list ()
{
  variable col = what_column ();
  insert ("<OrderedList>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<ListItem>\n");
  insert_spaces (col + 2 * SGML_INDENT);
  insert ("<Para>\n");
  insert_spaces (col + 2 * SGML_INDENT);
  push_spot ();
  insert ("\n");
  insert_spaces (col + 2 * SGML_INDENT);
  insert ("</Para>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("</ListItem>\n");
  insert ("</OrderedList>\n");
  pop_spot ();
}

define sgml_varlistentry ()
{
  variable col = what_column ();
  insert ("<VarListEntry>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Term>");
  push_spot (); % we'll do pop_spot () later
  insert ("</Term>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<ListItem>\n");
  insert_spaces (col + 2 * SGML_INDENT);
  sgml_para (NO_PUSH_SPOT);
  insert ("\n");
  insert_spaces (col + SGML_INDENT);
  insert ("</ListItem>\n");
  insert_spaces (col);
  insert ("</VarListEntry>\n");
}

define sgml_variable_list ()
{
  variable col = what_column ();
  insert ("<VariableList>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_varlistentry ();
  insert ("</VariableList>\n");
  pop_spot ();
}

define sgml_seglistitem (do_push_spot)
{
  variable col = what_column ();
  insert ("<SegListItem>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Seg>");
  if (do_push_spot)
    push_spot ();
  insert ("</Seg><Seg></Seg>\n");
  insert_spaces (col);
  insert ("</SegListItem>");
}

define sgml_segmented_list ()
{
  variable col = what_column ();
  insert ("<SegmentedList>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_title ("Title", PUSH_SPOT);
  insert_spaces (col + SGML_INDENT);
  insert ("<SegTitle></SegTitle>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<SegTitle></SegTitle>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_seglistitem (NO_PUSH_SPOT);
  insert ("\n");
  insert_spaces (col);
  insert ("</SegmentedList>\n");
  pop_spot ();
}

static define sgml_step (do_push_spot)
{
  variable col = what_column ();
  insert ("<Step>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_para (do_push_spot);
  insert ("\n");
  insert_spaces (col);
  insert ("</Step>");
  if (1 == do_push_spot)
    pop_spot ();
}

define sgml_procedure ()
{
  variable col = what_column ();
  insert ("<Procedure>\n");
  insert_spaces (col + SGML_INDENT);
  sgml_title ("Title", PUSH_SPOT);
  insert_spaces (col + SGML_INDENT);
  sgml_step (NO_PUSH_SPOT);
  insert ("\n");
  insert_spaces (col);
  insert ("</Procedure>\n");
  pop_spot ();
}

% tables

define sgml_table_row (table_col_int, do_push_spot)
{
  variable i, col = what_column ();
  
  insert ("<Row>\n");
  for (i = 0; i < table_col_int; i++) {
    insert_spaces (col + SGML_INDENT);
    insert ("<Entry>");
    if ( (do_push_spot) and (0 == i) )
      push_spot ();
    insert ("</Entry>\n");
  }
  insert_spaces (col);
  insert ("</Row>");
}

define sgml_align ()
{
  variable ch, align;
  
  flush 
    ("Align (Center cHar Justify Left Right)? ");
  ch = tolower (getkey ());
  switch (ch)
    {case 'c': align = "Center"; }
    {case 'h': align = "Char"; }
    {case 'j': align = "Justify"; }
    {case 'l': align = "Left"; }
    {case 'r': align = "Right"; }
  
  vinsert ("Align=\"%s\"", align);

}

% static
static variable table_columns = 4;

define sgml_table (informal_table)
{
  variable col = what_column ();
  variable i, ch, frame, type_of_table, table_col_str, ok;
  
  if (informal_table)
    type_of_table = "InformalTable";
  else
    type_of_table = "Table";
  
  flush 
    ("Frame (All, Bottom, None, Sides, Top, tOpbot)? ");
  ch = tolower (getkey ());
  switch (ch)
    {case 'a': frame = "All"; }
    {case 'b': frame = "Bottom"; }
    {case 'n': frame = "None"; }
    {case 's': frame = "Sides"; }
    {case 't': frame = "Top"; }
    {case 'o': frame = "TopBot"; }
  
  ok = 0;
  while (0 == ok) {
    table_col_str = read_mini ("Columns? ", Null_String, "4");
    table_columns = table_col_str [0] - '0';
    if ( (table_columns > 1) and (table_columns < 10) )
      ok = 1;
    !if (ok) {
      beep ();
      message ("Wrong value! ");
    }
  }
  
  vinsert ("<%s Frame=\"%s\">\n", type_of_table, frame);
  !if (informal_table)
    sgml_title ("Title", PUSH_SPOT);
  else
    push_spot ();
  insert_spaces (col + SGML_INDENT);
  insert (sprintf ("\n<TGroup Cols=\"%s\">\n", table_col_str));
  for (i = 0; i < table_columns; i++) {
    insert_spaces (col + SGML_INDENT);
    vinsert 
      ("<ColSpec ColName=\"col%s\" Align=\"Center\">\n", string (i+1) );
  }
  !if (informal_table) {
    insert_spaces (col + SGML_INDENT);
    insert ("<THead>\n");
    insert_spaces (col + SGML_INDENT);
    
    sgml_table_row (table_columns, NO_PUSH_SPOT);
    
    insert ("\n");
    insert_spaces (col + SGML_INDENT);
    insert ("</THead>\n");
    
    insert_spaces (col + SGML_INDENT);
    insert ("<TFoot>\n");
    insert_spaces (col + SGML_INDENT);
    insert ("<Row>\n");
    for (i = 0; i < table_columns; i++) {
      insert_spaces (col + 2 * SGML_INDENT);
      insert ("<Entry></Entry>\n");
    }
    insert_spaces (col + SGML_INDENT);
    insert ("</Row>\n");
    insert_spaces (col + SGML_INDENT);
    insert ("</TFoot>\n");
  }

  insert_spaces (col + SGML_INDENT);
  insert ("<TBody>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Row>\n");
  for (i = 0; i < table_columns; i++) {
    insert_spaces (col + 2 * SGML_INDENT);
    insert ("<Entry></Entry>\n");
  }
  insert_spaces (col + SGML_INDENT);
  insert ("</Row>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("</TBody>\n");

  insert_spaces (col + SGML_INDENT);
  insert ("</TGroup>\n");
  insert_spaces (col);
  vinsert ("</%s>\n", type_of_table);
  pop_spot ();
}

% figures

define sgml_figure ()
{
  variable i, col = what_column ();
  insert ("<Figure>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<Title>");
  push_spot ();
  insert ("</Title>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("<MediaObject>\n");
  for (i = 0; i < 2; i++) {
    insert_spaces (col + 2 * SGML_INDENT);
    insert ("<ImageObject>\n");
    insert_spaces (col + 3 * SGML_INDENT);
    insert ("<ImageData FileRef=\"\" Format=\"\">\n");
    insert_spaces (col + 2 * SGML_INDENT);
    insert ("</ImageObject>\n");
  }
  insert_spaces (col + 2 * SGML_INDENT);
  insert ("<TextObject>\n");
  insert_spaces (col + 3 * SGML_INDENT);
  insert ("<Phrase></Phrase>\n");
  insert_spaces (col + 2 * SGML_INDENT);
  insert ("</TextObject>\n");
  insert_spaces (col + SGML_INDENT);
  insert ("</MediaObject>\n");
  insert_spaces (col);
  insert ("</Figure>\n");
  pop_spot ();

}

% let's finish

$1 = "docbook";
!if (keymap_p ($1))
  make_keymap ($1);

definekey_reserved ("sgml_bskip_tag", "^B", $1);
definekey_reserved ("sgml_skip_tag", "^F", $1);
definekey_reserved ("sgml_para (1)", "p", $1);
definekey_reserved ("sgml_para (1)", "P", $1);
definekey_reserved ("insert (\"&dollar;\")", "$", $1);
definekey_reserved ("insert (\"&amp;\")", "&", $1);
definekey_reserved ("sgml_comment", ";", $1);
definekey_reserved ("insert (\"&hellip;\")", ".", $1);
definekey_reserved ("insert (\"&lt;\")", "<", $1);
definekey_reserved ("insert (\"&gt;\")", ">", $1);

definekey ("tex_insert_quote", "\"", $1);
definekey ("tex_insert_quote", "'", $1);
create_syntax_table ($1);
set_syntax_flags ($1, 8);

% defining keywords is not necessary, since all the highlighting is
% done by the second and third define_syntax (). Rough, but fairly nice.

define_syntax ("\"([{<", "\")]}>", '(', $1);
define_syntax ('<', '\\', $1);
define_syntax ('&', '\\', $1);
define_syntax ("0-9A-Za-z>/!", 'w', $1);
define_syntax ("<>", '<', $1);
define_syntax ("<!-", "-->", '%', $1);

define insert_sgml_screen ()
{
  % this function is redundant, but I can't find a way to pass
  % Width="80" through menu_append_item
  sgml_insert_tags ("<Screen Width=\"80\">\n", "</Screen>\n", 
		    PUSH_SPOT, NO_POP_SPOT);
}

define sgml_convert_to_ps ()
{
  variable mrk, cmd;
  mrk = create_user_mark;
  cmd = sprintf ("%s %s", SGML_Compile_PS_Cmd, mrk.buffer_name);
  cmd = read_mini ("Convert command: ", Null_String, cmd);
  compile (cmd);
}

define sgml_convert_to_html ()
{
  variable mrk, cmd;
  mrk = create_user_mark;
  cmd = sprintf ("%s %s", SGML_Compile_HTML_Cmd, mrk.buffer_name);
  cmd = read_mini ("Convert command: ", Null_String, cmd);
  compile (cmd);
}

define sgml_view (type) % "ps" or "html"
{
  variable mrk, cmd, tmp;
  mrk = create_user_mark;
  tmp = mrk.buffer_name;
  (tmp, ) = strreplace (tmp, "sgml", type,  -1);
  !if (strcmp (type, "ps"))
    cmd = sprintf ("%s %s &", SGML_View_PS_Cmd, tmp);
  else
    cmd = sprintf ("%s %s &", SGML_View_HTML_Cmd, tmp);
  cmd = read_mini ("View with: ", Null_String, cmd);
#ifdef UNIX
  system (cmd);
#else
  run_shell_cmd (cmd);
#endif
}


static define init_menu (menu)
{
  % character
  menu_append_popup (menu, "&Character");
  $1 = sprintf ("%s.&Character", menu);
  menu_append_item ($1, "<&Acronym>", "sgml_insert_tag (\"Acronym\", 1, 1)");
  menu_append_item ($1, "<&Citation>", 
		    "sgml_insert_tag (\"Citation\", 1, 1)");
  menu_append_item ($1, "<&Emphasis>", 
		    "sgml_insert_tag (\"Emphasis\", 1, 1)");
  menu_append_item ($1, "<&FirstTerm>", 
		    "sgml_insert_tag (\"FirstTerm\", 1, 1)");
  menu_append_item ($1, "<Foot&Note>", 
		    "sgml_insert_tag (\"FootNote\", 1, 1)");
  menu_append_item ($1, "<&SuperScript>", 
		    "sgml_insert_tag (\"SuperScript\", 1, 1)");
  menu_append_item ($1, "<S&ubScript>", 
		    "sgml_insert_tag (\"SubScript\", 1, 1)");
  % computer
  menu_append_popup (menu, "C&omputer");
  $1 = sprintf ("%s.C&omputer", menu);
  menu_append_item ($1, "<&Application>", 
		    "sgml_insert_tag (\"Application\", 1, 1)");
  menu_append_item ($1, "<Mouse&Button>", 
		    "sgml_insert_tag (\"MouseButton\", 1, 1)");
  menu_append_item ($1, "<&Command>", "sgml_insert_tag (\"Command\", 1, 1)");
  menu_append_item ($1, "&Directory", "sgml_directory ()");
  menu_append_item ($1, "<&EnVar>", 
		    "sgml_insert_tag (\"EnVar\", 1, 1)");
  menu_append_item ($1, "<&Filename>", 
		    "sgml_insert_tag (\"Filename\", 1, 1)");
  menu_append_item ($1, "<Program&Listing>", 
		    "sgml_insert_tag (\"ProgramListing\", 1, 1)");
  menu_append_item ($1, "<Computer&Output>", 
		    "sgml_insert_tag (\"ComputerOutput\", 1, 1)");
  menu_append_item ($1, "<C&onstant>", 
		    "sgml_insert_tag (\"Constant\", 1, 1)");
  menu_append_item ($1, "<&Prompt>", "sgml_insert_tag (\"Prompt\", 1, 1)");
  menu_append_item ($1, "<&Screen>", "insert_sgml_screen ()");
  menu_append_item ($1, "<&UserInput>", 
		    "sgml_insert_tag (\"UserInput\", 1, 1)");
  menu_append_item ($1, "<&VarName>", 
		    "sgml_insert_tag (\"VarName\", 1, 1)");
  % computer: popups
  menu_append_popup ($1, "&Menu");
  menu_append_popup ($1, "&Keys");
  menu_append_popup ($1, "CmdS&ynopsis");
  % computer/menu
  $1 = sprintf ("%s.C&omputer.&Menu", menu);
  menu_append_item ($1, "<&Accel>", "sgml_insert_tag (\"Accel\", 1, 1)");
  menu_append_item ($1, "<Menu&Choice>", "sgml_menuchoice ()");
  menu_append_item ($1, "<&GUIMenu>", "sgml_insert_tag (\"GUIMenu\", 1, 1)");
  menu_append_item ($1, "<GUIMenu&Item>", 
		    "sgml_insert_tag (\"GUIMenuItem\", 1, 1)");
  % computer/keys
  $1 = sprintf ("%s.C&omputer.&Keys", menu);
  menu_append_item ($1, "<Key&Cap>", "sgml_insert_tag (\"KeyCap\", 1, 1)");
  menu_append_item ($1, "<Key&Sym>", "sgml_insert_tag (\"KeySym\", 1, 1)");
  menu_append_item ($1, "<KeyC&ode>", "sgml_insert_tag (\"KeyCode\", 1, 1)");
  menu_append_item ($1, "<KeyCo&mbo>", "sgml_keycombo (1)");
  % computer/cmdsynopsis
  $1 = sprintf ("%s.C&omputer.CmdS&ynopsis", menu);
  menu_append_item ($1, "<&Arg>", "sgml_arg (1, 1)");
  menu_append_item ($1, "<&Break>", "insert (\"<Sbr>\n\");");
  menu_append_item ($1, "<Cmd&Synopsis>", "sgml_cmdsynopsis ()");
  menu_append_item ($1, "&Plain <Arg>", "sgml_arg_plain (1, 1)");
  menu_append_item ($1, "<&Group>", "sgml_group ()");
  menu_append_item ($1, "<&Replaceable>", 
		    "sgml_insert_tag (\"Replaceable\", 1, 1)");
  % environment
  menu_append_popup (menu, "&Environment");
  $1 = sprintf ("%s.&Environment", menu);
  menu_append_item ($1, "<&Blockquote>", "sgml_blockquote ()");
  menu_append_item ($1, "<&Caution>", "sgml_env (\"Caution\")");
  menu_append_item ($1, "<&Example>", "sgml_env (\"Example\")");
  menu_append_item ($1, "<&Important>", "sgml_env (\"Important\")");
  menu_append_item ($1, "<&Note>", "sgml_env (\"Note\")");
  menu_append_item ($1, "<&Tip>", "sgml_env (\"Tip\")");
  menu_append_item ($1, "<&Warning>", "sgml_env (\"Warning\")");
  % figure
  menu_append_item (menu, "<&Figure>", "sgml_figure ()");
  % header
  menu_append_popup (menu, "&Header");
  $1 = sprintf ("%s.&Header", menu);
  menu_append_item ($1, "<&Ackno>", "sgml_insert_tag (\"Ackno\", 1, 1)");
  menu_append_item ($1, "<A&uthor>", "sgml_author ()");
  menu_append_item ($1, "<&Copyright>", "sgml_copyright ()");
  menu_append_item ($1, "<Rev&History>", "sgml_revision_history ()");
  menu_append_item ($1, "<&Revision>", "sgml_revision (); pop_spot ()");
  menu_append_item ($1, "&Template", "sgml_template ()");
  % links
  menu_append_popup (menu, "Lin&k");
  $1 = sprintf ("%s.Lin&k", menu);
  menu_append_item ($1, "<&Anchor>", "sgml_anchor ()");
  menu_append_item ($1, "<&Email>", "sgml_insert_tag (\"Email\")");
  menu_append_item ($1, "<&Id>", "sgml_id (); pop_spot ()");
  menu_append_item ($1, "<&Url>", "sgml_url ()");
  menu_append_item ($1, "<&Xref>", "sgml_xref ()");
  % lists
  menu_append_popup (menu, "&List");
  $1 = sprintf ("%s.&List", menu);
  menu_append_item ($1, "<&ItemizedList>", "sgml_itemized_list ()");
  menu_append_item ($1, "<&Listitem>", "sgml_listitem (1); pop_spot ()");
  menu_append_item ($1, "<&OrderedList>", "sgml_ordered_list ()");
  menu_append_item ($1, "<&SegmentedList>", "sgml_segmented_list ()");
  menu_append_item ($1, "<&VariableList>", "sgml_variable_list ()");
  menu_append_item ($1, "<VarList&Entry>", 
		    "sgml_varlistentry (); pop_spot ()");
  menu_append_item ($1, "<Se&gListEntry>", 
		    "sgml_seglistitem  (1); pop_spot ()");
  menu_append_item ($1, "<&Procedure>", "sgml_procedure ()");
  menu_append_item ($1, "<S&tep>", "sgml_step (1)");
  % paragraph
  menu_append_item (menu, "<&Para>", "sgml_para (1); pop_spot ()");
  % sections
  menu_append_popup (menu, "&Section");
  $1 = sprintf ("%s.&Section", menu);
  menu_append_item ($1, "<&Appendix>", "sgml_appendix (1)");
  menu_append_item ($1, "<&Chapter>", "sgml_chapter (1)");
  menu_append_item ($1, "<&FormalPara>", "sgml_formalpara ()");
  menu_append_item ($1, "<Si&mpleSection>", "sgml_simple_section ()");
  menu_append_item ($1, "&Numbered Section", "sgml_sect (1, \"0\")");
  menu_append_item ($1, "<&Section>", "sgml_section ()");
  menu_append_item ($1, "<&Title>", 
		    "sgml_title (\"Title\", 1, 1); pop_spot ()");
  menu_append_item ($1, "<S&ubtitle>", 
		    "sgml_title (\"SubTitle\", 1, 1); pop_spot ()");
  % table
  menu_append_popup (menu, "&Table");
  $1 = sprintf ("%s.&Table", menu);
  menu_append_item ($1, "<&Align>", "sgml_align ()");
  menu_append_item ($1, "<&InformalTable>", "sgml_table (1)");
  menu_append_item ($1, "<&Table>", "sgml_table (0)");
  menu_append_item ($1, "<&Row>", 
		    "sgml_table_row (table_columns, 1, 1); pop_spot ()");
  % separator
  menu_append_separator (menu);
  % convert to...
  menu_append_item (menu, "Conv&ert To PostScript", "sgml_convert_to_ps ()");
  menu_append_item (menu, "Conve&rt To HTML", "sgml_convert_to_html ()");
  menu_append_item (menu, "&View PostScript", "sgml_view (\"ps\")");
  menu_append_item (menu, "V&iew HTML", "sgml_view (\"html\")");
}

%!%+
%\function{docbook_mode}
%\synopsis{docbook_mode}
%\usage{Void docbook_mode ();}
%\description
% This mode is designed to facilitate the editing of Docbook 3.1 SGML files.
% If a region is defined (i.e., if a mark is set), many SGML tags will
% insert around the region; e.g. '<Emphasis>' and '</Emphasis>'.
% Keybindings begin with ^C and are grouped according to function:
%     ^CC     character styles (emph, tt, etc)
%     ^CE     environments (quote, screen, etc)
%     ^CF     figure
%     ^CH     document header 
%     ^CL     lists (itemized, ordered, etc)
%     ^CK     links (urls, xref, etc)
%     ^CO     computer-related (command, constant, prompt, etc)
%     ^CP     paragraph
%     ^CS     sections
%     ^CT     table
% The Mode menu is also defined and offers the same functionality.
% Functions that affect this mode include:
%#v+
%  sgml_bskip_tag             ^C^B
%  sgml_skip_tag              ^C^F
%  sgml_comment               ESC ;
%#v-
% Variables affecting indentation include:
%#v+
%  SGML_INDENT (default: 2)
%#v-
% To change the value of SGML_INDENT, define the variable in .jedrc
% before loading docbook.sl:
%#v+
%  variable SGML_INDENT = 3;
%#v-
% Hooks: \var{sgml_mode_hook}
%!%-
define docbook_mode ()
{
  variable mode = "docbook";
  variable tmp;

   WRAP_INDENTS = 1; % you really want this

  set_mode (mode, 1); % wrap mode
  set_buffer_hook ("par_sep", &sgml_paragraph_separator);
  use_syntax_table (mode);
  use_keymap (mode);
  !if (blocal_var_exists (DOC_TYPE))
     define_blocal_var (DOC_TYPE, "");
  !if (blocal_var_exists (SECT_TYPE))
     define_blocal_var (SECT_TYPE, "");
  tmp = CASE_SEARCH;
  CASE_SEARCH = 0;
  if (fsearch ("book"))
    set_blocal_var ("b", DOC_TYPE);
  else
  if (fsearch ("<article>"))
    set_blocal_var ("a", DOC_TYPE);
  else
    set_blocal_var ("*", DOC_TYPE);
  CASE_SEARCH = tmp;
  set_blocal_var ("*", SECT_TYPE);
  mode_set_mode_info (mode, "init_mode_menu", &init_menu);
  run_mode_hooks ("docbook_mode_hook");
}

% --- End of file docbook.sl ---
