/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: xdelta.h 1.4.1.8.1.15 Mon, 06 Jul 1998 15:47:37 -0700 jmacd $
 */

#ifndef _XDELTA_H_
#define _XDELTA_H_

#ifdef _WIN32
#include <xdeltawin.h>
#else
#include <xdeltaconfig.h>
#endif

#ifdef HAVE_GLIB_H
#include <glib.h>
#else
#include <fakeglib.h>
#endif

#ifdef HAVE_GDBM_H
#include <gdbm.h>
#else
typedef struct {
  char *dptr;
  int   dsize;
} datum;
#endif

#ifdef HAVE_ZLIB_H
#include <zlib.h>
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#if 0
#define DEBUG_NORMALIZE
#undef DEBUG_HASH
#undef PURIFY
#undef DEBUG_XD
#undef DEBUG_PARAMS
#undef DEBUG_CKSUM
#undef DEBUG_INST
#undef DEBUG_GROW
#undef DEBUG_XDP
#undef XDSTATS
#endif

#ifdef _WIN32
#define FOPEN_READ_ARG "rb"
#define FOPEN_WRITE_ARG "wb"
#define FILE_SEPARATOR '\\'
#else
#define FOPEN_READ_ARG "r"
#define FOPEN_WRITE_ARG "w"
#define FILE_SEPARATOR '/'
#endif

typedef struct _MatchQuery      MatchQuery;
typedef struct _FromSegment     FromSegment;
typedef struct _MappedFile      MappedFile;

#ifdef HAVE_GDBM_H
typedef struct _XdFile          XdFile;

/* This is a serial number on the file format.  This is
 * because the 0.14->0.18 upgrade contained a super bogon
 * which caused some of the old files to break.  It is
 * put into created_by_version of the XdFile.
 *
 * 0.14 is retroactively set to 0.
 * 0.18-0.19 are 1
 * 0.20 and later are 2
 *
 * Starting at version 0.20, this is stored in the archive.
 */
#define XD_0_14_VERSION 0
#define XD_0_18_VERSION 1
#define XD_0_20_VERSION 2

#define XD_GDBM_VERSION XD_0_20_VERSION

struct _XdFile {
  GDBM_FILE dbf;
  gint versions;
  gint compress_level;
  gint created_by_version;
};
#endif

struct _MappedFile {
  guint8* seg;
  gint    len;
#ifdef HAVE_MMAP
  gint    fd; /* if < 0, allocated by malloc() */
#endif
};

datum xdelta_to_bytes            (MatchQuery* query);
datum xdelta_to_normalized_bytes (MatchQuery* query);
gint  xdelta_bytes_seg_count (datum dat);
datum xpatch                 (datum delta, datum* segs, gint nsegs);

void     emit_num_bytes (guint num, datum* dat);
gboolean get_num_bytes  (guint8 *seg, gint len, gint *index, gint* val);

char*       time_t_to_iso8601(time_t t);
time_t      file_timestamp (const char* name);

			     /* MAP.C */

MappedFile* map_file      (const gchar* filename);
void        unmap_file    (MappedFile* map);

MappedFile* unarchive_file (const char* name, int method);
gint        archive_file   (datum file, const char* name, int method);

gint        file_archive_type (const char* file, int* method);

			     /* MISC.C */

void        md5_buffer (gchar* buffer, gint buflen, gchar* digest);

gboolean    xd_unzmemcpy (guint8 *dest, gint *dest_len,
			  guint8 *source, gint source_len);
gboolean    xd_zmemcpy   (guint8 *dest, gint *dest_len,
			  guint8 *source, gint source_len,
			  gint compress_level);

void        xd_set_error_handler (void (* err) (const char* msg));

#if HAVE_VPRINTF
void        xd_error (const char* fmt, ...);
#else
#define xd_error printf
#endif
			    /* XDELTA.C */

GSList* xdelta (MatchQuery* query);

void        match_query_free (MatchQuery *query);
MatchQuery* match_query_new (gint (*break_segment) (MatchQuery* query,
						    const guint8* seg,
						    gint index,
						    gint length),
			     const guint8 *real_to_seg,
			     gint          real_to_len,
			     gint          size,
			     FromSegment  **from,
			     gint           from_count);

FromSegment* from_segment_new (const guint8 *real_to_seg,
			       gint          real_to_len);

FromSegment* from_segment_new_normalized (const guint8 *real_to_seg,
					  gint          real_to_len,
					  gint          sequence,
					  gboolean      is_delta);

void         from_segment_free       (FromSegment* from);
gint         from_segment_sequence   (FromSegment* from);
gboolean     from_segment_used       (FromSegment* from);
void         from_segment_clear_used (FromSegment* from);

#ifdef HAVE_GDBM_H
			      /* XD-GDBM.C */

XdFile*  xd_create (gchar* name);
XdFile*  xd_create_compressed (gchar* name, gint compress_level);
XdFile*  xd_open_read (gchar* name);
XdFile*  xd_open_write (gchar* name);
void     xd_close (XdFile* xd);

gchar* xd_get_date_str (XdFile* xd, gint ver);
gchar* xd_get_md5_raw  (XdFile* xd, gint ver);
gint   xd_get_len      (XdFile* xd, gint ver);
datum  xd_get_latest   (XdFile* xd);

gboolean xd_set_compress_level (XdFile* xd, gint compress_level);
gboolean xd_set_date (XdFile* xd, gint ver, time_t t);
gboolean xd_set_md5  (XdFile* xd, gint ver, gchar raw_md5[16]);
gboolean xd_set_len  (XdFile* xd, gint ver, gint len);
gboolean xd_set_latest (XdFile* xd, guint8* seg, gint len);
gboolean xd_set_versions (XdFile* xd, gint versions);

datum xd_get_delta   (XdFile* xd, gint delta);
gboolean xd_append_delta (XdFile* xd, guint8* seg, gint len);

gint xd_checkin (XdFile*, gchar* file);
gint xd_checkout (XdFile*, gchar* file, gint ver);

#ifdef XDSTATS
gint   xd_get_dlen      (XdFile* xd, gint ver);
gint   xd_get_dzlen     (XdFile* xd, gint ver);
gint   xd_get_usage     (XdFile* xd);
#endif

#endif

#endif
