# (c) Copyright 2014. CodeWeavers, Inc.

"""Updates a file, checking the signature if needed."""

import os
import shutil
import datetime
import time
import subprocess
import tempfile
import binascii

import cxlog
import cxutils
import cxurlget


def save_data_and_sig(lines_iterator, filename=None):
    """Separate the data from the signature information and save each in a
    separate file.

    Returns a (datafile, sigfile) tuple containing the filenames of the data
    file and its signature.
    """

    (database, dataext) = os.path.splitext(filename)
    database = os.path.basename(database)
    (datafd, datafile) = tempfile.mkstemp(suffix=dataext, prefix=database)
    datafh = os.fdopen(datafd, "wb")

    sigext = sigfile = sigfh = None
    for line in lines_iterator:
        if sigext is None:
            (before, sigstart, after) = line.partition(b"<!-- SHA2-256 Signature")
            if sigstart:
                sigext = ".sha256"
                line = after
            else:
                (before, sigstart, after) = line.partition(b"<!-- Signature")
                if sigstart:
                    sigext = ".sig"
                    line = after
            if sigext is not None and sigfh is None:
                if before:
                    datafh.write(before)
                (sigfd, sigfile) = tempfile.mkstemp(suffix=sigext, prefix=filename)
                sigfh = os.fdopen(sigfd, "wb")

        if sigfh:
            # We have a signature embedded in an XML comment.
            # Strip out the commenty bits.
            end = line.find(b"-->")
            if end >= 0:
                line = line[:end]
            line = line.strip()
            splitline = line.split(b"= ")
            if len(splitline) > 1:
                hexline = splitline[1].strip()
            else:
                hexline = line.strip()
            sigfh.write(binascii.unhexlify(hexline))
            if end >= 0:
                # Ignore everything after the first signature
                sigfh.close()
                sigfh = None
                break
        else:
            datafh.write(line)

    datafh.close()
    if sigfh:
        sigfh.close()

    return (datafile, sigfile)

def prepare(filename, urlname=None):
    """Unzip the file and/or extract its signature so it can be checked.

    Returns a (datafile, sigfile) tuple containing the filenames of the prepared
    file and its signature. If an error occurs, datafile will be None.
    """

    basename = filename
    gunzip_proc = None
    if urlname is None:
        urlname = filename
    if urlname.endswith(".gz"):
        if basename.endswith(".gz"):
            basename = basename[0:-3]
        import cxwhich
        gzip = cxwhich.which(os.environ["PATH"], "pigz")
        if gzip is None:
            gzip = "gzip"
        # Use single-letter flags for backward-compatibility.
        args = (gzip, "-d", "-c", filename)
        gunzip_proc = subprocess.Popen(args, stdout=subprocess.PIPE) # pylint: disable=R1732
        srcfh = gunzip_proc.stdout
    else:
        srcfh = open(filename, "rb") # pylint: disable=R1732

    (datafile, sigfile) = save_data_and_sig(srcfh, urlname)

    if gunzip_proc:
        gunzip_proc.communicate()
        if gunzip_proc.returncode != 0:
            cxlog.err("an error occurred while unzipping %s. This file will be ignored." % filename)
            if datafile:
                os.unlink(datafile)
            if sigfile:
                os.unlink(sigfile)
            return (None, None)
    else:
        srcfh.close()

    return (datafile, sigfile)


def is_signed(datafile, sigfile=None):
    """Returns True if the file has a valid signature."""

    if not sigfile:
        for ext in (".sha256", ".sig"):
            sigfile = datafile + ext
            if os.path.exists(sigfile):
                break
        else:
            return False
    elif not os.path.exists(sigfile):
        return False

    sigopt = "-sha1" if sigfile.endswith(".sig") else "-sha256"
    keyfile = os.path.join(cxutils.CX_ROOT, "share", "crossover", "data", "tie.pub")
    args = ["openssl", "dgst", sigopt, "-verify", keyfile, "-signature", sigfile, datafile]
    (retcode, _out, _err) = cxutils.run(args, stdout=cxutils.NULL, stderr=cxutils.NULL)
    if retcode == 0:
        return True

    cxlog.err("The file %s is signed, but the signature is invalid." % datafile)
    return False


def install(dstfile, tmpfile, tmpsig):
    """Replace filename and filename.sig with the specified temporary files so
    as to minimize the impact of race conditions and errors.
    Returns a tuple containing the filenames of the new data and signature
    files."""
    dstdir = os.path.dirname(dstfile)
    if not cxutils.mkdirs(dstdir):
        return (None, None)

    # Do a little dance so we can always find a destination file with a valid
    # signature even if we're interrupted.
    dstsig = None
    if tmpsig:
        shutil.move(tmpfile, dstfile + ".new")
        tmpfile = dstfile + ".new"

        if tmpsig is not None:
            (_path, sigext) = os.path.splitext(tmpsig)
            dstsig = dstfile + sigext
            # Move tmpsig into the destination directory before the final
            # rename, so the replacement is more likely to be atomic.
            shutil.move(tmpsig, dstsig + ".new")
            os.rename(dstsig + ".new", dstsig)
        # If we're interrupted here, then it's up to the caller to notice the
        # .new file on the next run and to move it into place.
    shutil.move(tmpfile, dstfile)
    return (dstfile, dstsig)


def update(filename, url, needs_signature=False, timeout=None, leeway=7200):
    """Updates the specified file from the URL and returns True if it was indeed
    updated. Returns False if the download failed or the file was already up to
    date."""

    exists = os.path.exists(filename)
    if exists:
        mtime = os.path.getmtime(filename)
        if leeway and time.time() < mtime + leeway:
            # No need for an update, the file is still recent enough
            return False
        last_modified = cxurlget.format_http_date(datetime.datetime.utcfromtimestamp(mtime))
    else:
        dirname = os.path.dirname(filename)
        if not cxutils.mkdirs(dirname):
            return False
        last_modified = None

    # Download the file if it is newer than the one we have.
    (basename, ext) = os.path.splitext(filename)
    basename = os.path.basename(basename)
    (newfd, newfile) = tempfile.mkstemp(suffix=ext, prefix=basename)
    newfh = os.fdopen(newfd, "w+b")
    getter = cxurlget.UrlGetter(url, newfh, last_modified=last_modified)
    getter.timeout = timeout
    getter.fetch()

    if not getter.finished:
        # The file did not change since last time
        os.unlink(newfile)
        if leeway and exists:
            # Update the timestamp so we don't try the download again
            # in the next leeway seconds
            os.utime(filename, None)
        return False

    (datafile, sigfile) = prepare(newfile, getter.basename)
    if datafile is None:
        success = False
    elif needs_signature:
        # Make sure that the downloaded file is properly signed.
        if sigfile is not None and is_signed(datafile, sigfile):
            success = True
        else:
            cxlog.warn("The newly downloaded %s file is not properly signed. Discarding." % filename)
            success = False
    else:
        success = True

    if success:
        # Replace filename and filename.sig with the new files.
        install(filename, datafile, sigfile)

    os.unlink(newfile)
    return success
