# (c) Copyright 2010-2013, 2015. CodeWeavers, Inc.

# for localization
from cxutils import cxgettext as _

import os

import checkgtk
checkgtk.check_gtk()

from gi.repository import Gtk

import cxguitools

import cxutils

def product_sort_key(product):
    version_tuple = tuple(int(x) for x in product['productversion'].replace("local", "").split('.'))
    return version_tuple, product['productid'], product['root']

class ProductSelectWindow:
    def __init__(self, products, appname):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("cxtie"))
        self.xml.connect_signals(self)

        self.dialog = self.xml.get_object('SelectProductDialog')
        self.dialog.set_title('')

        escaped_appname = cxutils.html_escape(appname)
        message = _("""<span weight="bold" size="larger">Which product would you like to use to install %(appname)s?</span>

You have multiple CrossOver products that can install %(appname)s. Please choose the product you would like to use from the list below.""") % {'appname': escaped_appname}
        self.xml.get_object('messageLabel').set_markup(message)

        self.product_store = Gtk.ListStore(str, str) # name, root

        product_names = {}
        for product in products:
            product_name = "%s %s" % (product['name'], product['publicversion'])
            if product_name in product_names:
                # duplicate product name; display the root dir to disambiguate
                if product_names[product_name] is not None:
                    existing_product = product_names[product_name]
                    existing_product_name = "%s %s (%s)" % (existing_product['name'], existing_product['publicversion'], existing_product['root'])
                    product_names[existing_product_name] = existing_product
                    product_names[product_name] = None
                product_name = "%s %s (%s)" % (product['name'], product['publicversion'], product['root'])
            product_names[product_name] = product

        # pylint: disable=C0201
        for product_name in list(product_names.keys()):
            if product_names[product_name] is None:
                del product_names[product_name]

        product_items = list(product_names.items())
        product_items.sort(key=product_sort_key)

        for name, product in product_items:
            self.product_store.append((name, product['root']))

        combo_widget = self.xml.get_object('productCombo')

        combo_widget.set_model(self.product_store)
        combo_widget.set_active(0)
        cell = Gtk.CellRendererText()
        combo_widget.pack_start(cell, True)
        combo_widget.add_attribute(cell, "text", 0)

    def get_cxroot(self):
        row = self.xml.get_object('productCombo').get_active_iter()
        root = self.product_store.get_value(row, 1)
        return root

def run(products, filename, appname):
    productselect = ProductSelectWindow(products, appname)
    response = productselect.dialog.run()
    if response == Gtk.ResponseType.OK:
        cxinstaller = os.path.join(productselect.get_cxroot(), 'bin', 'cxinstaller')
        # Use --c4pfile for compatibility with side-by-side installs
        os.execl(cxinstaller, cxinstaller, '--c4pfile', filename)
    return 0

def show_malware_dialog(c4pfile):
    cxguitools.show_malware_dialog(c4pfile, stop_loop=True)
    Gtk.main()
    return 43

def _invalid_file_response(_response):
    Gtk.main_quit()

def show_invalid_file_error(filename, exc):
    primary = _("Cannot open '%(filename)s'") % {'filename': filename}
    if isinstance(exc, IOError):
        secondary = exc.strerror
    elif exc is None:
        secondary = _("The file is not a valid CrossTie file.")
    else:
        secondary = str(exc)
    cxguitools.CXMessageDlg(secondary=secondary, primary=primary,
                            response_function=_invalid_file_response,
                            message_type=Gtk.MessageType.ERROR)
    Gtk.main()
    return 86

PRODUCT_NAMES = {
    'cxgames': _("CrossOver Games"),
    'cxoffice': _("CrossOver"),
    }

product_urls = {
    'cxgames': 'http://crossover.codeweavers.com/redirect/CXGUI_missProd_linux_cxgames',
    'cxoffice': 'http://crossover.codeweavers.com/redirect/CXGUI_missProd_linux_cxoffice',
    }

def _otherproduct_response(response, user_data):
    if response == 0:
        # Learn More
        cxutils.launch_url(user_data)
        return False
    # Close
    Gtk.main_quit()
    return True

def show_otherproduct_dialog(products, _filename, appname):
    productid = min(products)

    product_name = PRODUCT_NAMES.get(productid, productid)
    product_url = product_urls.get(productid, 'http://www.codeweavers.com/products/')

    primary = _("%s requires a different version of CrossOver than you have installed.") % appname
    secondary = _("To install %(appname)s, you will need %(product)s. Click 'Learn More' to visit our web site.") % {'appname': appname, 'product': product_name}

    cxguitools.CXMessageDlg(secondary=secondary, primary=primary,
                            button_array=[[_("Learn More"), 0],
                                          [Gtk.STOCK_CLOSE, 1]],
                            response_function=_otherproduct_response,
                            close_on_response=False,
                            user_data=product_url,
                            message_type=Gtk.MessageType.ERROR)
    Gtk.main()

    return 99
