﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/vpc-lattice/VPCLatticeRequest.h>
#include <aws/vpc-lattice/VPCLattice_EXPORTS.h>
#include <aws/vpc-lattice/model/RuleAction.h>
#include <aws/vpc-lattice/model/RuleMatch.h>

#include <utility>

namespace Aws {
namespace VPCLattice {
namespace Model {

/**
 */
class UpdateRuleRequest : public VPCLatticeRequest {
 public:
  AWS_VPCLATTICE_API UpdateRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateRule"; }

  AWS_VPCLATTICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID or ARN of the service.</p>
   */
  inline const Aws::String& GetServiceIdentifier() const { return m_serviceIdentifier; }
  inline bool ServiceIdentifierHasBeenSet() const { return m_serviceIdentifierHasBeenSet; }
  template <typename ServiceIdentifierT = Aws::String>
  void SetServiceIdentifier(ServiceIdentifierT&& value) {
    m_serviceIdentifierHasBeenSet = true;
    m_serviceIdentifier = std::forward<ServiceIdentifierT>(value);
  }
  template <typename ServiceIdentifierT = Aws::String>
  UpdateRuleRequest& WithServiceIdentifier(ServiceIdentifierT&& value) {
    SetServiceIdentifier(std::forward<ServiceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID or ARN of the listener.</p>
   */
  inline const Aws::String& GetListenerIdentifier() const { return m_listenerIdentifier; }
  inline bool ListenerIdentifierHasBeenSet() const { return m_listenerIdentifierHasBeenSet; }
  template <typename ListenerIdentifierT = Aws::String>
  void SetListenerIdentifier(ListenerIdentifierT&& value) {
    m_listenerIdentifierHasBeenSet = true;
    m_listenerIdentifier = std::forward<ListenerIdentifierT>(value);
  }
  template <typename ListenerIdentifierT = Aws::String>
  UpdateRuleRequest& WithListenerIdentifier(ListenerIdentifierT&& value) {
    SetListenerIdentifier(std::forward<ListenerIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID or ARN of the rule.</p>
   */
  inline const Aws::String& GetRuleIdentifier() const { return m_ruleIdentifier; }
  inline bool RuleIdentifierHasBeenSet() const { return m_ruleIdentifierHasBeenSet; }
  template <typename RuleIdentifierT = Aws::String>
  void SetRuleIdentifier(RuleIdentifierT&& value) {
    m_ruleIdentifierHasBeenSet = true;
    m_ruleIdentifier = std::forward<RuleIdentifierT>(value);
  }
  template <typename RuleIdentifierT = Aws::String>
  UpdateRuleRequest& WithRuleIdentifier(RuleIdentifierT&& value) {
    SetRuleIdentifier(std::forward<RuleIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The rule match.</p>
   */
  inline const RuleMatch& GetMatch() const { return m_match; }
  inline bool MatchHasBeenSet() const { return m_matchHasBeenSet; }
  template <typename MatchT = RuleMatch>
  void SetMatch(MatchT&& value) {
    m_matchHasBeenSet = true;
    m_match = std::forward<MatchT>(value);
  }
  template <typename MatchT = RuleMatch>
  UpdateRuleRequest& WithMatch(MatchT&& value) {
    SetMatch(std::forward<MatchT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The rule priority. A listener can't have multiple rules with the same
   * priority.</p>
   */
  inline int GetPriority() const { return m_priority; }
  inline bool PriorityHasBeenSet() const { return m_priorityHasBeenSet; }
  inline void SetPriority(int value) {
    m_priorityHasBeenSet = true;
    m_priority = value;
  }
  inline UpdateRuleRequest& WithPriority(int value) {
    SetPriority(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the action for the specified listener rule.</p>
   */
  inline const RuleAction& GetAction() const { return m_action; }
  inline bool ActionHasBeenSet() const { return m_actionHasBeenSet; }
  template <typename ActionT = RuleAction>
  void SetAction(ActionT&& value) {
    m_actionHasBeenSet = true;
    m_action = std::forward<ActionT>(value);
  }
  template <typename ActionT = RuleAction>
  UpdateRuleRequest& WithAction(ActionT&& value) {
    SetAction(std::forward<ActionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_serviceIdentifier;

  Aws::String m_listenerIdentifier;

  Aws::String m_ruleIdentifier;

  RuleMatch m_match;

  int m_priority{0};

  RuleAction m_action;
  bool m_serviceIdentifierHasBeenSet = false;
  bool m_listenerIdentifierHasBeenSet = false;
  bool m_ruleIdentifierHasBeenSet = false;
  bool m_matchHasBeenSet = false;
  bool m_priorityHasBeenSet = false;
  bool m_actionHasBeenSet = false;
};

}  // namespace Model
}  // namespace VPCLattice
}  // namespace Aws
