﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/backup-gateway/BackupGatewayRequest.h>
#include <aws/backup-gateway/BackupGateway_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace BackupGateway {
namespace Model {

/**
 */
class ListVirtualMachinesRequest : public BackupGatewayRequest {
 public:
  AWS_BACKUPGATEWAY_API ListVirtualMachinesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListVirtualMachines"; }

  AWS_BACKUPGATEWAY_API Aws::String SerializePayload() const override;

  AWS_BACKUPGATEWAY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the hypervisor connected to your virtual
   * machine.</p>
   */
  inline const Aws::String& GetHypervisorArn() const { return m_hypervisorArn; }
  inline bool HypervisorArnHasBeenSet() const { return m_hypervisorArnHasBeenSet; }
  template <typename HypervisorArnT = Aws::String>
  void SetHypervisorArn(HypervisorArnT&& value) {
    m_hypervisorArnHasBeenSet = true;
    m_hypervisorArn = std::forward<HypervisorArnT>(value);
  }
  template <typename HypervisorArnT = Aws::String>
  ListVirtualMachinesRequest& WithHypervisorArn(HypervisorArnT&& value) {
    SetHypervisorArn(std::forward<HypervisorArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of virtual machines to list.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListVirtualMachinesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The next item following a partial list of returned resources. For example, if
   * a request is made to return <code>maxResults</code> number of resources,
   * <code>NextToken</code> allows you to return more items in your list starting at
   * the location pointed to by the next token.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListVirtualMachinesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_hypervisorArn;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_hypervisorArnHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace BackupGateway
}  // namespace Aws
