////////////////////////////////////////////////////////////////////////////////////////
//
//  Copyright 2021 OVITO GmbH, Germany
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify it either under the
//  terms of the GNU General Public License version 3 as published by the Free Software
//  Foundation (the "GPL") or, at your option, under the terms of the MIT License.
//  If you do not alter this notice, a recipient may use your version of this
//  file under either the GPL or the MIT License.
//
//  You should have received a copy of the GPL along with this program in a
//  file LICENSE.GPL.txt.  You should have received a copy of the MIT License along
//  with this program in a file LICENSE.MIT.txt
//
//  This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND,
//  either express or implied. See the GPL or the MIT License for the specific language
//  governing rights and limitations.
//
////////////////////////////////////////////////////////////////////////////////////////

#include <ovito/crystalanalysis/CrystalAnalysis.h>
#include <ovito/crystalanalysis/modifier/elasticstrain/ElasticStrainModifier.h>
#include <ovito/particles/gui/modifier/analysis/StructureListParameterUI.h>
#include <ovito/gui/desktop/properties/BooleanParameterUI.h>
#include <ovito/gui/desktop/properties/BooleanRadioButtonParameterUI.h>
#include <ovito/gui/desktop/properties/FloatParameterUI.h>
#include <ovito/gui/desktop/properties/VariantComboBoxParameterUI.h>
#include <ovito/gui/desktop/properties/SubObjectParameterUI.h>
#include <ovito/gui/desktop/properties/ObjectStatusDisplay.h>
#include "ElasticStrainModifierEditor.h"

namespace Ovito::CrystalAnalysis {

IMPLEMENT_OVITO_CLASS(ElasticStrainModifierEditor);
SET_OVITO_OBJECT_EDITOR(ElasticStrainModifier, ElasticStrainModifierEditor);

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void ElasticStrainModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
    // Create the rollout.
    QWidget* rollout = createRollout(tr("Elastic strain calculation"), rolloutParams, "manual:particles.modifiers.elastic_strain");

    QVBoxLayout* layout = new QVBoxLayout(rollout);
    layout->setContentsMargins(4,4,4,4);
    layout->setSpacing(6);

    QGroupBox* structureBox = new QGroupBox(tr("Input crystal"));
    layout->addWidget(structureBox);
    QGridLayout* sublayout1 = new QGridLayout(structureBox);
    sublayout1->setContentsMargins(4,4,4,4);
    sublayout1->setSpacing(4);
    sublayout1->setColumnStretch(1,1);
    VariantComboBoxParameterUI* crystalStructureUI = new VariantComboBoxParameterUI(this, PROPERTY_FIELD(ElasticStrainModifier::inputCrystalStructure));
    crystalStructureUI->comboBox()->addItem(tr("Face-centered cubic (FCC)"), QVariant::fromValue((int)StructureAnalysis::LATTICE_FCC));
    crystalStructureUI->comboBox()->addItem(tr("Hexagonal close-packed (HCP)"), QVariant::fromValue((int)StructureAnalysis::LATTICE_HCP));
    crystalStructureUI->comboBox()->addItem(tr("Body-centered cubic (BCC)"), QVariant::fromValue((int)StructureAnalysis::LATTICE_BCC));
    crystalStructureUI->comboBox()->addItem(tr("Diamond cubic / Zinc blende"), QVariant::fromValue((int)StructureAnalysis::LATTICE_CUBIC_DIAMOND));
    crystalStructureUI->comboBox()->addItem(tr("Diamond hexagonal / Wurtzite"), QVariant::fromValue((int)StructureAnalysis::LATTICE_HEX_DIAMOND));
    sublayout1->addWidget(crystalStructureUI->comboBox(), 0, 0, 1, 2);

    FloatParameterUI* latticeConstantUI = new FloatParameterUI(this, PROPERTY_FIELD(ElasticStrainModifier::latticeConstant));
    sublayout1->addWidget(latticeConstantUI->label(), 1, 0);
    sublayout1->addLayout(latticeConstantUI->createFieldLayout(), 1, 1);

    _caRatioUI = new FloatParameterUI(this, PROPERTY_FIELD(ElasticStrainModifier::axialRatio));
    sublayout1->addWidget(_caRatioUI->label(), 2, 0);
    sublayout1->addLayout(_caRatioUI->createFieldLayout(), 2, 1);

    QGroupBox* outputParamsBox = new QGroupBox(tr("Output settings"));
    layout->addWidget(outputParamsBox);
    QGridLayout* sublayout2 = new QGridLayout(outputParamsBox);
    sublayout2->setContentsMargins(4,4,4,4);
    sublayout2->setSpacing(4);
    sublayout2->setColumnStretch(1, 1);
    sublayout2->setColumnMinimumWidth(0, 12);

    BooleanParameterUI* outputStrainTensorsUI = new BooleanParameterUI(this, PROPERTY_FIELD(ElasticStrainModifier::calculateStrainTensors));
    sublayout2->addWidget(outputStrainTensorsUI->checkBox(), 0, 0, 1, 2);

    BooleanRadioButtonParameterUI* pushStrainTensorsForwardUI = new BooleanRadioButtonParameterUI(this, PROPERTY_FIELD(ElasticStrainModifier::pushStrainTensorsForward));
    pushStrainTensorsForwardUI->buttonTrue()->setText(tr("in spatial frame"));
    pushStrainTensorsForwardUI->buttonFalse()->setText(tr("in lattice frame"));
    sublayout2->addWidget(pushStrainTensorsForwardUI->buttonTrue(), 1, 1);
    sublayout2->addWidget(pushStrainTensorsForwardUI->buttonFalse(), 2, 1);

    pushStrainTensorsForwardUI->setEnabled(false);
    connect(outputStrainTensorsUI->checkBox(), &QCheckBox::toggled, pushStrainTensorsForwardUI, &BooleanRadioButtonParameterUI::setEnabled);

    BooleanParameterUI* outputDeformationGradientsUI = new BooleanParameterUI(this, PROPERTY_FIELD(ElasticStrainModifier::calculateDeformationGradients));
    sublayout2->addWidget(outputDeformationGradientsUI->checkBox(), 3, 0, 1, 2);

    // Status label.
    layout->addWidget((new ObjectStatusDisplay(this))->statusWidget());

    // Structure list.
    StructureListParameterUI* structureTypesPUI = new StructureListParameterUI(this);
    layout->addSpacing(10);
    layout->addWidget(structureTypesPUI->tableWidget());

    connect(this, &PropertiesEditor::contentsChanged, this, &ElasticStrainModifierEditor::modifierChanged);
}

/******************************************************************************
* Is called each time the parameters of the modifier have changed.
******************************************************************************/
void ElasticStrainModifierEditor::modifierChanged(RefTarget* editObject)
{
    ElasticStrainModifier* modifier = static_object_cast<ElasticStrainModifier>(editObject);
    _caRatioUI->setEnabled(modifier &&
            (modifier->inputCrystalStructure() == StructureAnalysis::LATTICE_HCP ||
             modifier->inputCrystalStructure() == StructureAnalysis::LATTICE_HEX_DIAMOND));
}

}   // End of namespace
