﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/emr-serverless/EMRServerlessRequest.h>
#include <aws/emr-serverless/EMRServerless_EXPORTS.h>
#include <aws/emr-serverless/model/ApplicationState.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace EMRServerless {
namespace Model {

/**
 */
class ListApplicationsRequest : public EMRServerlessRequest {
 public:
  AWS_EMRSERVERLESS_API ListApplicationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListApplications"; }

  AWS_EMRSERVERLESS_API Aws::String SerializePayload() const override;

  AWS_EMRSERVERLESS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The token for the next set of application results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListApplicationsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of applications that can be listed.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListApplicationsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional filter for application states. Note that if this filter contains
   * multiple states, the resulting list will be grouped by the state.</p>
   */
  inline const Aws::Vector<ApplicationState>& GetStates() const { return m_states; }
  inline bool StatesHasBeenSet() const { return m_statesHasBeenSet; }
  template <typename StatesT = Aws::Vector<ApplicationState>>
  void SetStates(StatesT&& value) {
    m_statesHasBeenSet = true;
    m_states = std::forward<StatesT>(value);
  }
  template <typename StatesT = Aws::Vector<ApplicationState>>
  ListApplicationsRequest& WithStates(StatesT&& value) {
    SetStates(std::forward<StatesT>(value));
    return *this;
  }
  inline ListApplicationsRequest& AddStates(ApplicationState value) {
    m_statesHasBeenSet = true;
    m_states.push_back(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_nextToken;

  int m_maxResults{0};

  Aws::Vector<ApplicationState> m_states;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_statesHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMRServerless
}  // namespace Aws
