/**
 * <copyright>
 *
 * Service Data Objects
 * Version 1.0
 * Licensed Materials - Property of BEA and IBM
 *
 * (c) Copyright BEA Systems, Inc. and International Business Machines Corp 2003.  All rights reserved.
 *
 * </copyright>
 * 
 * $Id: DataGraph.java,v 1.1 2004/03/26 15:24:15 marcelop Exp $
 */
package commonj.sdo;


/**
 * A data graph is used to package a graph of {@link DataObject data objects} along with their
 * metadata, that is, data describing the data.
 * A data graph also contains a {@link #getChangeSummary change summary} 
 * which is used to record changes made to the objects in the graph.
 */
public interface DataGraph
{
  /**
   * Returns the root {@link DataObject data object} of this data graph.
   * @return the root data object.
   * @see DataObject#getDataGraph
   */
  DataObject getRootObject();

  /**
   * Creates a new root data object of the {@link #getType specified type},
   * replacing the existing root, if one exists.
   * @param namespaceURI namespace of the type.
   * @param typeName name of the type.
   * @return the new root.
   * @see #createRootObject(Type)
   * @see #getType(String, String)
   */
  DataObject createRootObject(String namespaceURI, String typeName);

  /**
   * Creates a new root data object of the specified type,
   * replacing the existing root, if one exists.
   * @param type the type of the new root.
   * @return the new root.
   * @see #createRootObject(String, String)
   */
  DataObject createRootObject(Type type);

  /**
   * Returns the {@link ChangeSummary change summary} associated with this data graph.
   * @return the change summary.
   * @see ChangeSummary#getDataGraph
   */
  ChangeSummary getChangeSummary();

  /**
   * Returns the {@link Type type} with the given the {@link Type#getURI() URI},
   * or contained by the resource at the given URI,
   * and with the given {@link Type#getName name}.
   * @param uri the namespace URI of a type or the location URI of a resource containing a type.
   * @param typeName name of a type.
   * @return the type with the corresponding namespace and name.
   */
  Type getType(String uri, String typeName);
}
