/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.server.internal.ui.explorer.content;

import java.io.IOException;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.wst.rdb.core.internal.ui.explorer.services.IVirtualNodeServiceFactory;
import org.eclipse.wst.rdb.core.internal.ui.explorer.virtual.IConnectionNode;
import org.eclipse.wst.rdb.core.internal.ui.explorer.virtual.IKnownConnectionNode;
import org.eclipse.wst.rdb.core.internal.ui.services.IDataToolsUIServiceManager;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinitionRegistry;
import org.eclipse.wst.rdb.internal.core.util.DatabaseProviderHelper;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.server.internal.ui.explorer.providers.ServerExplorerManager;
import org.eclipse.wst.rdb.server.internal.ui.services.IServerStatusDecorationService;
import org.eclipse.wst.rdb.server.internal.ui.services.IServicesManager;
import org.eclipse.wst.rdb.server.internal.ui.util.ServerUIDebugOptions;
import org.eclipse.wst.rdb.server.internal.ui.util.logging.Logger;
import org.eclipse.wst.rdb.server.internal.ui.util.resources.ResourceLoader;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;


/**
 * @author ljulien
 */
public class ServerExplorerInitializer
{
	private final static ResourceLoader resourceLoader = ResourceLoader.INSTANCE;
	private final static String SCHEMA_NODE = resourceLoader.queryString("DATATOOLS.SERVER.UI.EXPLORER.SCHEMA"); //$NON-NLS-1$
	private static final String DATABASE = resourceLoader.queryString("DATATOOLS.SERVER.UI.EXPLORER.DATABASE"); //$NON-NLS-1$
	private static final String PLUGIN_DIR = "Plugin_Directory/"; //$NON-NLS-1$
	private static final String PLUGIN_STATE_LOCATION = "Plugin_State_Location"; //$NON-NLS-1$

	private static final IVirtualNodeServiceFactory virtualNodeFactory = IDataToolsUIServiceManager.INSTANCE
			.getVirtualNodeServiceFactory();
	private static final DatabaseDefinitionRegistry defRegistry = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry();

	private ConnectionInfo info;

	private IConnectionNode getServerNode (ConnectionInfo info)
	{
	    IConnectionNode connection = null;
	    IKnownConnectionNode serverNode = IServicesManager.INSTANCE.getServerExplorerContentService().getKnownServerNode();
	    Object [] connections = serverNode.getChildrenArray();
	    for (int i = 0, n = connections.length; i < n; i++)
	    {
	        connection = (IConnectionNode) connections [i];
	        if (connection.getName().equals(info.getName()))
	        {
	            return connection;
	        }
	    }
	    return null;
	}
	
	/**
	 * Will add the New databases under the Server Node
	 * @param serverNode
	 * @param databases
	 */
	private void addNewDatabases(IConnectionNode serverNode, Database database)
	{
	    IServicesManager.INSTANCE.getServerExplorerContentService().addNode(serverNode, database);
	}

	/**
	 * Will save all the information relative to New Databases
	 */
	private void saveNewDatabaseConnection(ConnectionInfo connectionInfo, Database database, IConnectionNode serverNode)
	{
		try
        {
		    connectionInfo.setDatabaseName(database.getName());
            new ServerExplorerConfiguration().save(connectionInfo);
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
	}

	public boolean displayNewConnection (String serverName, ConnectionInfo connectionInfo)
	{
	    IConnectionNode node = getServerNode (connectionInfo);
	    if (node == null)
	    {
	        return displayNewConnection (serverName, connectionInfo, true);
	    }
	    else
	    {
			IServicesManager.INSTANCE.getServerExplorerContentService().connectServer(node);
	        IServerStatusDecorationService service = IServicesManager.INSTANCE.getServerStatusDecorationService();
	        if (service != null)
	        {
	            service.refreshDecoration(new Object [] {node});
	        }
			return true;
	    }
	}
	
	/**
	 * @param serverName
	 * @param databases
	 * @param connectionInfo
	 */
	private boolean displayNewConnection(String serverName, ConnectionInfo connectionInfo, boolean save)
	{
		IConnectionNode server = virtualNodeFactory.makeConnectionNode(	connectionInfo.getName(), 
		        												connectionInfo.getName(), 
		        												ServerExplorerManager.INSTANCE.getRootKnownServerNode(),
		        												connectionInfo);
		IServicesManager.INSTANCE.getServerExplorerContentService().addKnownServer(server);
		return true;
	}

	/**
	 * @return Will return the fully qualified path by substituing {Plugin_Directory} to its real value
	 */
	private String substitutePluginDirectory(String logicalPath, IConfigurationElement element)
	{
		try
        {
            return logicalPath.replaceFirst(PLUGIN_DIR, 
                    						Platform.resolve(Platform.getBundle(element.getDeclaringExtension().getNamespace()).getEntry("/")).getPath()); //$NON-NLS-1$
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        return null;
	}
	
	private boolean isBundleActivated (IConfigurationElement element)
	{
	    try
        {
            Bundle bundle = Platform.getBundle(element.getDeclaringExtension().getNamespace());
            if (bundle.getState() == Bundle.RESOLVED)
            {
                bundle.start();
            }
            return bundle.getState() == Bundle.ACTIVE;
        }
        catch (BundleException e)
        {
            e.printStackTrace();
        }
        return false;
	}
	
	private String substituteLocationDirectory (String logicalPath, IConfigurationElement element)
	{
	    String stateLocation = Platform.getStateLocation(Platform.getBundle(element.getDeclaringExtension().getNamespace())).toOSString();
		int index = logicalPath.indexOf(PLUGIN_STATE_LOCATION);
		if (index != -1)
		{
		    logicalPath = logicalPath.substring(0, index) + stateLocation + logicalPath.substring(index + PLUGIN_STATE_LOCATION.length());
		}
		return logicalPath;
	}
	
	/**
	 * @return The Connection Info object that will be used to retrieve the databases
	 */
	private ConnectionInfo getConnectionInfo(String product, String version, String connectionName)
	{
		ConnectionInfo info = RDBCorePlugin.getDefault().getConnectionManager().getConnectionInfo(connectionName);
		if(info == null) 
		{
			DatabaseDefinition dbDef = defRegistry.getDefinition(product, version);
			info = RDBCorePlugin.getDefault().getConnectionManager().createConnectionInfo(dbDef, connectionName);
		}
		ServerExplorerManager.INSTANCE.initializeConnectionInfo(info);
		return info;
	}

	private void enableLocalDatabase(IConfigurationElement[] configElements, String connectionName, String user, String password,
			String loadingPath, String driverClassName)
	{
		try
		{
			List databaseList = new ArrayList(configElements.length);
			for (int i = 0, n = configElements.length; i < n; i++)
			{
			    if (isBundleActivated (configElements[i]))
			    {
					String databaseName = configElements[i].getAttribute("name"); //$NON-NLS-1$
					info = getConnectionInfo(configElements[i].getAttribute("product"), configElements[i] //$NON-NLS-1$
							.getAttribute("version"), connectionName); //$NON-NLS-1$
					
					info.setDatabaseName(databaseName);
					info.setDriverClassName(driverClassName);
					info.setLoadingPath(substitutePluginDirectory(loadingPath, configElements[i]));
					
					info.setURL(substituteLocationDirectory(configElements[i].getAttribute("URL"), configElements[i])); //$NON-NLS-1$
	
					Properties ps = info.getProperties();
					ps.setProperty("user", user); //$NON-NLS-1$
					ps.setProperty("password", password); //$NON-NLS-1$
					
					Connection connection = info.connect();
					if (connection != null)
					{
					    info.setSharedConnection(connection);
					    new DatabaseProviderHelper().setDatabase(connection, info, databaseName);
					}
			    }
			}
		}
		catch (Exception e)
		{
			Logger.log(this, e, ServerUIDebugOptions.SERVER_EXPLORER_LOG);
		}
	}
	
	private void initializeLocalDatabase (IConfigurationElement config, String initializationClass)
	{
		try
		{
			config.createExecutableExtension("connectionInitialization");
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
	}

	/**
	 * Will initilize the Server Explorer when started with the Databases registered by Database Providers
	 */
	public void loadLocalRegisteredDatabases()
	{
		IExtensionRegistry pluginRegistry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint = pluginRegistry.getExtensionPoint("org.eclipse.wst.rdb.server.ui", //$NON-NLS-1$
				"ServerExplorerInitializationProvider"); //$NON-NLS-1$
		IExtension[] extensions = extensionPoint.getExtensions();
		for (int i = 0; i < extensions.length; ++i)
		{
			IConfigurationElement[] configElements = extensions[i].getConfigurationElements();
			for (int j = 0; j < configElements.length; ++j)
			{
				if (configElements[j].getName().equals("server")) //$NON-NLS-1$
				{
					String user = configElements[j].getAttribute("user"); //$NON-NLS-1$
					String password = configElements[j].getAttribute("password"); //$NON-NLS-1$
					String loadingPath = configElements[j].getAttribute("loadingPath"); //$NON-NLS-1$
					String driverClassName = configElements[j].getAttribute("driverClassName"); //$NON-NLS-1$
					String connectionName = configElements[j].getAttribute("serverName"); //$NON-NLS-1$
					String initializationClass = configElements[j].getAttribute("connectionInitialization"); //$NON-NLS-1$
					
					if (initializationClass != null)
					{
						initializeLocalDatabase(configElements[j], initializationClass);
					}
					enableLocalDatabase(configElements[j].getChildren("database"), connectionName, user, password, loadingPath, driverClassName); //$NON-NLS-1$
				}
			}
		}
	}
}
