<?php

declare(strict_types=1);
/**
 * SPDX-FileCopyrightText: 2019 Nextcloud GmbH and Nextcloud contributors
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */
namespace OCP\FullTextSearch\Model;

use OCP\FullTextSearch\IFullTextSearchProvider;

/**
 * Class ISearchTemplate
 *
 * This is a data transfer object that should be created by Content Provider
 * when the getSearchTemplate() method is called.
 *
 * The object will contain templates to be displayed, and the list of the different
 * options to be available to the user when they start a new search.
 *
 * The display of the Options is generated by the FullTextSearch app and Options
 * can be displayed in 2 places:
 *
 * - the navigation page of the app that generate the indexed content.
 *   (files, bookmarks, deck, mails, ...)
 * - the navigation page of the FullTextSearch app.
 *
 * Both pages will have different Options, and only the first one can integrate
 * a specific template.
 *
 * @see IFullTextSearchProvider::getSearchTemplate
 *
 * @since 16.0.0
 *
 */
interface ISearchTemplate {
	/**
	 * Set the class of the icon to be displayed in the left panel of the
	 * FullTextSearch navigation page, in front of the related Content Provider.
	 *
	 * @since 16.0.0
	 *
	 * @param string $class
	 *
	 * @return ISearchTemplate
	 */
	public function setIcon(string $class): ISearchTemplate;

	/**
	 * Get the class of the icon.
	 *
	 * @since 16.0.0
	 *
	 * @return string
	 */
	public function getIcon(): string;


	/**
	 * Set the path of a CSS file that will be loaded when needed.
	 *
	 * @since 16.0.0
	 *
	 * @param string $css
	 *
	 * @return ISearchTemplate
	 */
	public function setCss(string $css): ISearchTemplate;

	/**
	 * Get the path of the CSS file.
	 *
	 * @since 16.0.0
	 *
	 * @return string
	 */
	public function getCss(): string;


	/**
	 * Set the path of the file of a template that the HTML will be displayed
	 * below the Options.
	 * This should only be used if your Content Provider needs to set options in
	 * a way not generated by FullTextSearch
	 *
	 * @since 16.0.0
	 *
	 * @param string $template
	 *
	 * @return ISearchTemplate
	 */
	public function setTemplate(string $template): ISearchTemplate;

	/**
	 * Get the path of the template file.
	 *
	 * @since 16.0.0
	 *
	 * @return string
	 */
	public function getTemplate(): string;


	/**
	 * Add an option in the Panel that is displayed when the user start a search
	 * within the app that generate the content.
	 *
	 * @see ISearchOption
	 *
	 * @since 16.0.0
	 *
	 * @param ISearchOption $option
	 *
	 * @return ISearchTemplate
	 */
	public function addPanelOption(ISearchOption $option): ISearchTemplate;

	/**
	 * Get all options to be displayed in the Panel.
	 *
	 * @since 16.0.0
	 *
	 * @return ISearchOption[]
	 */
	public function getPanelOptions(): array;


	/**
	 * Add an option in the left panel of the FullTextSearch navigation page.
	 *
	 * @see ISearchOption
	 *
	 * @since 16.0.0
	 *
	 * @param ISearchOption $option
	 *
	 * @return ISearchTemplate
	 */
	public function addNavigationOption(ISearchOption $option): ISearchTemplate;

	/**
	 * Get all options to be displayed in the FullTextSearch navigation page.
	 *
	 * @since 16.0.0
	 *
	 * @return array
	 */
	public function getNavigationOptions(): array;
}
