"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importFileRoute = importFileRoute;
var _configSchema = require("@osd/config-schema");
var _constants = require("../../common/constants");
var _util = require("../utils/util");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function importFileRoute(router, config, fileProcessors, dataSourceEnabled) {
  router.post({
    path: '/api/data_importer/_import_file',
    options: {
      body: {
        maxBytes: config.maxFileSizeBytes,
        accepts: 'multipart/form-data',
        output: 'stream'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        indexName: _configSchema.schema.string({
          minLength: 1
        }),
        createMode: _configSchema.schema.boolean(),
        fileExtension: _configSchema.schema.string({
          minLength: 1
        }),
        dataSource: _configSchema.schema.maybe(_configSchema.schema.string()),
        delimiter: _configSchema.schema.maybe(_configSchema.schema.string({
          validate(value) {
            if (!_constants.CSV_SUPPORTED_DELIMITERS.includes(value)) {
              return `must be a supported delimiter`;
            }
          }
        }))
      }),
      body: _configSchema.schema.object({
        file: _configSchema.schema.stream(),
        mapping: _configSchema.schema.maybe(_configSchema.schema.string({
          minLength: 1
        }))
      })
    }
  }, async (context, request, response) => {
    const fileExtension = request.query.fileExtension.startsWith('.') ? request.query.fileExtension.slice(1) : request.query.fileExtension;
    const processor = fileProcessors.getFileProcessor(fileExtension);
    if (!processor) {
      return response.badRequest({
        body: `${fileExtension} is not a registered or supported filetype`
      });
    }
    const client = await (0, _util.decideClient)(dataSourceEnabled, context, request.query.dataSource);
    if (!!!client) {
      return response.notFound({
        body: 'Data source is not enabled or does not exist'
      });
    }
    try {
      const indexExists = await client.indices.exists({
        index: request.query.indexName
      });
      if (!request.query.createMode && !indexExists.body) {
        return response.notFound({
          body: `Index ${request.query.indexName} does not exist`
        });
      }
      if (request.query.createMode && indexExists.body) {
        return response.badRequest({
          body: `Index ${request.query.indexName} already exists`
        });
      }
    } catch (e) {
      return response.internalError({
        body: `Error checking if index exists: ${e}`
      });
    }
    if (request.query.createMode) {
      const mapping = request.body.mapping;
      try {
        await client.indices.create({
          index: request.query.indexName,
          ...(mapping && {
            body: {
              mappings: JSON.parse(mapping)
            }
          })
        });
      } catch (e) {
        return response.internalError({
          body: `Error creating index: ${e}`
        });
      }
    }
    const file = request.body.file;
    try {
      const message = await processor.ingestFile(file, {
        indexName: request.query.indexName,
        client,
        delimiter: request.query.delimiter,
        dataSourceId: request.query.dataSource
      });
      return response.ok({
        body: {
          message,
          success: message.failedRows.length < 1
        }
      });
    } catch (e) {
      return response.internalError({
        body: `Error ingesting file: ${e}`
      });
    }
  });
}