"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addClickToMetric = addClickToMetric;
exports.addRequestToMetric = addRequestToMetric;
exports.getMetrics = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("./constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const time2CountWin = new Map();

function addClickToMetric(element, counter = 'count') {
  // remove outdated key-value pairs
  trim();
  const timeKey = getKey(Date.now());

  const rollingCounter = time2CountWin.get(timeKey) || _lodash.default.cloneDeep(_constants.DEFAULT_ROLLING_COUNTER);

  const key = `click.${element}.${counter}`;

  _lodash.default.set(rollingCounter, key, _lodash.default.get(rollingCounter, key, 0) + 1);

  if (counter === 'count') {
    const basicCounterKey = `click.${element}.total`;

    _lodash.default.set(_constants.GLOBAL_BASIC_COUNTER, basicCounterKey, _lodash.default.get(_constants.GLOBAL_BASIC_COUNTER, basicCounterKey, 0) + 1);
  }

  time2CountWin.set(timeKey, rollingCounter);
}

function addRequestToMetric(component, request, counterNameOrError) {
  const counter = typeof counterNameOrError === 'object' ? checkErrorType(counterNameOrError) : counterNameOrError; // remove outdated key-value pairs

  trim();
  const timeKey = getKey(Date.now());

  const rollingCounter = time2CountWin.get(timeKey) || _lodash.default.cloneDeep(_constants.DEFAULT_ROLLING_COUNTER); // @ts-ignore not sure why 'request' can't be indexed


  rollingCounter[component][request][counter]++;

  if (counter === 'count') {
    // @ts-ignore
    _constants.GLOBAL_BASIC_COUNTER[component][request]['total']++;
  }

  time2CountWin.set(timeKey, rollingCounter);
}

const getMetrics = () => {
  const preTimeKey = getPreKey(Date.now());
  const rollingCounters = time2CountWin.get(preTimeKey);
  return buildMetrics(rollingCounters);
};

exports.getMetrics = getMetrics;

const checkErrorType = error => {
  if (error.statusCode && Math.floor(error.statusCode / 100) === 4) {
    return 'user_error';
  } else {
    return 'system_error';
  }
};

const trim = () => {
  if (time2CountWin.size > _constants.CAPACITY) {
    const currentKey = getKey(Date.now() - _constants.WINDOW * _constants.MILLIS_MULTIPLIER);
    time2CountWin.forEach((_value, key, map) => {
      if (key < currentKey) {
        map.delete(key);
      }
    });
  }
};

const getKey = milliseconds => {
  return Math.floor(milliseconds / _constants.MILLIS_MULTIPLIER / _constants.INTERVAL);
};

const getPreKey = milliseconds => {
  return getKey(milliseconds) - 1;
};

const isComponent = arg => {
  return Object.keys(_constants.COMPONENTS).includes(arg);
};

const buildMetrics = rollingCounters => {
  if (!rollingCounters) {
    rollingCounters = _constants.DEFAULT_ROLLING_COUNTER;
  }

  const basicMetrics = _lodash.default.merge(rollingCounters, _constants.GLOBAL_BASIC_COUNTER);

  const overallActionMetrics = {
    request_total: 0,
    request_count: 0,
    success_count: 0,
    failed_request_count_system_error: 0,
    failed_request_count_user_error: 0
  };
  Object.keys(basicMetrics).forEach(key => {
    if (isComponent(key)) {
      for (const counter of Object.values(basicMetrics[key])) {
        overallActionMetrics.request_count += (counter === null || counter === void 0 ? void 0 : counter.count) || 0;
        overallActionMetrics.request_total += (counter === null || counter === void 0 ? void 0 : counter.total) || 0;
        overallActionMetrics.failed_request_count_system_error += (counter === null || counter === void 0 ? void 0 : counter.system_error) || 0;
        overallActionMetrics.failed_request_count_user_error += (counter === null || counter === void 0 ? void 0 : counter.user_error) || 0;
      }
    }
  });
  overallActionMetrics.success_count = overallActionMetrics.request_count - (overallActionMetrics.failed_request_count_system_error + overallActionMetrics.failed_request_count_user_error);
  return { ...basicMetrics,
    ...overallActionMetrics
  };
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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