//! `GET /_matrix/identity/*/3pid/getValidated3pid`
//!
//! Determine if a given 3PID has been validated by a user.

pub mod v2 {
    //! `/v2/` ([spec])
    //!
    //! [spec]: https://spec.matrix.org/latest/identity-service-api/#get_matrixidentityv23pidgetvalidated3pid

    use js_int::UInt;
    use ruma_common::{
        api::{request, response, Metadata},
        metadata,
        thirdparty::Medium,
        OwnedClientSecret, OwnedSessionId,
    };

    const METADATA: Metadata = metadata! {
        method: GET,
        rate_limited: false,
        authentication: AccessToken,
        history: {
            1.0 => "/_matrix/identity/v2/3pid/getValidated3pid/",
        }
    };

    /// Request type for the `check_3pid_validity` endpoint.
    #[request]
    pub struct Request {
        /// The Session ID generated by the `requestToken` call.
        #[ruma_api(query)]
        pub sid: OwnedSessionId,

        /// The client secret passed to the `requestToken` call.
        #[ruma_api(query)]
        pub client_secret: OwnedClientSecret,
    }

    /// Response type for the `check_3pid_validity` endpoint.
    #[response]
    pub struct Response {
        /// The medium type of the 3PID.
        pub medium: Medium,

        /// The address of the 3PID being looked up.
        pub address: String,

        /// Timestamp, in milliseconds, indicating the time that the 3PID was validated.
        pub validated_at: UInt,
    }

    impl Request {
        /// Creates a `Request` with the given Session ID and client secret.
        pub fn new(sid: OwnedSessionId, client_secret: OwnedClientSecret) -> Self {
            Self { sid, client_secret }
        }
    }

    impl Response {
        /// Creates a `Response` with the given medium, address and validation timestamp.
        pub fn new(medium: Medium, address: String, validated_at: UInt) -> Self {
            Self { medium, address, validated_at }
        }
    }
}
