/* Notes *//*{{{C}}}*//*{{{*/
/*

The functions in this module implement common editor primitives.  They
respect the fold structure and reposition the screen cursor position.
If needed, they cause a full screen redraw.  There is only one condition
under which the editor may bypass this module and directly call buffer
gap functions for buffers currently associated with a display: If they
do not modify the buffer contents and if they do not move the cursor
further than the current line.

*/
/*}}}*/

/* #includes *//*{{{*/
#include "config.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <assert.h>
#include <ctype.h>
#include <stdarg.h> /* required by some curses.h implementations */
#ifdef HAVE_NCURSES_NCURSES_H
#include <ncurses/ncurses.h>
#else
#ifdef HAVE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif
#endif
#include <errno.h>
#include <fcntl.h>
#ifdef HAVE_GETTEXT
#include <libintl.h>
#define _(String) gettext(String)
#else
#define _(String) String
#endif
#include <limits.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef BROKEN_REALLOC
#define realloc(s,l) myrealloc(s,l)
#endif
#include <string.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>

#include "buffer.h"
#include "display.h"
#include "macro.h"
#include "misc.h"
#include "msgline.h"
/*}}}*/
/* #defines */ /*{{{*/
#define CO_L "co -l"
#define CI_U "ci -u"
/*}}}*/

/* variables */ /*{{{*/
static struct Display displays[MAXDISPLAYS];
static struct Display *current;

/* Terminal uses UTF-8 */
int ds_utf8term;

/* All buffers used for displays or attachable to displays. */
struct Buffer *buffers[MAXDISPLAYS];

/* Shell offers suspend. */
static int ds_has_suspend;
/*}}}*/

/* ds_chstr      -- convert current character to displayed string */ /*{{{*/
static const char *ds_chstr(const struct MetaChar *ch, int opened, unsigned int x, unsigned int tabsize, int *highlight, int *visible)
{
  static char control[]="^?";
  static char noprint[]="U+xxxxxx";
  static char character[5]="?";
  static char tab[MAXTAB+1]="";
  unsigned char c1 = ASCII(*ch);

  if (tab[0]!=' ') { memset(tab,' ',MAXTAB); tab[MAXTAB]='\0'; }
  if (c1=='\0')
  {
    switch (MARK(*ch))
    {
      case NUL:
      {
        if (highlight) *highlight=0;
        *visible=2;
        return "^@";
      }
      case QUOTE:
      {
        /* Invalid UTF-8 byte */
        if (highlight) *highlight=0;
        *visible=3;
        snprintf(noprint,sizeof(noprint),"\\%02x",(unsigned int)(unsigned char)QUOTED(*ch));
        return noprint;
      }
      case FOLDSTART:
      {
        if (highlight) *highlight=1;
        *visible=3;
        return (opened ? "{{{" : "...");
      }
      case FOLDEND:
      {
        if (highlight) *highlight=1;
        *visible=3;
        return "}}}";
      }
      default: assert(0); return (const char*)0;
    }
  }
  else if (tabsize && c1=='\t')
  {
    if (highlight) *highlight=0;
    *visible = tabsize-(x%tabsize);
    return (tab+MAXTAB-*visible);
  }
  else if (c1>='\1' && c1<='\37')
  {
    if (highlight) *highlight=0;
    *visible=2;
    control[1]=c1+'A'-1;
    return control;
  }
  else if (c1=='\177')
  {
    if (highlight) *highlight=0;
    *visible=2;
    control[1]='?';
    return control;
  }
  else if (!ds_utf8term && !bf_utf8 && !isprint((int)c1))
  {
    /* Single byte terminal with single byte characters */
    if (highlight) *highlight=0;
    *visible=3;
    snprintf(noprint,sizeof(noprint),"\\%02x",c1);
    return noprint;
  }
  else if (!ds_utf8term && bf_utf8 && (c1&0x80))
  {
    /* Single byte terminal with non-ASCII UTF-8 characters */
    if (highlight) *highlight=0;
    *visible=snprintf(noprint,sizeof(noprint),"U+%04lX",bf_uc(ch));
    return noprint;
  }
  else
  {
    size_t len;

    if (highlight) *highlight=0;
    *visible=1;
    len=bf_rcharlen(ch->c);
    memcpy(character,ch->c,len);
    character[len]='\0';
    return character;
  }  
}
/*}}}*/
/* hangup        -- save everything after a signal arrived */ /*{{{*/
static void hangup(int sig)
{
  int i;
  const struct passwd *pw;
  int fd[2];

  for (i=0; i<MAXDISPLAYS; ++i)
  {
    if (displays[i].buffer)
    {
      if (displays[i].buffer->mode&CHANGED)
      {
        char name[_POSIX_PATH_MAX];
        size_t len=strlen(displays[i].buffer->name);

        if (len>(_POSIX_PATH_MAX-5)) len=_POSIX_PATH_MAX-5;
        strncpy(name,displays[i].buffer->name,len);
        name[len]='\0';
        strcat(name+len,".hup");
        bf_save(displays[i].buffer,name,"{{{","}}}",(unsigned int*)0);
      }
      else ds_free(&displays[i]);
    }
  }
  ds_exit(0);
#ifdef HAVE_SENDMAIL
  if ((pw=getpwuid(getuid()))!=(struct passwd*)0 && pipe(fd)!=-1)
  {
    switch (fork())
    {
      case 0:
      {
        close(fd[1]);
        dup2(fd[0],0);
        close(fd[0]);
        execlp(SENDMAIL,"sendmail","-t",(const char*)0);
        exit(127);
      }
      case -1: break;
      default:
      {
        FILE *fp;

        close(fd[0]);
        fp=fdopen(fd[1],"w");
        fprintf(fp,"To: %s\n",pw->pw_name);
        fprintf(fp,_("Subject: Abnormal termination of fe caused by signal %d.\n"),sig);
        fprintf(fp,"\n");
        fprintf(fp,_("The following files contain the contents of unsaved buffers:\n"));
        for (i=0; i<MAXDISPLAYS; ++i)
        {
          if (displays[i].buffer)
          {
            char name[_POSIX_PATH_MAX];
            size_t len=strlen(displays[i].buffer->name);

            assert(displays[i].buffer->refcnt==2);
            if (len>(_POSIX_PATH_MAX-5)) len=_POSIX_PATH_MAX-5;
            strncpy(name,displays[i].buffer->name,len);
            name[len]='\0';
            strcat(name+len,".hup");
            fprintf(fp,"%s\n",name);
            ds_free(&displays[i]);
          }
        }
        fclose(fp);
      }
    }
  }
#endif
  exit(2);
}
/*}}}*/

/* ds_init       -- init editor */ /*{{{*/
int ds_init(void)
{
  int fd;
  struct termios termios;
  struct sigaction hupact;

  if (!isatty(0))
  {
    fd=dup(0); assert(fd!=-1);
    close(0);
    if (open("/dev/tty",O_RDWR)!=0)
    {
      fputs("Initialising terminal failed (no controlling tty).\n",stderr);
      exit(1);
    }
  }
  else fd=-1;
  if (getenv("TERM")==(const char*)0)
  {
    fputs("Initialising terminal failed (TERM variable not set).\n",stderr);
    exit(1);
  }
  if (tcgetattr(0, &termios)==-1)
  {
    fputs("Retrieving terminal attributes failed.\n",stderr);
    exit(1);
  }
  ds_has_suspend=(termios.c_cc[VSUSP]!=_POSIX_VDISABLE);
  initscr();
  noecho();
  raw();
  nonl();
#ifndef BROKEN_CURSES_IDLOK
  idlok(stdscr,TRUE);
#endif
#ifdef HAVE_IDCOK
  idcok(stdscr,TRUE);
#endif
  scrollok(stdscr,TRUE);
  keypad(stdscr,TRUE);
  clear();
  hupact.sa_handler=hangup;
  sigemptyset(&hupact.sa_mask);
  sigaddset(&hupact.sa_mask,SIGHUP);
  sigaddset(&hupact.sa_mask,SIGTERM);
  hupact.sa_flags=0;
  sigaction(SIGHUP,&hupact,(struct sigaction *)0);
  sigaction(SIGTERM,&hupact,(struct sigaction *)0);
  return fd;
}
/*}}}*/
/* ds_exit       -- exit editor */ /*{{{*/
void ds_exit(int regular)
{
  showmsg(regular ? _("[Bye]") : _("[Signal arrived, bye]"));
  scrl(1);
  refresh();
  echo();
  noraw();
  endwin();
}
/*}}}*/
/* ds_winch      -- window changed */ /*{{{*/
void ds_winch(void)
{
  struct Display *i,*bottom=(struct Display*)0;

  for (i=displays; i<displays+sizeof(displays)/sizeof(displays[0]); ++i)
  {
    if (i->buffer)
    {
      i->maxx=COLS;
      i->update=ALL;
      if (bottom==(struct Display*)0 || bottom->oriy+bottom->maxy<i->oriy+i->maxy) bottom=i;
    }
  }
  if (bottom->oriy<LINES-1) bottom->maxy=LINES-1-bottom->oriy;
}
/*}}}*/
/* ds_alloc      -- allocate display */ /*{{{*/
struct Display *ds_alloc(struct Buffer *b, unsigned int orix, unsigned int oriy, unsigned int maxx, unsigned int maxy)
{
  struct Display *d;
  int i;

  for (i=0; i<elementsof(displays) && displays[i].buffer; ++i);
  if (i==elementsof(displays)) return (struct Display*)0;
  d=&(displays[i]);
  d->buffer=b;
  ++d->buffer->refcnt;
  d->x=0;
  d->offx=0;
  d->y=0;
  d->maxx=maxx;
  d->maxy=maxy;
  d->orix=orix;
  d->oriy=oriy;
  d->update=ALL;
  d->mode=0;
  d->tabsize=8;
  d->cursor=bf_mark(b,-1);
  d->mark=bf_mark(b,-1);
  if (d->buffer->refcnt==1)
  {
    ++d->buffer->refcnt;
    for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i]==(struct Buffer*)0)
    {
      buffers[i]=b;
      break;
    }
  }
  d->titlebuf=(char*)0;
  d->titlebuflen=0;
  return d;
}
/*}}}*/
/* ds_realloc    -- allocate new buffer to existing display */ /*{{{*/
void ds_realloc(struct Display *d, struct Buffer *b)
{
  int i;

  if (d->buffer) --d->buffer->refcnt;
  d->buffer=b;
  ++d->buffer->refcnt;
  if (d->buffer->refcnt==1)
  {
    ++d->buffer->refcnt;
    for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i]==(struct Buffer*)0)
    {
      buffers[i]=b;
      break;
    }
  }
  d->update=ALL;
  d->cursor=bf_mark(d->buffer,-1);
  d->mark=bf_mark(d->buffer,-1);
}
/*}}}*/
/* ds_free       -- free display */ /*{{{*/
void ds_free(struct Display *d)
{
  int i;

  bf_unmark(d->buffer,d->cursor);
  bf_unmark(d->buffer,d->mark);
  if (--d->buffer->refcnt==1)
  {
    for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i]==d->buffer)
    {
      buffers[i]=(struct Buffer*)0;
      break;
    }
    bf_free(d->buffer);
  }
  d->buffer=(struct Buffer*)0;
  if (d->titlebuf) free(d->titlebuf);
  if (d==current) current=(struct Display*)0;
}
/*}}}*/
/* ds_drawline   -- draw current line */ /*{{{*/
void ds_drawline(struct Display *d, int cursor_line, int y, int opened)
{
  int cur_gapstart,cur_linestart;
  struct MetaChar ch;
  unsigned int curx, endx;
  unsigned int offx;

  move(d->oriy+y,d->orix);
  cur_gapstart=d->buffer->curch;
  bf_linestart(d->buffer,1);
  cur_linestart=d->buffer->curch;
  if (cursor_line) /* note x position and determine display offset *//*{{{*/
  {
    curx=0;
    while (bf_rchar(d->buffer,&ch) && ASCII(ch)!='\n')
    {
      int visible;

      if (d->buffer->curch==cur_gapstart) d->x=curx;
      ds_chstr(&ch,opened,curx,d->tabsize,(int*)0,&visible);
      curx+=visible;
      bf_forward(d->buffer);
    }
    if (d->buffer->curch==cur_gapstart) d->x=curx;
    while (d->buffer->curch>cur_linestart) bf_backward(d->buffer);
    while (d->buffer->curch<cur_linestart) bf_forward(d->buffer);
    if (d->x>d->offx+d->maxx-1-(d->mode&LINENO ? LINENOWIDTH : 0))
    {
      d->offx=d->x-d->maxx+1+(d->mode&LINENO ? LINENOWIDTH : 0);
      if (d->mode&SIDESCROLL) d->update=ALL;
    }
    else if (d->x<d->offx)
    {
      d->offx=d->x;
      if (d->mode&SIDESCROLL) d->update=ALL;
    }
  }
  /*}}}*/
  if ((d->mode&SIDESCROLL)==0 && y!=d->y) offx=0;
  else offx=d->offx;
  /* draw line *//*{{{*/
  endx=d->maxx+offx;
  if (d->mode&LINENO)
  {
    char s[LINENOWIDTH+1];

    sprintf(s,"%0*u:",LINENOWIDTH-1,d->buffer->cury+1);
    addstr(s);
    endx-=LINENOWIDTH;
  }
  for (curx=0; curx<endx && bf_rchar(d->buffer,&ch) && ASCII(ch)!='\n'; )
  {
    const char *s;
    int highlight,visible;
      
    s=ds_chstr(&ch,opened,curx,d->tabsize,&highlight,&visible);
    if (highlight && d->mode&BOLDMARKS) wattron(stdscr,A_BOLD);
    if (visible==1)
    {
      while (*s)
      {
        if (curx>=offx) addch((chtype)(unsigned char)*s);
        ++s;
      }
      ++curx;
    }
    else
    {
      while (*s && curx<endx)
      {
        if (curx>=offx) addch((chtype)(unsigned char)*s);
        ++s;
        ++curx;
      }
    }
    if (highlight) wattroff(stdscr,A_BOLD);
    if (bf_forward(d->buffer)==0) break;
  }
  if (curx>=endx)
  {
    move(d->oriy+y,d->orix+d->maxx-1);
    addch((chtype)'$');
  }
  else
  {
#if 0
    int clr;

    for (clr=curx; clr<endx; ++clr)
    {
      if (clr>=offx) addch(' ');
    }
#else
    clrtoeol();
#endif
  }
  /*}}}*/
  while (d->buffer->curch>cur_gapstart) bf_backward(d->buffer);
  while (d->buffer->curch<cur_gapstart) bf_forward(d->buffer);
}
/*}}}*/
/* ds_detach     -- detach display */ /*{{{*/
void ds_detach(struct Display *d)
{
  bf_unmark(d->buffer,d->cursor);
  bf_unmark(d->buffer,d->mark);
  --d->buffer->refcnt;
  d->buffer=(struct Buffer*)0;
  if (d==current) current=(struct Display*)0;
}
/*}}}*/
/* ds_redraw     -- redraw screen */ /*{{{*/
void ds_redraw(void)
{
  touchwin(curscr);
  wrefresh(curscr);
}
/*}}}*/
/* ds_set_current-- select current display */ /*{{{*/
void ds_set_current(struct Display *d)
{
  if (current) bf_mark(current->buffer,current->cursor);
  current=d;
  bf_tomark(current->buffer,current->cursor);
}
/*}}}*/
/* ds_get_current-- get current display */ /*{{{*/
struct Display *ds_get_current(void)
{
  return current;
}
/*}}}*/
/* ds_tocol      -- position cursor to a specific column */ /*{{{*/
void ds_tocol(int tox)
{
  struct MetaChar ch;
  int x,dx;

  ASCII(ch)='\0';
  x=0;
  while (bf_lchar(current->buffer,&ch) && ASCII(ch)!='\n' && bf_backward(current->buffer));
  if (ASCII(ch)=='\n' || current->buffer->gapstart==current->buffer->buffer)
  {
    while (tox)
    {
      if (bf_rchar(current->buffer,&ch)==0) break;
      if (ASCII(ch)=='\n') break;
      ds_chstr(&ch,1,x,current->tabsize,(int*)0,&dx);
      if (dx<=tox)
      {
        tox-=dx;
        x+=dx;
      }
      else tox=0;
      bf_forward(current->buffer);
    }
  }
}
/*}}}*/
/* ds_torow      -- position cursor to a specific row */ /*{{{*/
void ds_torow(struct Display *d, int row)
{
  d->y=row;
  d->update=(d->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_topos      -- go to buffer position */ /*{{{*/
void ds_topos(struct Display *d, const char *start)
{
  while (d->buffer->gapstart<start) bf_forward(d->buffer);
  while (d->buffer->gapstart>start) bf_backward(d->buffer);
  d->update=(d->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_refresh    -- refresh displays */ /*{{{*/
void ds_refresh(void)
{
  /* variables *//*{{{*/
  struct Display *d;
  char *cur_gapstart;
  int i,j;
  /*}}}*/

  bf_mark(current->buffer,current->cursor);
  for (i=0; i<elementsof(displays); ++i) if (displays[i].buffer && displays[i].update&MODIFIED)
  {
    for (j=0; j<elementsof(displays); ++j) if (displays[i].buffer && displays[j].buffer==displays[i].buffer && i!=j) displays[j].update=ALL;
  }
  for (i=0; i<elementsof(displays); ++i) if (displays[i].buffer)
  {
    d=&(displays[i]);
    do
    {
      bf_tomark(d->buffer,d->cursor);

      /* note where start of gap is *//*{{{*/
      cur_gapstart=d->buffer->gapstart;
      /*}}}*/
      if (d->update&LINE) ds_drawline(d,1,d->y,0);
      if (d->update&ALL)
      {
        /* variables */ /*{{{*/
        int opened;
        int cury;
        struct MetaChar ch;
        /*}}}*/

        d->update=NONE;
        /* draw current line and all previous lines until top of screen */ /*{{{*/
        cury=d->y+1;
        opened=0;
        do
        {
          --cury;
          /* draw current line *//*{{{*/
/* 0 means lines are not scrolled sideways when moving up from a scrolled line */
#if 0
          ds_drawline(d,cury==d->y,cury,opened);
#else
          ds_drawline(d,0,cury,opened);
#endif
          /*}}}*/
          /* go to beginning of it *//*{{{*/
          bf_linestart(d->buffer,1);
          /*}}}*/
          /* go one more back to get to end of previous line *//*{{{*/
          if (bf_backward(d->buffer)==0) break;
          /*}}}*/
          /* if this is an endfold, go to its matching startfold *//*{{{*/
          opened=1;
          while (bf_lchar(d->buffer,&ch))
          {
            if (ASCII(ch)=='\0' && MARK(ch)==FOLDEND)
            {
              unsigned int level;

              level=0;
              do
              {
                if (bf_backward(d->buffer)==0) break;
                if (bf_rchar(d->buffer,&ch)==0) break;
                if (ASCII(ch)=='\0' && MARK(ch)==FOLDSTART) --level;
                if (ASCII(ch)=='\0' && MARK(ch)==FOLDEND) ++level;
              } while (!(ASCII(ch)=='\0' && MARK(ch)==FOLDSTART && level==0));
              opened=0;
              break;
            }
            else if (ASCII(ch)=='\n') break;
            else bf_backward(d->buffer);
          }
          /*}}}*/
        } while (cury);
        /*}}}*/
        /* clear all lines above start of text */ /*{{{*/
        if (cury && d->mode&SHOWBEGEND)
        {
          const char *msg;
          int len;
          int clr;

          msg=_("{{{ Beginning of file");
          len=(int)strlen(msg);
          --cury;
          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(clr<len ? msg[clr] : ' ');
        }
        if (cury) { d->y-=cury; d->update=ALL; }
        else while (cury)
        {
          int clr;      

          --cury;
          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(' ');
        }
        /*}}}*/
        /* move behind current line */ /*{{{*/
        while (d->buffer->gapstart>cur_gapstart) bf_backward(d->buffer);
        while (d->buffer->gapstart<cur_gapstart) bf_forward(d->buffer);
        if (bf_isfold(d->buffer,FOLDSTART,LEFT|RIGHT))
        {
          int r;
        
bf_isfold(d->buffer,FOLDSTART,LEFT|RIGHT);
          bf_lineend(d->buffer,1);
          r=bf_tofoldend(d->buffer,1,(int*)0);
          assert(r);
        }
        /*}}}*/
        /* draw all lines below current line */ /*{{{*/
        bf_lineend(d->buffer,1);
        cury=d->y;
        for (; bf_forward(d->buffer) && cury<d->maxy-1; ++cury)
        {
          /* draw current line *//*{{{*/
          ds_drawline(d,0,cury+1,0);
          /*}}}*/
          /* if this is an startfold, go to its matching endfold *//*{{{*/
          if (bf_isfold(d->buffer,FOLDSTART,LEFT|RIGHT))
          {
            int r;

            bf_lineend(d->buffer,1);
            r=bf_tofoldend(d->buffer,1,(int*)0);
            assert(r);
          }
          /*}}}*/
          /* go to end of it *//*{{{*/
          bf_lineend(d->buffer,1);
          /*}}}*/
        }
        if (bf_lchar(d->buffer,&ch) && ASCII(ch)!='\n') ++cury;
        /*}}}*/
        /* clear all lines below end of text */ /*{{{*/
        if (cury<d->maxy-1 && d->mode&SHOWBEGEND)
        {
          const char *msg;
          int len;
          int clr;

          msg=_("}}} End of file");
          len=(int)strlen(msg);
          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(clr<len ? msg[clr] : ' ');
          ++cury;
        }
        for (; cury<d->maxy-1; ++cury)
        {
          int clr;

          move(d->oriy+cury,d->orix);
          for (clr=0; clr<d->maxx; ++clr) addch(' ');
        }
        /*}}}*/
      }
      else d->update=NONE;
      /* move back to old start of gap *//*{{{*/
      while (d->buffer->gapstart>cur_gapstart) bf_backward(d->buffer);
      while (d->buffer->gapstart<cur_gapstart) bf_forward(d->buffer);
      /*}}}*/
    } while (d->update&ALL);
    /* draw status line *//*{{{*/
    {
      int x;
      time_t t;
      static time_t ot=(time_t)0;
      static char timebuf[40];

      if (d->titlebuflen<80) d->titlebuf=malloc(d->titlebuflen=80);
      if (d->buffer->mode&CHANGED) strcpy(d->titlebuf," * ");
      else strcpy(d->titlebuf,"   ");
      strcat(d->titlebuf,"fe " VERSION " (");
      strcat(d->titlebuf,d->buffer->lang && d->buffer->lang->name[0] ? d->buffer->lang->name : _("No language"));
      if (bf_utf8)
      {
        strcat(d->titlebuf,",");
        strcat(d->titlebuf,"UTF-8");
      }
      if (d->buffer->mode&READONLY)
      {
        strcat(d->titlebuf,",");
        strcat(d->titlebuf,_("Read-only"));
      }
      if (d->mode&OVERWRITE)
      {
        strcat(d->titlebuf,",");
        strcat(d->titlebuf,_("Overwrite"));
      }
      if (d->mode&MAGIC)
      {
        strcat(d->titlebuf,",");
        strcat(d->titlebuf,_("Magic"));
      }
      strcat(d->titlebuf,") ");
      if (d->buffer->name)
      {
        size_t namelen;
        
        strcat(d->titlebuf,"\"");
        if (d->titlebuflen<(namelen=(80+strlen(d->buffer->name)))) d->titlebuf=realloc(d->titlebuf,d->titlebuflen=namelen);
        strcat(d->titlebuf,d->buffer->name);
        strcat(d->titlebuf,"\"");
      }
      else strcat(d->titlebuf,_("'unnamed'"));
      strcat(d->titlebuf," ");
      if (d->mode&POSITION)
      {
        sprintf(d->titlebuf+strlen(d->titlebuf),_("Line %d Column %d"),d->buffer->cury+1,d->x);
        strcat(d->titlebuf," ");
      }
      if (d->mode&TIME)
      {
        t=time((time_t*)0);
        if (t>ot)
        {
          struct tm *tm;

          tm=localtime(&t);
          assert(tm!=(struct tm*)0);
          ot=t+59-tm->tm_sec;
          strftime(timebuf,sizeof(timebuf),_("(%H:%M)"),tm);
        }
        strcat(d->titlebuf,timebuf);
      }
      if (d->titlebuflen>(size_t)d->maxx) d->titlebuf[d->maxx]='\0';
      move(d->oriy+d->maxy-1,d->orix);
      wattron(stdscr,A_REVERSE);
      addstr(d->titlebuf);
      for (x=strlen(d->titlebuf); x<d->maxx; ++x) addch(' ');
      wattroff(stdscr,A_REVERSE);
    }
    /*}}}*/
  }
  move(current->oriy+current->y,current->orix+current->x-current->offx+(current->mode&LINENO ? LINENOWIDTH : 0));
  bf_tomark(current->buffer,current->cursor);
}
/*}}}*/
/* ds_forward    -- move one character forward, skipping folds */ /*{{{*/
int ds_forward(struct Display *d)
{
  struct MetaChar ch;
  int res;
        
  if (bf_rchar(d->buffer,&ch)==0) return 0;
  if (ASCII(ch)=='\n') 
  {
    int olddy;

    olddy=d->y;
    if (bf_isfold(d->buffer,FOLDEND,LEFT|RIGHT)) d->update=(d->update&~UPDATEMASK)|ALL;
    if (d->y<d->maxy-2) ++d->y; else d->update=(d->update&~UPDATEMASK)|ALL;
    if (bf_isfold(d->buffer,FOLDSTART,LEFT|RIGHT))
    {
      char *gapstart;

      gapstart=d->buffer->gapstart;
      if (bf_tofoldend(d->buffer,1,(int*)0)==0 || bf_lineend(d->buffer,1)==0 || (res=bf_forward(d->buffer))==0)
      {
        while (d->buffer->gapstart>gapstart) bf_backward(d->buffer);
        d->y=olddy;
        return 0;
      }
    }
    else res=bf_forward(d->buffer);
    if ((d->mode&SIDESCROLL)==0 && d->offx)
    {
      d->offx=0;
      d->update=(d->update&~UPDATEMASK)|ALL;
    }
  }
  else res=bf_forward(d->buffer);
  if ((d->update&UPDATEMASK)==NONE) d->update|=LINE;
  return (res);
}
/*}}}*/
/* ds_backward   -- move one character backward, skipping folds *//*{{{*/
int ds_backward(struct Display *d)
{
  struct MetaChar ch;
      
  if (bf_lchar(d->buffer,&ch)==0) return 0;
  bf_backward(d->buffer);
  if (ASCII(ch)=='\n')
  {
    if (bf_isfold(d->buffer,FOLDSTART,LEFT|RIGHT)) d->update=(d->update&~UPDATEMASK)|ALL|LINE;
    if (d->y>0) --d->y; else d->update=(d->update&~UPDATEMASK)|ALL|LINE;
    if (bf_isfold(d->buffer,FOLDEND,LEFT|RIGHT))
    {
      bf_tofoldstart(d->buffer,0,(int*)0);
      bf_lineend(d->buffer,1);
    }
    if ((d->mode&SIDESCROLL)==0 && d->offx)
    {
      d->offx=0;
      d->update=(d->update&~UPDATEMASK)|ALL|LINE;
    }
  }
  if ((d->update&UPDATEMASK)==NONE) d->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_prevline   -- previous line, skipping folds *//*{{{*/
int ds_prevline(void) /* 0=failed, 1=success, 2=success & hit start */
{
  char *gapstart;
  int hitstart=0;

  gapstart=current->buffer->gapstart;
  if (current->buffer->cury==0 || bf_linestart(current->buffer,1)==0 || bf_backward(current->buffer)==0)
  {
    while (current->buffer->gapstart<gapstart) bf_forward(current->buffer);
    if (current->y==0 && current->mode&SHOWBEGEND)
    {
      current->y=1;
      current->update=(current->update&~UPDATEMASK)|ALL;
      return 1;
    }
    else return 0;
  }
  if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT))
  {
    if (bf_tofoldstart(current->buffer,0,(int*)0)==0)
    {
      while (current->buffer->gapstart<gapstart) bf_forward(current->buffer);
      if (current->y==0 && current->mode&SHOWBEGEND)
      {
        current->y=1;
        current->update=(current->update&~UPDATEMASK)|ALL;
        return 1;
      }
      else return 0;
    }
  }
  else if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT))
  {
    current->update=(current->update&~UPDATEMASK)|ALL;
    hitstart=1;
  }
  if ((current->mode&SIDESCROLL)==0 && current->offx)
  {
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  if (current->y>0) --current->y;
  else current->update=(current->update&~UPDATEMASK)|ALL;
  if (current->y==0 && current->buffer->cury>0)
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  ds_tocol(current->x);
  return 1+hitstart;
}
/*}}}*/
/* ds_nextline   -- next line, skipping folds *//*{{{*/
int ds_nextline(void)
{
  char *gapstart;

  gapstart=current->buffer->gapstart;
  if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT))
  {
    if (bf_lineend(current->buffer,1)==0 || bf_tofoldend(current->buffer,1,(int*)0)==0)
    {
      while (current->buffer->gapstart>gapstart) bf_backward(current->buffer);
      return 0;
    }
  }
  else if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT)) current->update=(current->update&~UPDATEMASK)|ALL;
  if (bf_lineend(current->buffer,1))
  { 
    if (bf_forward(current->buffer))
    {
      if (current->y<current->maxy-2) ++current->y;
      if (current->y==(current->maxy-2))
      {
        if (bf_lineend(current->buffer,1) && bf_forward(current->buffer))
        {
          current->update=(current->update&~UPDATEMASK)|ALL;
          current->y=(current->maxy-1)/2;
          bf_backward(current->buffer);
        }
      }
    }
    else
    {
      while (current->buffer->gapstart>gapstart) bf_backward(current->buffer);
      return 0;
    }
  }
  else
  {
    while (current->buffer->gapstart>gapstart) bf_backward(current->buffer);
    return 0;
  }
  if ((current->mode&SIDESCROLL)==0 && current->offx)
  {
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  ds_tocol(current->x);
  return 1;
}
/*}}}*/
/* ds_closefold  -- close fold *//*{{{*/
int ds_closefold(void)
{
  int back;

  if (bf_tofoldstart(current->buffer,1,&back))
  {
    if (back>current->y) current->y=0; else current->y-=back;
    ds_tocol(current->x);
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_openfold   -- open fold *//*{{{*/
int ds_openfold(void)
{
  if (bf_isfold(current->buffer,FOLDSTART,LEFT|RIGHT))
  {
    bf_lineend(current->buffer,1);
    if (bf_forward(current->buffer)==1) if (current->y<current->maxy-2) ++current->y;
    ds_tocol(current->x);
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_begin      -- go to begin of buffer *//*{{{*/
void ds_begin(void)
{
  bf_begin(current->buffer);
  current->y=(current->mode&SHOWBEGEND)!=0;
  current->update=(current->update&~UPDATEMASK)|ALL|LINE;
}
/*}}}*/
/* ds_end        -- go to end of buffer *//*{{{*/
void ds_end(void)
{
  bf_end(current->buffer);
  current->y=current->maxy-2;
  current->update=(current->update&~UPDATEMASK)|ALL|LINE;
}
/*}}}*/
/* ds_beginfold  -- go to beginning of current fold *//*{{{*/
int ds_beginfold(void)
{
  int back;

  if (bf_tofoldstart(current->buffer,1,&back)==1)
  {
    if (current->y>(back-1)) current->y-=(back-1);
    else
    {
      current->y=0;
      current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    }
    bf_lineend(current->buffer,1);
    bf_forward(current->buffer);
    ds_tocol(current->x);
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_endfold    -- go to end of current fold *//*{{{*/
int ds_endfold(void)
{
  int x;
  int forward;

  x=current->x;
  if (bf_tofoldend(current->buffer,1,&forward)==1)
  {
    if ((current->y+forward)<(current->maxy-1)) current->y+=forward;
    else
    {
      current->y=current->maxy-1;
      current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    }
    ds_prevline();
    ds_tocol(x);
    return 1;
  }
  else return 0;
}
/*}}}*/
/* ds_beginline  -- go to beginning of line */ /*{{{*/
void ds_beginline(struct Display *d)
{
  bf_linestart(d->buffer,0);
  if ((d->update&UPDATEMASK)==NONE) d->update|=LINE;
}
/*}}}*/
/* ds_endline    -- go to end of line */ /*{{{*/
void ds_endline(struct Display *d)
{
  bf_lineend(d->buffer,0);
  if ((d->update&UPDATEMASK)==NONE) d->update|=LINE;
}
/*}}}*/
/* ds_delmode    -- delete mode *//*{{{*/
void ds_delmode(struct Display *d, enum Displaymode mode)
{
  d->mode&=~mode;
  d->update=(d->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_addmode    -- add mode *//*{{{*/
void ds_addmode(struct Display *d, enum Displaymode mode)
{
  d->mode|=mode;
  d->update=(d->update&~UPDATEMASK)|ALL;
}
/*}}}*/
/* ds_tabsize    -- get/set tab size *//*{{{*/
unsigned int ds_tabsize(int tabsize)
{
  if (tabsize>=0)
  {
    current->tabsize=tabsize;
    current->update=(current->update&~UPDATEMASK)|ALL;
  }
  return current->tabsize;
}
/*}}}*/
/* ds_insertch   -- insert a character */ /*{{{*/
int ds_insertch(struct Display *d, const struct MetaChar *ch)
{
  assert(ASCII(*ch)!='\0' || MARK(*ch)==NUL || MARK(*ch)==QUOTE);

  if (d->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }

  if (d->mode&OVERWRITE)
  {
    struct MetaChar here;

    if (bf_rchar(d->buffer,&here)==0 || (ASCII(here)=='\0' && (MARK(here)!=NUL && MARK(here)!=QUOTE)) || ASCII(here)=='\n')
    {
      showmsg(_("[You can not overwrite this position]"));
      return 0;
    }
    bf_forward(d->buffer);
    bf_delete(d->buffer);
  }

  if (bf_insert(d->buffer,ch))
  {
    if (ASCII(*ch)=='\n')
    {
      if (d->y<current->maxy-2) ++d->y;
      d->update=ALL|MODIFIED|LINE;
    }
    if ((d->update&UPDATEMASK)==NONE) d->update|=LINE|MODIFIED;
    return 1;
  }
  else
  {
    showmsg(_("[Out of memory]"));
    return 0;
  }
}        
/*}}}*/
/* ds_deletech   -- delete a character *//*{{{*/
int ds_deletech(struct Display *d, int previous)
{
  struct MetaChar ch;

  if (d->buffer->mode&READONLY) showmsg(_("[Buffer is read-only]"));
  if (previous) /* delete previous character */ /*{{{*/
  {
    if (bf_lchar(d->buffer,&ch))
    {
      if (ASCII(ch)=='\n')
      {
        int isfold;

        isfold=bf_isfold(d->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT);
        bf_backward(d->buffer);
        if (isfold && bf_isfold(d->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT))
        {
          bf_forward(d->buffer);
          return 0;
        }
        bf_forward(d->buffer);
        if (d->y>0) --d->y;
        d->update=ALL|MODIFIED|LINE;
      }
      else if (ASCII(ch)=='\0' && MARK(ch)!=QUOTE && MARK(ch)!=NUL) return 0;
      if (bf_delete(d->buffer))
      {
        d->update|=MODIFIED;
        if ((d->update&UPDATEMASK)==NONE) d->update|=LINE;
        return 1;
      }
      else return 0;
    }
    else return 0;
  }
  /*}}}*/
  else
  /* delete character under cursor *//*{{{*/
  {
    int cy;

    cy=d->y;
    if (bf_forward(d->buffer)==0) return 0;
    if (ds_deletech(d,1)==0)
    {
      bf_backward(d->buffer);
      return 0;
    }
    d->y=cy;
    return 1;
  }
  /*}}}*/
}
/*}}}*/
/* ds_mark       -- set a mark *//*{{{*/
void ds_mark(void)
{
  bf_mark(current->buffer,current->mark);
}
/*}}}*/
/* ds_swapmark   -- swap cursor and mark */ /*{{{*/
void ds_swapmark(void)
{
  int mark;

  bf_mark(current->buffer,current->cursor);
  bf_tomark(current->buffer,current->mark);
  mark=current->mark;
  current->mark=current->cursor;
  current->cursor=mark;
  current->update|=ALL|LINE;
}
/*}}}*/
/* ds_fold       -- fold region *//*{{{*/
int ds_fold(void)
{
  int crossedLine,level;
  struct MetaChar ch;
  unsigned int distance,moved;

  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT)) return 0;
  moved=crossedLine=level=0;
  bf_mark(current->buffer,current->cursor);
  if (current->buffer->mark[current->cursor]>current->buffer->mark[current->mark])
  {
    unsigned int x;

    x=current->cursor;
    current->cursor=current->mark;
    current->mark=x;
    bf_tomark(current->buffer,current->cursor);
  }
  distance=current->buffer->mark[current->mark]-current->buffer->mark[current->cursor];
  for (moved=0; moved<distance; ++moved) /* scan if region crosses folds and contains at least one newline *//*{{{*/
  {
    bf_forward(current->buffer);
    if (bf_lchar(current->buffer,&ch))
    {
      if (ASCII(ch)=='\n') crossedLine=1;
      else if (ASCII(ch)=='\0') switch (MARK(ch))
      {
        case FOLDSTART: ++level; break;
        case FOLDEND: --level; break;
      }
    }
  }
  /*}}}*/
  if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT))
  {
    for (moved=0; moved<distance; ++moved) bf_backward(current->buffer);
    return 0;
  }
  current->update=ALL|MODIFIED|LINE;
  if (crossedLine && level==0)
  {
    bf_insert(current->buffer,&bf_foldend);
    ++distance;
    for (moved=0; moved<distance; ++moved) bf_backward(current->buffer);
    bf_insert(current->buffer,&bf_foldstart);
    return 1;
  }
  else
  {
    for (moved=0; moved<distance; ++moved) bf_backward(current->buffer);
    return 0;
  }
}
/*}}}*/
/* ds_unfold     -- unfold *//*{{{*/
int ds_unfold(void)
{
  struct MetaChar ch;

  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  bf_mark(current->buffer,current->cursor);
  bf_linestart(current->buffer,1);
  while (bf_rchar(current->buffer,&ch))
  {
    if (ASCII(ch)=='\0' && MARK(ch)==FOLDSTART)
    {
      bf_forward(current->buffer);
      bf_delete(current->buffer);
      bf_tofoldend(current->buffer,1,(int*)0);
      bf_delete(current->buffer);
      bf_tomark(current->buffer,current->cursor);
      current->update=ALL|MODIFIED|LINE;
      return 1;
    }
    else if (ASCII(ch)=='\0' && MARK(ch)==FOLDEND)
    {
      bf_forward(current->buffer);
      bf_delete(current->buffer);
      bf_tofoldstart(current->buffer,1,(int*)0);
      bf_forward(current->buffer);
      bf_delete(current->buffer);
      bf_tomark(current->buffer,current->cursor);
      current->update=ALL|MODIFIED|LINE;
      return 1;
    }
    else if (ASCII(ch)=='\n') break;
    if (bf_forward(current->buffer)==0) break;
  }
  bf_tomark(current->buffer,current->cursor);
  return 0;
}
/*}}}*/
/* ds_forword    -- forward word *//*{{{*/
int ds_forword(enum Wordaction action)
{
  struct MetaChar ch;
            
  if (bf_rchar(current->buffer,&ch)==0) return 0;
  if (isalnum((int)ASCII(ch)))
  do
  {
    if (ds_forward(current)==0) break;
    switch (action)
    {
      case NOTHING:
      {
        break;
      }
      case TOUPPER:
      {
        bf_delete(current->buffer);
        ASCII(ch)=toupper(ASCII(ch));
        bf_insert(current->buffer,&ch);
        break;
      }
      case TOLOWER:
      {
        bf_delete(current->buffer);
        ASCII(ch)=tolower(ASCII(ch));
        bf_insert(current->buffer,&ch);
        break;
      }
      case CAPITALISE:
      {
        bf_delete(current->buffer);
        ASCII(ch)=toupper(ASCII(ch));
        bf_insert(current->buffer,&ch);
        action=TOLOWER;
        break;
      }
      case DELETE:
      {
        bf_delete(current->buffer);
        break;
      }
      default: assert(0);
    }
    if (bf_rchar(current->buffer,&ch)==0) break;
  } while (isalnum((int)ASCII(ch)));
  else if (ASCII(ch)==' ' || ASCII(ch)=='\t' || ASCII(ch)=='\n')
  do
  {
    if (ds_forward(current)==0) break;
    if (action==DELETE) 
    {
      bf_delete(current->buffer);
      if (ASCII(ch)=='\n') current->update|=ALL|LINE;
    }
    if (bf_rchar(current->buffer,&ch)==0) break;
  } while (ASCII(ch)==' ' || ASCII(ch)=='\t' || ASCII(ch)=='\n');
  else
  {
    ds_forward(current);
    if (action==DELETE) bf_delete(current->buffer);
  }
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_backword   -- backward word *//*{{{*/
int ds_backword(enum Wordaction action)
{
  struct MetaChar ch;
            
  assert(action==NOTHING || action==DELETE);
  if (bf_lchar(current->buffer,&ch)==0) return 0;
  if (isalnum((int)ASCII(ch)))
  do
  {
    if (action==NOTHING) ds_backward(current);
    else bf_delete(current->buffer);
    if (bf_lchar(current->buffer,&ch)==0) break;
  } while (isalnum((int)ASCII(ch)));
  else if (ASCII(ch)==' ' || ASCII(ch)=='\t' || ASCII(ch)=='\n')
  do
  {
    if (action==NOTHING) ds_backward(current);
    else
    {
      bf_delete(current->buffer);
      if (ASCII(ch)=='\n') current->update|=ALL|LINE;
    }
    if (bf_lchar(current->buffer,&ch)==0) break;
  } while (ASCII(ch)==' ' || ASCII(ch)=='\t' || ASCII(ch)=='\n');
  else
  {
    if (action==NOTHING) ds_backward(current);
    else bf_delete(current->buffer);
  }
  if ((current->update&UPDATEMASK)==NONE) current->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_gotoline   -- go to line *//*{{{*/
int ds_gotoline(int line)
{
  unsigned int dx;

  dx=current->x;
  if (bf_gotoline(current->buffer,line)==0) return 0;
  current->update=(current->update&~UPDATEMASK)|ALL|LINE;
  current->y=(current->maxy-1)/2;
  ds_tocol(dx);
  return 1;
}
/*}}}*/
/* ds_transpose  -- transpose characters *//*{{{*/
int ds_transpose(struct Display *d)
{
  struct MetaChar lch,rch;

  if (d->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (bf_lchar(d->buffer,&lch)==0 || bf_rchar(d->buffer,&rch)==0)
  {
    showmsg(_("[These characters can not be transposed]"));
    return 0;
  }
  if (ASCII(lch)=='\n')
  {
    if (bf_isfold(d->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT) && ASCII(rch)=='\0' && (MARK(rch)==FOLDSTART || MARK(rch)==FOLDEND))
    {
      showmsg(_("[These characters can not be transposed]"));
      return 0;
    }
    d->update=ALL|MODIFIED|LINE;
  }
  if (ASCII(rch)=='\n')
  {
    bf_forward(d->buffer);
    if (bf_isfold(d->buffer,FOLDSTART|FOLDEND,LEFT|RIGHT) && ASCII(lch)=='\0' && (MARK(lch)==FOLDSTART || MARK(lch)==FOLDEND))
    {
      bf_backward(d->buffer);
      showmsg(_("[These characters can not be transposed]"));
      return 0;
    }
    bf_backward(d->buffer);
    if (d->y<d->maxy-2) ++d->y;
    d->update=ALL|MODIFIED|LINE;
  }
  bf_delete(d->buffer);
  bf_forward(d->buffer);
  bf_insert(d->buffer,&lch);
  if ((d->update&UPDATEMASK)==NONE) d->update|=LINE;
  return 1;
}
/*}}}*/
/* ds_strfsearch -- search string forward *//*{{{*/
int ds_strfsearch(const char *needle, size_t needlesize)
{
  if (bf_strfsearch(current->buffer,needle,needlesize))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_regfsearch -- search regexp forward *//*{{{*/
int ds_regfsearch(const regex_t *needle)
{
  if (bf_regfsearch(current->buffer,needle))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_strbsearch -- search string backward *//*{{{*/
int ds_strbsearch(const char *needle, size_t needlesize)
{
  if (bf_strbsearch(current->buffer,needle,needlesize))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_regbsearch -- search regexp backward *//*{{{*/
int ds_regbsearch(const regex_t *needle)
{
  if (bf_regbsearch(current->buffer,needle))
  {
    current->y=(current->maxy-1)/2;
    current->update=(current->update&~UPDATEMASK)|ALL|LINE;
    return 1;
  }
  else
  {
    return 0;
  }
}
/*}}}*/
/* ds_insertbuf  -- insert buffer at cursor position *//*{{{*/
int ds_insertbuf(struct Buffer *b, int markstart)
{
  int count,i;
  struct MetaChar ch;

  assert(b);
  if (current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,LEFT))
  {
    bf_begin(b);
    if (bf_isfold(b,FOLDSTART|FOLDEND,LEFT|RIGHT))
    {
      showmsg(_("[Insertion failed at this cursor position]"));
      return 0;
    }
  }
  else if (bf_isfold(current->buffer,FOLDSTART|FOLDEND,RIGHT))
  {
    bf_end(b);
    if (bf_isfold(b,FOLDSTART|FOLDEND,LEFT|RIGHT))
    {
      showmsg(_("[Insertion failed at this cursor position]"));
      return 0;
    }
  }
  bf_begin(b);
  for (count=0; bf_rchar(b,&ch); ++count)
  {
    if (ASCII(ch)=='\n') current->update=ALL|MODIFIED|LINE;
    if (bf_insert(current->buffer,&ch)==0)
    {
      while (count>0) bf_delete(current->buffer);
      showmsg(_("[Out of memory]"));
      return 0;
    }
    bf_forward(b);
  }
  if (markstart)
  {
    for (i=0; i<count; ++i) bf_backward(current->buffer);
    bf_mark(current->buffer,current->mark);
    for (i=0; i<count; ++i) bf_forward(current->buffer);
  }
  if ((current->update&UPDATEMASK)==NONE) current->update|=(LINE|MODIFIED);
  return 1;
}
/*}}}*/
/* ds_covebuf    -- copy or move region into new buffer *//*{{{*/
struct Buffer *ds_covebuf(struct Display *d, int delete, int copy)
{
  /* variables *//*{{{*/
  struct Buffer *b;
  int level;
  unsigned int distance,moved;
  int lf;
  int joinedLine;
  struct MetaChar ch;
  /*}}}*/

  assert(d!=(struct Display*)0);
  if (delete && current->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if ((b=bf_alloc())==(struct Buffer*)0) return (struct Buffer*)0;
  bf_mark(d->buffer,d->cursor);
  if (d->buffer->mark[d->cursor]>d->buffer->mark[d->mark])
  {
    unsigned int x;

    x=d->cursor;
    d->cursor=d->mark;
    d->mark=x;
    bf_tomark(d->buffer,d->cursor);
  }
  distance=d->buffer->mark[d->mark]-d->buffer->mark[d->cursor];
  lf=bf_isfold(d->buffer,FOLDSTART|FOLDEND,LEFT);
  joinedLine=level=0;
  for (moved=0; moved<distance; ++moved) /* scan if region crosses folds, copy region if needed *//*{{{*/
  {
    bf_forward(d->buffer);
    if (bf_lchar(d->buffer,&ch))
    {
      if (ASCII(ch)=='\0') switch (MARK(ch))
      {
        case FOLDSTART: ++level; break;
        case FOLDEND: --level; break;
      }
      if (level==0 && ASCII(ch)=='\n' && delete) joinedLine=1;
      if (copy && bf_insert(b,&ch)==0)
      {
        bf_free(b);
        bf_tomark(d->buffer,d->cursor);
        return (struct Buffer*)0;
      }
    }
    else assert(0);
  }
  /*}}}*/
  if (level || (delete && lf && bf_isfold(d->buffer,FOLDSTART|FOLDEND,RIGHT))) /* cross fold or deleting would violate fold structure *//*{{{*/
  {
    bf_free(b);
    bf_tomark(d->buffer,d->cursor);
    return (struct Buffer*)0;
  }
  /*}}}*/
  if (delete)
  {
    for (moved=0; moved<distance; ++moved) bf_delete(d->buffer);
  }
  if (delete)
  {
    if (joinedLine) d->update=ALL|MODIFIED|LINE;
    else if ((d->update&UPDATEMASK)==NONE) d->update|=(LINE|MODIFIED);
  }
  b->lang=d->buffer->lang;
  return b;
}
/*}}}*/
/* ds_moveline   -- move line beginning with cursor into new buffer *//*{{{*/
struct Buffer *ds_moveline(struct Display *d)
{
  struct MetaChar ch;

  if (d->buffer->mode&READONLY)
  {
    showmsg(_("[Buffer is read-only]"));
    return 0;
  }
  if (!bf_rchar(d->buffer,&ch) || bf_isfold(d->buffer,FOLDEND,RIGHT))
  {
    showmsg(_("[You can not kill this line]"));
    return (struct Buffer*)0;
  }
  bf_mark(d->buffer,d->mark);
  if (bf_isfold(d->buffer,FOLDSTART,RIGHT))
  {
    int r;

    r=bf_lineend(d->buffer,1);
    assert(r);
    r=bf_tofoldend(d->buffer,1,(int*)0);
    assert(r);
    r=bf_lineend(d->buffer,1);
    assert(r);
  }
  else
  {
    if (ASCII(ch)=='\n') bf_forward(d->buffer);
    else bf_lineend(d->buffer,1);
  }
  return (ds_covebuf(d,1,1));
}
/*}}}*/
/* ds_indent     -- copy current indentation into new buffer *//*{{{*/
struct Buffer *ds_indent(void)
{
  struct Buffer *b;
  struct MetaChar ch;

  if ((b=bf_alloc())==(struct Buffer*)0) return (struct Buffer*)0;
  bf_mark(current->buffer,current->cursor);
  bf_linestart(current->buffer,1);
  while (bf_rchar(current->buffer,&ch) && (ASCII(ch)=='\t' || ASCII(ch)==' ') && (current->buffer->curch<current->buffer->mark[current->cursor]))
  {
    if (bf_insert(b,&ch)==0 || bf_forward(current->buffer)==0)
    {
      bf_free(b);
      return (struct Buffer*)0;
    }
  }
  bf_tomark(current->buffer,current->cursor);
  return b;
}
/*}}}*/
/* ds_reshape    -- reshape display *//*{{{*/
void ds_reshape(int orix, int oriy, int maxx, int maxy)
{
  current->orix=orix;
  current->oriy=oriy;
  current->maxx=maxx;
  current->maxy=maxy;
  if (current->y>=current->maxy-1) current->y=current->maxy-2;
  current->update=ALL;
}
/*}}}*/
/* ds_prevpage   -- move to previous page but stay inside fold */ /*{{{*/
int ds_prevpage(void)
{
  int i,hitstart;

  for (i=0; i<current->maxy-1; ++i)
  {
    if ((hitstart=ds_prevline())==0) return 0;
    else if (hitstart==2) return 2;
  }
  return 1;
}
/*}}}*/
/* ds_nextpage   -- move to next page but stay inside fold */ /*{{{*/
int ds_nextpage(void)
{
  int i;

  for (i=0; i<current->maxy-1; ++i)
  {
    if (ds_nextline()==0) return 0;
    else if (bf_isfold(current->buffer,FOLDEND,LEFT|RIGHT)) return 2;
  }
  return 1;
}
/*}}}*/
/* ds_gotofence  -- go to matching fence */ /*{{{*/
int ds_gotofence(void)
{
  struct MetaChar ch;

  if (bf_rchar(current->buffer,&ch))
  {
    char l,r;
    int level,forw;

    l=ASCII(ch);
    switch (ASCII(ch))
    {
      case '(': r=')'; forw=1; goto join;
      case '{': r='}'; forw=1; goto join;
      case '[': r=']'; forw=1; goto join;
      case ')': r='('; forw=0; goto join;
      case '}': r='{'; forw=0; goto join;
      case ']': r='['; forw=0;
      join:
      {
        level=1;
        do
        {
          if ((forw ? bf_forward(current->buffer) : bf_backward(current->buffer))==0 || bf_rchar(current->buffer,&ch)==0)
          {
            bf_tomark(current->buffer,current->cursor);
            return 0;
          }
          if (ASCII(ch)==r) --level;
          else if (ASCII(ch)==l) ++level;
        }
        while (level);
        current->update=(current->update&~UPDATEMASK)|ALL|LINE;
        return 1;
      }
      default: return 0;
    }
  }
  else return 0;
}
/*}}}*/
/* ds_shell      -- start a sub shell */ /*{{{*/
const char *ds_shell(const char *msg, const char *cmd)
{
  pid_t pid;
  struct sigaction interrupt;

  refresh();
  interrupt.sa_flags=0;
  sigemptyset(&interrupt.sa_mask);
  interrupt.sa_handler=SIG_IGN;
  sigaction(SIGINT,&interrupt,(struct sigaction *)0);
  sigaction(SIGQUIT,&interrupt,(struct sigaction *)0);
  switch (pid=fork())
  {
    /*      -1 */ /*{{{*/
    case -1: return strerror(errno);
    /*}}}*/
    /*       0 */ /*{{{*/
    case 0:
    {
      if (msg) showmsg(msg);
      move(LINES-1,0);
      refresh();
      reset_shell_mode();
      write(1,"\n",1);
      interrupt.sa_handler=SIG_DFL;
      sigaction(SIGINT,&interrupt,(struct sigaction *)0);
      sigaction(SIGQUIT,&interrupt,(struct sigaction *)0);
      if (cmd) execlp(getshell(),getshell(),"-c",cmd,(const char*)0);
      else execlp(getshell(),getshell(),(const char*)0);
      exit(127);
      break;
    }
    /*}}}*/
    /* default */ /*{{{*/
    default:
    {
      pid_t r;
      int status;
      static char buf[20];
      const char *cont;

      while ((r=wait(&status))!=-1 && r!=pid);
      reset_prog_mode();
      interrupt.sa_handler=SIG_DFL;
      sigaction(SIGINT,&interrupt,(struct sigaction *)0);
      sigaction(SIGQUIT,&interrupt,(struct sigaction *)0);
      if (msg==(const char*)0)
      {
        struct MacroChar mc;

        cont=_("[Press any key to continue]");
        write(1,cont,strlen(cont));
        mc_get(&mc,ds_utf8term);
      }
      touchwin(curscr);
      wrefresh(curscr);
      sprintf(buf,"%d",status);
      return (status ? buf : (const char*)0);
    }
    /*}}}*/
  }
}
/*}}}*/
/* ds_suspend    -- suspend fe */ /*{{{*/
void ds_suspend(void)
{
  if (ds_has_suspend)
  {
    showmsg(_("[Suspended]"));
    move(LINES-1,0);
    refresh();
    reset_shell_mode();
    write(1,"\n",1);
    kill(getpid(),SIGSTOP);
    alarm(0); /* and hope for the best */
    reset_prog_mode();
    touchwin(curscr);
    wrefresh(curscr);
    showmsg((const char*)0);
  }
  else
  {
    showmsg(_("[Shell does not offer suspend]"));
  }
}
/*}}}*/
/* ds_compose    -- compose character from two characters */ /*{{{*/
const struct MetaChar *ds_compose(char a, char b)
{
  /* variables */ /*{{{*/
  static struct
  {
    char a,b;
    struct MetaChar iso88591;
    struct MetaChar utf8;
  }
  compose [] =
  {
#define C (char)
    { '+', '+', {{ '#' }},    {{ '#' }} },           /* #  number sign            */
    { 'a', 'a', {{ '@' }},    {{ '@' }} },           /* @  commercial at          */
    { '(', '(', {{ '[' }},    {{ '[' }} },           /* [  opening bracket        */
    { ')', ')', {{ ']' }},    {{ ']' }} },           /* ]  closing bracket        */
    { '(', '-', {{ '{' }},    {{ '{' }} },           /* {  opening brace          */
    { ')', '-', {{ '}' }},    {{ '}' }} },           /* }  closing brace          */
    { '/', '/', {{ '\\' }},   {{ '\\' }} },          /* \  backslash              */
    { '/', '^', {{ '|' }},    {{ '|' }} },           /* |  vertical bar           */
    { ' ', ' ', {{ '\240' }}, {{ C 0xc2,C 0xa0 }} }, /*   no-break space         */
    { '|', '|', {{ '\246' }}, {{ C 0xc2,C 0xa6 }} }, /*    broken bar             */
    { '-', '-', {{ '\255' }}, {{ C 0xc2,C 0xad }} }, /*   soft hyphen            */
    { '.', '.', {{ '.' }}, {{ C 0xe2,C 0x80,C 0xa6 }} }, /*   ellipsis            */
    { 'u', '/', {{ '\265' }}, {{ C 0xc2,C 0xb5 }} }, /*   micro sign             */
    { 'm', 'u', {{ '\265' }}, {{ C 0xc2,C 0xb5 }} }, /*   micro sign             */
    { '!', '!', {{ '\241' }}, {{ C 0xc2,C 0xa1 }} }, /*   inverted !             */
    { '?', '?', {{ '\277' }}, {{ C 0xc2,C 0xbf }} }, /*   inverted ?             */
    { 'c', '/', {{ '\242' }}, {{ C 0xc2,C 0xa2 }} }, /*   cent sign              */
    { 'c', '|', {{ '\242' }}, {{ C 0xc2,C 0xa2 }} }, /*   cent sign              */
    { 'l', '-', {{ '\243' }}, {{ C 0xc2,C 0xa3 }} }, /*   pound sign             */
    { 'l', '=', {{ '\243' }}, {{ C 0xc2,C 0xa3 }} }, /*   pound sign             */
    { 'x', 'o', {{ '\244' }}, {{ C 0xc2,C 0xa4 }} }, /*   currency sign          */
    { 'x', '0', {{ '\244' }}, {{ C 0xc2,C 0xa4 }} }, /*   currency sign          */
    { 'e', 'u', {{ '\244' }}, {{ C 0xe2,C 0x82,C 0xac }} }, /*   euro sign       */
    { 'e', '=', {{ '\244' }}, {{ C 0xe2,C 0x82,C 0xac }} }, /*   euro sign       */
    { 'y', '-', {{ '\245' }}, {{ C 0xc2,C 0xa5 }} }, /*   yen sign               */
    { 'y', '=', {{ '\245' }}, {{ C 0xc2,C 0xa5 }} }, /*   yen sign               */
    { 's', 'o', {{ '\247' }}, {{ C 0xc2,C 0xa7 }} }, /*   section sign           */
    { 's', '!', {{ '\247' }}, {{ C 0xc2,C 0xa7 }} }, /*   section sign           */
    { 's', '0', {{ '\247' }}, {{ C 0xc2,C 0xa7 }} }, /*   section sign           */
    { 'p', '!', {{ '\266' }}, {{ C 0xc2,C 0xb6 }} }, /*   pilcrow sign           */
    { '"', '"', {{ '\250' }}, {{ C 0xc2,C 0xa8 }} }, /*    diaeresis              */
    { ':', ':', {{ '\250' }}, {{ C 0xc2,C 0xa8 }} }, /*    diaeresis              */
    { '^', '-', {{ '\257' }}, {{ C 0xc2,C 0xaf }} }, /*   macron                 */
    { '^', '_', {{ '\257' }}, {{ C 0xc2,C 0xaf }} }, /*   macron                 */
    { '\'','\'',{{ '\264' }}, {{ C 0xc2,C 0xb4 }} }, /*    acute accent           */
    { ',', ',', {{ '\270' }}, {{ C 0xc2,C 0xb8 }} }, /*    cedilla                */
    { 'c', 'o', {{ '\251' }}, {{ C 0xc2,C 0xa9 }} }, /*   copyright sign         */
    { 'c', '0', {{ '\251' }}, {{ C 0xc2,C 0xa9 }} }, /*   copyright sign         */
    { 'r', 'o', {{ '\256' }}, {{ C 0xc2,C 0xae }} }, /*   registered sign        */
    { 'a', '_', {{ '\252' }}, {{ C 0xc2,C 0xaa }} }, /*   feminine ordinal       */
    { 'o', '_', {{ '\272' }}, {{ C 0xc2,C 0xba }} }, /*   masculine ordinal      */
    { '<', '<', {{ '\253' }}, {{ C 0xc2,C 0xab }} }, /*   opening angle brackets */
    { '>', '>', {{ '\273' }}, {{ C 0xc2,C 0xbb }} }, /*   closing angle brakets  */
    { '0', '^', {{ '\260' }}, {{ C 0xc2,C 0xb0 }} }, /*   degree sign            */
    { '1', '^', {{ '\271' }}, {{ C 0xc2,C 0xb9 }} }, /*   superscript 1          */
    { '2', '^', {{ '\262' }}, {{ C 0xc2,C 0xb2 }} }, /*   superscript 2          */
    { '3', '^', {{ '\263' }}, {{ C 0xc2,C 0xb3 }} }, /*   superscript 3          */
    { '+', '-', {{ '\261' }}, {{ C 0xc2,C 0xb1 }} }, /*   plus or minus sign     */
    { '1', '4', {{ '\274' }}, {{ C 0xc2,C 0xbc }} }, /*   fraction one-quarter   */
    { '1', '2', {{ '\275' }}, {{ C 0xc2,C 0xbd }} }, /*   fraction one-half      */
    { '3', '4', {{ '\276' }}, {{ C 0xc2,C 0xbe }} }, /*   fraction three quarter */
    { '.', '^', {{ '\267' }}, {{ C 0xc2,C 0xb7 }} }, /*   middle dot             */
    { '-', '|', {{ '\254' }}, {{ C 0xc2,C 0xac }} }, /*   not sign               */
    { '-', ',', {{ '\254' }}, {{ C 0xc2,C 0xac }} }, /*   not sign               */
    { 'x', 'x', {{ '\327' }}, {{ C 0xc3,C 0x97 }} }, /*   multiplication sign    */
    { ':', '-', {{ '\367' }}, {{ C 0xc3,C 0xb7 }} }, /*   division sign          */
    { '`', 'A', {{ '\300' }}, {{ C 0xc3,C 0x80 }} }, /*   A grave                */
    { '`', 'a', {{ '\340' }}, {{ C 0xc3,C 0xa0 }} }, /*   a grave                */
    { '\'','A', {{ '\301' }}, {{ C 0xc3,C 0x81 }} }, /*   A acute                */
    { '\'','a', {{ '\341' }}, {{ C 0xc3,C 0xa1 }} }, /*   a acute                */
    { '^', 'A', {{ '\302' }}, {{ C 0xc3,C 0x82 }} }, /*   A circumflex           */
    { '^', 'a', {{ '\342' }}, {{ C 0xc3,C 0xa2 }} }, /*   a circumflex           */
    { '~', 'A', {{ '\303' }}, {{ C 0xc3,C 0x83 }} }, /*   A tilde                */
    { '~', 'a', {{ '\343' }}, {{ C 0xc3,C 0xa3 }} }, /*   a tilde                */
    { ':', 'A', {{ '\304' }}, {{ C 0xc3,C 0x84 }} }, /*   A diaeresis            */
    { '"', 'A', {{ '\304' }}, {{ C 0xc3,C 0x84 }} }, /*   A diaeresis            */
    { ':', 'a', {{ '\344' }}, {{ C 0xc3,C 0xa4 }} }, /*   a diaeresis            */
    { '"', 'a', {{ '\344' }}, {{ C 0xc3,C 0xa4 }} }, /*   a diaeresis            */
    { 'A', 'O', {{ '\305' }}, {{ C 0xc3,C 0x85 }} }, /*   A ring                 */
    { 'A', '0', {{ '\305' }}, {{ C 0xc3,C 0x85 }} }, /*   A ring                 */
    { 'A', '*', {{ '\305' }}, {{ C 0xc3,C 0x85 }} }, /*   A ring                 */
    { 'a', 'o', {{ '\345' }}, {{ C 0xc3,C 0xa5 }} }, /*   a ring                 */
    { 'a', '0', {{ '\345' }}, {{ C 0xc3,C 0xa5 }} }, /*   a ring                 */
    { 'a', '*', {{ '\345' }}, {{ C 0xc3,C 0xa5 }} }, /*   a ring                 */
    { 'A', 'E', {{ '\306' }}, {{ C 0xc3,C 0x86 }} }, /*   AE ligature            */
    { 'a', 'e', {{ '\346' }}, {{ C 0xc3,C 0xa6 }} }, /*   ae ligature            */
    { ',', 'C', {{ '\307' }}, {{ C 0xc3,C 0x87 }} }, /*   C cedilla              */
    { ',', 'c', {{ '\347' }}, {{ C 0xc3,C 0xa7 }} }, /*   c cedilla              */
    { '`', 'E', {{ '\310' }}, {{ C 0xc3,C 0x88 }} }, /*   E grave                */
    { '`', 'e', {{ '\350' }}, {{ C 0xc3,C 0xa8 }} }, /*   e grave                */
    { '\'','E', {{ '\311' }}, {{ C 0xc3,C 0x89 }} }, /*   E acute                */
    { '\'','e', {{ '\351' }}, {{ C 0xc3,C 0xa9 }} }, /*   e acute                */
    { '^', 'E', {{ '\312' }}, {{ C 0xc3,C 0x8a }} }, /*   E circumflex           */
    { '^', 'e', {{ '\352' }}, {{ C 0xc3,C 0xaa }} }, /*   e circumflex           */
    { ':', 'E', {{ '\313' }}, {{ C 0xc3,C 0x8b }} }, /*   E diaeresis            */
    { '"', 'E', {{ '\313' }}, {{ C 0xc3,C 0x8b }} }, /*   E diaeresis            */
    { ':', 'e', {{ '\353' }}, {{ C 0xc3,C 0xab }} }, /*   e diaeresis            */
    { '"', 'e', {{ '\353' }}, {{ C 0xc3,C 0xab }} }, /*   e diaeresis            */
    { '`', 'I', {{ '\314' }}, {{ C 0xc3,C 0x8c }} }, /*   I grave                */
    { '`', 'i', {{ '\354' }}, {{ C 0xc3,C 0xac }} }, /*   i grave                */
    { '\'','I', {{ '\315' }}, {{ C 0xc3,C 0x8d }} }, /*   I acute                */
    { '\'','i', {{ '\355' }}, {{ C 0xc3,C 0xad }} }, /*   i acute                */
    { '^', 'I', {{ '\316' }}, {{ C 0xc3,C 0x8e }} }, /*   I circumflex           */
    { '^', 'i', {{ '\356' }}, {{ C 0xc3,C 0xae }} }, /*   i circumflex           */
    { ':', 'I', {{ '\317' }}, {{ C 0xc3,C 0x8f }} }, /*   I diaeresis            */
    { '"', 'I', {{ '\317' }}, {{ C 0xc3,C 0x8f }} }, /*   I diaeresis            */
    { ':', 'i', {{ '\357' }}, {{ C 0xc3,C 0xaf }} }, /*   i diaeresis            */
    { '"', 'i', {{ '\357' }}, {{ C 0xc3,C 0xaf }} }, /*   i diaeresis            */
    { '-', 'D', {{ '\320' }}, {{ C 0xc3,C 0x90 }} }, /*   capital eth            */
    { '-', 'd', {{ '\360' }}, {{ C 0xc3,C 0xb0 }} }, /*   small eth              */
    { '~', 'N', {{ '\321' }}, {{ C 0xc3,C 0x91 }} }, /*   N tilde                */
    { '~', 'n', {{ '\361' }}, {{ C 0xc3,C 0xb1 }} }, /*   n tilde                */
    { '`', 'O', {{ '\322' }}, {{ C 0xc3,C 0x92 }} }, /*   O grave                */
    { '`', 'o', {{ '\362' }}, {{ C 0xc3,C 0xb2 }} }, /*   o grave                */
    { '\'','O', {{ '\323' }}, {{ C 0xc3,C 0x93 }} }, /*   O acute                */
    { '\'','o', {{ '\363' }}, {{ C 0xc3,C 0xb3 }} }, /*   o acute                */
    { '^', 'O', {{ '\324' }}, {{ C 0xc3,C 0x94 }} }, /*   O circumflex           */
    { '^', 'o', {{ '\364' }}, {{ C 0xc3,C 0xb4 }} }, /*   o circumflex           */
    { '~', 'O', {{ '\325' }}, {{ C 0xc3,C 0x95 }} }, /*   O tilde                */
    { '~', 'o', {{ '\365' }}, {{ C 0xc3,C 0xb5 }} }, /*   o tilde                */
    { ':', 'O', {{ '\326' }}, {{ C 0xc3,C 0x96 }} }, /*   O diaeresis            */
    { '"', 'O', {{ '\326' }}, {{ C 0xc3,C 0x96 }} }, /*   O diaeresis            */
    { ':', 'o', {{ '\366' }}, {{ C 0xc3,C 0xb6 }} }, /*   o diaeresis            */
    { '"', 'o', {{ '\366' }}, {{ C 0xc3,C 0xb6 }} }, /*   o diaeresis            */
    { '/', 'O', {{ '\330' }}, {{ C 0xc3,C 0x98 }} }, /*   O slash                */
    { '/', 'o', {{ '\370' }}, {{ C 0xc3,C 0xb8 }} }, /*   o slash                */
    { '`', 'U', {{ '\331' }}, {{ C 0xc3,C 0x99 }} }, /*   U grave                */
    { '`', 'u', {{ '\371' }}, {{ C 0xc3,C 0xb9 }} }, /*   u grave                */
    { '\'','U', {{ '\332' }}, {{ C 0xc3,C 0x9a }} }, /*   U acute                */
    { '\'','u', {{ '\372' }}, {{ C 0xc3,C 0xba }} }, /*   u acute                */
    { '^', 'U', {{ '\333' }}, {{ C 0xc3,C 0x9b }} }, /*   U circumflex           */
    { '^', 'u', {{ '\373' }}, {{ C 0xc3,C 0xbb }} }, /*   u circumflex           */
    { ':', 'U', {{ '\334' }}, {{ C 0xc3,C 0x9c }} }, /*   U diaeresis            */
    { '"', 'U', {{ '\334' }}, {{ C 0xc3,C 0x9c }} }, /*   U diaeresis            */
    { ':', 'u', {{ '\374' }}, {{ C 0xc3,C 0xbc }} }, /*   u diaeresis            */
    { '"', 'u', {{ '\374' }}, {{ C 0xc3,C 0xbc }} }, /*   u diaeresis            */
    { '\'','Y', {{ '\335' }}, {{ C 0xc3,C 0x9d }} }, /*   Y acute                */
    { '\'','y', {{ '\375' }}, {{ C 0xc3,C 0xbd }} }, /*   y acute                */
    { 'T', 'H', {{ '\336' }}, {{ C 0xc3,C 0x9e }} }, /*   capital thorn          */
    { 't', 'h', {{ '\376' }}, {{ C 0xc3,C 0xbe }} }, /*   small thorn            */
    { 's', 's', {{ '\337' }}, {{ C 0xc3,C 0x9f }} }, /*   German small sharp s   */
    { 's', 'z', {{ '\337' }}, {{ C 0xc3,C 0x9f }} }, /*   German small sharp s   */
    { '"', 'y', {{ '\377' }}, {{ C 0xc3,C 0xbf }} }, /*   y diaeresis            */
    { ':', 'y', {{ '\377' }}, {{ C 0xc3,C 0xbf }} }, /*   y diaeresis            */
    { 'i', 'j', {{ '\377' }}, {{ C 0xc3,C 0xbf }} }, /*   y diaeresis            */
    { '\0','\0', {{ 0 }}, {{ C 0 }} }
#undef C
  };

  int i;
  /*}}}*/  

  if (b=='\0')
  {
    /* Check if a is a possible compose character at all. */
    for (i=0; compose[i].a && a!=compose[i].a; ++i);
  }
  else
  {
    for (i=0; compose[i].a && (a!=compose[i].a || b!=compose[i].b) && (a!=compose[i].b || b!=compose[i].a); ++i);
  }
  if (!compose[i].a) return (struct MetaChar*)0;
  return ds_utf8term ? &compose[i].utf8 : &compose[i].iso88591;
}
/*}}}*/
/* ds_describe   -- describe line */ /*{{{*/
void ds_describe(void)
{
  struct MetaChar ch;
  const char *describe;
  char buf[16];
  unsigned int asc;

  if (bf_rchar(current->buffer,&ch))
  {
    asc=(unsigned)ASCII(ch);
    if (asc=='\0') switch (MARK(ch))
    {
      case NUL:
      {
        describe="'0'='\0'";
        break;
      }
      case QUOTE:
      {
        /* no UTF-8 character */
        snprintf(buf,sizeof(buf),"'%u'='\\%o'",(unsigned char)QUOTED(ch),(unsigned char)QUOTED(ch));
        describe=buf;
        break;
      }
      case FOLDSTART: describe=_("Opening fold mark"); break;
      case FOLDEND: describe=_("Closing fold mark"); break;
      default: assert(0);
    }
    else
    {
      if (asc>=' ' && asc<127)
      {
        /* printable ASCII character */
        snprintf(buf,sizeof(buf),"'%c'",asc);
        describe=buf;
      }
      else if (asc<' ' || asc==127)
      {
        /* non-printable ASCII character */
        snprintf(buf,sizeof(buf),"'%u'='\\%o'",(unsigned char)asc,(unsigned char)asc);
        describe=buf;
      }
      else if (bf_utf8)
      {
        /* UTF-8 encoded codepoint */
        snprintf(buf,sizeof(buf),"U+%04lX",bf_uc(&ch));
        describe=buf;
      }
      else
      {
        snprintf(buf,sizeof(buf),"'%u'='%o'",(unsigned char)asc,(unsigned char)asc);
        describe=buf;
      }
    }
  }
  else describe=_("End of text");

  showmsg(_("[Line %d Column %d %s]"),current->buffer->cury+1,current->x,describe);
}
/*}}}*/
/* ds_setlang    -- set language */ /*{{{*/
void ds_setlang(void)
{
  static MenuChoice menu[LANGUAGES];
  int i;

  if (!menu[0].str)
  {
    for (i=0; i<LANGUAGES-1; ++i)
    {
      const char *lang;
      char *choice;

      if (*(lang=language[i].name)=='\0') lang=_("No language");
      choice=malloc(strlen(lang)+2);
      strcpy(choice+1,lang);
      choice[0]=tolower(lang[0]);
      menu[i].str=choice;
      menu[i].c='\0';
    }
    menu[LANGUAGES-1].str=(char*)0;
  }
  i=menuline(_("Language:"),menu,current->buffer->lang-language);
  if (i>=0) bf_lang(current->buffer,&(language[i]));
}
/*}}}*/
