// Copyright 2023 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "generator/internal/discovery_proto_export_file.h"
#include "generator/internal/codegen_utils.h"
#include "google/cloud/internal/absl_str_replace_quiet.h"
#include "google/cloud/testing_util/status_matchers.h"
#include <gmock/gmock.h>

namespace google {
namespace cloud {
namespace generator_internal {
namespace {

using ::testing::Eq;

TEST(DiscoveryProtoExportFileTest, FormatFile) {
  auto constexpr kExpectedHeaderFile =
      R"""(// Copyright $copyright_year$ Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Generated by the Codegen C++ plugin.
// If you make any local changes, they will be lost.

// The contents of internal/common_xxx.proto files are subject to change, and it
// is not recommended to include their respective headers directly. Including
// the <service>.pb.h file transitively provides all the necessary proto
// definitions. However, it is not IWYU friendly. Including this header file
// uses IWYU pragmas to export the internal/common_xxx.pb.h files as an
// IWYU-friendly alternative.

#ifndef GOOGLE_CLOUD_CPP_RELATIVE_FILE_PATH_H
#define GOOGLE_CLOUD_CPP_RELATIVE_FILE_PATH_H

// IWYU pragma: begin_exports
#include <google/cloud/compute/v1/internal/common_001.pb.h>
#include <google/cloud/compute/v1/internal/common_002.pb.h>
// IWYU pragma: end_exports

#endif  // GOOGLE_CLOUD_CPP_RELATIVE_FILE_PATH_H
)""";

  DiscoveryProtoExportFile f(
      "my_output_file_path.h", "relative_file_path.h",
      {"google/cloud/compute/v1/internal/common_001.proto",
       "google/cloud/compute/v1/internal/common_002.proto"});
  std::stringstream os;
  auto result = f.FormatFile(os);
  ASSERT_STATUS_OK(result);
  EXPECT_THAT(os.str(), Eq(absl::StrReplaceAll(
                            kExpectedHeaderFile,
                            {{"$copyright_year$", CurrentCopyrightYear()}})));
}

}  // namespace
}  // namespace generator_internal
}  // namespace cloud
}  // namespace google

// We define main here due to
// https://github.com/protocolbuffers/protobuf/issues/12887.
int main(int argc, char** argv) {
  ::testing::InitGoogleTest(&argc, argv);
  return RUN_ALL_TESTS();
}
