/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include "mongo/base/string_data.h"
#include "mongo/bson/bsonmisc.h"
#include "mongo/bson/bsonobj.h"
#include "mongo/bson/bsonobjbuilder.h"
#include "mongo/db/operation_context.h"
#include "mongo/db/ops/single_write_result_gen.h"
#include "mongo/db/ops/write_ops.h"
#include "mongo/db/ops/write_ops_gen.h"
#include "mongo/db/repl/oplog_entry.h"
#include "mongo/db/repl/oplog_entry_gen.h"

namespace mongo {

class BSONObjBuilder;
class OperationContext;

const BSONObj kWouldChangeOwningShardSentinel(BSON("$wouldChangeOwningShard" << 1));

template <typename OplogEntryType>
bool isWouldChangeOwningShardSentinelOplogEntry(const OplogEntryType& oplogEntry) {
    return (oplogEntry.getOpType() == repl::OpTypeEnum::kNoop) &&
        (oplogEntry.getObject().woCompare(kWouldChangeOwningShardSentinel) == 0) &&
        oplogEntry.getObject2() && oplogEntry.getObject2()->isEmpty();
}

/**
 * Returns the single write result corresponding to the given oplog entry for document update. I.e.,
 * the single write result that would have been returned by the statement that would have resulted
 * in the given oplog entry. The oplog entries are assumed to be properly formed and have the
 * correct op type.
 * Uses the 'upsertedId' for an upsert operation if it's provided.
 */
SingleWriteResult parseOplogEntryForUpdate(
    const repl::OplogEntry& entry, const boost::optional<BSONElement>& upsertedId = boost::none);

/**
 * Populates the passed-in builder with the result of a findAndModify based on the oplog entries
 * generated by the operation.
 */
write_ops::FindAndModifyCommandReply parseOplogEntryForFindAndModify(
    OperationContext* opCtx,
    const write_ops::FindAndModifyCommandRequest& request,
    const repl::OplogEntry& oplogEntry);

}  // namespace mongo
