/* $Id$ */
/*
 * functions to make X11 BDF font bold.
 */
#if defined HAVE_LIMITS_H
#include <limits.h>
#endif /* HAVE_LIMITS_H */
#if !defined CHAR_BIT
#define CHAR_BIT 8
#endif /* !CHAR_BIT */
#include "common.h"
#include "bold.h"

#define RADIXBITS CHAR_BIT*sizeof(unsigned int)
#define BDF_LEN   (RADIXBITS/8)
#define BIT_SHIFT (RADIXBITS-1)

struct bold_bitmap {
  size_t num;
  unsigned int *d[3];
};

static void hex_get_line(unsigned int d[], const char *s, size_t len);
static size_t bdf2hex(unsigned int *h, const char *s, size_t len);
static void shl(unsigned int dout[], unsigned int din[], size_t num);
static void shr(unsigned int dout[], unsigned int din[], size_t num);
static void bitcalc(unsigned int out[], unsigned int d1[], unsigned int d2[], unsigned int d3[], size_t num);


/*
 * recast dummy for font property.
 */
int recast_dummy(char *dst, int *width, int *height, int *dw, int dy, const char *src)
{
  unsigned int w, h;

  if (sscanf(src, "%*s%u%u%*d%*d", &w, &h) != 2) {
    return FALSE;
  }
  *width = w;
  *height = h;
  strcpy(dst, src);
  return TRUE;
}

BoldBitmap *boldBitmap_new(size_t len)
{
  BoldBitmap *bitmap;
  size_t num, i;

  num = (len-1)/BDF_LEN+1;
  bitmap = XMALLOC(BoldBitmap, 1);
  bitmap->num = num;
  for (i = 0; i < 3; i++) {
    bitmap->d[i] = XMALLOC(unsigned int, num);
  }
  return bitmap;
}

void boldBitmap_delete(BoldBitmap *p)
{
  size_t i;

  for (i = 0; i < 3; i++) {
    XFREE(p->d[i]);
  }
  XFREE(p);
}

/*
 * output BITMAP image.
 */
void boldBitmap_output(FILE *fp, BoldBitmap *bitmap, size_t len)
{
  unsigned int *d, b0;
  size_t i, l;

  d = bitmap->d[0];
  for (i = 0; i < len/BDF_LEN; i++) {
    fprintf(fp, "%0*x", BDF_LEN*2, d[i]);
  }
  l = len%BDF_LEN;
  if (l > 0) {
    b0 = RADIXBITS-8*l;
    fprintf(fp, "%0*x", l*2, (d[i] >> b0));
  }
  fprintf(fp, "\n");
}

/*
 * create bold font BITMAP image.
 */
void boldBitmap_make_bold(BoldBitmap *bitmap, const char *s, size_t len, int bdir, int pile)
{
  unsigned int *d, *bold, *tmp;
  size_t num;

  num = (len-1)/BDF_LEN+1;
  d = bitmap->d[0];
  bold = bitmap->d[1];
  tmp = bitmap->d[2];
  if (bdir) {
    /* left shifted image */
    hex_get_line(d, s, len);
    shl(bold, d, num);
  } else {
    /* right shifted image */
    hex_get_line(bold, s, len);
    shr(d, bold, num);
  }
  if (pile) {
    /* left edge */
    shl(tmp, bold, num);
    bitcalc(d, tmp, d, bold, num);
  } else {
    /* right edge */
    shr(tmp, d, num);
    bitcalc(d, tmp, bold, d, num);
  }
}

static void hex_get_line(unsigned int d[], const char *s, size_t len)
{
  unsigned int h;
  size_t i, l;

  for (i = 0; i < len/BDF_LEN; i++) {
    l = bdf2hex(&h, s, BDF_LEN*2);
    d[i] = h;
    s += l;
  }
  l = len%BDF_LEN;
  if (l > 0) {
    bdf2hex(&h, s, l*2);
    d[i] = h;
  }
}

static size_t bdf2hex(unsigned int *h, const char *s, size_t len)
{
  unsigned int b, bs;
  size_t i;

  bs = RADIXBITS;
  *h = 0x0;
  for (i = 0; i < len; i++) {
    switch (*s) {
    case '\0':
      return i;
    case '0':
      b = 0x0;
      break;
    case '1':
      b = 0x1;
      break;
    case '2':
      b = 0x2;
      break;
    case '3':
      b = 0x3;
      break;
    case '4':
      b = 0x4;
      break;
    case '5':
      b = 0x5;
      break;
    case '6':
      b = 0x6;
      break;
    case '7':
      b = 0x7;
      break;
    case '8':
      b = 0x8;
      break;
    case '9':
      b = 0x9;
      break;
    case 'A':
    case 'a':
      b = 0xa;
      break;
    case 'B':
    case 'b':
      b = 0xb;
      break;
    case 'C':
    case 'c':
      b = 0xc;
      break;
    case 'D':
    case 'd':
      b = 0xd;
      break;
    case 'E':
    case 'e':
      b = 0xe;
      break;
    case 'F':
    case 'f':
      b = 0xf;
      break;
    default:
      b = 0x0;
      break;
    }
    bs -= 4;
    if (bs > 0) {
      b <<= bs;
    }
    *h |= b;
    s++;
  }
  return i;
}

/*
 * shift left.
 */
static void shl(unsigned int dout[], unsigned int din[], size_t num)
{
  unsigned int c, d;
  int i;

  c = 0x0;
  for (i = num-1; i >= 0; i--) {
    d = c;
    c = (din[i] >> BIT_SHIFT) & 0x1;
    dout[i] = (din[i] << 1) | d;
  }
}

/*
 * shift right.
 */
static void shr(unsigned int dout[], unsigned int din[], size_t num)
{
  unsigned int c, d;
  size_t i;

  c = 0x0;
  for (i = 0; i < num; i++) {
    d = c;
    c = (din[i] & 0x1) << BIT_SHIFT;
    dout[i] = (din[i] >> 1) | d;
  }
}

/*
 * bit operation.
 */
static void bitcalc(unsigned int out[], unsigned int d1[], unsigned int d2[], unsigned int d3[], size_t num)
{
  size_t i;

  for (i = 0; i < num; i++) {
    out[i] = (~d1[i] & d2[i]) | d3[i];
  }
}
