/*
    Copyright (C) 1994,1995,1996 Uwe Ohse
	Copyright (C) 1988 ejp@ausmelb.oz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Contact: uwe@ohse.de, Uwe Ohse @ DU3 (mausnet)
*/

/*
	Dynamic hashing, after CACM April 1988 pp 446-457, by Per-Ake Larson.
	Coded into C, with minor code improvements, and with hsearch(3) interface,
	by ejp@ausmelb.oz, Jul 26, 1988: 13:16;
	also, hcreate/hdestroy routines added to simulate hsearch(3).

	These routines simulate hsearch(3) and family, with the important
	difference that the hash table is dynamic - can grow indefinitely
	beyond its original size (as supplied to hcreate()).

	Performance appears to be comparable to that of hsearch(3).
	The 'source-code' options referred to in hsearch(3)'s 'man' page
	are not implemented; otherwise functionality is identical.

	Compilation controls:
	HASH_DEBUG controls some informative traces, mainly for debugging.
	HASH_STATISTICS causes HashAccesses and HashCollisions to be maintained;
	when combined with HASH_DEBUG, these are displayed by hdestroy().

    Modified by Uwe Ohse:
       31.1.1994: Many tables, no more static vars.
        2.7.1994: hsearch,hcreate,hdestroy: gone
                  some smaller cleanups.
      April 1996: range-checks for directory-array added (large initial
                  tablesizes did overflow the directory).
                  Added DYN_ENTER_OR_OVERWRITE, DYN_OVERWRITE and
                  DYN_DELETE.
                  ExpandTable was never called (must be a mistake 
                  i made 1994).
*/


/*
AUTO_NEED MACRO AC_HEADER_STDC
*/
#include "config.h"

#ifdef STDC_HEADERS
#include <string.h>
#include <stdlib.h>
#endif

#include <stdio.h>
#include "dynhash.h"			/* instead of search.h, fb 26.04.93 */
#include <assert.h>


/*
	Constants
*/

#define SegmentSize              256
#define SegmentSizeShift           8   /* log2(SegmentSize) */
#define DirectorySize            256
#define DirectorySizeShift         8   /* log2(DirectorySize) */
#define Prime1                    37
#define Prime2               1048583L
#define DefaultMaxLoadFactor       5
#define OverFullLookupAfter       10

/*
	Fast arithmetic, relying on powers of 2,
*/

#define MOD(x,y)				((x) & ((y)-1))

/*
	local data templates
*/

typedef struct element
{
	/*
		The user only sees the first two fields,
		as we pretend to pass back only a pointer to DYN_ENTRY.
		{S}he doesn't know what else is in here.
	*/
	unsigned long crc;
    union
    {
        long nr;
        void *ptr;
    } u;
	struct element	*Next;	/* secret from user */

} Element,*Segment;


typedef struct
{
	long            p;             /* Next bucket to be split */
	long            maxp;          /* upper bound on p during expansion */
	long             KeyCount;      /* current # keys	*/
	long            SegmentCount;  /* current # segments */
	long            MinLoadFactor;
	long            MaxLoadFactor;
	Segment          *Directory[DirectorySize];
	unsigned long   NewAdded;
#ifdef HASH_STATISTICS
	long HashAccesses;
	long HashCollisions;
	long HashExpansions;
#endif
} HashTable;

typedef unsigned long Address;

/*
	Internal routines
*/

static inline Address	Hash P__((HashTable *Table,unsigned long crc));
static void ExpandTable P__((HashTable *Table));

/*
	Code
*/

static inline Address 
Hash(HashTable *Table,unsigned long crc)
{
	Address 	h,address;

	h = crc % Prime2;
	address = MOD(h,Table->maxp); 
	/*
	 * address is unsigned long, p is short.
	 */
	if (address < (Address) (Table->p))
		address = MOD(h,(Table->maxp << 1));	/* h % (2*Table->maxp) */
	return(address);
}

void *
hhcreate(unsigned long count)
{
	size_t i;
	HashTable *Table;

	/*
		Adjust count to be nearest higher power of 2,
		minimum SegmentSize, then convert into segments.
	*/
	i = SegmentSize;
	while (i < count)
		i <<= 1;

	count = (i >> SegmentSizeShift);
	if (count > DirectorySize)
		count=DirectorySize;

	Table = (HashTable *) calloc(sizeof(HashTable),1);
	if (Table == NULL)
		return NULL;
#if 0
	/* resets are redundant - already done by calloc(3) */
	Table->SegmentCount = Table->p = Table->KeyCount = 0;
#if HASH_STATISTICS
	Table->HashAccesses=0;
	Table->HashCollisions=0;
#endif
#endif
	/*
		Allocate initial 'i' segments of buckets
	*/
	for (i = 0; i < count; i++)
	{
		Table->Directory[i] = (Segment*)calloc(sizeof(Segment),SegmentSize);
		if (Table->Directory[i] == NULL)
		{
			hhdestroy(Table);
			return NULL;
		}
		Table->SegmentCount++;
	}
	Table->maxp = count << SegmentSizeShift;
	Table->MinLoadFactor = 1;
	Table->MaxLoadFactor = DefaultMaxLoadFactor;
#if HASH_DEBUG
	fprintf(
			stderr,
			"[hcreate] Table %lx count %d maxp %d SegmentCount %d\r\n",
			Table,
			count,
			Table->maxp,
			Table->SegmentCount
			);
#endif
	return((void *)Table);
}



void 
hhdestroy(void *handle)
{
	int 		i,j;
	Segment 	*s;
	Element 	*p,*q;
	HashTable 	*Table;

	if (!handle)
		return;
	Table=(HashTable *)handle;

	for (i = 0; i < Table->SegmentCount; i++)
	{
		/* test probably unnecessary */
		if ((s = Table->Directory[i]) != NULL)
		{
			for (j = 0; j < SegmentSize; j++)
			{
				p = s[j];
				while (p != NULL)
				{
					q = p->Next;
					free((char*)p);
					p = q;
				}
			}
			free(Table->Directory[i]);
		}
	}
#if HASH_STATISTICS && HASH_DEBUG
	fprintf(stderr,"[hdestroy] Accesses %ld Collisions %ld Expansions %ld\r\n",
			Table->HashAccesses,
			Table->HashCollisions,
			Table->HashExpansions);
#endif
	free(handle);
}

DYN_ENTRY *
hhsearch(void *handle,DYN_ENTRY *item,int action)
/* action is: FIND/ENTER */
{
	Address 	h;
	Segment 	*CurrentSegment;
	size_t	SegmentIndex;
	size_t	SegmentDir;
	Segment 	*p,q;
	HashTable 	*Table;

	assert(handle != NULL);
	Table=(HashTable *)handle;

#if HASH_STATISTICS
	Table->HashAccesses++;
#endif
	h = Hash(Table,item->crc);
	SegmentDir = (h >> SegmentSizeShift) % DirectorySize;
	SegmentIndex = MOD(h,SegmentSize);
	/* 
	 * valid segment ensured by Hash()
	 */
	CurrentSegment = Table->Directory[SegmentDir];
	assert(CurrentSegment != NULL); 	/* bad failure if tripped */
	p = &CurrentSegment[SegmentIndex];
	q = *p;
	if (action==DYN_DELETE)
	{
		Segment *last_p=NULL;

		/*
			Follow collision chain
		*/
		while (q != NULL && q->crc != item->crc)
		{
			last_p=p;
			p = &q->Next;
			q = *p;
#if HASH_STATISTICS
			Table->HashCollisions++;
#endif
		}
		if (q==NULL)
			return NULL;
		if (last_p)
			(*last_p)->Next=q->Next;
		else
			*p=q->Next;
		free(q);
		return (NULL);
	}
	/*
		Follow collision chain
	*/
	while (q != NULL && q->crc != item->crc)
	{
		p = &q->Next;
		q = *p;
#if HASH_STATISTICS
		Table->HashCollisions++;
#endif
	}
	if ( action == DYN_FIND)
		return ((DYN_ENTRY *)q);
	else if ( action == DYN_ENTER ) {
		if (q != NULL)
			return ((DYN_ENTRY *)q);
		q = (Element*)malloc(sizeof(Element));
		if ( q == NULL)
			return NULL;
	}
	else /* DYN_OVERWRITE */ {
		if (q != NULL)
		{
			memcpy(&(q->u),&(item->u),sizeof(item->u));
			return ((DYN_ENTRY *)q);
		}
		q = (Element*)malloc(sizeof(Element));
		if ( q == NULL)
			return NULL;
	}
	*p = q; 				/* link into chain */
	/*
	 * Initialize new element
	 */
	q->crc = item->crc;
	memcpy(&(q->u),&(item->u),sizeof(item->u));
	q->Next = NULL;
	/*
	 * Table over-full?
	 */
	if (++(Table->NewAdded)>OverFullLookupAfter)
	{
		long loadfactor;
		Table->KeyCount++;
		loadfactor=Table->KeyCount / (Table->SegmentCount << SegmentSizeShift);

		if (loadfactor > Table->MaxLoadFactor)
			ExpandTable(Table);	/* doesn't affect q 	*/
	}
	return((DYN_ENTRY*)q);
}

static void 
ExpandTable(HashTable *Table)
{
	Address 	NewAddress;
	size_t 	OldSegmentIndex,NewSegmentIndex;
	size_t	OldSegmentDir,NewSegmentDir;
	Segment 	*OldSegment,*NewSegment;
	Element 	*Current,**Previous,**LastOfNew;

	if (Table->maxp + Table->p < (DirectorySize << SegmentSizeShift))
	{
#if HASH_STATISTICS
		Table->HashExpansions++;
#endif
		/*
		 * Locate the bucket to be split
		 */
		OldSegmentDir = Table->p >> SegmentSizeShift;
		OldSegment = Table->Directory[OldSegmentDir];
		OldSegmentIndex = MOD(Table->p,SegmentSize);
		/*
		 * Expand address space; if necessary create a new segment
		 */
		NewAddress = Table->maxp + Table->p;
		NewSegmentDir = NewAddress >> SegmentSizeShift;
		NewSegmentIndex = MOD(NewAddress,SegmentSize);
		if (NewSegmentIndex == 0)
			Table->Directory[NewSegmentDir] = (Segment*)calloc(sizeof(Segment),SegmentSize);
		NewSegment = Table->Directory[NewSegmentDir];
		/*
			Adjust state variables
		*/
		Table->p++;
		if (Table->p == Table->maxp)
		{
			Table->maxp <<= 1;	/* Table->maxp *= 2 	*/
			Table->p = 0;
		}
		Table->SegmentCount++;
		/*
			Relocate records to the new bucket
		*/
		Previous = &OldSegment[OldSegmentIndex];
		Current = *Previous;
		LastOfNew = &NewSegment[NewSegmentIndex];
		*LastOfNew = NULL;
		while (Current != NULL)
		{
			if (Hash(Table,Current->crc) == NewAddress)
			{
				/*
					Attach it to the end of the new chain
				*/
				*LastOfNew = Current;
				/*
					Remove it from old chain
				*/
				*Previous = Current->Next;
				LastOfNew = &Current->Next;
				Current = Current->Next;
				*LastOfNew = NULL;
			}
			else
			{
				/*
					leave it on the old chain
				*/
				Previous = &Current->Next;
				Current = Current->Next;
			}
		}
	}
}
#ifdef TEST

#ifndef NR
#define NR 100000
#endif
#ifndef START
#define START 5000
#endif

int main(void)
{
	DYN_ENTRY new;
	int i;
	void *handle;
	DYN_ENTRY *found;
	int ec=0;
	
	handle=hhcreate(START);
	puts("1->Enter");
	for (i=0;i<NR;i++)
	{
		new.crc=i;
		new.u.nr=i;
		if (!hhsearch(handle,&new,DYN_ENTER))
		{
			fprintf(stderr,"cannot enter %d\n",i);
			ec=1;
		}
	}
	puts("2->find");
	for (i=0;i<NR;i++)
	{
		new.crc=i;
		new.u.nr=-1;
		if (!(found=hhsearch(handle,&new,DYN_FIND)))
		{
			fprintf(stderr,"cannot find %d\n",i);
			ec=1;
		}
		else if (found->u.nr!=i)
		{
			fprintf(stderr,"found %d, but nr==%d\n",i,found->u.nr);
			ec=1;
		}
	}
	puts("3->overwrite 50%");
	for (i=0;i<NR;i+=2)
	{
		new.crc=i;
		new.u.nr=-i;
		if (!(found=hhsearch(handle,&new,DYN_ENTER_OR_OVERWRITE)))
		{
			fprintf(stderr,"cannot enter %d\n",i);
			ec=1;
		}
	}
	puts("4->find 50%");
	for (i=0;i<NR;i+=2)
	{
		new.crc=i;
		if (!(found=hhsearch(handle,&new,DYN_FIND)))
		{
			fprintf(stderr,"cannot enter %d\n",i);
			ec=1;
		}
		else if (found->u.nr!=-i)
		{
			fprintf(stderr,"found %d, but nr==%d\n",i,found->u.nr);
			ec=1;
		}
	}
	puts("5->overwrite 50%");
	for (i=0;i<NR;i+=2)
	{
		new.crc=i;
		new.u.nr=i;
		if (!(found=hhsearch(handle,&new,DYN_OVERWRITE)))
		{
			fprintf(stderr,"cannot overwrite %d\n",i);
			ec=1;
		}
	}
	puts("6->find 50%");
	for (i=0;i<NR;i+=2)
	{
		new.crc=i;
		if (!(found=hhsearch(handle,&new,DYN_FIND)))
		{
			fprintf(stderr,"cannot enter %d\n",i);
			ec=1;
		}
		else if (found->u.nr!=i)
		{
			fprintf(stderr,"found %d, but nr==%d\n",i,found->u.nr);
			ec=1;
		}
	}
	puts("7->find other 50%");
	for (i=1;i<NR;i+=2)
	{
		new.crc=i;
		if (!(found=hhsearch(handle,&new,DYN_FIND)))
		{
			fprintf(stderr,"cannot enter %d\n",i);
		}
		else if (found->u.nr!=i)
		{
			fprintf(stderr,"found %d, but nr==%d\n",i,found->u.nr);
			ec=1;
		}
	}
	puts("8->delete 50%");
	for (i=0;i<NR;i+=2)
	{
		new.crc=i;
		hhsearch(handle,&new,DYN_DELETE);
	}
	puts("9->find deleted 50%");
	for (i=0;i<NR;i+=2)
	{
		new.crc=i;
		if ((found=hhsearch(handle,&new,DYN_FIND))!=NULL)
		{
			fprintf(stderr,"%d is not deleted\n",i);
			ec=1;
		}
	}
	puts("10->find still existing 50%");
	for (i=1;i<NR;i+=2)
	{
		new.crc=i;
		if ((found=hhsearch(handle,&new,DYN_FIND))==NULL)
		{
			fprintf(stderr,"%d is deleted\n",i);
			ec=1;
		}
	}
	puts("11->reenter deleted 50%");
	for (i=0;i<NR;i+=2)
	{
		new.crc=i;
		if ((found=hhsearch(handle,&new,DYN_ENTER))==NULL)
		{
			fprintf(stderr,"cannot enter %d\n",i);
			ec=1;
		}
	}
	hhdestroy(handle);
	exit(ec);
}
#endif
