# Copyright (c) HashiCorp, Inc.
# SPDX-License-Identifier: BUSL-1.1

packer {
  required_version = ">= 1.5.4"
}

variable "aws_region" {
  type    = string
  default = "${env("AWS_DEFAULT_REGION")}"
}

variable "dd_api_key" {
  type    = string
  default = "${env("DD_API_KEY")}"
}

data "amazon-ami" "autogenerated_1" {
  filters = {
    architecture                       = "x86_64"
    "block-device-mapping.volume-type" = "gp2"
    name                               = "ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64-server-*"
    root-device-type                   = "ebs"
    virtualization-type                = "hvm"
  }
  most_recent = true
  owners      = ["099720109477"]
  region      = "${var.aws_region}"
}

source "amazon-ebs" "ubuntu18-ami" {
  ami_description             = "An Ubuntu 18.04 AMI that has Consul installed."
  ami_name                    = "consul-ubuntu-${ formatdate("YYYY-MM-DD", timestamp()) }T${ formatdate("hh-mm", timestamp()) }-{{uuid}}"
  associate_public_ip_address = true
  instance_type               = "t2.micro"
  region                      = "${var.aws_region}"
  source_ami                  = "${data.amazon-ami.autogenerated_1.id}"
  ssh_username                = "ubuntu"
  tags = {
      Environment = "consul-load-test"
  }
}

build {
  sources = ["source.amazon-ebs.ubuntu18-ami"]

  provisioner "shell" {
    inline = ["mkdir -p /home/ubuntu/scripts"]
  }

  provisioner "file" {
    destination  = "/home/ubuntu"
    pause_before = "30s"
    source       = "${path.root}/scripts"
  }

  provisioner "shell" {
    inline       = ["/home/ubuntu/scripts/setup-systemd-resolved"]
    pause_before = "30s"
  }

  provisioner "shell" {
    inline = ["DD_AGENT_MAJOR_VERSION=7 DD_API_KEY=${var.dd_api_key} bash -c \"$(curl -L https://raw.githubusercontent.com/DataDog/datadog-agent/master/cmd/agent/install_script.sh)\""]
  }

  provisioner "shell" {
    environment_vars = ["DD_API_KEY=${var.dd_api_key}"]
    execute_command  = "sudo sh -c '{{ .Vars }} {{ .Path }}'"
    script           = "${path.root}/scripts/move-files.sh"
  }

}
