/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __TERRAIN_H
#define __TERRAIN_H

#include <cstdlib>
#include <cstdio>
#include <vector>
#include <time.h>
#include <cmath>
#include <queue>
#include <cassert>
#include <algorithm>
#include "globals.h"
#include "coord.h"
#include "block.h"
#include "game_obj.h"

using namespace std;

#define CASTLE_RADIUS 1
#define CASTLE_LENGTH ((CASTLE_RADIUS * 2) + 1)
#define CASTLE_WALL_RADIUS (CASTLE_RADIUS + 3)
#define CASTLE_MINDIST  ( CASTLE_WALL_RADIUS*2 + 3) // minimum distance from castle center to walls

/**
 * Class for a single tile.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class tile
{
 public:
  /** Type of the tile. */
  int type;
  /** Owner of the tile. */
  int owner;
  /** Checked-status of the tile. */
  bool checked;
  /** Conquered-status of the tile. */
  bool conquered;
  /** damage of the tile */
  int damage;
  /** Available types of the tile. */
  enum tile_type {
    TYPE_EMPTY = 0, // plain grass
    TYPE_WALL, // castle wall
    TYPE_CASTLE_A, // castle area
    TYPE_CASTLE_C, // castle center
    TYPE_CANNON_A, // cannon area
    TYPE_CANNON_C, // cannon center
    TYPE_CANNON_A_DEST,
    TYPE_CANNON_C_DEST,
    TYPE_BIG_CANNON_A, //big cannon area
    TYPE_BIG_CANNON_C, //big cannon center
    TYPE_BIG_CANNON_A_DEST, 
    TYPE_BIG_CANNON_C_DEST
  };
  /**
   * Constructor. Constructs a tile of given type and owner.
   * \param type Type of the tile.
   * \param owner Owner of the tile.
   */
  tile(int _type, int _owner);
  /**
   * Default constructor. Constructs a tile of type TYPE_EMPTY and owner OWNER_NOONE.
   */
  tile();

  void hit(const ammo &a) ;
  //{ damage+= AMMO_DAMAGE; }
  inline int get_damage () const
    { return damage; }
};

/**
 * Class representing the whole playing areas "background".
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class terrain
{
 protected:
  /** Tiles of the background. */
  tile tiles[YTILES][XTILES];
  /** Number of castles in the area. */
  int num_castles;
  /** Changed tiles since the last get_changed_tiles(). */
  vector<coord> changed_tiles;
  /**
   * Initializes the terrain with given number of castles.
   * \param num_castles The number of castles in the terrain.
   */
  void init_terrain (int num_castles);

 public:
  /**
   * Constructor. Creates a terrain with given number of castles.
   * \param num_castles Number of castles to place in the created terrain.
   */
  terrain(int num_castles = DEFAULT_NUM_CASTLES);

  coord get_center_tile (const coord &c);
  /**
   * Checks if some of the the surrounding tiles contain tiles of type
   * TYPE_WALL belonging to the player.
   * \param c Coordinates of the location to be checked.
   * \param player_id Index of the player to check the tiles for.
   * \return -1 if adjacent contain foreign wall-tiles, 1 if no foreign and some own, and 0 if neither.
   */
  int check_surrounding (const coord &c, int player_id);
  /**
   * Inserts a castle.
   * \param c Coordinates of the castle to be inserted.
   * \param owner Owning player of the castle.
   */
  void insert_castle (const coord &c, int owner);
  /**
   * Checks the surrounding square for occupied tiles.
   * \param c Coordinates of the center of the square to be checked.
   * \param radius Radius of the square.
   * \return True if the square is clear, false if not.
   */
  bool is_square_clear (const coord &c, int radius);
  /**
   * Checks if the tile at the given coordinates is clear.
   * \param c Coordinates to be checked.
   * \return True if it is clear, false if not.
   */
  bool is_tile_clear (const coord &c);
  /**
   * Sets tile at the given coordinates.
   * \param c Coordinates of the tile to be set.
   * \param t Tile the tile is to be set.
   */
  void set_tile (const coord &c, tile &t);
  /**
   * Sets the tile at the given coordinates.
   * \param c Coordinates of the tile to be set.
   * \param type New type of the tile.
   * \param owner New owner of the tile.
   */
  void set_tile (const coord &c, int type, int owner);
  /**
   * Gives a tile.
   * \param c Coordinates of the tile to be given.
   * \return Tile in the given location.
   */
  tile& get_tile (const coord &c);
  /**
   * Gives castle centers.
   * \return a vector of coordinates of tiles containing castle centers.
   */
  vector<coord> get_castle_centers() const;
  /**
   * Updates the conquered-status of the tiles.
   */
  void check_conquered();
  /**
   * Gives the conquered tiles of certain type of a player.
   * \param type Type of tiles to be given.
   * \param owner Owner of the tiles to be given.
   * \return A vector of pointers to the tiles.
   */
  vector<const tile *> conquered_tiles(int type, int owner);
  /**
   * Gives coordinates of the conquered tiles.
   * \param type Type of the tiles to be given.
   * \param owner Player whose conquered tiles are to be taken.
   * \return A vector of coordinates of the tiles.
   */
  vector<coord> conq_tiles (int type, int owner);
   /**
    * Gives a vector of coordinates to the changed tiles since the last get_changed_tiles().
    * \return A vector of coordinates.
    */
  vector<coord> get_changed_tiles();
  /**
   * Removes dead players owned wall tiles & player ownage from castles & cannons.
   * \param index Player index who has dies.
   */
  void dead_player_remove(int index);

  void hit (const coord &c, const ammo &a);
};


#endif
