/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo, Marko Mattila

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "game_client.h"

const char * game_client::character_names[4] = { "Tervonen", "Klemola", "Orovuo", "Mattila" };

game_client::game_client (string *names, IPaddress *_server, SDL_Surface *screen, TTF_Font *font, int _local_players, gfx *_graphix, starfield_efect *se, audio *au, config *cfg)
  : gui_screen (screen, font, _graphix, se, au, cfg)
{
  serv_conn =0;
  local_players =_local_players;
  mode = ID_MODE_NOTACTIVE;

  graphix = _graphix;
  stats   = new game_statistic();

  server_ip = _server;
  for(int pn=0; pn <local_players;pn++)
    {
      player_index[pn] = pn;
      players[pn] = player((connects *)1, names[pn]); // use '1' as connects to indicate non-null "connected"
      
      // Add statistic data for this player 'pn' is used later as an id
      //stats->add_player(new player_statistic(pn));
      //fprintf(stdout, "Adding player with index: %d\n", pn);

      //set default keys
      //set_keys(SDLK_o, SDLK_p, SDLK_UP, SDLK_DOWN, SDLK_RIGHT, SDLK_LEFT,SDLK_1, pn);
      set_keys( cfg->get_key( pn, config::_KEY_ACTION1 ), cfg->get_key( pn, config::_KEY_ACTION2 ),
          cfg->get_key( pn, config::_KEY_UP ), cfg->get_key( pn, config::_KEY_DOWN ),
          cfg->get_key( pn, config::_KEY_RIGHT ), cfg->get_key( pn, config::_KEY_LEFT ),
          cfg->get_key( pn, config::_KEY_TALK ), pn );
    }
  // set player 1 & 2 keys
  // Umm, aren't these useless? Weren't these keys *already* set in the
  // loop above?
  //set_keys(SDLK_o, SDLK_p, SDLK_UP , SDLK_DOWN, SDLK_RIGHT, SDLK_LEFT, SDLK_1, 0);
  //set_keys(SDLK_v, SDLK_b, SDLK_w , SDLK_s, SDLK_d, SDLK_a, SDLK_2, 1);
  set_keys( cfg->get_key( config::PLAYER_1, config::_KEY_ACTION1 ), cfg->get_key( config::PLAYER_1, config::_KEY_ACTION2 ),
      cfg->get_key( config::PLAYER_1, config::_KEY_UP ), cfg->get_key( config::PLAYER_1, config::_KEY_DOWN ),
      cfg->get_key( config::PLAYER_1, config::_KEY_RIGHT ), cfg->get_key( config::PLAYER_1, config::_KEY_LEFT ),
      cfg->get_key( config::PLAYER_1, config::_KEY_TALK ), 0 );
  set_keys( cfg->get_key( config::PLAYER_2, config::_KEY_ACTION1 ), cfg->get_key( config::PLAYER_2, config::_KEY_ACTION2 ),
      cfg->get_key( config::PLAYER_2, config::_KEY_UP ), cfg->get_key( config::PLAYER_2, config::_KEY_DOWN ),
      cfg->get_key( config::PLAYER_2, config::_KEY_RIGHT ), cfg->get_key( config::PLAYER_2, config::_KEY_LEFT ),
      cfg->get_key( config::PLAYER_2, config::_KEY_TALK ), 1 );
   

  wins = 0;
  game_on=  false;
  round = 1;


  for(int i = 0; i < MAX_PLAYERS; i++)
    talk_txt[i] = "";
  talk = false;
  talker = 1;
  
}

game_client::~game_client()
{
  disconnect();

  while (!outp_queue.empty())
    {
      msg *m = outp_queue.front();
      outp_queue.pop();
      delete m;
    }

  delete stats;
}

void game_client::disconnect()
{
  if (serv_conn)
    {
      serv_conn->disconnect();
      delete serv_conn;
      serv_conn = 0;
    }
}

void game_client::connect() throw (string &)
{
  msg *m = 0;
  vector<string> nams;
  for (int ii=0;ii<local_players;ii++)
    nams.push_back(players[player_index[ii]].get_name());
  
  try {
    if (serv_conn) 
      disconnect();
    serv_conn = new client_tcpnet(server_ip);
    vector<string> tmpv;
    msg_clienthello sendm (nams);
    serv_conn->send(&sendm);
    m = serv_conn->recv();
    msg_serverhello *msh = dynamic_cast<msg_serverhello *>(m);
    if (msh == 0)
      throw string ("Error: received message type not serverhello.");
	
    const vector<int> &indv = msh->get_player_indices();
    for (unsigned int i=0;i<indv.size();i++)
      {
	player_index[i] = indv[i];
	if (player_index[i] != indv[i])
	  {
	    player p1 (players[player_index[i]]);
	    player p2 (players[indv[i]]);
	    players[player_index[i]] = p2;
	    players[indv[i]] = p1;	  	  
	  }
      }
    // add socket to socketset to be able to check its status
  }
  catch (string &st) {
    if (m) { delete m; m = 0; }
#ifdef DEBUG
    printf ("Error performing handshake: %s\n", st.c_str());
#endif
    delete serv_conn;
    serv_conn = 0;
    throw st;
  }
  if (m) { delete m; m = 0;}
#ifdef DEBUG
  printf ("Connected to server.\n");
#endif
}

void game_client::connect_local(monitor_queue<msg *> **in_q, monitor_queue<msg*> ** out_q) throw (string &)
{
  vector<string> nams;
  for (int ii=0;ii<local_players;ii++)
    nams.push_back(players[player_index[ii]].get_name());
  
  if (serv_conn) 
    disconnect();
  client_localnet *cl = new client_localnet();

  monitor_queue<msg *> *in = cl->get_in_queue();
  monitor_queue<msg *> *out = cl->get_out_queue();

  serv_conn = cl;

  *in_q = in;
  *out_q = out;

#ifdef DEBUG
  printf ("Connected to LOCAL server.\n");
#endif
}


void game_client::data_transfer()
{
  
  msg *m = 0;
  
  // read messages
  while (serv_conn->has_msgs())
    {
      try {
	m = serv_conn->recv();
      }
      catch (string &s) {
	//printf ("Disconnected\n");
	disconnect();
	break;
      }

      msg_player_disconnected *pdm = dynamic_cast<msg_player_disconnected *>(m);
      if (pdm)
	{
	  try {
	    handle_player_disconnect(pdm->get_playerid());	  
	  }
	  catch (last_player_exception e)
	    {
	      delete pdm;
	      throw e;
	    }
	  delete pdm;
	  continue;
	}

      switch (mode)
	{
	case ID_MODE_NOTACTIVE:
	  if (m->get_id() == msg::START_CSELECT)
	    {
	      mode_change = true;
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);

	      //printf ("TALKMESSAGE: %s\n",mmc->get_txt().c_str());
	    }
	  else if(m->get_id() == msg::START_GAME)
	    {
	      msg_start_game *sgm = dynamic_cast<msg_start_game *>(m);
	      //printf("Here we go!\n");
	      wins=sgm->get_wins();
#ifdef DEBUG
	      printf("It takes %i wins to win the game\n",wins);
#endif
	    }
	  else if (m->get_id() == msg::PLAYERINFO)
	    {
	      msg_playerinfo *mpi = dynamic_cast<msg_playerinfo *>(m);
	      int index = mpi->get_index();
	      players[index] = player((connects *) 1, mpi->get_name());
	      // Add a new player to the statistics.
	      // If player exists already, the player won't be added.
	      stats->add_player(new player_statistic(index));
#ifdef DEBUG
	      printf ("Added player %s to index %d\n", mpi->get_name().c_str(), index);
#endif
	    }
	  else if (m->get_id() == msg::CASTLE_LOCATION)
	    {
	      msg_castle_location *mcl = dynamic_cast<msg_castle_location *>(m);
#ifdef DEBUG
	      printf ("Castle in coords ( %d , %d ) - owner %d\n", 
		      mcl->get_position().getTX(), mcl->get_position().getTY(), mcl->get_owner());
#endif
	      terra->insert_castle (mcl->get_position(), OWNER_NOONE);
	      if (mcl->get_owner() != OWNER_NOONE)
		home_castles[mcl->get_owner()] = mcl->get_position();
	    }
	  else
	    {
#ifdef DEBUG
	      printf ("Unknown msg: %d\n", m->get_id());
#endif
	      assert(0); // unknown msg
	    }
	  break;
	case ID_MODE_PLACEC:
	  if (m->get_id() == msg::MOVE_CANNONPLACER)
	    {
	      msg_move_cannonplacer *mmc = dynamic_cast<msg_move_cannonplacer *>(m);
	      cannon_placers[mmc->get_owner()] = mmc->get_position();
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);
	    }
	  else if (m->get_id() == msg::PLACE_CANNON)
	    {
	      msg_place_cannon *mpc = dynamic_cast<msg_place_cannon *>(m);
	      int own = mpc->get_owner();
	      coord c = cannon_placers[own];
	      terra->set_tile (c, tile::TYPE_CANNON_C, own);
	      c.incTX();
	      terra->set_tile (c, tile::TYPE_CANNON_A, own);
	      c.incTY();
	      terra->set_tile (c, tile::TYPE_CANNON_A, own);
	      c.decTX();
	      terra->set_tile (c, tile::TYPE_CANNON_A, own);

	      //play cannon down sound. This is when it's not local player
	      aud->play_cannon_down_sound(players[own].get_character());
			  
	      players[own].set_kredits(players[own].get_kredits() - CANNON_COST);
	    }
	  else if (m->get_id() == msg::PLACE_BIG_CANNON)
	    {
	      msg_place_big_cannon *mpc = dynamic_cast<msg_place_big_cannon *>(m);
	      int own = mpc->get_owner();
	      coord c = cannon_placers[own];
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_C, own);
	      c.incTX();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      c.incTY();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      c.decTX();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      c.incTY();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      c.incTX();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      c.incTX();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      c.decTY();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      c.decTY();
	      terra->set_tile (c, tile::TYPE_BIG_CANNON_A, own);
	      
	      //play cannon down sound. This is when it's not local player
	      aud->play_cannon_down_sound(players[own].get_character());
			  

	      players[own].set_kredits(players[own].get_kredits() - BIG_CANNON_COST);
	    }
	  else if (m->get_id() == msg::CHANGE_CANNON)
	    {
	      msg_change_cannon *mpc = dynamic_cast<msg_change_cannon *>(m);
	      int owner = mpc->get_owner();
	      if (cannon_size[owner]==NORMAL)
		cannon_size[owner]=BIG;
	      else
		cannon_size[owner]=NORMAL;
	      
	    }
	  else if (m->get_id() == msg::START_SHOOT)
	    {
	      mode_change = true;
	    }
	  else
	    {
#ifdef DEBUG
	      printf ("Unknown msg: %d\n", m->get_id());
#endif
	    }
	  break;
	case ID_MODE_SHOOT:
	  if (m->get_id() == msg::START_REPAIR)
	    {
	      mode_change = true;
	    }
	  else if (m->get_id() == msg::SHOOT)
	    {
	      msg_shoot *ms = dynamic_cast<msg_shoot *>(m);
	      int shid = ms->get_shooter(); 
	      shoot (ms->get_from(), ms->get_to(), shid, ms->get_ammosize());
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);
	    }
	  else if (m->get_id() == msg::MOVE_CURSOR)
	    {
	      msg_move_cursor *mmu = dynamic_cast<msg_move_cursor *>(m);
	      cursors[mmu->get_owner()] = mmu->get_position();
	      cursor_lock[mmu->get_owner()]=mmu->get_lock();
	    }
	  else
	    {
#ifdef DEBUG
	      printf ("Unknown msg: %d\n", m->get_id());
#endif
	    }
	  break;
	case ID_MODE_REPAIR:
	  if (m->get_id() == msg::START_ENDROUND)
	    {
	      mode_change = true;
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);
	    }
	  else if (m->get_id() == msg::MOVE_BLOCKPLACER)
	    {
	      msg_move_blockplacer *mmb = dynamic_cast<msg_move_blockplacer *>(m);
	      int own = mmb->get_owner();
	      block_placers[own].set_center(mmb->get_position());
	    }
	  else if (m->get_id() == msg::PLACE_BLOCK)
	    {
	      msg_place_block *mpb = dynamic_cast<msg_place_block *>(m);
	      int own = mpb->get_owner();
	      place_block (block_placers[own], own);
	      // gc->block_placers[m.owner].set_center(gc->block_placers[m.owner].center);
	      block_placers[own].set_type(mpb->get_next_type());
	      block_placers[own].set_orientation(mpb->get_next_orientation());
	    }
	  else if (m->get_id() == msg::ROTATE_BL)
	    {
	      msg_rotate_block *mrb = dynamic_cast<msg_rotate_block *>(m);
	      int own = mrb->get_owner();
	      aud->play_block_rotate_sound();
	      block_placers[own].rotate();
	    }
	  else
	    {
#ifdef DEBUG
	    printf("Unknown message: %d\n", m->get_id());
#endif
	    }
	  break;
	case ID_MODE_ENDR:
	  if (m->get_id() == msg::START_PLACECAN)
	    {
	      mode_change = true;
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);
	    }
	  else if ( m->get_id() == msg::START_ENDGAME)
	    {
	      mode_change = true;
	    }
	  else
	    {
#ifdef DEBUG
	    printf ("Unknown message: %d\n", m->get_id());
#endif
	    }
	  break;
	case ID_MODE_ENDGAME:
	  if(m->get_id() == msg::START_GAME)
	    {
	      mode_change = true;
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);
	    }
	  else if(m->get_id() == msg::GAME_OVER)//game is over for good
	    {
	      game_on = false;
	      mode_change = true;
	    }
	  else
	    {
#ifdef DEBUG
	    printf ("Unknown message: %d\n", m->get_id());
#endif
	    }
	  break;
	case ID_MODE_CHSELECT:
	  if (m->get_id() == msg::START_CSELECT)
	    {
	      mode_change = true;
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);
	    }
	  else if(m->get_id()==msg::MOVE_CHARACTERCHOOSER)
	    {
	      msg_move_characterchooser *mmc = dynamic_cast<msg_move_characterchooser *>(m);
	      character_choosers[mmc->get_owner()] = mmc->get_position();
	    }	  
	  else if(m->get_id()==msg::CHOOSE_CHARACTER)
	    {
	     
	      msg_choose_character *mmc = dynamic_cast<msg_choose_character *>(m);
	      players[mmc->get_owner()].put_character(mmc->get_character());
	      
	      if(mmc->get_character()==NOONE)
		cursor_lock[mmc->get_owner()]=false;//open cursor lock if no character
	      else
		{
		  	      
		  char teksti[150];
		  sprintf(teksti,"Character chosen: %s has chosen %s",
			  players[mmc->get_owner()].get_name().c_str(),
			  character_names[mmc->get_character()]);
			  //players[mmc->get_owner()].get_name().c_str());
		  
		  string txt(teksti);

		  aud->play_select_castle_sound(mmc->get_character());

		  cursor_lock[mmc->get_owner()]=true;		  
		  objs.push_back(new text_object (-1, txt));	    
		}


	    }
	  else
	    {
#ifdef DEBUG
	      printf ("Unknown message: %d\n", m->get_id());
#endif
	    }
	  break;
	case ID_MODE_CSELECT:
	  if (m->get_id() == msg::MOVE_CHOOSER)
	    {
	      msg_move_chooser *mmc = dynamic_cast<msg_move_chooser *>(m);
	      process_castlechooser_move(mmc->get_owner(), mmc->get_position());
	    }
	  else if(m->get_id()==msg::TALKMESSAGE)
	    {
	      msg_talkmessage *mmc = dynamic_cast<msg_talkmessage *>(m);
	      process_talkmessage(mmc);
	    }
	  else if (m->get_id() == msg::CHOOSER_LOCK)
	    {
	      msg_chooser_lock *mcl = dynamic_cast<msg_chooser_lock *>(m);
	      int own = mcl->get_owner();
	      chooser_lock (own);
	    }
	  else if (m->get_id() == msg::START_PLACECAN)
	    {
	      mode_change = true;
	    }
	  else
	    {
#ifdef DEBUG
	    printf ("Unknown message: %d\n", m->get_id());
#endif
	    }
	  break;
	default:
	  // should never get here
	  assert(0);
	}
      delete m;
    }
  // send all pending messages
  while (!outp_queue.empty()) 
    {
      msg *sm = outp_queue.front();
      outp_queue.pop();
      serv_conn->send(sm);
      delete sm;
    }
}

void game_client::mode_character_select() throw (exception)
{
  mode = ID_MODE_CHSELECT;
  mode_change = false;

  bool talk = false;
  string txt="";
  int talker =-1;
  
  for(int x=0;x<MAX_PLAYERS;x++)
    players[x].put_character(NOONE);
  
  SDL_EnableKeyRepeat (100, 100);
  
  graphix->draw_bg(terra,players);
  graphix->update();
  
  Uint32 t_start = SDL_GetTicks();
  Uint32 o_time = t_start;
  
  //tss kohtaa nm on characterchoosereita. Pit vaihtaa myhemmin kuntoon.
  for (int i=0;i<MAX_PLAYERS;i++)
    {
      if(players[i].get_name().size() > 0)
	character_choosers[i] = coord::tc(20+i*8,10);
      else
	character_choosers[i] = coord::tc(-1,-1);
    }

  // unlock all cursors
  for(int ll=0;ll<MAX_PLAYERS;ll++)
    cursor_lock[ll]=false;
  
  while (!mode_change)
    {
      data_transfer();
      
      o_time = t_start;
      t_start = SDL_GetTicks();
      Uint32 advance_time = t_start-o_time;
      
      move_objects(advance_time);
      process_objects();
      
      graphix->update_bg(terra,players,false,false);

      graphix->draw_characters(MAX_PLAYERS);
      graphix->draw_characterchoosers(character_choosers, players, MAX_PLAYERS, cursor_lock);

      //draw talkmode window and txt
      graphix->draw_talkmode(talk,txt,players[talker]);


      graphix->draw_objects(objs, players);
      graphix->draw_bar(-1.0f, "Choose your CHARACTER!");
      graphix->update();
      
      SDL_Event ev;
      while (SDL_PollEvent(&ev))
	{
	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&txt,ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    txt.erase();
		}
	      else
		{
		  for(int pn =0; pn<local_players;pn++)
		    {
		      if(!cursor_lock[player_index[pn]])
			{
			  if(ev.key.keysym.sym == keyDOWN[pn])
			    move_characterchooser(ev.key.keysym.sym, pn);
			  else if(ev.key.keysym.sym == keyRIGHT[pn])
			    move_characterchooser(ev.key.keysym.sym, pn);
			  else if(ev.key.keysym.sym == keyUP[pn])
			    move_characterchooser(ev.key.keysym.sym, pn);
			  else if(ev.key.keysym.sym == keyLEFT[pn])
			    move_characterchooser(ev.key.keysym.sym, pn);
			  else if(ev.key.keysym.sym == keyA[pn])
			    {
			      cursor_lock[player_index[pn]]=true;
			      choose_character(pn);
			    }
			  else if(ev.key.keysym.sym == keyB[pn])
			    {
			      cursor_lock[player_index[pn]]=true;
			      choose_character(pn);
			    }
			  else if(ev.key.keysym.sym == keyTALK[pn])
			    {
			      talk = true;
			      talker = player_index[pn];
			    }
			  else if(ev.key.keysym.sym == SDLK_ESCAPE)
			    play_ended();
			}
		      else if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			}
		      else if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		    }
		}
	    }
	}
    } // end turn-loop
  //build_default_walls();

  //kerrotaan pelaajille, kuinka monta voittoa tarvitaan ja monesko er on menossa
  char voitot[100];
  sprintf(voitot,"You need to win %i rounds to win the game",wins);
  string txt1(voitot);
  objs.push_back(new text_object (coord::pc(10,40),1, txt1, DEFAULT_TEXT_DURATION+1000));	  
  
  sprintf(voitot,"ROUND %i FIGHT!", round);
  string txt2(voitot);
  objs.push_back(new text_object (coord::pc(260,200),1, txt2, DEFAULT_TEXT_DURATION));	    
}

void game_client::mode_castle_select() throw (exception)
{
  mode = ID_MODE_CSELECT;
  mode_change = false;

  //  printf( "Entered CASTLE SELECT mode\n" );
  
  SDL_EnableKeyRepeat (100, 100);
  
  graphix->draw_bg(terra,players);
  graphix->update();
  
  Uint32 start = SDL_GetTicks();
  Uint32 t_start = start;
  Uint32 o_time = t_start;
  bool chooser_lockreq[MAX_PLAYERS];
  for (int i=0;i<MAX_PLAYERS;i++)
    chooser_lockreq[i] = false;
  while (SDL_GetTicks() - start < CASTLE_SELECT_LENGTH && !mode_change)
    {
      data_transfer();

      float tleft = (CASTLE_SELECT_LENGTH - (SDL_GetTicks() - start)) / 1000.0f;

      o_time = t_start;
      t_start = SDL_GetTicks();
      Uint32 advance_time = t_start-o_time;
      
      move_objects(advance_time);
      process_objects();
      
      graphix->update_bg(terra,players,false,false);
      graphix->draw_castlechoosers(home_castles, players, MAX_PLAYERS);

      //draw talkmode window and txt
      graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

      graphix->draw_objects(objs, players);
      graphix->draw_bar(tleft, "Choose your planet!");
      graphix->update();
      
      SDL_Event ev;
      while (SDL_PollEvent(&ev))
	{
	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    talk_txt[talker].erase();
		}
	      
	      else
		{
		  for(int pn =0; pn<local_players;pn++)
		    {
		      if (!own_chooser_locked(pn) && !chooser_lockreq[pn]) 
			{
			  /*switch-lauet korjatt if-lausilla, koska keyA ja keyB ei ole vakioita*/
			  if(ev.key.keysym.sym == keyDOWN[pn])
			    {
			      outp_queue.push(new msg_move_chooser_req(player_index[pn], true));
			    }
			  else if(ev.key.keysym.sym == keyRIGHT[pn])
			    {
			      outp_queue.push(new msg_move_chooser_req(player_index[pn], true));
			    }
			  else if(ev.key.keysym.sym == keyUP[pn])
			    {
			      outp_queue.push(new msg_move_chooser_req(player_index[pn], false));
			    }
			  else if(ev.key.keysym.sym == keyLEFT[pn])
			    {
			      outp_queue.push(new msg_move_chooser_req(player_index[pn], false));
			    }
			  else if(ev.key.keysym.sym == keyA[pn])
			    {
			      // choose current castle
			      chooser_lockreq[pn] = true;
			      outp_queue.push (new msg_chooser_lock(player_index[pn]));
			    }
			  else if(ev.key.keysym.sym == keyB[pn])
			    {
			      // choose current castle
			      chooser_lockreq[pn] = true;
			      outp_queue.push (new msg_chooser_lock(player_index[pn]));
			    }
			  else if(ev.key.keysym.sym == keyTALK[pn])
			    {
			      talk = true;
			      talker = player_index[pn];
			    }
			  else if(ev.key.keysym.sym == SDLK_ESCAPE)
			    play_ended();
			}
		      else if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			}
		      else if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		    }
		}
	    }
	}
    } 
  // if we haven't locked choosers, lock them.
  for(int pn =0; pn<local_players;pn++)
    if (!own_chooser_locked(pn) && !chooser_lockreq[pn])
      outp_queue.push(new msg_chooser_lock(player_index[pn]));
  
  while (!mode_change)
    {
      data_transfer();

      o_time = t_start;
      t_start = SDL_GetTicks();
      Uint32 advance_time = t_start-o_time;
      
      move_objects(advance_time);
      process_objects();
      
      graphix->update_bg(terra,players,false,false);

      //draw talkmode window and txt
      graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

      graphix->draw_objects(objs, players);
      graphix->draw_bar(-1.0f, "Waiting for other players to get ready.");
      graphix->update();

      // go through key events
      SDL_Event ev;
      while (SDL_PollEvent(&ev))
	{
	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    talk_txt[talker].erase();
		}
	      else
		{
		  for(int pn=0; pn <local_players;pn++)
		    {
		      if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		      else if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			}
		    }
		}
	    }
	}      
    }

  build_default_walls();
}

void game_client::activate(int signum)
{
  game_loop(this);
    // if starfield_efect not in objs, re-add it
  for (unsigned int i=0;i<objs.size();i++)
    if (typeid(*(objs[i])) == typeid(starfield_efect)) return;
  objs.push_back(starfield);
  return;
}

bool game_client::own_chooser_locked(int player_number)
{
  vector<int>::iterator it = locked_choosers.begin();
  
  while (it != locked_choosers.end())
    if (*it++ == player_index[player_number])
      return true;
  return false;
}

void game_client::mode_shooting() throw (exception)
{
  int i = 0;
  mode = ID_MODE_SHOOT;
  mode_change = false;
  aud->play_shooting_music();


  
  //  printf( "Entered SHOOTING mode\n" );
  
  SDL_EnableKeyRepeat (0, 0); // disable repeat, we have to do all by ourselved to get smooth cursor-move

  bool three_sec_warning_played=false;

  bool d_pressed[MAX_PLAYERS]=
    {false,false,false,false};
  bool u_pressed[MAX_PLAYERS]=
    {false,false,false,false};
  bool r_pressed[MAX_PLAYERS]=
    {false,false,false,false};
  bool l_pressed[MAX_PLAYERS]=
    {false,false,false,false};
  bool b_pressed[MAX_PLAYERS]=
    {false,false,false,false};
  bool a_pressed[MAX_PLAYERS]=
    {false,false,false,false};

  // unlock all cursors
  for(int ll=0;ll<MAX_PLAYERS;ll++)
    cursor_lock[ll]=false;
  
  // center cursors to castle centers. first init to -1 (=not in use)
  for (i=0;i<MAX_PLAYERS;i++)
    {
      if ( !players[i].is_alive())
	cursors[i].setTX(-1);
      else {
	cursors[i] = home_castles[i] + coord::pc(TILE_WIDTH / 2, TILE_HEIGHT / 2);
      }
    }


  //tst puuttuu kokonaan isot tykit
  for(int pn=0;pn<local_players;pn++)
    {
      vector<coord> tmpv = terra->conq_tiles(tile::TYPE_CANNON_C, player_index[pn]);
      own_cannons[player_index[pn]].clear();
      for (unsigned int i=0;i<tmpv.size();i++)
	{
	  coord cc = coord::tc(tmpv[i].getTX()+1, tmpv[i].getTY()+1);
	  vector<cannon_in*>::iterator ite = own_cannons[player_index[pn]].begin();
	  cannon* kanu = new cannon(cc, player_index[pn]);
	  //alempi vaihettiin thn voi olla, ett sy muistia. Toivottavasti ei
	  //printf("kanuunan paikka %i, %i\n", cc.getPX(), cc.getPY());
	  own_cannons[player_index[pn]].push_back(kanu);
	  //own_cannons[player_index[pn]].insert(ite, (cannon_in*)&kanu);
	  //  own_cannons[player_index[pn]].insert(own_cannons[player_index[pn]].begin(), cannon(cc, player_index[pn]));
	  
	  //const coord &cen = kanu->get_position();
	  
	  //printf("onko viel kanuunan paikka %i, %i\n ", cen.getPX(), cen.getPY() );
	}
      //isot
      vector<coord> tmpx = terra->conq_tiles(tile::TYPE_BIG_CANNON_C, player_index[pn]);
      for (unsigned int i=0;i<tmpx.size();i++)
	{
	  coord cc = coord::tc(tmpx[i].getTX()+1, tmpx[i].getTY()+1);
	  vector<cannon_in*>::iterator ite = own_cannons[player_index[pn]].begin();
	  big_cannon* kanu = new big_cannon(cc, player_index[pn]);
	  own_cannons[player_index[pn]].push_back(kanu);
	}
    }

  /* siltn hetki
  //tst puuttuu kokonaan isot tykit
  for(int pn=0;pn<local_players;pn++)
    {
      vector<coord> tmpv = terra->conq_tiles(tile::TYPE_CANNON_C, player_index[pn]);
      own_cannons[player_index[pn]].clear();
      for (unsigned int i=0;i<tmpv.size();i++)
	{
	  coord cc = coord::tc(tmpv[i].getTX()+1, tmpv[i].getTY()+1);
	  vector<cannon_in*>::iterator ite = own_cannons[player_index[pn]].begin();
	  cannon* kanu = new cannon(cc, player_index[pn]);
	  //alempi vaihettiin thn voi olla, ett sy muistia. Toivottavasti ei
	  printf("kanuunan paikka %i, %i\n", cc.getPX(), cc.getPY());
	  own_cannons[player_index[pn]].push_back(kanu);
	  //own_cannons[player_index[pn]].insert(ite, (cannon_in*)&kanu);
	  //  own_cannons[player_index[pn]].insert(own_cannons[player_index[pn]].begin(), cannon(cc, player_index[pn]));
	  
	  const coord &cen = kanu->get_position();
	  
	  printf("onko viel kanuunan paikka %i, %i\n ", cen.getPX(), cen.getPY() );
	}
    }
  */


  graphix->draw_bg(terra,players);
  graphix->update();
  
  Uint32 start = SDL_GetTicks();
  Uint32 o_time = start;
  Uint32 t_start = start;
  
  clear_event_queue();
  
  
  while (SDL_GetTicks() - start < SHOOTING_LENGTH)
    {
      
      data_transfer();
      
      float tleft = (SHOOTING_LENGTH - (SDL_GetTicks() - start)) / 1000.0f;
      
      o_time = t_start;
      t_start = SDL_GetTicks();
      Uint32 advance_time = t_start-o_time;
      
      move_objects(advance_time);
      process_objects();
      i++;
      
      // poll events
      SDL_Event ev;
      while (SDL_PollEvent(&ev))
	{

	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    talk_txt[talker].erase();
		}
	      
	      else
		{
		  for(int pn=0;pn<local_players;pn++)
		    {
		      /*switch-lauet korjatt if-lausilla, koska keyA ja keyB ei ole vakioita*/
		      if(ev.key.keysym.sym == keyDOWN[pn])
			d_pressed[pn] = true;
		      else if(ev.key.keysym.sym == keyRIGHT[pn])
			r_pressed[pn] = true;
		      else if(ev.key.keysym.sym == keyUP[pn])
			u_pressed[pn] = true;
		      else if(ev.key.keysym.sym == keyLEFT[pn])
			l_pressed[pn] = true;
		      else if(ev.key.keysym.sym == keyA[pn])
			{
			  shoot(cursors[player_index[pn]], player_index[pn]);
			  a_pressed[pn] = true;
			}
		      else if(ev.key.keysym.sym == keyB[pn])
			b_pressed[pn] = true;
		      else if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			  for(int ppn=0;ppn<local_players;ppn++)
			    {
			      d_pressed[ppn] = false;
			      r_pressed[ppn] = false;
			      u_pressed[ppn] = false;
			      l_pressed[ppn] = false;
			      a_pressed[ppn] = false;
			      b_pressed[ppn] = false;
			      cursor_lock[player_index[ppn]] = false;
			    }
			}
		      else if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		      //else
		      //break; //tupla break?!?
		    }
		}
	    }
	  if (ev.type == SDL_KEYUP && !talk)
	    {
	      for(int pn=0;pn<local_players;pn++)
		{
		  /*switch-lauet korjatt if-lausilla, koska keyA ja keyB ei ole vakioita*/
		  if(ev.key.keysym.sym == keyDOWN[pn])
		    d_pressed[pn] = false;
		  else if(ev.key.keysym.sym == keyRIGHT[pn])
		    r_pressed[pn] = false;
		  else if(ev.key.keysym.sym == keyUP[pn])
		    u_pressed[pn] = false;
		  else if(ev.key.keysym.sym == keyLEFT[pn])
		    l_pressed[pn] = false;
		  else if(ev.key.keysym.sym == keyA[pn])
		    {
		      a_pressed[pn] = false;
		      cursor_lock[player_index[pn]] = false;
		    }
		  else if(ev.key.keysym.sym == keyB[pn])
		    b_pressed[pn] = false;
		  //else
		  //break;//tupla break
		}
	    }
	} // end poll events

      double move = 0.0;
      
      for(int pn=0;pn<local_players;pn++)
	{
	  coord &c=cursors[player_index[pn]];
	  
	  if (b_pressed[pn])
	    move = CURSOR_MOVE_FAST;
	  else move = CURSOR_MOVE;
	  int move_am = (int)(move*advance_time);
	  if (move_am < 1) move_am = 1;
	  
	  // move cursor according to keys pressed
	  if (u_pressed[pn] && c.getPY() > 0+move_am)
	    {
	      c.subPY(move_am);
	    }
	  else if (d_pressed[pn] && c.getPY() < SCR_HEIGHT-1-move_am)
	    {
	      c.addPY(move_am);
	    }
	  if (l_pressed[pn] && c.getPX() > 0+move_am)
	    {
	      c.subPX(move_am);
	    }
	  else if (r_pressed[pn] && c.getPX() < SCR_WIDTH-1-move_am)
	    {
	      c.addPX(move_am);
	    }
	  
	  //	  printf ("curs move %d\n", (int) move*advance_time);
	  // send new cursor position if has changed
	  
	  
	  if (u_pressed[pn] || d_pressed[pn] || l_pressed[pn] || r_pressed[pn] || !a_pressed[pn])
	    outp_queue.push (new msg_move_cursor(c, player_index[pn], cursor_lock[player_index[pn]]));
	  else if (u_pressed[pn] || d_pressed[pn] || l_pressed[pn] || r_pressed[pn] || a_pressed[pn])
	    outp_queue.push (new msg_move_cursor(c, player_index[pn], cursor_lock[player_index[pn]]));
	}	  
      
      graphix->update_bg(terra, players,true,true);

      //draw talkmode window and txt
      graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

      graphix->draw_objects(objs, players);
      graphix->draw_bar(tleft, "Shoot enemy walls & cannons!");


      //play three sec left warning sound
      if(tleft<=3 && !three_sec_warning_played)
	{
	  aud->play_three_sec_warning_sound();
	  three_sec_warning_played=true;
	}
	
      
      //lasketaan vapaana olevat cannonit
      for(int pn=0;pn<local_players;pn++)
      {
	  int nc=0;
	  vector<cannon_in*>::iterator vi;
	  for (vi = own_cannons[player_index[pn]].begin(); vi != own_cannons[player_index[pn]].end(); ++vi)
	    {
	      if((*vi)->is_ready())
		nc++;
	    }
	  //piirretn kursorit
	  graphix->draw_cursors(cursors, MAX_PLAYERS, cursor_lock, nc, player_index[pn], players);
	}
      graphix->update();
    }
  
  // wait until the flying objects have landed.
  // no talkmode key grabbing here yet because that condition. Remember to fix this!
  while (objs.size() > 0)
    {
      data_transfer();
      o_time = t_start;
      t_start = SDL_GetTicks();
      Uint32 advance_time = t_start-o_time;
      
      move_objects(advance_time);
      process_objects();
      
      graphix->update_bg(terra, players,true,true);

      //draw talkmode window and txt
      graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

      graphix->draw_objects(objs, players);
      graphix->draw_bar(-1.0f, "Shooting end. Waiting for objects to reach ground.");
      graphix->update();
      


      // go through key events
      SDL_Event ev;
      while (SDL_PollEvent(&ev))
	{
	  if (ev.type == SDL_KEYDOWN)
	    {
	      switch (ev.key.keysym.sym)
		{
		case SDLK_ESCAPE:
		  play_ended();
		default:
		  break;
		}
	    }
	}
    }
  
  outp_queue.push (new msg_end());
  
  while (!mode_change)
    {
      data_transfer();
      o_time = t_start;
      t_start = SDL_GetTicks();
      
      Uint32 advance_time = t_start-o_time;
      
      move_objects(advance_time);
      process_objects();
      
      graphix->update_bg(terra, players,true,true);

      //draw talkmode window and txt
      graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

      graphix->draw_objects(objs, players);
      graphix->draw_bar(-1.0f, "Shooting end. Waiting for other players to get ready.");
      graphix->update();
      
      // go through key events
      SDL_Event ev;
      while (SDL_PollEvent(&ev))
	{
	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    talk_txt[talker].erase();
		}
	      else
		{
		  for(int pn=0; pn <local_players;pn++)
		    {
		      if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		      else if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			}
		    }
		}
	    }
	}      
    }
}

void game_client::mode_place_cannons() throw (exception)
{
  int i=0;
  mode = ID_MODE_PLACEC;
  mode_change = false;

  aud->play_cannonplace_music();

  bool three_sec_warning_played=false;
  
  //  printf( "Entered PLACE CANNONS mode\n" );
  
  SDL_EnableKeyRepeat (100, 50);
  
  terra->check_conquered();

  
  for (i=0;i<MAX_PLAYERS;i++)
    {
      if ( players[i].is_alive())
	{
	  // Give player 2 kredits for each castle
	  vector<const tile *> cas = terra->conquered_tiles(tile::TYPE_CASTLE_C, i);
	  int numberofcastles = cas.size();
	  int con = terra->conquered_tiles(tile::TYPE_EMPTY,i).size();
	  players[i].set_kredits((players[i].get_kredits() + numberofcastles * CASTLE_INCOME)+con);

	  cannon_size[i]= NORMAL;	  
	}
      else 
	{
	  players[i].set_kredits(0);
	}

      // init cannon placers to (-1,-1)
      cannon_placers[i] = coord::tc(-1,-1);
    }
  
  // set cannon placers to home castle centers
  for (i=0;i<MAX_PLAYERS;i++)
    {
      if (players[i].is_alive())
	cannon_placers[i] = home_castles[i];
    }
  
  // mode loop!
  Uint32 t = SDL_GetTicks();
  Uint32 old_time = t;
  Uint32 new_time = t;
  Uint32 advance_time = t;
  
  clear_event_queue();
  while (SDL_GetTicks() - t < (Uint32) PLACE_CANNON_LENGTH && !mode_change)
    {
      data_transfer();
      float tleft = (PLACE_CANNON_LENGTH - (SDL_GetTicks() - t)) / 1000.0f;
      old_time = new_time;
      new_time = SDL_GetTicks();
      advance_time = new_time - old_time;
      
      move_objects(advance_time);
      process_objects();
      
      graphix->update_bg(terra,players,false,false);


      //if(cannon_size[player_index[pn]]==NORMAL)
      graphix->draw_cannonplacers(terra, cannon_placers, MAX_PLAYERS, players, cannon_size);
      //else//for big cannons
      
      //draw talkmode window and txt
      graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

      graphix->draw_objects(objs, players);
      graphix->draw_bar(tleft, "Place cannons!");
      graphix->update();

      //play three sec left warning sound
      if(tleft<=3 && !three_sec_warning_played)
	{
	  aud->play_three_sec_warning_sound();
	  three_sec_warning_played=true;
	}

      SDL_Event ev;
      
      // go through key events
      while (SDL_PollEvent(&ev))
	{

	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    talk_txt[talker].erase();
		}
	      
	      else
		{
		  for(int pn=0;pn<local_players;pn++)
		    {
		      
		      if(ev.key.keysym.sym == keyDOWN[pn])
			{
			  if (players[player_index[pn]].get_kredits()>0)
			    move_cannonplacer(ev.key.keysym.sym, pn);     
			}
		      else if(ev.key.keysym.sym == keyRIGHT[pn])
			{
			  if (players[player_index[pn]].get_kredits()>0)
			    move_cannonplacer(ev.key.keysym.sym, pn);
			}
		      else if(ev.key.keysym.sym == keyUP[pn])
			{
			  if (players[player_index[pn]].get_kredits()>0)
			    move_cannonplacer(ev.key.keysym.sym, pn);
			}
		      else if(ev.key.keysym.sym == keyLEFT[pn])
			{
			  if (players[player_index[pn]].get_kredits()>0)
			    move_cannonplacer(ev.key.keysym.sym, pn);
			}
		      else if(ev.key.keysym.sym == keyA[pn])
			{
			  if(cannon_size[player_index[pn]]==NORMAL)
			    {
			      if (players[player_index[pn]].get_kredits()>0)
				place_cannon(pn);
			    }
			  else if(cannon_size[player_index[pn]]==BIG)
			    {
			      if (players[player_index[pn]].get_kredits()>0)
				place_big_cannon(pn);
			    }
			}
		      else if(ev.key.keysym.sym == keyB[pn])
			{
			  if (players[player_index[pn]].get_kredits()>0)
			    change_cannon(player_index[pn]);
			}
		      else if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			}
		      else if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		    }

		}
	    } // end while
	} // end turn loop
    }
  // if we ran out of time, tell server we're done
  if (!mode_change)
    {
      data_transfer();
      outp_queue.push (new msg_end());
    }
  
  while (!mode_change)
    { // delay until server has given permission to change mode.
      data_transfer();
    }
}


void game_client::move_characterchooser (SDLKey sym, int pn)
{

  coord & c = character_choosers[player_index[pn]];

  //voi olla kohdissa 10,10 10,15 15,10 ja 15,15 

  if(sym == keyUP[pn])
    {
      return;
      //if (c.getTY() <= 10) return;
      //c = coord::tc(c.getTX(),c.getTY()-8);
      //c.decTY();
    }
  
  else if(sym == keyDOWN[pn])
    {
      return;
      //if (c.getTY() >= 15) return; // -2 because cannons are 2x2!
      //c = coord::tc(c.getTX(),c.getTY()+8);
      //c.incTY();   
    }
  else if(sym == keyRIGHT[pn])
    {
      if (c.getTX() >= 40) return; // -2 because cannons are 2x2!
      c = coord::tc(c.getTX()+8,c.getTY());
      //c.incTX();
    }
  else if(sym == keyLEFT[pn])
    {
      if (c.getTX() <= 20) return;
      c = coord::tc(c.getTX()-8,c.getTY());
      //c.decTX();
    }
  outp_queue.push (new msg_move_characterchooser(c, player_index[pn]));
  return;
}

void game_client::change_cannon(int owner)
{
  //change local cannon
  if (cannon_size[owner]==NORMAL)
    cannon_size[owner]=BIG;
  else
    cannon_size[owner]=NORMAL;  

  //send message that it we are changeing cannon
  outp_queue.push (new msg_change_cannon(owner));
  return;  
}

void game_client::move_cannonplacer (SDLKey sym, int pn)
{
  coord & c = cannon_placers[player_index[pn]];
  
  if(sym == keyUP[pn])
    {
      if (c.getTY() <= 0) return;
      c.decTY();
    }
  
  else if(sym == keyDOWN[pn])
    {
      if (c.getTY() >= YTILES-2) return; // -2 because cannons are 2x2!
      c.incTY();   
    }
  else if(sym == keyRIGHT[pn])
    {
      if (c.getTX() >= XTILES-2) return; // -2 because cannons are 2x2!
      c.incTX();
    }
  else if(sym == keyLEFT[pn])
    {
      if (c.getTX() <= 0) return;
      c.decTX();
    }
  outp_queue.push (new msg_move_cannonplacer(c, player_index[pn]));
  return;
}

void game_client::move_blockplacer (SDLKey sym)
{

  for(int pn=0;pn<local_players;pn++)
    {
      bool moved = false;
      block &b = block_placers[player_index[pn]];
      
      if(sym == keyUP[pn])
	moved = b.move(coord::tc(0, -1));
      else if(sym == keyDOWN[pn])
	moved = b.move(coord::tc(0, 1));
      else if(sym == keyRIGHT[pn])
	moved = b.move(coord::tc(1, 0));
      else if(sym == keyLEFT[pn])
	moved = b.move(coord::tc(-1, 0));
      const coord &c = b.get_center();
      if (moved)
	outp_queue.push (new msg_move_blockplacer(c, player_index[pn]));
    }
  return;
}

void game_client::choose_character(int pn)
{
  coord & c = character_choosers[player_index[pn]];

  int cha=NOONE;

  //this is going to change later
  if(c.getTX()==20)
    cha=ROCMAN;
  else if(c.getTX()==28)
    cha=ANUZ;
  else if(c.getTX()==36)
    cha=FIREMAN;
  else if(c.getTX()==44)
    cha=DOD;
  else
    assert(0);
    
  outp_queue.push (new msg_choose_character(player_index[pn],c,cha));
  return;

}
  
bool game_client::place_cannon(int pn)
{
  
  if (players[player_index[pn]].get_kredits() < CANNON_COST)
    return false;


  coord &c = cannon_placers[player_index[pn]];
  
  tile& t = terra->get_tile (c);
  if (!t.conquered || t.owner != player_index[pn] || t.type != tile::TYPE_EMPTY)
    return false;
  
  tile& t2 = terra->get_tile (c+coord::tc(1,0));
  if (!t2.conquered || t2.owner != player_index[pn] || t2.type != tile::TYPE_EMPTY)
    return false;
  
  tile& t3 = terra->get_tile (c+coord::tc(0,1));
  if (!t3.conquered || t3.owner != player_index[pn] || t3.type != tile::TYPE_EMPTY)
    return false;
  
  tile& t4 = terra->get_tile (c+coord::tc(1,1));
  if (!t4.conquered || t4.owner != player_index[pn] || t4.type != tile::TYPE_EMPTY)
    return false;
  
  players[player_index[pn]].set_kredits(players[player_index[pn]].get_kredits()-CANNON_COST);


  t.type = tile::TYPE_CANNON_C;
  t2.type = tile::TYPE_CANNON_A;
  t3.type = tile::TYPE_CANNON_A;
  t4.type = tile::TYPE_CANNON_A;
  
  terra->set_tile (c+coord::tc(1,0), t2);
  terra->set_tile (c+coord::tc(0,1), t3);
  terra->set_tile (c+coord::tc(1,1), t4);
  // note! these have to be placed in this way to get correct drawing behaviour
  terra->set_tile (c, t);
  
  outp_queue.push (new msg_place_cannon(player_index[pn], c));

  bool ended = true;
  for (int i=0;i<local_players;i++)
    if(players[player_index[i]].get_kredits() >= BIG_CANNON_COST && players[player_index[i]].is_alive() || (players[player_index[i]].get_kredits() >= CANNON_COST && players[player_index[i]].is_alive()))
      {
	ended = false;
	break;
      }

  if (ended)
    outp_queue.push (new msg_end());

  //own_cannons[pn]=;

  //play cannon down sound. This is for local players. For remote players this can found in data_tracfer
  aud->play_cannon_down_sound(players[player_index[pn]].get_character());

  return true;
}

bool game_client::place_big_cannon(int pn)
{
  if (players[player_index[pn]].get_kredits() < BIG_CANNON_COST)
    return false;
  
  coord &c = cannon_placers[player_index[pn]];
  
  tile& t = terra->get_tile (c);
  if (!t.conquered || t.owner != player_index[pn] || t.type != tile::TYPE_EMPTY)
    return false;
  
  tile& t2 = terra->get_tile (c+coord::tc(1,0));
  if (!t2.conquered || t2.owner != player_index[pn] || t2.type != tile::TYPE_EMPTY)
    return false;
  
  tile& t3 = terra->get_tile (c+coord::tc(0,1));
  if (!t3.conquered || t3.owner != player_index[pn] || t3.type != tile::TYPE_EMPTY)
    return false;
  
  tile& t4 = terra->get_tile (c+coord::tc(1,1));
  if (!t4.conquered || t4.owner != player_index[pn] || t4.type != tile::TYPE_EMPTY)
    return false;

  tile& t5 = terra->get_tile (c+coord::tc(2,0));
  if (!t5.conquered || t5.owner != player_index[pn] || t5.type != tile::TYPE_EMPTY)
    return false;

  tile& t6 = terra->get_tile (c+coord::tc(2,1));
  if (!t6.conquered || t6.owner != player_index[pn] || t6.type != tile::TYPE_EMPTY)
    return false;

  tile& t7 = terra->get_tile (c+coord::tc(2,2));
  if (!t7.conquered || t7.owner != player_index[pn] || t7.type != tile::TYPE_EMPTY)
    return false;

  tile& t8 = terra->get_tile (c+coord::tc(0,2));
  if (!t8.conquered || t8.owner != player_index[pn] || t8.type != tile::TYPE_EMPTY)
    return false;

  tile& t9 = terra->get_tile (c+coord::tc(1,2));
  if (!t9.conquered || t9.owner != player_index[pn] || t9.type != tile::TYPE_EMPTY)
    return false;
  
  // ok, all tiles where cannon is to be placed are clear & conquered. place cannon.
  
  players[player_index[pn]].set_kredits(players[player_index[pn]].get_kredits()-BIG_CANNON_COST);

  t.type = tile::TYPE_BIG_CANNON_C;
  t2.type = tile::TYPE_BIG_CANNON_A;
  t3.type = tile::TYPE_BIG_CANNON_A;
  t4.type = tile::TYPE_BIG_CANNON_A;
  t5.type = tile::TYPE_BIG_CANNON_A;
  t6.type = tile::TYPE_BIG_CANNON_A;
  t7.type = tile::TYPE_BIG_CANNON_A;
  t8.type = tile::TYPE_BIG_CANNON_A;
  t9.type = tile::TYPE_BIG_CANNON_A;
  
  terra->set_tile (c+coord::tc(1,0), t2);
  terra->set_tile (c+coord::tc(0,1), t3);
  terra->set_tile (c+coord::tc(1,1), t4);
  terra->set_tile (c+coord::tc(2,0), t5);
  terra->set_tile (c+coord::tc(0,2), t6);
  terra->set_tile (c+coord::tc(2,1), t7);
  terra->set_tile (c+coord::tc(1,2), t8);
  terra->set_tile (c+coord::tc(2,2), t9);

  // note! these have to be placed in this way to get correct drawing behaviour
  terra->set_tile (c, t);
  
  outp_queue.push (new msg_place_big_cannon(player_index[pn], c));


  bool ended = true;
  for (int i=0;i<local_players;i++)
    if(players[player_index[i]].get_kredits() >= BIG_CANNON_COST && players[player_index[i]].is_alive() || (players[player_index[i]].get_kredits() >= CANNON_COST && players[player_index[i]].is_alive()))
      {
	ended = false;
	break;
      }

  if (ended)
    outp_queue.push (new msg_end());

  //own_cannons[pn]=;

  //play cannon down sound. This is for local players. For remote players this can found in data_tracfer
  aud->play_cannon_down_sound(players[player_index[pn]].get_character());

  return true;
}
void game_client::mode_inactive() throw (exception)
{
  mode_change = false;
  mode = ID_MODE_NOTACTIVE;


  //0.5 versioo varten nopee korjaus
  if(round>1)
    outp_queue.push (new msg_end());
  
  //  printf( "Entered INACTIVE mode\n" );
  
  // init home castles to coord(-1,-1)
  for (int i=0;i<MAX_PLAYERS;i++)
    home_castles[i] = coord::tc(-1,-1);
  
  // create default terrain, and fill in server-given castle positions in comm thread.
  terra = new terrain (0);
  
  
  graphix->draw_bg(terra,players);


  //  printf("pzq tarkistus 1\n");
  //init all things needed for intro
  int xx=SCR_WIDTH;
  int kulma=0;
  coord c = coord::pc(xx, 400);
  
  Uint32 start_time = SDL_GetTicks();
  Uint32 o_time = start_time;
  Uint32 t_start = start_time;

  while (!mode_change) 
    {
      if(!game_on)
	{
	  o_time = t_start;
	  t_start = SDL_GetTicks();
	  Uint32 advance_time = t_start - o_time;
	  
	  move_objects(advance_time);
	  process_objects();
	  
	  if(xx<-1260)
	    xx=SCR_WIDTH;
	  else
	    xx--;
	  
	  c.setPX(xx);
	  
	  kulma+=5;
	  kulma%=360;
	  
	  
	  
	  const char* names[MAX_PLAYERS]={players[0].get_name().c_str(),players[1].get_name().c_str(),players[2].get_name().c_str(),players[3].get_name().c_str()};
	  graphix->update_bg(terra,players, false, false);

	  //draw talkmode window and txt
	  graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

	  graphix->draw_objects(objs, players);
	  graphix->draw_intro(c, kulma, names , MAX_PLAYERS);
	  graphix->update();
	}
      data_transfer();      
      // go through key events
      SDL_Event ev;

      while (SDL_PollEvent(&ev))
	{
	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    talk_txt[talker].erase();
		}
	      
	      else
		{
		  for(int pn =0; pn<local_players;pn++)
		    {
		      if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			}
		      else if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		    }
		}
	    }
	}
    }

  // clear objects .. removes also starfield effect from there
  objs.clear();
  
  game_on=true;//game on.
  
  // playfield initialized!
  //  printf ("Initial playfield synchronized with server.\n");
  // draw background.
  //graphix->draw_bg(terra,players);
  //graphix->update();
  graphix->clear_updates(terra);
  
  if(round>1)//if it's round one this comes in after player has choosen her character
    {
      //kerrotaan pelaajille, kuinka monta voittoa tarvitaan ja monesko er on menossa
      char voitot[100];
      sprintf(voitot,"You need to win %i rounds to win the game",wins);
      string txt1(voitot);
      objs.push_back(new text_object (coord::pc(10,40),1, txt1, DEFAULT_TEXT_DURATION+1000));	  
      
      sprintf(voitot,"ROUND %i FIGHT!", round);
      string txt2(voitot);
      objs.push_back(new text_object (coord::pc(260,200),1, txt2, DEFAULT_TEXT_DURATION));	    
    }
  
  
}

void game_client::shoot(const coord &from, const coord &to, int shooter, int ammosize)
{
  cursor_lock[shooter]=true;
  objs.insert(objs.end(), new ammo (from, to, shooter, ammosize));

  // fire cannon sound effect
  aud->play_fire_sound(players[shooter].get_character());

  //  printf ("shoot from %d %d to %d %d\n", from.x, from.y, to.x, to.y);
}

void game_client::shoot(const coord &to, int pn)
{

  int ammosize=NORMAL;
  //printf("shoooot");
  
  cannon_in *can = 0;

  //printf("omien kanuunojen mr %i\n", own_cannons[pn].size());
  //vector<cannon_in*>::iterator ite = own_cannons[pn].begin();
  //const coord &en=(*ite)->get_position();
  //printf("kanuunan paikka %i, %i\n", en.getPX(), en.getPY() );

  for (unsigned int i=0;i<own_cannons[pn].size();i++)
    {
      cannon_in* &c = own_cannons[pn][i];
      if (c->is_ready())
	{
	  can = own_cannons[pn][i];
	  break;
	}
    }
  if (can == 0) return;
  
  cursor_lock[pn]=true;
  
  const coord &cen = can->get_position();
  
  if(typeid(*can)==typeid(big_cannon))
    ammosize=BIG;
  else
    ammosize=NORMAL;

  //printf("kanuunan paikka %i, %i", cen.getPX(), cen.getPY() );
      
  // get random scatter
  double r1 = rand() / (double) RAND_MAX;
  double r2 = rand() / (double) RAND_MAX;
  double d1 = rand() / (double) RAND_MAX;
  double d2 = rand() / (double) RAND_MAX;
  int x=0,y=0;
  for (int i=0;i<20;i++)
    if (scatterprob[i] >= r1) 
      {
	x = i;
	break;
      }
  for (int i=0;i<20;i++)
    if (scatterprob[i] >= r2) 
      {
	y = i;
	break;
      }
  
  if (d1 <= 0.5) x = 0-x;
  if (d2 <= 0.5) y = 0-y;
  //  printf ("x %d, y %d\n", x, y);
  
  coord rto = coord::pc(to.getPX()+x, to.getPY()+y);
  
  shoot (cen, rto,pn, ammosize);
  can->set_ready(false);
  
  outp_queue.push (new msg_shoot(cen, rto, pn, ammosize));

}

void game_client::mode_repair() throw (exception)
{
  mode = ID_MODE_REPAIR;
  mode_change = false;

  aud->play_repair_music();
  SDL_EnableKeyRepeat (100, 50);

  bool three_sec_warning_played=false;
  //  printf ("Entered REPAIR mode\n");

  // center blockplacers to castle centers. first init to -1 (=not in use)
  for (int i=0;i<MAX_PLAYERS;i++)
    {
      block_placers[i].set_orientation(0);
      block_placers[i].set_type(block::TYPE_L);
      if (!players[i].is_alive())
	{
	  coord c = coord::tc(-1,-1);
	  block_placers[i].set_center(c);
	  continue;
	}
      block_placers[i].set_center (home_castles[i]);
      // affirm that home castles are "in control" still in this phase.
      coord co = home_castles[i];
      tile t (tile::TYPE_CASTLE_C, i);
      tile ta (tile::TYPE_CASTLE_A, i);
      terra->set_tile(co+coord::tc(-1,-1), ta);
      terra->set_tile(co+coord::tc(-1,0), ta);
      terra->set_tile(co+coord::tc(-1,1), ta);
      terra->set_tile(co+coord::tc(0,-1), ta);
      terra->set_tile(co+coord::tc(0,1), ta);
      terra->set_tile(co+coord::tc(1,-1), ta);
      terra->set_tile(co+coord::tc(1,0), ta);
      terra->set_tile(co+coord::tc(1,1), ta);
      terra->set_tile(co, t);
    }
  
  graphix->draw_bg(terra,players);
  graphix->update();
  
  clear_event_queue();
  
  Uint32 curt = SDL_GetTicks();
  Uint32 o_time = curt;
  Uint32 t_start = curt;

  while (SDL_GetTicks() - curt < REPAIR_LENGTH)
    {
      data_transfer();
      float tleft = (REPAIR_LENGTH - (SDL_GetTicks() - curt)) / 1000.0f;
      
      o_time = t_start;
      t_start = SDL_GetTicks();
      Uint32 advance_time = t_start - o_time;
      
      move_objects(advance_time);
      process_objects();
      
      graphix->update_bg(terra, players,true, false);
      graphix->draw_blockplacers (block_placers, MAX_PLAYERS, players);

      //draw talkmode window and txt
      graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

      graphix->draw_objects(objs, players);
      graphix->draw_bar(tleft, "Repair and build walls.");
      graphix->update();

      //play three sec left warning sound
      if(tleft<=3 && !three_sec_warning_played)
	{
	  aud->play_three_sec_warning_sound();
	  three_sec_warning_played=true;
	}

      
      // go through key events
      SDL_Event ev;
      while (SDL_PollEvent(&ev))
	{
	  if (ev.type == SDL_KEYDOWN)
	    {
	      if(talk)
		{
		  talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		  if(!talk)
		    talk_txt[talker].erase();
		}
	      
	      else
		{
		  for(int pn=0; pn <local_players;pn++)
		    {
		      
		      if(ev.key.keysym.sym == keyDOWN[pn])
			move_blockplacer(ev.key.keysym.sym);
		      else if(ev.key.keysym.sym == keyRIGHT[pn])
			move_blockplacer(ev.key.keysym.sym);
		      else if(ev.key.keysym.sym == keyUP[pn])
			move_blockplacer(ev.key.keysym.sym);
		      else if(ev.key.keysym.sym == keyLEFT[pn])
			move_blockplacer(ev.key.keysym.sym);
		      else if(ev.key.keysym.sym == keyB[pn])
			{
			  aud->play_block_rotate_sound();
			  block_placers[player_index[pn]].rotate();
			  outp_queue.push (new msg_rotate_block(player_index[pn]));
			}
		      else if(ev.key.keysym.sym == keyA[pn])
			{
			  if (place_block(block_placers[player_index[pn]], player_index[pn]))
			    {
			      block &pb = block_placers[player_index[pn]];
			      coord c = pb.get_center();
			      block b (c);
			      outp_queue.push (new msg_place_block(player_index[pn], b.get_type(), 
										 b.get_orientation()));
			      block_placers[player_index[pn]] = b;
			    }
			}
		      else if(ev.key.keysym.sym == keyTALK[pn])
			{
			  talk = true;
			  talker = player_index[pn];
			}
		      else if(ev.key.keysym.sym == SDLK_ESCAPE)
			play_ended();
		      //else
		      //abreak; //tupla break?=!
		    }
		  
		}
	    }
	} // end while
    }
  
  outp_queue.push (new msg_end());
  
  // odotetaan ett kaikki haluaa jatkaa
  while (!mode_change) {

    data_transfer();
    o_time = t_start;
    t_start = SDL_GetTicks();
    Uint32 advance_time = t_start - o_time;
    
    move_objects(advance_time);
    process_objects();
    
    graphix->update_bg(terra, players, false, false);
    //graphix->draw_blockplacers (block_placers, MAX_PLAYERS);

    //draw talkmode window and txt
    graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

    graphix->draw_objects(objs, players);
    graphix->draw_bar(-1.0f, "Waiting for other players to get ready.");
    graphix->update();


    // go through key events
    SDL_Event ev;
    while (SDL_PollEvent(&ev))
      {
	if (ev.type == SDL_KEYDOWN)
	  {
	    if(talk)
	      {
		talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		if(!talk)
		  talk_txt[talker].erase();
	      }
	    else
	      {
		for(int pn=0; pn <local_players;pn++)
		  {
		    if(ev.key.keysym.sym == SDLK_ESCAPE)
		      play_ended();
		    else if(ev.key.keysym.sym == keyTALK[pn])
		      {
			talk = true;
			talker = player_index[pn];
			}
		  }
		}
	  }
      }          
  }
}

bool game_client::mode_endround() throw (exception)
{
  int count=0;
  mode = ID_MODE_ENDR;;
  mode_change = false;
  bool some_dead = false;

  aud->play_intro_music();
  
  //  printf( "Entered ENROUND mode\n" );
  graphix->draw_bg(terra,players);
  graphix->update();

  terra->check_conquered();

  // check dead players.
  for (int i=0;i<MAX_PLAYERS;i++)
    {
      if (players[i].is_alive())
	{
	  vector<const tile *> castles = terra->conquered_tiles(tile::TYPE_CASTLE_C, i);
	  if (castles.size() == 0)
	    {
#ifdef DEBUG
	      printf ("Player %d is dead\n", i);
#endif
	      players[i].kill();
	      some_dead = true;
	      terra->dead_player_remove(i);
	    }
	}
    }
  
  Uint32	start_time = SDL_GetTicks();
  Uint32 o_time = start_time;
  Uint32 t_start = start_time;

  while ( ( SDL_GetTicks() - start_time ) < ENDROUND_LENGTH && !mode_change ) {
    
    data_transfer();
    o_time = t_start;
    t_start = SDL_GetTicks();
    Uint32 advance_time = t_start-o_time;
    
    move_objects(advance_time);
    process_objects();
    
    graphix->update_bg(terra, players,false,false);
    coord co = coord::pc(10,20);

    //draw talkmode window and txt
    graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

    graphix->draw_objects(objs, players);
    draw_text(screen, font, "Dead players:", co );
    
    count = 0;
    
    int dead_count = 0;
    
    for ( int i = 0; i < MAX_PLAYERS; i++ ) 
      {
	if (!players[i].is_alive() && players[i].get_name().size() > 0)
	  {
	    char name[NAME_MAXLEN + 1];
	    strcpy( name, players[i].get_name().c_str());
	    coord co = coord::pc(10, 20*(dead_count+2));
	    draw_text(screen, font, name, co);
	    dead_count++;
	  }
	else if ( players[i].get_name().size() > 0)
	  {
	    count++;
	  }
      }
    if (dead_count == 0)
      draw_text(screen, font, "None", coord::pc(10, 40));

    graphix->draw_bar (-1.0, "Game round end.");
    graphix->update();
    
    // go through key events
    SDL_Event ev;
    while (SDL_PollEvent(&ev))
      {
	if (ev.type == SDL_KEYDOWN)
	  {
	    if(talk)
	      {
		talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		if(!talk)
		  talk_txt[talker].erase();
	      }
	    else
	      {
		for(int pn=0; pn <local_players;pn++)
		  {
		    if(ev.key.keysym.sym == SDLK_ESCAPE)
		      play_ended();
		    else if(ev.key.keysym.sym == keyTALK[pn])
		      {
			talk = true;
			talker = player_index[pn];
		      }
		  }
	      }
	  }
      }
  }
  
  outp_queue.push (new msg_end());
  
  while (!mode_change) {
    data_transfer();
    o_time = t_start;
    t_start = SDL_GetTicks();
    Uint32 advance_time = t_start-o_time;
    
    move_objects(advance_time);
    process_objects();
    
    graphix->update_bg(terra, players,false, false);


    //draw talkmode window and txt
    graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

    graphix->draw_objects(objs, players);

    graphix->draw_bar (-1.0, "Waiting for other players to get ready.");
    graphix->update();
    

    // go through key events
    SDL_Event ev;
    while (SDL_PollEvent(&ev))
      {
	
	if (ev.type == SDL_KEYDOWN)
	  {
	    if(talk)
	      {
		talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		if(!talk)
		  talk_txt[talker].erase();
	      }
	    
	    else
	      {
		for(int pn=0; pn <local_players;pn++)
		  {
		    
		    if(ev.key.keysym.sym == SDLK_ESCAPE)
		      play_ended();
		    else if(ev.key.keysym.sym == keyTALK[pn])
		      {
			talk = true;
			talker = player_index[pn];
		      }
		  }
	      }
	  }
      }
  }
  
  if ( count < 2 && some_dead)
    return false;
  
  return true;
}

void game_client::mode_endgame() throw (exception)
{
  mode = ID_MODE_ENDGAME;
  mode_change = false;
  
  //  printf ("Entered ENDGAME mode\n");

 
 
  Uint32 t = SDL_GetTicks();
  
  graphix->set_bg_fademask(false, 0.75);
  graphix->draw_bg(terra,players);
  
  Uint32 o_time = SDL_GetTicks();
  Uint32 t_start = o_time;
  
  
  // get winner for this round
  string ts ("GAME STATS: ");

  int i = 0;
  for (i=0;i<MAX_PLAYERS;i++)
    {
      if (players[i].is_alive())
	{
	  ts += players[i].get_name();
	  players[i].win();
	  break;
	}
    }
  if (i == MAX_PLAYERS)
    ts += string("No winners this round!");
  else
    ts += string(" gets one win!");
  
  //send winner and her wins to server
  //  for(int pn=0; pn<local_players;pn++)
  //  outp_queue[player_index[pn]].push(msg_player_stats(i, msg(msg::MSG_PLAYER_STATS,i, players[i].wins));
  // DONT SEND!!!
  // CLIENT CAN CALCULATE ALL ITSELF EH? :)
  
  /* lets calculate if game is over*/
  int mostwins=0;
  int winner_index=0;
  for (int i=0; i < MAX_PLAYERS; i++)
    {
      if(players[i].get_wins() > mostwins)
	{
	  mostwins = players[i].get_wins();
	  winner_index = i;
	}
    }

  //  fprintf(stdout, "END GAME\n");
  


  if(mostwins == wins)
    {
      game_on=false;
      //      printf( "GAME OVER\n");
    }
  
  //draw stats to screen
  objs.push_back(new text_object (coord::pc(10,20),1, ts, ENDGAME_LENGTH));
  int y_coord = 0;;
  for(int x=0; x < MAX_PLAYERS; x++)
    {
      if(players[x].get_name().size() > 0)
	{
	  char voitot[NAME_MAXLEN+20];
	  sprintf(voitot,"%s has %i wins", players[x].get_name().c_str(), players[x].get_wins());
	  string txt(voitot);
	  objs.push_back(new text_object (coord::pc(10,40+(20*x)),1, txt, ENDGAME_LENGTH));	  
	  y_coord = 40 + (20*x);
	}
    }

  //draw total winner to screen
  if(!game_on)
    {
      string txt1 (string("Player ")+players[winner_index].get_name()+string(" is the winner!"));
      string txt2 (string("GAME OVER"));
      objs.push_back(new text_object (coord::pc(220,220),1, txt1, ENDGAME_LENGTH));
      objs.push_back(new text_object (coord::pc(270,240),1, txt2, ENDGAME_LENGTH));
    }
  

  while ( (( SDL_GetTicks() - t ) < ENDGAME_LENGTH) && !mode_change) {
    data_transfer();
    o_time = t_start;
    t_start = SDL_GetTicks();
    Uint32 advance_time = t_start-o_time;
    
    move_objects(advance_time);
    process_objects();
    
    graphix->update_bg(terra, players,false, false);

    // Draw game statistics for every player
    graphix->draw_player_statistics(players, stats->get_statistics(), coord::pc(10, y_coord + 25));

    //draw talkmode window and txt
    graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);

    graphix->draw_objects(objs, players);
    
    if (game_on)
      graphix->draw_bar (-1.0f, "Game round end. Get ready for more!");
    else{
      graphix->draw_bar (-1.0f, "Game end. Congratulations to the winner!");
    }

    graphix->update();


    // go through key events
    SDL_Event ev;
    while (SDL_PollEvent(&ev))
      {
	
	if (ev.type == SDL_KEYDOWN)
	  {
	    if(talk)
	      {
		talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		if(!talk)
		  talk_txt[talker].erase();
	      }
	    
	    else
	      {
		for(int pn=0; pn <local_players;pn++)
		  {
		    
		    if(ev.key.keysym.sym == SDLK_ESCAPE)
		      play_ended();
		    else if(ev.key.keysym.sym == keyTALK[pn])
		      {
			talk = true;
			talker = player_index[pn];
		      }
		  }
	      }
	  }
      }    
  }

  //we are ready  
  outp_queue.push (new msg_end());
  
  round++;

  //  mosaic *moz = new mosaic(screen);

  while (!mode_change) {
    data_transfer();
    
    //graphix->update_bg(terra, false, false);
    //graphix->draw_objects(objs);
    //if(game_on)
    //graphix->draw_bar (-1.0f, "Waiting for other players to get ready.");

    //    moz->update();

    //draw talkmode window and txt
    //graphix->draw_talkmode(talk,talk_txt[talker],players[talker]);
    //graphix->draw_objects(objs);

    graphix->update();


    // go through key events
    SDL_Event ev;
    while (SDL_PollEvent(&ev))
      {
	
	if (ev.type == SDL_KEYDOWN)
	  {
	    if(talk)
	      {
		talk = handle_talk(&talk_txt[talker],ev.key.keysym.sym,ev.key.keysym.mod,talker);
		if(!talk)
		  talk_txt[talker].erase();
	      }
	    
	    else
	      {
		for(int pn=0; pn <local_players;pn++)
		  {
		    
		    if(ev.key.keysym.sym == SDLK_ESCAPE)
		      play_ended();
		    else if(ev.key.keysym.sym == keyTALK[pn])
		      {
			talk = true;
			talker = player_index[pn];
		      }
		  }
	      }
	  }
      }        
  }
  
  //  delete moz;
  process_objects();
  graphix->update();
  graphix->unset_bg_fademask();
}

bool game_client::place_block(block &b, int owner)
{
  // first check if the player has enough kredits to place a block.
  if(players[owner].get_kredits() < BLOCK_COST)
    return false;

  const vector<coord> vec = b.get_tiles();
  vector<coord>::const_iterator it = vec.begin();
  const coord cen = b.get_center();
  
  for(int pn=0;pn<local_players;pn++)
    {
      if (owner == player_index[pn]) 
	{
	  
	  // first, check that all tiles clear.
	  while (it != vec.end())
	    {
	      coord c = *it++;
	      tile t = terra->get_tile (c+cen);
	      if (t.type != tile::TYPE_EMPTY) return false;
	      if (t.conquered && t.owner != owner) return false;
	      if (!((c+cen).getTX() >= 0 && (c+cen).getTY() >= 0 && (c+cen).getTX() < XTILES && (c+cen).getTY() < YTILES)) return false; // pzq
	    }
	  
	  // then, check that some of the adjacent tiles contain own blocks
	  it = vec.begin();
	  bool ok = false;
	  while (it != vec.end()) 
	    {
	      const coord &c = *it++;
	      int ret = terra->check_surrounding (c+cen, player_index[pn]);
	      if (ret == -1) return false; // not happening actually anymore, as we allow adjacent enemy blocks.
	      if (ret == 1) ok = true;
	    }
	  if (!ok)
	    return false;
	}
    }
      // then, perform the actual placement of the tiles
      it = vec.begin();
      while (it != vec.end())
	{
	  coord c = coord(*it++);
	  c+=cen;
	  terra->set_tile (c, tile::TYPE_WALL, owner);
	}
      
      //  printf ("placing block of owner %d type %d x %d y %d orient %d\n", 
      //	  owner, b.get_type(), cen.x, cen.y, b.get_orientation());
      // finally, update conquered tiles
      terra->check_conquered();
    
      //block down sound effect
      aud->play_block_down_sound();

      //decrease kredist
      players[owner].set_kredits(players[owner].get_kredits() - BLOCK_COST);

  return true;
}

void game_client::build_default_walls() {

  vector<coord> vec = terra->get_castle_centers();
  int x, y;
  for (unsigned int i=0;i<vec.size();i++)
    {  
      coord &c = vec[i];
      tile &t = terra->get_tile(c);
      if ( t.owner == OWNER_NOONE )
	continue;
      
      // North
      y = c.getTY() - CASTLE_WALL_RADIUS;
      for ( x = c.getTX() - CASTLE_WALL_RADIUS; x < c.getTX() + CASTLE_WALL_RADIUS;
	  x++ )
	{
	  coord c1 = coord::tc(x,y);
	  tile t1(tile::TYPE_WALL, t.owner);
	  terra->set_tile( c1, t1);
	}
      
      // East
      x = c.getTX() + CASTLE_WALL_RADIUS;
      for ( y = c.getTY() - CASTLE_WALL_RADIUS; y < c.getTY() + CASTLE_WALL_RADIUS; y++ )
	{
	  coord c1 = coord::tc(x,y);
	  tile t1(tile::TYPE_WALL, t.owner);
	  terra->set_tile( c1, t1);
	}
      
      // South
      y = c.getTY() + CASTLE_WALL_RADIUS;
      for ( x = c.getTX() + CASTLE_WALL_RADIUS; x > c.getTX() - CASTLE_WALL_RADIUS; x-- )
	{
	  coord c1 = coord::tc(x,y);
	  tile t1(tile::TYPE_WALL, t.owner);
	  terra->set_tile( c1, t1);
	}
      
      // West
      x = c.getTX() - CASTLE_WALL_RADIUS;
      for ( y = c.getTY() + CASTLE_WALL_RADIUS; y > c.getTY() - CASTLE_WALL_RADIUS; y-- )
	{
	  coord c1 = coord::tc(x,y);
	    tile t1(tile::TYPE_WALL, t.owner);
	    terra->set_tile( c1, t1);
	  }
    }
}

void game_client::chooser_lock (int player)
{
  if (find(locked_choosers.begin(), locked_choosers.end(), player) == locked_choosers.end())
    {
      locked_choosers.push_back (player);
      coord co = home_castles[player];
      tile t (tile::TYPE_CASTLE_C, player);
      tile ta (tile::TYPE_CASTLE_A, player);
      terra->set_tile(co+coord::tc(-1,-1), ta);
      terra->set_tile(co+coord::tc(-1,0), ta);
      terra->set_tile(co+coord::tc(-1,1), ta);
      terra->set_tile(co+coord::tc(0,-1), ta);
      terra->set_tile(co+coord::tc(0,1), ta);
      terra->set_tile(co+coord::tc(1,-1), ta);
      terra->set_tile(co+coord::tc(1,0), ta);
      terra->set_tile(co+coord::tc(1,1), ta);
      terra->set_tile(co, t);

      //select castle sound
      aud->play_select_castle_sound(players[player].get_character());
    }
}

void game_client::clearallfornextround()
{
  locked_choosers.erase(locked_choosers.begin(),locked_choosers.end());
  for (int i=0;i<MAX_PLAYERS;i++)
    { 
      if(players[i].get_name().size() > 0)
	{
	  players[i].rebirth();
	}
    }
  stats->reinit();
}


int game_client::game_loop(void *data)
{
  game_client *gc = (game_client *) data;

  try {
    do{
    gc->mode_inactive();

    if(gc->round<2)
      gc->mode_character_select();

    gc->mode_castle_select();
    
    while (true)
      {
	gc->mode_place_cannons();
	gc->mode_shooting();
	gc->mode_repair();
	if (!gc->mode_endround())
	  break;
      }
    gc->mode_endgame();
    gc->clearallfornextround();
    }while(gc->game_on);
  }
  catch (exception &e)
    {
      if (typeid(e) == typeid(esc_exception))
	{
	  gc->errors.push_back (string ("ESC pressed, exited game."));
	}
      else if (typeid(e) == typeid(last_player_exception))
	{
	  gc->errors.push_back (string ("Only 1 player left, exited game."));
	}
      gc->outp_queue.push(new msg_disconnect());
      gc->data_transfer();
      gc->disconnect();
      return 0;
    }
  return 0;
}

void game_client::process_castlechooser_move (int owner, const coord & co)
{
  home_castles[owner] = co;
}

void game_client::clear_event_queue()
{
  SDL_Event ev;
  while (SDL_PollEvent(&ev))
    { }
}

void game_client::handle_player_disconnect(int index) throw (last_player_exception)
{
  string txt (string("Player ")+players[index].get_name()+string(" disconnected."));
  objs.push_back(new text_object (index, txt, DEFAULT_TEXT_DURATION));
  
  // Remove player statistics
  stats->remove_player(index);

#ifdef DEBUG
  fprintf(stdout, "Removing player with index %d from statistic.\n", index);
#endif

  players[index].disconnect();
  vector<int>::iterator it = find(locked_choosers.begin(), locked_choosers.end(), index);
  if (it != locked_choosers.end())
    locked_choosers.erase(it);
  block_placers[index].set_center(coord::tc(-1,-1));
  home_castles[index] = coord::tc(-1,-1);
  terra->dead_player_remove(index);

  if (get_no_players() == 1 && mode != ID_MODE_NOTACTIVE)
    throw last_player_exception();

}

int game_client::get_no_players() const
{
  int c=0;
  for (int i=0;i<MAX_PLAYERS;i++)
    if (players[i].is_connected())
      c++;
  return c;
}

void game_client::set_keys(int _keyA, int _keyB, int _keyUP, int  _keyDOWN, int _keyRIGHT, int _keyLEFT, int _keyTALK, int player_num)
{
  keyA[player_num] = _keyA;
  keyB[player_num] = _keyB;
  keyUP[player_num] = _keyUP;
  keyDOWN[player_num] = _keyDOWN;
  keyRIGHT[player_num] = _keyRIGHT;
  keyLEFT[player_num] = _keyLEFT;
  keyTALK[player_num] = _keyTALK;
}


void game_client::process_objects()
{
  vector<game_obj*>::iterator it = objs.begin();
  while (it != objs.end())
    {
      game_obj *o = *it++;
      assert(o != NULL); // musn't be null
      
      if (!o->is_finished())
        continue;

      // delete object
      it = objs.erase(it - 1);

      if (typeid (*o) == typeid (ammo))
        {
          ammo *a = dynamic_cast<ammo *>(o);
          for(int pn=0;pn<local_players;pn++)
            {
              if (a->get_owner() == player_index[pn])
                {
                  // go through all, check which cannon has it.
                  cannon_in *can = 0;
                  for (unsigned int i=0;i<own_cannons[player_index[pn]].size();i++)
                    {
                      cannon_in* &tc = own_cannons[player_index[pn]][i];
                      const coord &p1 = tc->get_position();
                      const coord &p2 = a->get_start_position();
                      if (p1 == p2)
                        can = tc;
                    }
                  if (can) // if the cannon isn't destroyed, set it as ready
                    can->set_ready(true);
                }
	    }
	  // increse overall ammo amount
	  stats->update_player_nr_all_shots(a->get_owner());


	  // get tile where it landed
	  const coord &pos = a->get_position();
	  tile& t = terra->get_tile(pos);
	  int old_ow = t.owner;
	  int old_pl = MAX_PLAYERS;
	  if (old_ow < MAX_PLAYERS)
	    old_pl = players[old_ow].get_character();
	  
	  if (t.type == tile::TYPE_WALL)
	    {
	      
	      /* Update hits to the statistic*/
	      stats->update_player_nr_hits(a->get_owner());

	      t.type = tile::TYPE_EMPTY;
	      t.owner = OWNER_NOONE;
	      t.conquered = false;
	      
	      terra->set_tile (a->get_position(), t);
	      
	      const coord &tc = a->get_position();

	     
	      explosion_efect *e = new explosion_efect(tc, graphix->get_tileimage(a->get_owner(), gfx::TILEINDEX_WALL));
	      
	      objs.insert(objs.end(), e);

	      //hit sound effect
	      aud->play_hit_sound(old_pl);
	      
	      terra->check_conquered();
	    }
	  else if (t.type == tile::TYPE_CANNON_A || t.type == tile::TYPE_CANNON_C)
	    {
	      //          printf ("%d %d\n", pos.x, pos.y);


	      //hit cannon sound effect
	      aud->play_hit_cannon_sound(old_pl);
			
		  /* hit to the cannon, add to the statistics */
	      stats->update_player_nr_cannon_hits(a->get_owner());

	      coord cc = terra->get_center_tile(pos);
	      terra->hit(cc, *a);
	      tile &t = terra->get_tile(cc);
	      if (t.get_damage() >= CANNON_HP) // was destroyed, check more ..
		{
		  // Destroyed cannon.. update statistics
		  stats->update_player_nr_destroyed_cannons(a->get_owner());
		  anim_explosion_obj *a = new anim_explosion_obj (pos, exp_anim_container::EXP_MED);
		  objs.push_back(a);
		  // if it was ours, it doesn't shoot anymore!
		  for(int pn=0;pn<local_players;pn++)
		    {
                      vector<cannon_in*>::iterator it = own_cannons[player_index[pn]].begin();
                      coord c  = cc;
                      c.incTX();
                      c.incTY();
                      while (it != own_cannons[player_index[pn]].end())
                        {
                          if (c.tile_eq((*it)->get_position()))
                            {
                              own_cannons[player_index[pn]].erase(it);
                              break;
                            }
                          it++;
                        }
		    }
		}
	      else // not destroyed
		{
		  
		  anim_explosion_obj *a = new anim_explosion_obj (pos, exp_anim_container::EXP_SMALL);
		  objs.push_back(a);
		}
	    }
	  //for big cannons
	  else if (t.type == tile::TYPE_BIG_CANNON_A || t.type == tile::TYPE_BIG_CANNON_C)
	    {
	      //          printf ("%d %d\n", pos.x, pos.y);


	      //hit cannon sound effect
	      aud->play_hit_cannon_sound(old_pl);
			
		  /* hit to the cannon, add to the statistics */
	      stats->update_player_nr_cannon_hits(a->get_owner());

	      coord cc = terra->get_center_tile(pos);
	      terra->hit(cc, *a);
	      tile &t = terra->get_tile(cc);
	      if (t.get_damage() >= BIG_CANNON_HP) // was destroyed, check more ..
		{
		  // Destroyed cannon.. update statistics
		  stats->update_player_nr_destroyed_cannons(a->get_owner());
		  anim_explosion_obj *a = new anim_explosion_obj (pos, exp_anim_container::EXP_MED);
		  objs.push_back(a);
		  // if it was ours, it doesn't shoot anymore!
		  for(int pn=0;pn<local_players;pn++)
		    {
                      vector<cannon_in*>::iterator it = own_cannons[player_index[pn]].begin();
                      coord c  = cc;
                      c.incTX();
                      c.incTY();
                      while (it != own_cannons[player_index[pn]].end())
                        {
                          if (c.tile_eq((*it)->get_position()))
                            {
                              own_cannons[player_index[pn]].erase(it);
                              break;
                            }
                          it++;
                        }
		    }
		}
	      else // not destroyed
		{
		  
		  anim_explosion_obj *a = new anim_explosion_obj (pos, exp_anim_container::EXP_SMALL);
		  objs.push_back(a);
		}
	    }
            
	} // end if type ammo
      else if (typeid (*o) == typeid(text_object)) // is of type text_object
	{
	}
      else if (typeid(*o) == typeid(explosion_efect))
	{
	}
      // delete the actual object.
      delete o;
    }
  
}


void game_client::draw_castleowners()
{
  vector<coord> cent = terra->get_castle_centers();

  for (unsigned int i=0;i<cent.size();i++)
    {
      coord &c = cent[i];
      tile &t = terra->get_tile(c);
      if (t.owner == OWNER_NOONE)
	continue;
      graphix->draw_castle_ownertext(players[t.owner], c);
    }
}

void game_client::play_ended()
{
  /* we are using only the reverse mosaic at the moment
  mosaic *moz2 = new mosaic(screen, screen, SDL_GetTicks(), false);
  while(!moz2->is_finished())
    {
      moz2->update(SDL_GetTicks());
      graphix->update();
    }
  delete moz2;
  */

  //fade mask is on if we push ESC in mode_end_game() the mask is erased
  graphix->unset_bg_fademask();

  throw esc_exception();//pois
}

bool game_client::handle_talk(string* txt, SDLKey sym, SDLMod mod,int index)
{
  switch (sym)
    {
    case SDLK_RETURN:
      {
	if(txt->length()>0)
	  {
	    outp_queue.push (new msg_talkmessage(index,string(txt->c_str())));
	    
	    if(mode == ID_MODE_NOTACTIVE || mode == ID_MODE_CHSELECT || mode == ID_MODE_CSELECT || mode == ID_MODE_ENDGAME)
	      {
		char teksti[150];
		sprintf(teksti,"<%s>%s",
			players[index].get_name().c_str(),
			txt->c_str());
		string txt2(teksti);
		objs.push_back(new text_object (-1, txt2,ERRTEXT_DUR));
	      }
	    else
	      {
		
		char teksti[150];
		sprintf(teksti,"%s",txt->c_str());
		string txt2(teksti);
		
		vector<game_obj *>::iterator it = objs.begin();
		objs.insert(it,new balloon_object (home_castles[index],index,txt2,ERRTEXT_DUR,players[index].get_color()));
		
	      }
	  }

	return false;
      }
    case SDLK_BACKSPACE:
      if (txt->length()>0)
	txt->erase(txt->length()-1, 1);
      return true;
    case SDLK_RSHIFT:
    case SDLK_LSHIFT:
      return true;
    default:
      if ( isprint( sym ) && txt->length()<50)
	txt->insert( txt->length(), 1, ( mod & ( KMOD_LSHIFT | KMOD_RSHIFT ) ) ? toupper( sym ) : sym );
    }
  return true;
}

vector<string> game_client::get_errors()
{
  vector<string> res = errors;
  for (unsigned int i=0;i<objs.size();i++)
    {
      game_obj * o = objs[i];
      text_object *to = dynamic_cast<text_object *>(o);
      if (to)
	{
	  if (to->has_unknown_position())
	    res.push_back(to->get_text());
	}
    }
  return res;
}

void game_client::process_talkmessage(msg_talkmessage* mmc)
{

  if(mode == ID_MODE_NOTACTIVE || mode == ID_MODE_CHSELECT || mode == ID_MODE_CSELECT || mode == ID_MODE_ENDGAME)
    {
      char teksti[150];
      sprintf(teksti,"<%s>%s",
	      players[mmc->get_owner()].get_name().c_str(),
	      mmc->get_txt().c_str());
      string txt2(teksti);
      objs.push_back(new text_object (-1, txt2,TALKTEXT_DUR));
    }
  else
    {

  char teksti[150];
  sprintf(teksti,"%s",mmc->get_txt().c_str());
  string txt2(teksti);  
  //objs.push_back(new text_object (-1, txt2));
  //text_object(const coord & pos, int owner, const string &text, Uint32 duration, SDL_Color col);
  vector<game_obj *>::iterator it = objs.begin();
  objs.insert(it,new balloon_object (home_castles[mmc->get_owner()],mmc->get_owner(),txt2,TALKTEXT_GAME_DUR,players[mmc->get_owner()].get_color()));

  /*
  char teksti[150];
  sprintf(teksti,"<%s> %s",
	  players[mmc->get_owner()].get_name().c_str(),
	  mmc->get_txt().c_str());
  string txt2(teksti);
  objs.push_back(new text_object (mmc->get_owner(), txt2, players[mmc->get_owner()].get_color(), (Uint32) ERRTEXT_DUR));
  */
    }
}
