# Copyright (C) 2016-2022 The ESPResSo project
# Copyright (C) 2014 Olaf Lenz
#
# This file is part of ESPResSo.
#
# ESPResSo is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# ESPResSo is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# This script generates code_info.pyx
#
import inspect
import sys
import os
# find featuredefs.py
moduledir = os.path.dirname(inspect.getfile(inspect.currentframe()))
sys.path.append(os.path.join(moduledir, '..', '..', 'config'))
import featuredefs

if len(sys.argv) != 3:
    print(f"Usage: {sys.argv[0]} DEFFILE PYXFILE", file=sys.stderr)
    exit(2)

deffilename, cfilename = sys.argv[1:3]

print(f"Reading definitions from {deffilename}...")
defs = featuredefs.defs(deffilename)
print("Done.")

# generate cpp-file
print(f"Writing {cfilename}...")
cfile = open(cfilename, 'w')

cfile.write("""
# This file is autogenerated by gen_code_info.py.
# DO NOT EDIT MANUALLY, CHANGES WILL BE LOST

include "myconfig.pxi"
from . import utils

def features():
    \"\"\"Returns list of features compiled into ESPResSo core\"\"\"

    f = []
""")

for feature in defs.allfeatures:
    cfile.write(f"\n    IF {feature} == 1:\n        f.append(\"{feature}\")\n")

cfile.write(f"""
    return sorted(f)

def all_features():
    return {defs.allfeatures}


cdef extern from "version.hpp":
    cdef const char * ESPRESSO_BUILD_TYPE


def build_type():
    \"\"\"Prints the CMake build type.
    Can be e.g. Debug, Release, RelWithAssert, RelWithDebInfo, Coverage, etc.
    \"\"\"
    return utils.to_str(ESPRESSO_BUILD_TYPE)  # pylint: disable=undefined-variable


from libcpp.string cimport string
from libcpp.vector cimport vector

IF SCAFACOS:
    cdef extern from "script_interface/scafacos/scafacos.hpp" namespace "ScriptInterface::Scafacos":
        vector[string] available_methods()

def scafacos_methods():
    \"\"\"Lists long-range methods available in the ScaFaCoS library.\"\"\"
    scafacos_features = []
    IF SCAFACOS == 1:
        method_names = available_methods()
        for method_name in method_names:
            scafacos_features.append(method_name.decode('ascii'))
    return scafacos_features
""")

cfile.close()
print("Done.")
