/***************************************************************************
                          linker.cpp  -  description
                             -------------------
    begin                : lun avr 25 2005
    copyright            : (C) 2005 by Gibaud Alain
    email                : alain.gibaud@free.fr
 ***************************************************************************/

#include "linker.h"
#include "utility.h"
#include "compiler.h"
#include "portability.h"

#include <sstream>
#include <vector>
#include <algorithm>
#include <cstdio>
#include <fstream>

using namespace std ;

Linker::Linker()
{
}
Linker::~Linker()
{
}
/** request ref symbol */
void Linker::request ( const string& ref )
{
  _refs[ref] |= Needed ;
}
/** true if ref symbol is requested */
bool Linker::isNeeded ( const string& ref )
{
  map<string,int>::iterator i = _refs.find ( ref ) ;
  if ( i ==  _refs.end() ) return false ;
  int state = ( *i ).second ;
  return ( state & Needed ) && ! ( state & Found )   ;
}

void Linker::satisfy ( const string& ref )
{
  _refs[ref] |= Found ;
}

/** returns the list of undefined symbols */
stringList Linker::undefinedList()
{
  stringList u ;

  map<string,int>::iterator i  ;
  for ( i = _refs.begin() ; i != _refs.end() ; ++i )
  {
    int stat = ( *i ).second ;
    if ( ( stat & Needed ) && ! ( stat & Found ) ) u.push_back ( ( *i ).first ) ;
  }
  return u ;
}

const char *Linker::doLink()
{
  int cnt = 0 ;
  unsigned int i ;
  string mess ;
  const char *err = 0 ;

  // open output file
  _out.open ( _outfile.c_str() ) ;
  if ( !_out.is_open() ) return "Link: Can't open output file." ;

  // generate prolog
  err = makeProlog() ;
  if ( err ) return err ;

  request ( "__config" ) ;  // module for config bits
  request ( "__idlocs" ) ;  // module for idlocs bytes
  request ( "cstart" ) ; // root of all applications (calls main() )
  request ( "C18_lo_pri_ISR" ) ; // user can provide his own interrupt code
  request ( "C18_hi_pri_ISR" ) ; // user can provide his own interrupt code


  // add standard libraries
  if ( find ( objfiles().begin(), objfiles().end(), "float.slb" ) ==  objfiles().end() )
    objfiles().push_back ( "float.slb" ) ; //
  if ( find ( objfiles().begin(), objfiles().end(), "rtl.slb" ) ==  objfiles().end() )
    objfiles().push_back ( "rtl.slb" ) ; //

  // add standard lib paths
  libPaths().push_back ( Portability::cpikLibDir() ) ;

  // test input files
  for ( i = 0 , cnt = 0 ; i < objfiles().size() ; ++i )
  {
    string obj = utility::find ( objfiles() [i], libPaths() );

    if ( obj == "" )
    {
      mess =  "Link: file \"" +   objfiles() [i] + "\" does not exist." ;
      return utility::setMessage ( mess ) ;
    }
    else
      objfiles() [i] = obj ;

  }


  do
  {
    cnt = doOnePassLink();
  }
  while ( cnt != 0 ) ;

  outSegment ( _cdata );
  outSegment ( _code );
  _code.clear();
  // output initialization table for statically initialized data
  makeInitializationTable() ;
  makeUdataClear() ;

  outSegment ( _code ); // output the code generated by initialization


  _out << "\tCBLOCK\n\t_UDATA_BEGIN:0\n\tENDC" << endl;
  outSegment ( _udata );
  _out << "\tCBLOCK\n\t_UDATA_END:0\n\tENDC" << endl;
  _out << "_UDATA_SIZE equ _UDATA_END-_UDATA_BEGIN" << endl;

  _out << "\tCBLOCK\n\t_IDATA_BEGIN:0\n\tENDC" << endl;
  outSegment ( _idata );


  makeEpilog() ;
  _out.close() ;

  stringList undef = undefinedList() ;

  if ( undef.size() != 0 )
  {
    mess =  "Link: Undefined symbol(s):\n" ;
    for ( unsigned int i = 0 ; i < undef.size() ; ++i )
    {
      mess += undef[i] ; mess += "\n" ;
    }
    return utility::setMessage ( mess ) ;
  }

  return 0 ;
}
/* )+1 */
int Linker::countZeroes ( vector<string> &vals, int from, int step )
{
  int cnt=0;
  for ( unsigned int i=0; i<vals.size(); i++, from+=step )
  {
    string val=vals[from];
    if ( val=="0x00"
         || val=="0x0"
         || val=="0"
       )
    {
      cnt++;
    }
    else
    {
      break; // break at first non zero byte
    }
  }
  return cnt;
}

int Linker::link1 ( const string& f )
{
  int cpt , kk ;
  enum Tag r;
  ifstream in ( f.c_str() )  ;
  vector<string> vals ;
  vector<string> block ;
  string name ;
  vector<string> *segment=&_code;

  if ( !in.is_open() ) return 0 ;
  string line , modname;

  for ( cpt = 0 , kk = 0;  getline ( in, line, '\n' ) ; ++kk )
  {

    r = decodeTag ( line, modname, vals, &segment ) ;
    if ( r == 1 && isNeeded ( modname ) )
    {
      ++cpt ;
      satisfy ( name=modname ) ;
      vals.clear() ;
      block.clear() ;
      while ( getline ( in, line, '\n' ) )
      {
        r = decodeTag ( line, modname,vals ) ;
        if ( r == Reference ) request ( modname ) ;
        else if ( r == EndOfDefinition ) break ; // ;<->
        else if ( r == StartOfDefinition ) continue ; // ;<+xxxx> this is an error, ignore
        else if ( r == InitData ) ; // <=> nothing to do, values are stored into vals
        else
        {
          block.push_back ( line );
//           _out << line << endl ; // this is normal (program) line, copy it to output
        }
      }
      // end record : if initialization needed, store info
      if ( vals.size() != 0 )
      {
        unsigned int zeroesAtBegin=countZeroes ( vals,0,1 );
        unsigned int zeroesAtEnd=countZeroes ( vals,vals.size()-1,-1 );
        if ( zeroesAtBegin == vals.size() ) // initialized by only zeroes
        {
          // place it in the udata segment
          segment=&_udata;
        }
        else if ( zeroesAtBegin+zeroesAtEnd > 6 )
        {
          // initialized with many zeroes, so we can place it to udata and initialize only the needed block
          segment=&_udata;
          for ( unsigned int i=zeroesAtBegin; i<vals.size()-zeroesAtEnd; i++ )
            _inits[name+"+"+utility::toString ( zeroesAtBegin ) ].push_back ( vals[i] ) ;
        }
        else if ( segment == &_idata )
        {
          // this is initialized variable
          // fixme - if the variable (long structure) is partially initialized, threat it differently
          for ( unsigned int i=0; i<vals.size(); i++ ) _inits["_IDATA_BEGIN"].push_back ( vals[i] ) ;
        }
        else if ( segment == &_code )
        {
          // old style initialized data, thread it as code and append init data to the table
          _inits[name] = vals ;
        }
      }
      for ( unsigned int i=0; i<block.size(); i++ ) segment->push_back ( block[i] ) ;
    }
    // other tags must be ignored in this context
  }

  in.close() ;
  return cpt ;
}
/** extract tag from a line
return :
  0=normal line,
  1=start of definition,
  2=end of definition,
  3=reference
  4=initialization data
*/
enum Linker::Tag Linker::decodeTag ( const string& line, string& tag, vector<string>& data, vector<string> **segment )
{
  enum Tag r = NormalLine  ;
  tag.clear() ;
  string val ;

  if ( line[0] == ';' && line[1] == '<' )
  {
    if ( line[2] == '-' && line[3] == '>' )
      r = EndOfDefinition ;
    else if ( line[2] == '+' ) // symbol creation
    {
      unsigned int i;
      for ( i = 3 ; line[i] != '>' && line[i] != '|'  ; ++i )  tag += line[i] ;
      if ( line[i]=='|' )
      {
        string seg;
        for ( i++ ; line[i] != '>' ; ++i )  seg += line[i] ;
        if ( segment )
          *segment= ( seg=="IDATA" ) ?&_idata:
                    ( seg=="UDATA" ) ?&_udata:
                    ( seg=="CDATA" ) ?&_cdata:
                    &_code;
      }
      r = StartOfDefinition ;
    }
    else if ( line[2] == '?' ) // external reference
    {
      for ( unsigned int i = 3 ; line[i] != '>' ; ++i )  tag += line[i] ;
      r = Reference ;
    }
    else if ( line[2] == '=' ) // initialization record
    {
      istringstream ss ( line ) ;
      int cpt ;
      ss.ignore ( 3 ) ;
      for ( cpt = 0 ; ( ss >> val ) && val != ">" ; ++cpt )
      {
        data.push_back ( val ) ;
      }
      if ( cpt )  r = InitData ;
    }
  }
  return r ;
}
/** No descriptions */
const char * Linker::makeProlog()
{
  _out << ";" << endl ;
  _out << "; cpik C compiler V" ;
  _out << c18.version()  ;
  _out << " by Alain Gibaud and Josef Pavlik" << endl ;
  _out << "; (alain.gibaud@free.fr)" << endl ;
  _out << ";" << endl ;

  _out << "\tPROCESSOR  "<<   deviceName()  << endl ;
  _out << "\tinclude \"" <<  deviceName() << ".inc\"" <<endl ;

  return includeFile ( "cpik.prolog" ) ? 0 : "Can't open \"cpik.prolog\"" ;
}
/** No descriptions */
string& Linker::deviceName()
{
  return _devicename ;
}
/** No descriptions */
bool Linker::includeFile ( const string& name )
{
  string fname = Portability::cpikLibDir() + Portability::dirSeparator() + name ;
  ifstream  f ( fname.c_str() ) ;
  string line ;

  if ( !f.is_open() ) return false ;
  while ( getline ( f, line, '\n' ) )
  {
    _out << line << endl ;
  }

  return true ;
  // file closed by destructor
}
/** No descriptions */
const char * Linker::makeEpilog()
{
  _out << "\tCBLOCK" << endl ;
  _out << "\tsoft_stack_start:1" << endl ;
  _out << "\tENDC" << endl ;
  _out << "\tend" << endl ;
  return 0 ;
}
/** Do one pass link, and returns the number of satisfied symbols
 */
int Linker::doOnePassLink()
{
  int cnt ;
  unsigned int i ;
  for ( i = 0 , cnt = 0 ; i < objfiles().size() ; ++i )
    cnt += link1 ( objfiles() [i] ) ;
  return cnt ;
}
/** No descriptions */
stringList& Linker::libPaths()
{
  return _libpaths ;
}


void Linker::outSegment ( vector<string> &s )
{
  for ( unsigned int i=0; i<s.size(); i++ )
  {
    _out << s[i] << endl;
  }
}

void Linker::makeUdataClear()
{
  if ( _udata.size() == 0 )
  {
    // no initialization needed, put a dummy clearer
    _out << "C18___udata_clear" << endl << "\treturn 0" << endl ;
    return  ;
  }

  // add loader code, if not already loaded
  if ( ! isFound ( "C18___udata_clear" ) )
  {
    request ( "C18___udata_clear" ) ;
    doOnePassLink() ;
  }
}

/** output initialization table for statically initialized data
Table Format:
  <byte count of this entry> (2 bytes)
  <RAM memory addr to be initialized>  (2 bytes)
  <the bytes>  (padded with 0 if odd byte count)

Table end is marked by a zero byte count

*/
void Linker::makeInitializationTable()
{
  if ( _inits.size() == 0 )
  {
    // no initialization needed, put a dummy loader
    _out << "C18___static_loader" << endl << "\treturn 0" << endl ;
    return  ;
  }

  // add loader code, if not already loaded
  if ( ! isFound ( "C18___static_loader" ) )
  {
    request ( "C18___static_loader" ) ;
    doOnePassLink() ;
  }


  map< string , vector<string> >::iterator p ;

  _out << ";\n__static_init_table" << endl ;

  for ( p = _inits.begin() ; p != _inits.end() ; ++p )
  {
    string entname = ( *p ).first ;
    vector<string>& v = ( *p ).second ;
    _out << dec << ";\n\tdb LOW(" << v.size() << "), HIGH(" << v.size() << "-1)+1" << endl ;
    _out << "\tdw " << entname << endl ;

    if ( v.size() & 1 ) // odd ?
      v.push_back ( string ( "0x0" ) ) ;
    for ( unsigned  int k = 0 ; k < v.size() ;  ++k )
    {
      if ( k == 0 )
        _out << "\tdb ";
      else if ( k % 8 == 0 )
        _out << endl << "\tdb ";

      _out << hex <<  v[k] ;
      if ( k % 8 != 7 && k != v.size()-1 ) _out << "," ;
    }
    _out << endl ;
  }
  _out << ";\n\tdw 0x0" << dec << endl ;
}
/** True if symbol already loaded
 */
bool Linker::isFound ( const string& n )
{
  map<string,int>::iterator i = _refs.find ( n ) ;
  if ( i ==  _refs.end() ) return false ;
  return ( ( *i ).second & Found )   ;
}

/**
 bool deviceMap(const string& zone, int& min, int& max) ;

 search for information about a specific memory zone
 in the lnk device file
 returns true is information found, else false

 Since gputils 1.0.0, it is tricky to read this file
 because it must be preprocessed.
 Hopefully the non relevant lines contain symbolic values that will make the
 line decoding to fail : these lines will be ignored
*/
bool Linker::getDeviceMap ( const string& key, int& min, int& max )
{
  string name ( deviceName(),1 )  ; // remove the first letter (p)
  // uses .lkr files from gputils 1.0.0
  name = Portability::cpikLkrDir() +Portability::dirSeparator() +name+"_g.lkr" ;

  ifstream in ( name.c_str() ) ;

  if ( in.is_open() )
  {
    string line ;
    char zone[30], codepage[30] ;

    int mini, maxi ;

    for ( ; getline ( in, line ) ; )
    {
//        CODEPAGE   NAME=page       START=0x0               END=0x7FF7
      int k = sscanf ( line.c_str(),
           "%29s%*[ \t]NAME=%29s%*[ \n]START=0x%x%*[ \n]END=0x%x",
          codepage,  zone, &mini,&maxi ) ;
      if ( k == 4 && key==zone  && string(codepage) == "CODEPAGE")
      {
        // found !!
 //         cout << "memory information found in " << name << endl ;
        min = mini ; max = maxi ; return true ;
      }
    }

  }

  return false ;
}
